//*****************************************************************************
//                              PnlGnuCapOP.cpp                               *
//                             -----------------                              *
// Started     : 06/11/2003                                                   *
// Last Update : 13/04/2010                                                   *
// Copyright   : (C) 2003 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapOP.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapOP, PnlAnaBase )

  EVT_RADIOBOX( ID_RBX_SCALE,   PnlGnuCapOP::OnScale   )
  EVT_CHOICE  ( ID_CHO_SRCNAME, PnlGnuCapOP::OnSrcName )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argunebt List:
//   poWin - A pointer to the parent window

PnlGnuCapOP::PnlGnuCapOP( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrEng( eSIMR_GNUCAP );
  bSetAnaType( eCMD_OP );

  Create( );

  bClear( );
}

//*****************************************************************************
// Destructor.

PnlGnuCapOP::~PnlGnuCapOP( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapOP::Create( void )
{
  int  ix, iy;

  // Set the sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" Operating Point Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Temperature") );
  m_oPnlStop   .bSetName( wxT("Stop Temperature") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( eUNITS_TEMP );
  m_oPnlStop .bSetUnitsType( eUNITS_TEMP );
  m_oPnlStep .bSetUnitsType( eUNITS_TEMP );

  // Create the scale controls but disable the scale option GNU-Cap doesn't
  // support
  PnlAnaBase::CreateScale( );
  m_oRbxScale.Enable( eSCALE_OCT, FALSE );

  PnlAnaBase::CreateSrc( );  // Create input source controls
}

//*****************************************************************************
// Initialize the step scale.

void  PnlGnuCapOP::InitScale( void )
{
  switch( m_oRbxScale.GetSelection( ) )
  {
    case eSCALE_LIN :
      m_oPnlStep.bSetName( wxT("Step Increment") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_FLT );
      m_oPnlStep.bSetParms( 1.0, 0.0, 100.0, 1.0, 10.0 );
      m_oPnlStep.bShowUnits( TRUE );
      break;

    case eSCALE_LOG :
      m_oPnlStep.bSetName( wxT("Step Multiplier") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_FLT );
      m_oPnlStep.bSetParms( 1.01, 0.0, 100.0, 0.01, 0.1 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    case eSCALE_DEC :
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 1000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    default :
      break;
  }
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapOP::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )   bRtn = FALSE;

  // Set the step scale type and default sweep values
  if( ! bSetScale( eSCALE_LIN ) ) bRtn = FALSE;
  m_oPnlStep.bSetValue( (float) 1.0 );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - A SimnGnuCap object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapOP::bLoad( SimnGnuCap & roSimn )
{
  bool  bRtn=TRUE;

  // Load the components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VIRLC") );

  // Set the source component
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;

  // Set the step scale (do this before setting the sweep step)
  if( roSimn.m_oCmdOP.m_eScale != eSCALE_NONE )
    m_oRbxScale.SetSelection( roSimn.m_oCmdOP.m_eScale );

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdOP.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdOP.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdOP.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  if( roSimn.eGetAnaType( ) == eCMD_OP )
  {
    // Set the parameters to derive
    m_oCbxVoltage.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] );
    m_oCbxCurrent.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] );
    m_oCbxPower  .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] );
    m_oCbxResist .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] );
  }

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimn - A SimnGnuCap object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapOP::bSave( SimnGnuCap & roSimn )
{
  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdOP.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdOP.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdOP.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the step scale
  roSimn.m_oCmdOP.m_eScale = (eScaleType) m_oRbxScale.GetSelection( );

  // Modify the component to be used as the sweep source
  roSimn.m_oCpntSwpSrc.m_osValue = m_oPnlSrcLvl.rosGetValue( );

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_OP;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Create the command strings
  roSimn.m_oCmdOP.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdOP.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdOP.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapOP::OnScale( wxCommandEvent & roEvtCmd )
{
  InitScale( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapOP::OnSrcName( wxCommandEvent & roEvtCmd )
{
  wxString  os1;

  // Set the units type
  os1 = m_oChoSrcName.GetStringSelection( );
  m_oPnlSrcLvl.bSetUnitsType( Component::eGetUnitsType( os1 ) );

  // Set the source value
  if( m_oChoSrcName.GetStringSelection( ) == wxT("None") )
    m_oPnlSrcLvl.bSetValue( (double) 0.0 );
  else if( m_oPnlSrcLvl.dfGetValue( ) == 0.0 )
    m_oPnlSrcLvl.bSetValue( (double) 1.0 );

  // Permit the base class to process this event as well
  roEvtCmd.Skip( );
}

//*****************************************************************************
