//*****************************************************************************
//                              CmdGnuCapOPT.cpp                              *
//                             ------------------                             *
// Started     : 11/09/2006                                                   *
// Last Update : 16/04/2010                                                   *
// Copyright   : (C) 2006 by M.S.Waters                                       *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/commands/CmdGnuCapOPT.hpp"

//*****************************************************************************
// Constructor.

CmdGnuCapOPT::CmdGnuCapOPT( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdGnuCapOPT::~CmdGnuCapOPT( )
{
}

//*****************************************************************************
// Check that the object attributes are valid.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapOPT::bValidate( void )
{
  double  df1;
  long    li1;

  CmdBase::bValidate( );

  if( ! ConvertType::bStrToDFlt( m_osABSTOL,   &df1 ) )
    SetErrMsg( wxT("Invalid value for ABSTOL.") );
  if( ! ConvertType::bStrToDFlt( m_osCHGTOL,   &df1 ) )
    SetErrMsg( wxT("Invalid value for CHGTOL.") );
  if( ! ConvertType::bStrToDFlt( m_osDEFAD,    &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFAD.")  );
  if( ! ConvertType::bStrToDFlt( m_osDEFAS,    &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFAS.")  );
  if( ! ConvertType::bStrToDFlt( m_osDEFL,     &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFL.")   );
  if( ! ConvertType::bStrToDFlt( m_osDEFW,     &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFW.")   );
  if( ! ConvertType::bStrToDFlt( m_osGMIN,     &df1 ) )
    SetErrMsg( wxT("Invalid value for GMIN.")   );
  if( ! ConvertType::bStrToDFlt( m_osRELTOL,   &df1 ) )
    SetErrMsg( wxT("Invalid value for RELTOL.") );
  if( ! ConvertType::bStrToDFlt( m_osTNOM,     &df1 ) )
    SetErrMsg( wxT("Invalid value for TNOM.")   );
  if( ! ConvertType::bStrToDFlt( m_osTRTOL,    &df1 ) )
    SetErrMsg( wxT("Invalid value for TRTOL.")  );
  if( ! ConvertType::bStrToDFlt( m_osVNTOL,    &df1 ) )
    SetErrMsg( wxT("Invalid value for VNTOL.")  );

  if( ! ConvertType::bStrToDFlt( m_osDAMPMAX,  &df1 ) )
    SetErrMsg( wxT("Invalid value for DAMPMAX.")  );
  if( ! ConvertType::bStrToDFlt( m_osDAMPMIN,  &df1 ) )
    SetErrMsg( wxT("Invalid value for DAMPMIN.")  );
  if( ! ConvertType::bStrToDFlt( m_osDTMIN,    &df1 ) )
    SetErrMsg( wxT("Invalid value for DTMIN.")    );
  if( ! ConvertType::bStrToDFlt( m_osDTRATIO,  &df1 ) )
    SetErrMsg( wxT("Invalid value for DTRATIO.")  );
  if( ! ConvertType::bStrToDFlt( m_osFLOOR,    &df1 ) )
    SetErrMsg( wxT("Invalid value for FLOOR.")    );
  if( ! ConvertType::bStrToDFlt( m_osLIMIT,    &df1 ) )
    SetErrMsg( wxT("Invalid value for LIMIT.")    );
  if( ! ConvertType::bStrToDFlt( m_osSEED,     &df1 ) )
    SetErrMsg( wxT("Invalid value for SEED.")     );
  if( ! ConvertType::bStrToDFlt( m_osSHORT,    &df1 ) )
    SetErrMsg( wxT("Invalid value for SHORT.")    );
  if( ! ConvertType::bStrToDFlt( m_osTEMPAMB,  &df1 ) )
    SetErrMsg( wxT("Invalid value for TEMPAMB.")  );
  if( ! ConvertType::bStrToDFlt( m_osTRREJECT, &df1 ) )
    SetErrMsg( wxT("Invalid value for TRREJECT.") );
  if( ! ConvertType::bStrToDFlt( m_osTRSTEPG,  &df1 ) )
    SetErrMsg( wxT("Invalid value for TRSTEPG.")  );
  if( ! ConvertType::bStrToDFlt( m_osTRSTEPS,  &df1 ) )
    SetErrMsg( wxT("Invalid value for TRSTEPS.")  );
  if( ! ConvertType::bStrToDFlt( m_osVFLOOR,   &df1 ) )
    SetErrMsg( wxT("Invalid value for VFLOOR.")   );
  if( ! ConvertType::bStrToDFlt( m_osVMAX,     &df1 ) )
    SetErrMsg( wxT("Invalid value for VMAX.")     );
  if( ! ConvertType::bStrToDFlt( m_osVMIN,     &df1 ) )
    SetErrMsg( wxT("Invalid value for VMIN.")     );
  if( ! ConvertType::bStrToDFlt( m_osWCZERO,   &df1 ) )
    SetErrMsg( wxT("Invalid value for WCZERO.")   );

  if( ! ConvertType::bStrToLong( m_osITL1,     &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL1.") );
  if( ! ConvertType::bStrToLong( m_osITL2,     &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL2.") );
  if( ! ConvertType::bStrToLong( m_osITL4,     &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL4.") );

  if( ! ConvertType::bStrToLong( m_osDAMPST,   &li1 ) )
    SetErrMsg( wxT("Invalid value for DAMPST.")   );
  if( ! ConvertType::bStrToLong( m_osHARMS,    &li1 ) )
    SetErrMsg( wxT("Invalid value for HARMS.")    );
  if( ! ConvertType::bStrToLong( m_osITL7,     &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL7.")     );
  if( ! ConvertType::bStrToLong( m_osITL8,     &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL8.")     );
  if( ! ConvertType::bStrToLong( m_osNUMDGT,   &li1 ) )
    SetErrMsg( wxT("Invalid value for NUMDGT.")   );
  if( ! ConvertType::bStrToLong( m_osTRANSITS, &li1 ) )
    SetErrMsg( wxT("Invalid value for TRANSITS.") );

  if( m_osMETHOD.IsEmpty( ) )
    SetErrMsg( wxT("Invalid value for METHOD.") );
  if( m_osMODE  .IsEmpty( ) )
    SetErrMsg( wxT("Invalid value for MODE.") );
  if( m_osORDER .IsEmpty( ) )
    SetErrMsg( wxT("Invalid value for ORDER.") );

  return( bIsValid( ) );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapOPT::bClear( void )
{
  CmdBase::bClear( );

  m_eSimEng  = eSIMR_GNUCAP;
  m_eCmdType = eCMD_OPT;

  ConvertType::bDFltToStr( GCP_ABSTOL  , m_osABSTOL   );
  ConvertType::bDFltToStr( GCP_CHGTOL  , m_osCHGTOL   );
  ConvertType::bDFltToStr( GCP_DEFAD   , m_osDEFAD    );
  ConvertType::bDFltToStr( GCP_DEFAS   , m_osDEFAS    );
  ConvertType::bDFltToStr( GCP_DEFL    , m_osDEFL     );
  ConvertType::bDFltToStr( GCP_DEFW    , m_osDEFW     );
  ConvertType::bDFltToStr( GCP_GMIN    , m_osGMIN     );
  ConvertType::bDFltToStr( GCP_ITL1    , m_osITL1     );
  ConvertType::bDFltToStr( GCP_ITL2    , m_osITL2     );
  ConvertType::bDFltToStr( GCP_ITL4    , m_osITL4     );
  ConvertType::bDFltToStr( GCP_RELTOL  , m_osRELTOL   );
  ConvertType::bDFltToStr( GCP_TNOM    , m_osTNOM     );
  ConvertType::bDFltToStr( GCP_TRTOL   , m_osTRTOL    );
  ConvertType::bDFltToStr( GCP_VNTOL   , m_osVNTOL    );

  ConvertType::bDFltToStr( GCP_DAMPMAX , m_osDAMPMAX  );
  ConvertType::bDFltToStr( GCP_DAMPMIN , m_osDAMPMIN  );
  ConvertType::bDFltToStr( GCP_DAMPST  , m_osDAMPST   );
  ConvertType::bDFltToStr( GCP_DTMIN   , m_osDTMIN    );
  ConvertType::bDFltToStr( GCP_DTRATIO , m_osDTRATIO  );
  ConvertType::bDFltToStr( GCP_FLOOR   , m_osFLOOR    );
  ConvertType::bDFltToStr( GCP_HARMS   , m_osHARMS    );
  ConvertType::bDFltToStr( GCP_ITL7    , m_osITL7     );
  ConvertType::bDFltToStr( GCP_ITL8    , m_osITL8     );
  ConvertType::bDFltToStr( GCP_LIMIT   , m_osLIMIT    );
  ConvertType::bDFltToStr( GCP_NUMDGT  , m_osNUMDGT   );
  ConvertType::bDFltToStr( GCP_SEED    , m_osSEED     );
  ConvertType::bDFltToStr( GCP_SHORT   , m_osSHORT    );
  ConvertType::bDFltToStr( GCP_TEMPAMB , m_osTEMPAMB  );
  ConvertType::bDFltToStr( GCP_TRANSITS, m_osTRANSITS );
  ConvertType::bDFltToStr( GCP_TRREJECT, m_osTRREJECT );
  ConvertType::bDFltToStr( GCP_TRSTEPG , m_osTRSTEPG  );
  ConvertType::bDFltToStr( GCP_TRSTEPS , m_osTRSTEPS  );
  ConvertType::bDFltToStr( GCP_VFLOOR  , m_osVFLOOR   );
  ConvertType::bDFltToStr( GCP_VMAX    , m_osVMAX     );
  ConvertType::bDFltToStr( GCP_VMIN    , m_osVMIN     );
  ConvertType::bDFltToStr( GCP_WCZERO  , m_osWCZERO   );

  m_bBYPASS   = GCP_BYPASS;
  m_bCSTRAY   = GCP_CSTRAY;
  m_bINCMODE  = GCP_INCMODE;
  m_bLUBYPASS = GCP_LUBYPASS;
  m_bRSTRAY   = GCP_RSTRAY;
  m_bTRACEL   = GCP_TRACEL;
  m_bVBYPASS  = GCP_VBYPASS;

  m_osMETHOD  = GCP_METHOD;
  m_osMODE    = GCP_MODE;
  m_osORDER   = GCP_ORDER;

  return( TRUE );
}

//*****************************************************************************
// Parse the OPTIONS command string.
//
// Eg.s : .OPTIONS NOPAGE
//        .OPTIONS NOPAGE ABSTOL=1.20p NOBYP CHGTOL=12.00f NOCSTRAY
//                 DAMPMAX=900.00m DAMPMIN=600.00m DAMPST=1 DEFAD=11.00f
//                 DEFAS=11.00f DEFL=120.00u DEFW=120.00u DTMIN=1.10p
//                 DTRATIO=1.10Giga FLOOR=1.10E-21 GMIN=1.20p HARMS=10 NOINC
//                 ITL1=200 ITL2=70 ITL4=30 ITL7=2 ITL8=200 LIMIT=11.00Giga
//                 NOLUBYP METHOD=EULER MODE=ANALOG NUMDGT=6 ORDER=FORWARD
//                 RELTOL=3.00m RSTRAY SEED=2.00 SHORT=110.00n TEMPAMB=30.00
//                 TNOM=35.00 NOTRACEL TRANSITS=3 TRREJECT=600.00m TRSTEPG=3.00
//                 TRSTEPS=9.00 TRTOL=9.00 VBYPASS VFLOOR=1.10f VMAX=40.00
//                 VMIN=-20.00 VNTOL=3.00u WCZERO=1.10n
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapOPT::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1, os2;
  size_t             szt1;

  // Clear the object attributes
  os1 = (wxString &) *this;
  bClear( );
  assign( os1 );

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) < 2 )      return( bValidate( ) );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 4 ).Upper( );
  if( os1 != wxT(".OPT") )            return( bValidate( ) );

  // Extract each parameter value
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the field name and the associated value
    os1 = ostk1.GetNextToken( );
    os2 = wxT("");
    if( ( szt1=os1.find( wxT("=") ) ) != wxString::npos )
    {
      os2 = os1.Right( os1.Length( )-szt1-1 );
      os1 = os1.Left( szt1 );
    }

    // Set the object attribute values
    if(      os1 == wxT("ABSTOL")   ) m_osABSTOL   = os2;
    else if( os1 == wxT("BYPASS")   ) m_bBYPASS    = TRUE;
    else if( os1 == wxT("CHGTOL")   ) m_osCHGTOL   = os2;
    else if( os1 == wxT("CSTRAY" )  ) m_bCSTRAY    = TRUE;
    else if( os1 == wxT("DAMPMAX")  ) m_osDAMPMAX  = os2;
    else if( os1 == wxT("DAMPMIN")  ) m_osDAMPMIN  = os2;
    else if( os1 == wxT("DAMPST")   ) m_osDAMPST   = os2;
    else if( os1 == wxT("DEFAD")    ) m_osDEFAD    = os2;
    else if( os1 == wxT("DEFAS")    ) m_osDEFAS    = os2;
    else if( os1 == wxT("DEFL")     ) m_osDEFL     = os2;
    else if( os1 == wxT("DEFW")     ) m_osDEFW     = os2;
    else if( os1 == wxT("DTMIN")    ) m_osDTMIN    = os2;
    else if( os1 == wxT("DTRATIO")  ) m_osDTRATIO  = os2;
    else if( os1 == wxT("FLOOR")    ) m_osFLOOR    = os2;
    else if( os1 == wxT("GMIN")     ) m_osGMIN     = os2;
    else if( os1 == wxT("HARMS")    ) m_osHARMS    = os2;
    else if( os1 == wxT("INCMODE")  ) m_bINCMODE   = TRUE;
    else if( os1 == wxT("ITL1")     ) m_osITL1     = os2;
    else if( os1 == wxT("ITL2")     ) m_osITL2     = os2;
    else if( os1 == wxT("ITL4")     ) m_osITL4     = os2;
    else if( os1 == wxT("ITL7")     ) m_osITL7     = os2;
    else if( os1 == wxT("ITL8")     ) m_osITL8     = os2;
    else if( os1 == wxT("LIMIT")    ) m_osLIMIT    = os2;
    else if( os1 == wxT("LUBYPASS") ) m_bLUBYPASS  = TRUE;
    else if( os1 == wxT("METHOD")   ) m_osMETHOD   = os2;
    else if( os1 == wxT("MODE")     ) m_osMODE     = os2;
    else if( os1 == wxT("NOBYP")    ) m_bBYPASS    = FALSE;
    else if( os1 == wxT("NOCSTRAY") ) m_bCSTRAY    = FALSE;
    else if( os1 == wxT("NOINC")    ) m_bINCMODE   = FALSE;
    else if( os1 == wxT("NOLUBYP")  ) m_bLUBYPASS  = FALSE;
    else if( os1 == wxT("NOPAGE")   ) ;
    else if( os1 == wxT("NORSTRAY") ) m_bRSTRAY    = FALSE;
    else if( os1 == wxT("NOTRACEL") ) m_bTRACEL    = FALSE;
    else if( os1 == wxT("NOVBYP")   ) m_bVBYPASS   = FALSE;
    else if( os1 == wxT("NUMDGT")   ) m_osNUMDGT   = os2;
    else if( os1 == wxT("ORDER")    ) m_osORDER    = os2;
    else if( os1 == wxT("RELTOL")   ) m_osRELTOL   = os2;
    else if( os1 == wxT("RSTRAY")   ) m_bRSTRAY    = TRUE;
    else if( os1 == wxT("SEED")     ) m_osSEED     = os2;
    else if( os1 == wxT("SHORT")    ) m_osSHORT    = os2;
    else if( os1 == wxT("TEMPAMB")  ) m_osTEMPAMB  = os2;
    else if( os1 == wxT("TNOM")     ) m_osTNOM     = os2;
    else if( os1 == wxT("TRACEL")   ) m_bTRACEL    = TRUE;
    else if( os1 == wxT("TRANSITS") ) m_osTRANSITS = os2;
    else if( os1 == wxT("TRREJECT") ) m_osTRREJECT = os2;
    else if( os1 == wxT("TRSTEPG")  ) m_osTRSTEPG  = os2;
    else if( os1 == wxT("TRSTEPS")  ) m_osTRSTEPS  = os2;
    else if( os1 == wxT("TRTOL")    ) m_osTRTOL    = os2;
    else if( os1 == wxT("VBYPASS")  ) m_bVBYPASS   = TRUE;
    else if( os1 == wxT("VFLOOR")   ) m_osVFLOOR   = os2;
    else if( os1 == wxT("VMAX")     ) m_osVMAX     = os2;
    else if( os1 == wxT("VMIN")     ) m_osVMIN     = os2;
    else if( os1 == wxT("VNTOL")    ) m_osVNTOL    = os2;
    else if( os1 == wxT("WCZERO")   ) m_osWCZERO   = os2;
    else                              return( bValidate( ) );
  }

  return( bValidate( ) );
}

//*****************************************************************************
// Format the OPTIONS command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapOPT::bFormat( void )
{
  wxString  os1, os2;
  double    df1;
  long      li1;

  os1 = wxT(".OPTIONS NOPAGE");

  ConvertType::bStrToDFlt( m_osABSTOL, &df1 );
  if( df1 != GCP_ABSTOL )           os1 << wxT(" ABSTOL=")   << m_osABSTOL;

  os2 = m_bBYPASS ? wxT(" BYPASS") : wxT(" NOBYP");
  if( m_bBYPASS != GCP_BYPASS )     os1                      << os2;

  ConvertType::bStrToDFlt( m_osCHGTOL, &df1 );
  if( df1 != GCP_CHGTOL )           os1 << wxT(" CHGTOL=")   << m_osCHGTOL;

  os2 = m_bCSTRAY ? wxT(" CSTRAY") : wxT(" NOCSTRAY");
  if( m_bCSTRAY != GCP_CSTRAY )     os1                      << os2;

  ConvertType::bStrToDFlt( m_osDAMPMAX, &df1 );
  if( df1 != GCP_DAMPMAX )          os1 << wxT(" DAMPMAX=")  << m_osDAMPMAX;

  ConvertType::bStrToDFlt( m_osDAMPMIN, &df1 );
  if( df1 != GCP_DAMPMIN )          os1 << wxT(" DAMPMIN=")  << m_osDAMPMIN;

  ConvertType::bStrToLong( m_osDAMPST, &li1 );
  if( li1 != GCP_DAMPST )           os1 << wxT(" DAMPST=")   << m_osDAMPST;

  ConvertType::bStrToDFlt( m_osDEFAD, &df1 );
  if( df1 != GCP_DEFAD )            os1 << wxT(" DEFAD=")    << m_osDEFAD;

  ConvertType::bStrToDFlt( m_osDEFAS, &df1 );
  if( df1 != GCP_DEFAS )            os1 << wxT(" DEFAS=")    << m_osDEFAS;

  ConvertType::bStrToDFlt( m_osDEFL, &df1 );
  if( df1 != GCP_DEFL )             os1 << wxT(" DEFL=")     << m_osDEFL;

  ConvertType::bStrToDFlt( m_osDEFW, &df1 );
  if( df1 != GCP_DEFW )             os1 << wxT(" DEFW=")     << m_osDEFW;

  ConvertType::bStrToDFlt( m_osDTMIN, &df1 );
  if( df1 != GCP_DTMIN )            os1 << wxT(" DTMIN=")    << m_osDTMIN;

  ConvertType::bStrToDFlt( m_osDTRATIO, &df1 );
  if( df1 != GCP_DTRATIO )          os1 << wxT(" DTRATIO=")  << m_osDTRATIO;

  ConvertType::bStrToDFlt( m_osFLOOR, &df1 );
  if( df1 != GCP_FLOOR )            os1 << wxT(" FLOOR=")    << m_osFLOOR;

  ConvertType::bStrToDFlt( m_osGMIN, &df1 );
  if( df1 != GCP_GMIN )             os1 << wxT(" GMIN=")     << m_osGMIN;

  ConvertType::bStrToLong( m_osHARMS, &li1 );
  if( li1 != GCP_HARMS )            os1 << wxT(" HARMS=")    << m_osHARMS;

  os2 = m_bINCMODE ? wxT(" INCMODE") : wxT(" NOINC");
  if( m_bINCMODE != GCP_INCMODE )   os1                      << os2;

  ConvertType::bStrToLong( m_osITL1, &li1 );
  if( li1 != GCP_ITL1 )             os1 << wxT(" ITL1=")     << m_osITL1;

  ConvertType::bStrToLong( m_osITL2, &li1 );
  if( li1 != GCP_ITL2 )             os1 << wxT(" ITL2=")     << m_osITL2;

  ConvertType::bStrToLong( m_osITL4, &li1 );
  if( li1 != GCP_ITL4 )             os1 << wxT(" ITL4=")     << m_osITL4;

  ConvertType::bStrToLong( m_osITL7, &li1 );
  if( li1 != GCP_ITL7 )             os1 << wxT(" ITL7=")     << m_osITL7;

  ConvertType::bStrToLong( m_osITL8, &li1 );
  if( li1 != GCP_ITL8 )             os1 << wxT(" ITL8=")     << m_osITL8;

  ConvertType::bStrToDFlt( m_osLIMIT, &df1 );
  if( df1 != GCP_LIMIT )            os1 << wxT(" LIMIT=")    << m_osLIMIT;

  os2 = m_bLUBYPASS ? wxT(" LUBYP") : wxT(" NOLUBYP");
  if( m_bLUBYPASS != GCP_LUBYPASS ) os1                      << os2;

  os2 = m_osMETHOD.Upper( );
  if( os2.CmpNoCase( GCP_METHOD ) ) os1 << wxT(" METHOD=")   << os2;

  os2 = m_osMODE.Upper( );
  if( os2.CmpNoCase( GCP_MODE ) )   os1 << wxT(" MODE=")     << os2;

  ConvertType::bStrToLong( m_osNUMDGT, &li1 );
  if( li1 != GCP_NUMDGT )           os1 << wxT(" NUMDGT=")   << m_osNUMDGT;

  os2 = m_osORDER.Upper( );
  if( os2.CmpNoCase( GCP_ORDER ) )  os1 << wxT(" ORDER=")    << os2;

  ConvertType::bStrToDFlt( m_osRELTOL, &df1 );
  if( df1 != GCP_RELTOL )           os1 << wxT(" RELTOL=")   << m_osRELTOL;

  os2 = ( m_bRSTRAY ? wxT(" RSTRAY") : wxT(" NORSTRAY") );
  if( m_bRSTRAY != GCP_RSTRAY )     os1                      << os2;

  ConvertType::bStrToDFlt( m_osSEED, &df1 );
  if( df1 != GCP_SEED )             os1 << wxT(" SEED=")     << m_osSEED;

  ConvertType::bStrToDFlt( m_osSHORT, &df1 );
  if( df1 != GCP_SHORT )            os1 << wxT(" SHORT=")    << m_osSHORT;

  ConvertType::bStrToDFlt( m_osTEMPAMB, &df1 );
  if( df1 != GCP_TEMPAMB )          os1 << wxT(" TEMPAMB=")  << m_osTEMPAMB;

  ConvertType::bStrToDFlt( m_osTNOM, &df1 );
  if( df1 != GCP_TNOM )             os1 << wxT(" TNOM=")     << m_osTNOM;

  os2 = ( m_bTRACEL ? wxT(" TRACEL") : wxT(" NOTRACEL") );
  if( m_bTRACEL != GCP_TRACEL )     os1                      << os2;

  ConvertType::bStrToDFlt( m_osTRANSITS, &df1 );
  if( df1 != GCP_TRANSITS )         os1 << wxT(" TRANSITS=") << m_osTRANSITS;

  ConvertType::bStrToDFlt( m_osTRREJECT, &df1 );
  if( df1 != GCP_TRREJECT )         os1 << wxT(" TRREJECT=") << m_osTRREJECT;

  ConvertType::bStrToDFlt( m_osTRSTEPG, &df1 );
  if( df1 != GCP_TRSTEPG )          os1 << wxT(" TRSTEPG=")  << m_osTRSTEPG;

  ConvertType::bStrToDFlt( m_osTRSTEPS, &df1 );
  if( df1 != GCP_TRSTEPS )          os1 << wxT(" TRSTEPS=")  << m_osTRSTEPS;

  ConvertType::bStrToDFlt( m_osTRTOL, &df1 );
  if( df1 != GCP_TRTOL )            os1 << wxT(" TRTOL=")    << m_osTRTOL;

  os2 = ( m_bVBYPASS ? wxT(" VBYPASS") : wxT(" NOVBYP") );
  if( m_bVBYPASS != GCP_VBYPASS )   os1                      << os2;

  ConvertType::bStrToDFlt( m_osVFLOOR, &df1 );
  if( df1 != GCP_VFLOOR )           os1 << wxT(" VFLOOR=")   << m_osVFLOOR;

  ConvertType::bStrToDFlt( m_osVMAX, &df1 );
  if( df1 != GCP_VMAX )             os1 << wxT(" VMAX=")     << m_osVMAX;

  ConvertType::bStrToDFlt( m_osVMIN, &df1 );
  if( df1 != GCP_VMIN )             os1 << wxT(" VMIN=")     << m_osVMIN;

  ConvertType::bStrToDFlt( m_osVNTOL, &df1 );
  if( df1 != GCP_VNTOL )            os1 << wxT(" VNTOL=")    << m_osVNTOL;

  ConvertType::bStrToDFlt( m_osWCZERO, &df1 );
  if( df1 != GCP_WCZERO )           os1 << wxT(" WCZERO=")   << m_osWCZERO;

  assign( os1 );

  return( bValidate( ) );
}

//*****************************************************************************
// Copy the contents of a CmdNgSpiceOPT object.
//
// Argument List :
//   roCmdOPT - A reference to a CmdNgSpiceOPT object
//
// Return Values :
//   A reference to this object

CmdGnuCapOPT & CmdGnuCapOPT::operator = ( const CmdNgSpiceOPT & roCmdOPT )
{
  (CmdBase &) *this = (CmdBase &) roCmdOPT;

  m_osABSTOL  = roCmdOPT.m_osABSTOL;
  m_osCHGTOL  = roCmdOPT.m_osCHGTOL;
  m_osDEFAD   = roCmdOPT.m_osDEFAD;
  m_osDEFAS   = roCmdOPT.m_osDEFAS;
  m_osDEFL    = roCmdOPT.m_osDEFL;
  m_osDEFW    = roCmdOPT.m_osDEFW;
  m_osGMIN    = roCmdOPT.m_osGMIN;
  m_osITL1    = roCmdOPT.m_osITL1;
  m_osITL2    = roCmdOPT.m_osITL2;
  m_osITL4    = roCmdOPT.m_osITL4;
  m_osMETHOD  = roCmdOPT.m_osMETHOD;
  m_osRELTOL  = roCmdOPT.m_osRELTOL;
  m_osTNOM    = roCmdOPT.m_osTNOM;
  m_osTRTOL   = roCmdOPT.m_osTRTOL;
  m_osVNTOL   = roCmdOPT.m_osVNTOL;

  m_osTEMPAMB = roCmdOPT.m_osTEMP;

  bFormat( );

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  CmdGnuCapOPT::Print( const wxString & rosPrefix )
{
  CmdBase::Print( rosPrefix + wxT("CmdBase::") );

  cout << rosPrefix   .mb_str( ) << "m_osABSTOL   : "
       << m_osABSTOL  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osCHGTOL   : "
       << m_osCHGTOL  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDEFAD    : "
       << m_osDEFAD   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDEFAS    : "
       << m_osDEFAS   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDEFL     : "
       << m_osDEFL    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDEFW     : "
       << m_osDEFW    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osGMIN     : "
       << m_osGMIN    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osITL1     : "
       << m_osITL1    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osITL2     : "
       << m_osITL2    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osITL4     : "
       << m_osITL4    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osMETHOD   : "
       << m_osMETHOD  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osRELTOL   : "
       << m_osRELTOL  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTNOM     : "
       << m_osTNOM    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTRTOL    : "
       << m_osTRTOL   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osVNTOL    : "
       << m_osVNTOL   .mb_str( ) << '\n';

  cout << rosPrefix   .mb_str( ) << "m_bBYPASS    : "
       << ( m_bBYPASS ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_bCSTRAY    : "
       << ( m_bCSTRAY ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDAMPMAX  : "
       << m_osDAMPMAX .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDAMPMIN  : "
       << m_osDAMPMIN .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDAMPST   : "
       << m_osDAMPST  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDTMIN    : "
       << m_osDTMIN   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osDTRATIO  : "
       << m_osDTRATIO .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osFLOOR    : "
       << m_osFLOOR   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osHARMS    : "
       << m_osHARMS   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_bINCMODE   : "
       << ( m_bINCMODE ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osITL7     : "
       << m_osITL7    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osITL8     : "
       << m_osITL8    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osLIMIT    : "
       << m_osLIMIT   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_bLUBYPASS  : "
       << ( m_bLUBYPASS ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osMODE     : "
       << m_osMODE    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osNUMDGT   : "
       << m_osNUMDGT  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osORDER    : "
       << m_osORDER   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_bRSTRAY    : "
       << ( m_bRSTRAY ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osSEED     : "
       << m_osSEED    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osSHORT    : "
       << m_osSHORT   .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTEMPAMB  : "
       << m_osTEMPAMB .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_bTRACEL    : "
       << ( m_bTRACEL ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTRANSITS : "
       << m_osTRANSITS.mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTRREJECT : "
       << m_osTRREJECT.mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTRSTEPG  : "
       << m_osTRSTEPG .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osTRSTEPS  : "
       << m_osTRSTEPS .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_bVBYPASS   : "
       << ( m_bVBYPASS ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osVFLOOR   : "
       << m_osVFLOOR  .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osVMAX     : "
       << m_osVMAX    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osVMIN     : "
       << m_osVMIN    .mb_str( ) << '\n';
  cout << rosPrefix   .mb_str( ) << "m_osWCZERO   : "
       << m_osWCZERO  .mb_str( ) << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_CMDGCPOPT

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdGnuCapOPT Test Utility"
       << "\n     Version 1.02 (04/06/2009)\n";

  // Create a NG-SPICE OPTIONS command object
  CmdGnuCapOPT  oCmd_OPT;

  // Use the following command example to check the formatter and the parser :
  osCmd << wxT(".OPTIONS NOPAGE ABSTOL=1.20p NOBYP CHGTOL=12.00f NOCSTRAY ")
        << wxT("DAMPMAX=900.00m DAMPMIN=600.00m DAMPST=1 DEFAD=11.00f ")
        << wxT("DEFAS=11.00f DEFL=120.00u DEFW=120.00u DTMIN=1.10p ")
        << wxT("DTRATIO=1.10Giga FLOOR=1.10E-21 GMIN=1.20p HARMS=10 NOINC ")
        << wxT("ITL1=200 ITL2=70 ITL4=30 ITL7=2 ITL8=200 LIMIT=11.00Giga ")
        << wxT("NOLUBYP METHOD=EULER MODE=ANALOG NUMDGT=6 ORDER=FORWARD ")
        << wxT("RELTOL=3.00m RSTRAY SEED=2.00 SHORT=110.00n TEMPAMB=30.00 ")
        << wxT("TNOM=35.00 NOTRACEL TRANSITS=3 TRREJECT=600.00m TRSTEPG=3.00 ")
        << wxT("TRSTEPS=9.00 TRTOL=9.00 VBYPASS VFLOOR=1.10f VMAX=40.00 ")
        << wxT("VMIN=-20.00 VNTOL=3.00u WCZERO=1.10n");

  // Set things up for a formatter test
  oCmd_OPT.bClear( );
  oCmd_OPT.m_osABSTOL   = wxT("1.20p");
  oCmd_OPT.m_bBYPASS    = FALSE;
  oCmd_OPT.m_osCHGTOL   = wxT("12.00f");
  oCmd_OPT.m_bCSTRAY    = FALSE;
  oCmd_OPT.m_osDAMPMAX  = wxT("900.00m");
  oCmd_OPT.m_osDAMPMIN  = wxT("600.00m");
  oCmd_OPT.m_osDAMPST   = wxT("1");
  oCmd_OPT.m_osDEFAD    = wxT("11.00f");
  oCmd_OPT.m_osDEFAS    = wxT("11.00f");
  oCmd_OPT.m_osDEFL     = wxT("120.00u");
  oCmd_OPT.m_osDEFW     = wxT("120.00u");
  oCmd_OPT.m_osDTMIN    = wxT("1.10p");
  oCmd_OPT.m_osDTRATIO  = wxT("1.10Giga");
  oCmd_OPT.m_osFLOOR    = wxT("1.10E-21");
  oCmd_OPT.m_osGMIN     = wxT("1.20p");
  oCmd_OPT.m_osHARMS    = wxT("10");
  oCmd_OPT.m_bINCMODE   = FALSE;
  oCmd_OPT.m_osITL1     = wxT("200");
  oCmd_OPT.m_osITL2     = wxT("70");
  oCmd_OPT.m_osITL4     = wxT("30");
  oCmd_OPT.m_osITL7     = wxT("2");
  oCmd_OPT.m_osITL8     = wxT("200");
  oCmd_OPT.m_osLIMIT    = wxT("11.00Giga");
  oCmd_OPT.m_bLUBYPASS  = FALSE;
  oCmd_OPT.m_osMETHOD   = wxT("EULER");
  oCmd_OPT.m_osMODE     = wxT("ANALOG");
  oCmd_OPT.m_osNUMDGT   = wxT("6");
  oCmd_OPT.m_osORDER    = wxT("FORWARD");
  oCmd_OPT.m_osRELTOL   = wxT("3.00m");
  oCmd_OPT.m_bRSTRAY    = TRUE;
  oCmd_OPT.m_osSEED     = wxT("2.00");
  oCmd_OPT.m_osSHORT    = wxT("110.00n");
  oCmd_OPT.m_osTEMPAMB  = wxT("30.00");
  oCmd_OPT.m_osTNOM     = wxT("35.00");
  oCmd_OPT.m_bTRACEL    = FALSE;
  oCmd_OPT.m_osTRANSITS = wxT("3");
  oCmd_OPT.m_osTRREJECT = wxT("600.00m");
  oCmd_OPT.m_osTRSTEPG  = wxT("3.00");
  oCmd_OPT.m_osTRSTEPS  = wxT("9.00");
  oCmd_OPT.m_osTRTOL    = wxT("9.00");
  oCmd_OPT.m_bVBYPASS   = TRUE;
  oCmd_OPT.m_osVFLOOR   = wxT("1.10f");
  oCmd_OPT.m_osVMAX     = wxT("40.00");
  oCmd_OPT.m_osVMIN     = wxT("-20.00");
  oCmd_OPT.m_osVNTOL    = wxT("3.00u");
  oCmd_OPT.m_osWCZERO   = wxT("1.10n");
  cout << "\nRun Formatter     : " << ( oCmd_OPT.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_OPT.bClear( );
  oCmd_OPT.bSetString( osCmd );
  cout << "\nRun Parser        : " << ( oCmd_OPT.bParse( ) ? "OK" : "FAULT" );
  oCmd_OPT.bFormat( );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_CMDGCPOPT

//*****************************************************************************
