//*****************************************************************************
//                               PnlAnaBase.cpp                               *
//                              ----------------                              *
// Started     : 26/04/2004                                                   *
// Last Update : 12/04/2010                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "base/PnlAnaBase.hpp"

//*****************************************************************************
// Allocate storage for static data members.

bool       PnlAnaBase::m_bSyncSwpSrcs = FALSE;
Component  PnlAnaBase::m_tCpntSwpSrcs;

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlAnaBase, wxPanel )

  EVT_CHOICE( ID_CHO_SRCNAME, PnlAnaBase::OnSrcName )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List :
//   poWin - A pointer to the parent window

PnlAnaBase::PnlAnaBase( wxWindow * poWin ) : wxPanel( poWin )
{
  m_eSimEng  = eSIMR_NONE;
  m_eAnaType = eCMD_NONE;

  CreateBase( );  // Create the analysis panel
  bClear( );      // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlAnaBase::~PnlAnaBase( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bClear( void )
{
  bool  bRtn=TRUE;

  // Set sweep default values
  if( ! m_oPnlStart.bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bClear( ) ) bRtn = FALSE;

  // Set input source default values
  if( bIsCreated( eCTLGRP_SIGSRC ) && m_oSbxSrc.IsShown( ) )
  {
    m_oChoSrcName.Clear( );
    m_oChoSrcName.Append( wxT("None") );
    m_oChoSrcName.SetSelection( 0 );
    if( m_oPnlSrcLvl.GetParent( ) != NULL )
    {
      m_oPnlSrcLvl.bSetValue( (float) 0.0 );
      m_oPnlSrcLvl.bSetUnitsType( eUNITS_NONE );
    }
  }

  // Set parameters check box default values
  m_oCbxVoltage.SetValue( TRUE );
  m_oCbxCurrent.SetValue( FALSE );
  m_oCbxPower  .SetValue( FALSE );
  m_oCbxResist .SetValue( FALSE );

  // Set the analysis temperature default value
  if( bIsCreated( eCTLGRP_TEMP ) && m_oSbxTemp.IsShown( ) )
    if( ! m_oPnlTemp.bClear( ) )
      bRtn = FALSE;

  m_osErrMsg.Empty( ); // Clear the error string

  return( bRtn );
}

//*****************************************************************************
// Create the basic display objects.

void  PnlAnaBase::CreateBase( void )
{
  if( bIsCreated( eCTLGRP_BASE ) ) return;

  // Create and add sweep parameter labels
  m_oSbxSwpPars.Create( this, ID_UNUSED, wxT(" Sweep Parameters "),
                        wxPoint( 6, 5 ), wxSize( 330, 120 ) );
  m_oPnlStart.bCreate( this, ID_PNL_START, 120, wxPoint( 18, 28 ) );
  m_oPnlStop .bCreate( this, ID_PNL_STOP,  120, wxPoint( 18, 58 ) );
  m_oPnlStep .bCreate( this, ID_PNL_STEP,  120, wxPoint( 18, 88 ) );
  m_oPnlStart.bSetName( wxT("Start Value") );
  m_oPnlStop .bSetName( wxT("Stop Value") );
  m_oPnlStep .bSetName( wxT("Step Increment") );
  m_oSbxSwpPars.SetOwnFont( FONT_SLANT );

  // Create and add simulation parameter check boxes
  m_oSbxCalcPars.Create( this, ID_UNUSED, wxT(" Parameters "),
                         wxPoint( 343, 5 ), wxSize( 105, 122 ) );
  m_oCbxVoltage .Create( this, ID_CBX_VOLT, wxT("Voltage"),
                         wxPoint( 349, 25 ) );
  m_oCbxCurrent .Create( this, ID_CBX_CURR, wxT("Current"),
                         wxPoint( 349, 50 ) );
  m_oCbxPower   .Create( this, ID_CBX_PWR,  wxT("Power"),
                         wxPoint( 349, 75 ) );
  m_oCbxResist  .Create( this, ID_CBX_RES,  wxT("Resistance"),
                         wxPoint( 349, 100 ) );
  m_oSbxCalcPars.SetOwnFont( FONT_SLANT );

  // Create and add .OPTIONS configuration dialog button
  m_oBtnOPTIONS.Create( this, ID_BTN_OPTIONS, wxT(".OPTIONS ..."),
                        wxPoint( 343, 137 ), wxSize( 105, 29 ) );
}

//*****************************************************************************
// Create the scale display objects.

void  PnlAnaBase::CreateScale( void )
{
  if( bIsCreated( eCTLGRP_SCALE ) ) return;

  // Create and add the scale radio buttons
  wxString  osScale[ 4 ] = { wxT("Lin  "), wxT("Log "), wxT("Dec "), wxT("Oct ") };
  m_oRbxScale.Create( this, ID_RBX_SCALE, wxT(" Step Scale "),
                      wxPoint( 13, 123 ), wxDefaultSize, 4, osScale, 4 );
  m_oRbxScale.SetSelection( eSCALE_LIN );
}

//*****************************************************************************
// Create the initial conditions display objects.

void  PnlAnaBase::CreateInitC( void )
{
  if( bIsCreated( eCTLGRP_INITC ) ) return;

  // Create and add the initial conditions radio buttons
  wxString  osInitC[ 3 ] = { wxT("Warm "), wxT("Use ICs"), wxT("Cold ") };
  m_oRbxInitC.Create( this, ID_UNUSED, wxT(" Initial Conditions "),
                      wxPoint( 13, 123 ), wxDefaultSize, 3, osInitC, 3 );
  m_oRbxInitC.SetSelection( eINITC_WARM );
}

//*****************************************************************************
// Create the source component display objects.

void  PnlAnaBase::CreateSrc( void )
{
  if( bIsCreated( eCTLGRP_SIGSRC ) ) return;

  // Create and add input source
  m_oSbxSrc    .Create( this, ID_UNUSED, wxT(" Signal Source "),
                        wxPoint(  6, 175 ), wxSize( 330, 55 ) );
  m_oChoSrcName.Create( this, ID_CHO_SRCNAME,
                        wxPoint( 18, 195 ), wxSize( 121, -1 ) );
  m_oPnlSrcLvl.bCreate( this, ID_PNL_SRCLVL, -1, wxPoint( 143, 193 ) );

  m_oSbxSrc.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Create the simulation parameter complex part display objects.

void  PnlAnaBase::CreateCpxPrt( void )
{
  if( bIsCreated( eCTLGRP_CPXPRT ) ) return;

  // Create and add simulation parameter complex part check boxes
  m_oSbxCpxPrt.Create( this, ID_UNUSED,    wxT(" Complex Parts "),
                       wxPoint( 455,  5 ), wxSize( 110, 146 ) );
  m_oCbxMag   .Create( this, ID_CBX_MAG,   wxT("Magnitude"),
                       wxPoint( 461, 24 ) );
  m_oCbxPhase .Create( this, ID_CBX_PHASE, wxT("Phase"),
                       wxPoint( 461, 49 ) );
  m_oCbxReal  .Create( this, ID_CBX_REAL,  wxT("Real Part"),
                       wxPoint( 461, 74 ) );
  m_oCbxImag  .Create( this, ID_CBX_IMAG,  wxT("Imag. Part"),
                       wxPoint( 461, 99 ) );
  m_oCbxMagDb .Create( this, ID_CBX_MAGDB, wxT("Mag. in dB"),
                       wxPoint( 461,124 ) );

  m_oSbxCpxPrt.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Create the analysis temperature display objects.

void  PnlAnaBase::CreateTemp( void )
{
  if( bIsCreated( eCTLGRP_TEMP ) ) return;

  // Create and add analysis temperature
  m_oSbxTemp. Create( this, ID_UNUSED, wxT(" Temperature "),
                      wxPoint( 343, 175 ), wxSize( 205, 55 ) );
  m_oPnlTemp.bCreate( this, ID_PNL_TEMP, -1,
                      wxPoint( 355, 193 ) );
  m_oPnlTemp.bSetUnitsType( eUNITS_TEMP );

  m_oSbxTemp.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Test if a certain group of controls has been created.
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlAnaBase::bIsCreated( enum eTypeCtlGrp eCtlGrp )
{
  wxWindow * poCtlGrp;

  if( eCtlGrp<eCTLGRP_FST || eCtlGrp>eCTLGRP_LST ) return( FALSE );

  switch( eCtlGrp )
  {
    case eCTLGRP_BASE   : poCtlGrp = &m_oSbxSwpPars; break;
    case eCTLGRP_SCALE  : poCtlGrp = &m_oRbxScale;   break;
    case eCTLGRP_INITC  : poCtlGrp = &m_oRbxInitC;   break;
    case eCTLGRP_SIGSRC : poCtlGrp = &m_oChoSrcName; break;
    case eCTLGRP_CPXPRT : poCtlGrp = &m_oSbxCpxPrt;  break;
    case eCTLGRP_TEMP   : poCtlGrp = &m_oPnlTemp;    break;
    default             :                          return( FALSE );
  }

  return( poCtlGrp->GetParent( ) != NULL );
}

//*****************************************************************************
// Set the state of the step scale radio box.
//
// Argument List :
//   eScale - The enumerated scale specifier
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlAnaBase::bSetScale( eScaleType eScale )
{
  if( eScale<eSCALE_FST || eScale>eSCALE_LST )      return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxScale.GetCount( ) < (uint) eScale + 1 ) return( FALSE );
#else
  if( m_oRbxScale.GetCount( ) < (int)  eScale + 1 ) return( FALSE );
#endif

  m_oRbxScale.SetSelection( (int) eScale );

  InitScale( );

  return( TRUE );
}

//*****************************************************************************
// Set the state of the initial conditions radio box.
//
// Argument List :
//   eInitC - The enumerated initial conditions specifier
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlAnaBase::bSetInitC( eInitCType eInitC )
{
  if( eInitC<eINITC_FST || eInitC>eINITC_LST )      return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxInitC.GetCount( ) < (uint) eInitC + 1 ) return( FALSE );
#else
  if( m_oRbxInitC.GetCount( ) < (int)  eInitC + 1 ) return( FALSE );
#endif

  m_oRbxInitC.SetSelection( (int) eInitC );

  return( TRUE );
}

//*****************************************************************************
// Load the source choice box with the components names which may be chosen as
// a source.
//
// Argument List :
//   roSimn   - A reference to an array on Component objects
//   osPrefix - A string containing the first letter of the desired components

void  PnlAnaBase::LoadSrcNames( ArrayComponent & roaCpnts, wxString osPrefixes )
{
  wxString  os1;
  wxChar    oc1;
  uint      ui1;

  // Clear the signal source choice box
  m_oChoSrcName.Clear( );
  m_oChoSrcName.Append( wxT("None") );

  // Load the selected components into the signal source choice box
  for( ui1=0; ui1<roaCpnts.GetCount( ); ui1++ )
  {
    os1 = roaCpnts.Item( ui1 ).m_osName;
    if( os1.IsEmpty( ) )                        continue;

    oc1 = os1.GetChar( 0 );
    if( osPrefixes.Find( oc1 ) == wxNOT_FOUND ) continue;

    m_oChoSrcName.Append( os1 );
  }

  // Select the default component
  m_oChoSrcName.SetStringSelection( wxT("None") );
}

//*****************************************************************************
// Set the source component.
//
// Argument List :
//   roCpnt - A component object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSrcCpnt( Component & roCpnt )
{
  // Can't do anything if the signal source controls haven't been created
  if( ! bIsCreated( eCTLGRP_SIGSRC ) )                         return( FALSE );

  if( roCpnt.bIsValid( ) )
  {
    // Set the source component label, value and units
    if( ! m_oChoSrcName.SetStringSelection( roCpnt.m_osName )  ) return( FALSE );
    if( ! m_oPnlSrcLvl.bSetUnitsType( roCpnt.eGetUnitsType() ) ) return( FALSE );
    if( ! m_oPnlSrcLvl.bSetValue( roCpnt.rosGetNumValue( ) )   ) return( FALSE );
  }
  else
  {
    // If the components isn't valid set default values
    m_oChoSrcName.SetStringSelection( wxT("None") );
    m_oPnlSrcLvl.bSetValue( (float) 0.0 );
    m_oPnlSrcLvl.bSetUnitsType( eUNITS_NONE );
  }

  return( roCpnt.m_osName == wxT("None") );
}

//*****************************************************************************
// Select the simulator engine to be used.
//
// Argument list :
//   eSimEng - The simulator engine type
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSimrEng( eSimrType eSimEng )
{
  if( eSimEng<eSIMR_FST || eSimEng>eSIMR_LST ) return( FALSE );

  m_eSimEng = eSimEng;

  return( TRUE );
}

//*****************************************************************************
// Select the analysis to be performed.
//
// Argument List :
//   eAnalysis - The analysis type
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetAnaType( eCmdType eAnaType )
{
  if( eAnaType<eCMD_ANA_FST || eAnaType>eCMD_ANA_LST ) return( FALSE );

  m_eAnaType = eAnaType;

  return( TRUE );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Source component choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlAnaBase::OnSrcName( wxCommandEvent & roEvtCmd )
{
  // Set the sweep source component
  if( m_oChoSrcName.GetSelection( ) > 0 )
  {
    // Get the sweep source name
    m_tCpntSwpSrcs.bSetName( m_oChoSrcName.GetStringSelection( ) );

    // Get the sweep source value
    if( m_oPnlSrcLvl.GetParent( ) != NULL )
         m_tCpntSwpSrcs.bSetValue( m_oPnlSrcLvl.rosGetValue( ) );
    else m_tCpntSwpSrcs.bSetValue( wxT("1.0") );
  }
  else m_tCpntSwpSrcs.bClear( );
}

//*****************************************************************************
