//*****************************************************************************
//                               PnlGnuCapAC.cpp                              *
//                              -----------------                             *
//  Started     : 18/08/2003                                                  *
//  Last Update : 29/02/2008                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapAC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapAC, PnlAnaBase )

  EVT_RADIOBOX( ID_RBX_STEPSCALE,  PnlGnuCapAC::OnStepScale  )
  EVT_CHOICE  ( ID_CHO_SIGSRCCPNT, PnlGnuCapAC::OnSigSrcCpnt )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapAC::PnlGnuCapAC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrType( eSIMR_GNUCAP );
  bSetAnaType ( eANA_AC );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapAC::~PnlGnuCapAC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapAC::Create( void )
{
  int  ix, iy;

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" AC Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Frequency") );
  m_oPnlStop   .bSetName( wxT("Stop Frequency") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlStop .bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlStep .bSetUnitsType( ChoUnits::eUNITS_FREQ );

  // Create and add the scale radio buttons
  wxString  osScale[ 4 ] = { wxT("Lin  "), wxT("Log "), wxT("Dec "), wxT("Oct ") };
  m_oRbxStepScale.Create( this, ID_RBX_STEPSCALE, wxT(" Step Scale "),
                          wxPoint( 13, 123 ), wxDefaultSize, 4, osScale, 4 );
  bSetStepScale( eSCALE_LIN );

  CreateCpxPrt( );  // Create the simulation parameter complex part check boxes
  CreateTemp( );    // Create the analysis temperature controls
  CreateSigSrc( );  // Create the input source controls
}

//*****************************************************************************
// Set the state of the step scale radio box.
//
// Argument List:
//   eScale - The enumerated scale specifier
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  PnlGnuCapAC::bSetStepScale( eScaleType eScale )
{
  wxCommandEvent  oEvtCmd;

  if( eScale<eSCALE_FST || eScale>eSCALE_LST )          return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxStepScale.GetCount( ) < (uint) eScale + 1 ) return( FALSE );
#else
  if( m_oRbxStepScale.GetCount( ) < (int)  eScale + 1 ) return( FALSE );
#endif

  m_oRbxStepScale.SetSelection( (int) eScale );

  OnStepScale( oEvtCmd );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapAC::bClear( void )
{
  bool  bRtnValue=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )       bRtnValue = FALSE;

  // Set default step scale type and sweep values
  m_oPnlStart.bSetValue( (float)   1.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)  10.0 );
  m_oPnlStep .bShowUnits( TRUE );
  m_oPnlStart.bSetUnits( wxT("kHz") );
  m_oPnlStop .bSetUnits( wxT("kHz") );
  m_oPnlStep .bSetUnits( wxT("kHz") );

  // Set default scale value
  if( ! bSetStepScale( eSCALE_DEC ) ) bRtnValue = FALSE;

  // Set the complex part check box default values
  m_oCbxMag  .SetValue( TRUE );
  m_oCbxPhase.SetValue( FALSE );
  m_oCbxReal .SetValue( FALSE );
  m_oCbxImag .SetValue( FALSE );
  m_oCbxMagDb.SetValue( TRUE );

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapAC::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Load the components into the choice box
  LoadSigSrcCpnts( roSim, wxT("VI") );

  // Don't go any further if the analysis type doesn't match
  if( roSim.eGetAnaType( ) != eGetAnaType( ) ) return( TRUE );

  // Set the source component
  if( ! PnlAnaBase::bSetSigSrc( roSim ) )      bRtnValue = FALSE;

  // Set the sweep scale
  m_oRbxStepScale.SetSelection( roSim.iGetSwpScale( ) );

  // Set the complex parts to derive
  m_oCbxMag  .SetValue( roSim.bGetOutCpx( eCPX_MAG   ) );
  m_oCbxPhase.SetValue( roSim.bGetOutCpx( eCPX_PHASE ) );
  m_oCbxReal .SetValue( roSim.bGetOutCpx( eCPX_REAL  ) );
  m_oCbxImag .SetValue( roSim.bGetOutCpx( eCPX_IMAG  ) );
  m_oCbxMagDb.SetValue( roSim.bGetOutCpx( eCPX_MAGDB ) );
  if( !m_oCbxPhase.GetValue() && !m_oCbxReal.GetValue() && !m_oCbxImag.GetValue() )
  {
    m_oCbxMag  .SetValue( TRUE );
    m_oCbxMagDb.SetValue( TRUE );
  }

  // Perform any base class load tasks
  if( ! PnlAnaBase::bLoad( roSim ) )           bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapAC::bSave( Simulation & roSim )
{
  wxString  os1;
  size_t    sz1;
  bool      b1;

  // Execute base class save tasks
  PnlAnaBase::bSave( roSim );

  // Set the component to be used as the sweep source
  if( m_oPnlSigSrcLvl.dfGetValue( ) != 0.0 )
  {
    os1 = roSim.rosGetSigSrc( );
    sz1 = os1.Find( wxT(' '), TRUE );
    os1.insert( sz1, wxT(" GENERATOR(1) AC") );
    if( ! roSim.bSetSigSrc( os1 ) )
      SetErrMsg( wxT("Signal source component couldn't be set.") );
  }
  else SetErrMsg( wxT("Signal source component value of zero is not permitted.") );

  // Store the complex parts of the parameters to derive
  roSim.bSetOutCpx( eCPX_MAG,   m_oCbxMag  .GetValue( ) );
  roSim.bSetOutCpx( eCPX_PHASE, m_oCbxPhase.GetValue( ) );
  roSim.bSetOutCpx( eCPX_REAL,  m_oCbxReal .GetValue( ) );
  roSim.bSetOutCpx( eCPX_IMAG,  m_oCbxImag .GetValue( ) );
  roSim.bSetOutCpx( eCPX_MAGDB, m_oCbxMagDb.GetValue( ) );
  for( sz1=eCPX_MAG, b1=FALSE; sz1<=eCPX_IMAG; sz1++ )
    if( roSim.bGetOutCpx( (eCpxType) sz1 ) ) b1 = TRUE;
  if( ! b1 ) SetErrMsg( wxT("No complex parts have been selected.") );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapAC::OnStepScale( wxCommandEvent & roEvtCmd )
{
  switch( m_oRbxStepScale.GetSelection( ) )
  {
    case eSCALE_LIN:
      m_oPnlStep.bSetName( wxT("Step Increment") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_FLT );
      m_oPnlStep.bSetUnitsType( ChoUnits::eUNITS_FREQ );
      m_oPnlStep.bShowUnits( TRUE );
      break;
    case eSCALE_LOG:
      m_oPnlStep.bSetName( wxT("Step Multiplier") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_FLT );
      m_oPnlStep.bSetParms( 1.1, 0.01, 10.0, 1.0, 10.0 );
      m_oPnlStep.bShowUnits( FALSE );
      break;
    case eSCALE_DEC:
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;
    case eSCALE_OCT:
      m_oPnlStep.bSetName( wxT("Steps / Octave") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;
    default:
      break;
  }
}

//*****************************************************************************
// Signal source component choice box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapAC::OnSigSrcCpnt( wxCommandEvent & roEvtCmd )
{
  bSetSigSrcUnits( );

  if( m_oChoSigSrcCpnt.GetStringSelection( ) == wxT("None") )
    m_oPnlSigSrcLvl.bSetValue( (double) 0.0 );
  else if( m_oPnlSigSrcLvl.dfGetValue( ) == 0.0 )
    m_oPnlSigSrcLvl.bSetValue( (double) 1.0 );
}

//*****************************************************************************
