//*****************************************************************************
//                               CmdGnuCapAC.cpp                              *
//                              -----------------                             *
//  Started     : 21/02/2008                                                  *
//  Last Update : 22/02/2008                                                  *
//  Copyright   : (C) 2008 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/commands/CmdGnuCapAC.hpp"

//*****************************************************************************
// Constructor.

CmdGnuCapAC::CmdGnuCapAC( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdGnuCapAC::~CmdGnuCapAC( )
{
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .AC 1.00K 100.00K BY 10.00K TE 27.00 BASIC
//        .AC 1.00K 100.00K TI 10.00 TE 27.00 BASIC
//        .AC 1.00K 100.00K DE 10 TE 27.00 BASIC
//        .AC 1.00K 100.00K OC 10 TE 27.00 BASIC
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapAC::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  double             df1, df2;
  long               li1;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  *this = os1;

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) != 8 ) return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 3 ).Upper( );
  if( os1 != wxT(".AC") )         return( FALSE );
  m_osName = wxT("AC");

  m_bIsOk = TRUE;

  // Extract the start value
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStart = (float) df1;
  else                            m_bIsOk = FALSE;

  // Is the next field a number?
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df2 )  )
  {
    // Extract the stop value
    if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStop = (float) df1;
    else                          m_bIsOk = FALSE;

    // Extract the sweep type: linear, log or steps per decade
    os1 = ostk1.GetNextToken( ).Upper( );
    if(      os1 == wxT("BY") ) m_eScale = eSCALE_LIN;
    else if( os1 == wxT("TI") ) m_eScale = eSCALE_LOG;
    else if( os1 == wxT("DE") ) m_eScale = eSCALE_DEC;
    else if( os1 == wxT("OC") ) m_eScale = eSCALE_OCT;
    else                          m_bIsOk = FALSE;

    // Extract the step size/count
    os1 = ostk1.GetNextToken( );
    if( m_eScale < eSCALE_DEC )
    {
      if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStep = (float) df1;
      else                        m_bIsOk = FALSE;
    }
    else
    {
      if( ConvertType::bStrToInt( os1, &li1 ) ) m_fStep = (float) li1;
      else                        m_bIsOk = FALSE;
    }

    os1 = ostk1.GetNextToken( );
  }
  else
  {
    m_fStop = (float) df1;
    m_fStep = 1.0;
    m_eScale = eSCALE_LIN;
  }

  // Extract the analysis temperature
  if( os1.Upper( ) == wxT("TE") )
  {
    os1 = ostk1.GetNextToken( );
    if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fTempC = (float) df1;
    else                          m_bIsOk = FALSE;
  }

  return( m_bIsOk );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapAC::bFormat( void )
{
  wxString  osCmd, os1;

  m_bIsOk = FALSE;

  // Set the command name
  osCmd = wxT(".AC");
  if( m_osName.IsEmpty( ) ) m_osName = wxT("AC");

  // Set sweep parameters
  if( ! ConvertType::bFltToStrEng( (double) m_fStart, os1 ) )  return( FALSE );
  osCmd << wxT(' ') << os1;
  if( ! ConvertType::bFltToStrEng( (double) m_fStop,  os1 ) )  return( FALSE );
  osCmd << wxT(' ') << os1;
  switch( m_eScale )
  {
    case eSCALE_LIN : osCmd << wxT(" BY"); break;
    case eSCALE_LOG : osCmd << wxT(" TI"); break;
    case eSCALE_DEC : osCmd << wxT(" DE"); break;
    case eSCALE_OCT : osCmd << wxT(" OC"); break;
    default         :                                          return( FALSE );
  }
  if( m_eScale < eSCALE_DEC )
  {
    if( ! ConvertType::bFltToStrEng( (double) m_fStep, os1 ) ) return( FALSE );
    osCmd << wxT(' ') << os1;
  }
  else osCmd << wxT(' ') << (int) lroundf( m_fStep );

  // Set analysis temperature
  if( ! ConvertType::bFltToStrEng( (double) m_fTempC, os1 ) )  return( FALSE );
  osCmd << wxT(" TE ") << os1;

  // Append format modifier
  osCmd << wxT(" BASIC");

  *this = osCmd;

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapAC::bClear( void )
{
  CmdBase::bClear( );

  m_fStart = -FLT_MAX;
  m_fStop  = -FLT_MAX;
  m_fStep  = -FLT_MIN;

  m_eScale = eSCALE_NONE;

  m_fTempC = -FLT_MAX;

  return( TRUE );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   psPrefix - A prefix to every line displayed (usually just spaces)

void  CmdGnuCapAC::Print( const char * psPrefix )
{
  CmdBase::Print( psPrefix );

  cout << psPrefix << "m_fStart : " << m_fStart << '\n';
  cout << psPrefix << "m_fStop  : " << m_fStop << '\n';
  cout << psPrefix << "m_fStep  : " << m_fStep << '\n';

  cout << psPrefix << "m_eScale : ";
  switch( m_eScale )
  {
    case eSCALE_LIN  : cout << "eSCALE_LIN\n";  break;
    case eSCALE_LOG  : cout << "eSCALE_LOG\n";  break;
    case eSCALE_DEC  : cout << "eSCALE_DEC\n";  break;
    case eSCALE_OCT  : cout << "eSCALE_OCT\n";  break;
    case eSCALE_NONE : cout << "eSCALE_NONE\n"; break;
  }

  cout << psPrefix << "m_fTempC : " << m_fTempC << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdGnuCapAC Test Utility"
       << "\n    Version 0.80 (22/02/2008)\n";

  // Create a GNU-CAP AC command object
  CmdGnuCapAC  tCmd_AC;

  // Use the following command example to check the formatter and the parser :
  osCmd = wxT(".AC 1.00K 200.00K TI 1.01 TE 30.00 BASIC");

  // Set things up for a formatter test
  tCmd_AC.bClear( );
  tCmd_AC.m_fStart = 1.0E3;
  tCmd_AC.m_fStop  = 200.0E3;
  tCmd_AC.m_fStep  = 1.01;
  tCmd_AC.m_eScale = eSCALE_LOG;
  tCmd_AC.m_fTempC = 30.0;
  cout << "\nRun Formatter    : " << ( tCmd_AC.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format  : " << ( tCmd_AC == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\ntCmd_AC Contents : " << tCmd_AC.mb_str( ) << '\n';

  // Set things up for a parser test
  tCmd_AC.bClear( );
  tCmd_AC = osCmd;
  cout << "\nRun Parser       : " << ( tCmd_AC.bParse( ) ? "OK" : "FAULT" );
  tCmd_AC.bFormat( );
  cout << "\nTest Cmd Format  : " << ( tCmd_AC == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\ntCmd_AC Contents : " << tCmd_AC.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
