#include "kifuViewer.h"
#include "moveList.h"

#include "moveGeneratorDialog.h"
#include "kifuAnalyzer.h"
#include "dualEvaluationDialog.h"

#include "gpsshogi/gui/board.h"
#include "gpsshogi/gui/util.h"

#include "osl/search/searchMonitor.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/record/csaIOError.h"
#include "osl/record/kisen.h"
#include "osl/record/ki2.h"

#include <qlayout.h>
#include <QListWidget>
#include <QTableWidget>
#include <QThread>
#include <Q3ListView>
#include <QDateTime>
#include <qtextcodec.h>
#include <qtimer.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qpushbutton.h>

#include <iostream>

volatile bool KifuViewer::analyzeOnlineEnabled = false;
volatile bool KifuViewer::stateForAnalysisChanged = false;
boost::scoped_ptr<osl::NumEffectState> KifuViewer::stateForAnalysis;
boost::scoped_ptr<KifuViewer::AnalysisOnlineThread> KifuViewer::thread;
KifuViewer::AnalysisOnlineDialog *KifuViewer::analysisOnlineDialog = 0;
static QMutex analysisOnlineMutex;


KifuViewer::KifuViewer(QWidget *parent, const char *name)
  : BoardTabChild(parent, name), lastReadTime(0),
    index(-1), manualIndex(-1), ignoreMoveSignal(false),
    watchFileEnabled(false)
{
  list = new MoveList(this);
  players = new QListWidget(this);
  timeView = new QTableWidget(2, 1, this);
  moveInfoView = new QListWidget(this);
  commentView = new QTextBrowser(this);
  updateTime();

  QVBoxLayout *eastLayout = new QVBoxLayout;
  eastLayout->addWidget(players, 2);
  eastLayout->addWidget(timeView, 2);
  eastLayout->addWidget(moveInfoView, 5);
  eastLayout->addWidget(commentView, 6);
  QHBoxLayout *mainLayout = new QHBoxLayout(this);
  mainLayout->addWidget(list, 1);
  mainLayout->addWidget(board, 4);
  mainLayout->addLayout(eastLayout, 2);

  timer = new QTimer(this);

  connect(list, SIGNAL(currentRowChanged(int)),
	  this, SLOT(updateIndex(int)));
  connect(board, SIGNAL(moved(osl::Move)),
	  this, SLOT(boardMoved(osl::Move)));
  connect(board, SIGNAL(statusChanged()),
	  this, SLOT(updateTime()));
  connect(board, SIGNAL(statusChanged()),
	  this, SLOT(notifyState()));
  connect(timer, SIGNAL(timeout()),
	  this, SLOT(reloadFile()));
  connect(timer, SIGNAL(timeout()),
	  this, SLOT(updateStatusForAnalysis()));
  connect(this, SIGNAL(statusChanged(const osl::state::SimpleState &,
				     const osl::stl::vector<osl::Move> &,
				     int, osl::Move)),
	  this, SLOT(prepareUpdateForAnalysis(const osl::state::SimpleState &,
					      const osl::stl::vector<osl::Move> &,
					      int, osl::Move)));
  connect(this, SIGNAL(analyzeOnlineDisabled()), this, SLOT(stopAnalysisOnline()));

  timer->start(1000, false);
}

KifuViewer::~KifuViewer()
{
  setAnalyzeOnline(false);
}

void KifuViewer::forward()
{
  if (index >= (int) list->numMoves() - 1)
    return;

  if (manualMoves.size() > 0)
  {
    updateStateToIndex(index);
  }

  manualMoves.clear();

  osl::Move move = list->getMove(++index);

  if (!move.isPass())
  {
    ignoreMoveSignal = true;
    board->move(move);
    ignoreMoveSignal = false;
    list->setCurrentItem(list->item(index));
  }
  manualIndex = -1;
  emit indexUpdate(index);
}

void KifuViewer::backward()
{
  if (manualIndex > 0)
  {
    manualIndex--;
    manualMoves.pop_back();
    updateState();
    return;
  }
  else if (manualIndex == 0)
  {
    manualIndex = -1;
    manualMoves.clear();
    updateState();
    return;
  }

  if (index < 0)
  {
    return;
  }

  updateStateToIndex(--index);
  emit indexUpdate(index);
}

void KifuViewer::updateStateToIndex(int n)
{
  if (n < (int)list->numMoves())
  {
    osl::NumEffectState state(initialState);
    for (int i = 0; i <= n; i++)
    {
      state.makeMove(list->getMove(i));
    }
    board->setState(state, list->getMove(n));
    list->setCurrentItem(list->item(index));
  }
}

void KifuViewer::updateState()
{
  osl::NumEffectState state(initialState);
  for (int i = 0; i <= index; i++)
  {
    state.makeMove(list->getMove(i));
  }
  for (int i = 0; i <= manualIndex && i < (int)manualMoves.size(); i++)
  {
    state.makeMove(manualMoves[i]);
  }
  if (manualIndex >= 0 && manualIndex < (int)manualMoves.size())
    board->setState(state, manualMoves[manualIndex]);
  else
    board->setState(state);
  if (index == -1)
  {
    list->setCurrentItem(list->item(0));
  }
  else
  {
    list->setCurrentItem(list->item(index));
  }
}

void KifuViewer::boardMoved(osl::Move move)
{
  if (ignoreMoveSignal)
    return;
  if (move == list->getMove(index + 1) && manualMoves.size() == 0)
  {
    index++;
  }
  else
  {
    manualMoves.push_back(move);
    manualIndex = manualMoves.size() - 1;
  }
}

void KifuViewer::updateTime()
{
  timeView->clear();
  QStringList headers, vheaders;
  headers << QString::fromUtf8("消費時間");
  vheaders << QString::fromUtf8("先手  ") << QString::fromUtf8("後手  ");
  timeView->setHorizontalHeaderLabels(headers);
  timeView->setVerticalHeaderLabels(vheaders);
  QTableWidgetItem *white_time = new QTableWidgetItem(secondToTime(consumedTime(osl::WHITE)));
  white_time->setFlags(Qt::NoItemFlags);
  white_time->setTextAlignment(Qt::AlignRight);
  QTableWidgetItem *black_time = new QTableWidgetItem(secondToTime(consumedTime(osl::BLACK)));
  black_time->setFlags(Qt::NoItemFlags);
  black_time->setTextAlignment(Qt::AlignRight);
  timeView->setItem(0, 0, black_time);
  timeView->setItem(1, 0, white_time);

  moveInfoView->clear();
  commentView->clear();
  if (0 <= index && manualIndex == -1)
  {
    if (index < (int) comments.size())
      commentView->setText(comments[index]);
    if (index < (int) move_info.size())
    {
      const osl::record::SearchInfo& info = move_info[index];
      new QListWidgetItem(QString::fromUtf8("評価値 ") +
                          QString("%1").arg(info.value), moveInfoView);
      for (size_t i = 0; i < info.moves.size(); ++i) {
        new QListWidgetItem(gpsshogi::gui::Util::moveToString(info.moves[i]),
                            moveInfoView);
      }
    }
  }
}

int KifuViewer::consumedTime(osl::Player player) const
{
  int time = 0;
  for (int i = (player == osl::BLACK) ? 0 : 1; i <= index; i += 2)
  {
    time += list->getTime(i);
  }
  return time;
}

QString KifuViewer::secondToTime(int time)
{
  int hour = 0;
  int min = 0;
  int second = time;

  if (second >= 60)
  {
    min = second / 60;
    second = second % 60;
    if (min >= 60)
    {
      hour = min / 60;
      min = min % 60;
    }
  }
  QString str;
  str.sprintf("%2d %s", second, "秒");
  if (min > 0 || hour > 0)
  {
    QString minStr;
    minStr.sprintf("%2d %s ", min, "分");
    str.prepend(minStr);
    if (hour > 0)
    {
      QString hourStr;
      hourStr.sprintf("%2d %s ", hour, "時間");
      str.prepend(hourStr);
    }
  }
  return str;
}

void KifuViewer::reloadFile()
{
  if (! watchFileEnabled)
    return;
  QMutexLocker lk(&kifuFileMutex);
  if (kifuFile->reloadIfChanged())
  {
    open(kifuFile.get());
    toLastState();
  }
}

class OnlineViewItem : public Q3ListViewItem
{
  int depth;
  int relative_value;
public:
  OnlineViewItem(Q3ListView *pvList, int depth, double elapsed, size_t node_count, 
		 int value, QString pv, osl::Player turn)
    : Q3ListViewItem(pvList, QString("%1").arg(depth, 2),
		     QString("%1").arg(elapsed, 0, 'f', 1),
		     QString("%1").arg(node_count, 6),
		     QString("%1").arg(value*osl::playerToMul(turn), 5), 
		     pv),
      depth(depth), relative_value(value)
  {
  }
  int compare(Q3ListViewItem *i, int col, bool ascending) const
  {
    if (col == 0) {
      const double me = depth*1e6 + relative_value;
      const double other = ((OnlineViewItem*)i)->depth*1e6+((OnlineViewItem*)i)->relative_value;
      if (me < other)
	return 1;
      if (me == other)
	return 0;
      return -1;
    }
    return Q3ListViewItem::compare(i, col, ascending);
  }
};

class KifuViewer::AnalysisOnlineDialog
  : public QDialog, public osl::misc::Align16New
{
  QLabel *status;
  Q3ListView *pvList;
  QTextCodec *codec;
  osl::NumEffectState root;
  osl::HashKey root_key;
  osl::Move cur_move;
  size_t node_count;
  double elapsed, memory;
  static QRect lastGeometry;
public:
  volatile bool closed;
  explicit AnalysisOnlineDialog(const osl::NumEffectState& state, 
				QWidget *parent = 0, const char *name = 0)
    : QDialog(parent, name), status(0), pvList(0),
      codec(QTextCodec::codecForName("EUC-JP")), 
      root(state), root_key(state), 
      node_count(0), elapsed(0), memory(0), closed(false)
  {
    QVBoxLayout *layout = new QVBoxLayout(this);
    status = new QLabel("Searching");
    layout->addWidget(status);

    pvList = new Q3ListView(this);
    pvList->addColumn("Depth");
    pvList->addColumn("Elapsed");
    pvList->addColumn("Node");
    pvList->addColumn("Value");
    pvList->addColumn("Principal Variation");
    pvList->setColumnAlignment(0, Qt::AlignRight);
    pvList->setColumnAlignment(1, Qt::AlignRight);
    pvList->setColumnAlignment(2, Qt::AlignRight);
    pvList->setColumnAlignment(3, Qt::AlignRight);
    pvList->setAllColumnsShowFocus(true) ;
    layout->addWidget(pvList);

    QPushButton *button = new QPushButton(this);
    button->setText("&Finish");
    layout->addWidget(button);
    connect(button, SIGNAL(clicked()), this, SLOT(accept()));
  }
  void updateStatus()
  {
    if (! cur_move.isNormal() || closed)
      return;
    QMutexLocker lk(&analysisOnlineMutex);
    std::string euc_move = osl::record::ki2::show(cur_move, root);
    status->setText(QString::fromUtf8("探索中 %1 %2秒 NPS %3 Memory %4%")
		    .arg(codec->toUnicode(euc_move.c_str(), euc_move.length()))
		    .arg(elapsed, 0, 'g', 3).arg((int)(node_count/elapsed))
		    .arg((int)memory)
		    );
  }
  void setState(const osl::NumEffectState& state)
  {
    root = state;
    root_key = osl::HashKey(state), 
    node_count = 0;
    elapsed = memory = 0;
    pvList->setUpdatesEnabled(false);
    pvList->clear();
    pvList->setUpdatesEnabled(true);
  }
  void showPV(const osl::HashKey& key,
	      int depth, size_t node_count, double elapsed, int value,
	      const std::vector<osl::Move>& pv, const char *additional_message="",
	      const std::vector<char> *pv_threatmate=0)
  {
    if (closed || key != root_key)
      return;
    std::string euc_pv = pv_threatmate
      ? osl::record::ki2::show(&*pv.begin(), &*pv.end(),
			       &*pv_threatmate->begin(), &*pv_threatmate->end(),
			       root)
      : osl::record::ki2::show(&*pv.begin(), &*pv.end(), root);
    {
      QMutexLocker lk(&analysisOnlineMutex);
      pvList->setUpdatesEnabled(false);
      Q3ListViewItem *item =
	new OnlineViewItem(pvList, depth, elapsed, node_count, 
			   value,
			   codec->toUnicode(euc_pv.c_str(), euc_pv.length())+additional_message,
			   root.turn());
      pvList->sort();
      pvList->setUpdatesEnabled(true);
      pvList->ensureItemVisible(item);
    }
    this->node_count = node_count;
    this->elapsed = elapsed;
    updateStatus();
  }
  void rootMove(const osl::HashKey& key, osl::Move cur)
  {
    if (closed || key != root_key)
      return;
    cur_move = cur;
    // Too many rootMove() can be called in a short period.
    // Thus, updateStatus() will be delayed until timeInfo() comes.
    // updateStatus();
  }
  void rootForcedMove(const osl::HashKey& key, osl::Move the_move)
  {
    if (closed || key != root_key)
      return;
    std::vector<osl::Move> pv;
    pv.push_back(the_move);
    showPV(key, 100, this->node_count, this->elapsed, 0, pv, " (forced move)");
  }
  void rootLossByCheckmate(const osl::HashKey& key)
  {
    if (closed || key != root_key)
      return;
    std::vector<osl::Move> pv;
    showPV(key, 100, this->node_count, this->elapsed, -10000000, pv, "(loss by checkmate)");
  }	      
  void timeInfo(const osl::HashKey& key, size_t node_count, double elapsed)
  {
    if (closed || key != root_key)
      return;
    this->node_count = node_count;
    this->elapsed = elapsed;
    updateStatus();
  }
  void hashInfo(const osl::HashKey& key, double ratio)
  {
    if (closed || key != root_key)
      return;
    memory = ratio*100;
  }
  void showEvent(QShowEvent *event)
  {
    setGeometry(lastGeometry);
    if (geometry().width() < 100 || geometry().height() < 100)
      resize(std::max(geometry().width(), 100), 
	     std::max(geometry().height(),100));
    QDialog::showEvent(event);
  }
  void hideEvent(QHideEvent *event)
  {
    lastGeometry = geometry();
    QDialog::hideEvent(event);
  }
};

QRect KifuViewer::AnalysisOnlineDialog::lastGeometry(100, 100, 700, 300);
class OnlineSearchMonitor : public osl::search::SearchMonitor
{
  KifuViewer::AnalysisOnlineDialog *dialog;
  QTextCodec *codec;
  osl::HashKey key;
  int cur_depth;
  enum { MinimumDepth = 4 };
public:
  OnlineSearchMonitor(KifuViewer::AnalysisOnlineDialog *d, 
		      const osl::HashKey& k)
    : dialog(d), codec(QTextCodec::codecForName("EUC-JP")), key(k),
      cur_depth(0)
  {
  }
  void showPV(int depth, size_t node_count, double elapsed, int value, osl::Move cur, const osl::Move *first, const osl::Move *last,
	      const bool *threatmate_first, const bool *threatmate_last)
  {
    std::vector<osl::Move> pv;
    std::vector<char> pv_threatmate;
    if (first == 0 || *first != cur) {
      pv.push_back(cur);
      pv_threatmate.push_back(false);
    }
    while (first != last) {
      pv.push_back(*first++);
      if (threatmate_first != threatmate_last)
	pv_threatmate.push_back(*threatmate_first++);
    }
    cur_depth = std::max(cur_depth, depth);
    if (cur_depth >= MinimumDepth || abs(value) >= 1000000)
      dialog->showPV(key, depth, node_count, elapsed, value, pv, "", &pv_threatmate);
  }
  void rootMove(osl::Move cur)
  {
    if (cur_depth >= MinimumDepth)
      dialog->rootMove(key, cur);    
  }
  void rootFirstMove(osl::Move cur)
  {
    if (cur_depth >= MinimumDepth)
      dialog->rootMove(key, cur);    
  }
  void timeInfo(size_t node_count, double elapsed)
  {
    if (cur_depth >= MinimumDepth)
      dialog->timeInfo(key, node_count, elapsed);
  }		      
  void hashInfo(double ratio)
  {
    if (cur_depth >= MinimumDepth)
      dialog->hashInfo(key, ratio);
  }
  void rootForcedMove(osl::Move the_move)
  {
    dialog->rootForcedMove(key, the_move);
  }
  void rootLossByCheckmate()
  {
    std::cerr << "rootLossByCheckmate\n";
    dialog->rootLossByCheckmate(key);
  }	      
};

struct KifuViewer::AnalysisOnlineThread : public QThread, public osl::misc::Align16New
{
  osl::NumEffectState input;
  boost::scoped_ptr<osl::game_playing::AlphaBeta2OpenMidEndingEvalPlayer> player;
  QMutex mutex;
  volatile bool stop;
  boost::shared_ptr<OnlineSearchMonitor> monitor;
  osl::HashKey key;
  AnalysisOnlineThread(const osl::NumEffectState& initial) 
    : input(initial), stop(false), key(initial)
  {
  }
  void run()
  {
    while (! stop) {
      boost::scoped_ptr<osl::game_playing::GameState> state;
      osl::HashKey last_key;
      {
	QMutexLocker lk(&mutex);
	state.reset(new osl::game_playing::GameState(input));
	player.reset(new osl::game_playing::AlphaBeta2OpenMidEndingEvalPlayer);
	monitor.reset(new OnlineSearchMonitor(analysisOnlineDialog, key));
	player->addMonitor(monitor);
	player->setDepthLimit(2000, 400, 200);
	player->setNodeLimit(std::numeric_limits<size_t>::max());
	player->setTableLimit(std::numeric_limits<size_t>::max(), 200);
	player->enableMultiPV(100);
	player->setVerbose(0);
	last_key = key;
      }
      osl::MoveWithComment ret
	= player->selectBestMove(*state, 0,0,3600);
      while (! stop && last_key == key)
	boost::this_thread::sleep(boost::posix_time::milliseconds(500));
    }
  }
  void setState(const osl::NumEffectState& state)
  {
    const osl::HashKey new_key(state);
    QMutexLocker lk(&mutex);
    if (key == new_key)
      return;
    input = state;
    key = new_key;
    if (player && player->canStopSearch())
      player->stopSearchNow();
  }
  void finish()
  {
    QMutexLocker lk(&mutex);
    stop = true;
    if (player && player->canStopSearch())
      player->stopSearchNow();
    wait();
  }
};

void KifuViewer::prepareUpdateForAnalysis(const osl::state::SimpleState &origin,
					 const osl::stl::vector<osl::Move> &moves,
					 int /*limit*/, osl::Move /*next_move*/)
{
  if (! analyzeOnlineEnabled)
    return;
  osl::state::NumEffectState state(origin);
  for (size_t i = 0; i < moves.size(); i++)
    state.makeMove(moves[i]);
  QMutexLocker lk(&analysisOnlineMutex);
  if (! stateForAnalysis)
    stateForAnalysis.reset(new osl::NumEffectState(state));
  else
    *stateForAnalysis = state;
  stateForAnalysisChanged = true;
}

void KifuViewer::updateStatusForAnalysis()
{
  if (! analyzeOnlineEnabled)
    return;
  QMutexLocker lk(&analysisOnlineMutex);
  if (! stateForAnalysisChanged)
    return;
  static time_t prev = 0;
  time_t now = time(0);
  if (now - prev < 1)
    return;
  prev = now;
  if (analysisOnlineDialog) {
    analysisOnlineDialog->setState(*stateForAnalysis);
  } else {
    analysisOnlineDialog = new AnalysisOnlineDialog(*stateForAnalysis, this);
    connect(analysisOnlineDialog, SIGNAL(accepted()), this, SIGNAL(analyzeOnlineDisabled()));
    connect(analysisOnlineDialog, SIGNAL(rejected()), this, SIGNAL(analyzeOnlineDisabled()));
    analysisOnlineDialog->show();
  }

  if (! thread) {
    thread.reset(new AnalysisOnlineThread(*stateForAnalysis));
    thread->start();
  }
  else
    thread->setState(*stateForAnalysis);
  stateForAnalysisChanged = false;
}

void KifuViewer::stopAnalysisOnline()
{
  if (analyzeOnlineEnabled)
    setAnalyzeOnline(false);
}


void KifuViewer::setAnalyzeOnline(bool enable)
{
  analyzeOnlineEnabled = enable;
  if (thread && !enable) {
    thread->finish();
    thread.reset();
  }
  if (analysisOnlineDialog && !enable) {
    QMutexLocker lk(&analysisOnlineMutex);
    AnalysisOnlineDialog *dialog = analysisOnlineDialog;
    dialog->closed = true;
    analysisOnlineDialog = 0;
    dialog->close();
  }
}

void KifuViewer::clearState()
{
  comments.clear();
  move_info.clear();
  players->clear();
  list->clear();
  manualMoves.clear();
  index = -1;
  manualIndex = -1;
}

void KifuViewer::resetState()
{
  clearState();
  emit stateReset();
}

void KifuViewer::openEvalGraph(bool force)
{
  int move_info_count=0;
  if (! force)
  {
    for (size_t i=0; i< move_info.size(); ++i)
      if (move_info[i].value)
	++move_info_count;
  }
  if (force || move_info_count > 0)
  {
    DualEvaluationDialog *eval = new DualEvaluationDialog(this);
    eval->setInfo(move_info);
    connect(this, SIGNAL(indexUpdate(int)), eval, SLOT(setIndex(int)));
    connect(eval, SIGNAL(selected(int)), this, SLOT(updateIndex(int)));
    connect(this, SIGNAL(stateReset()), eval, SLOT(accept()));
    eval->show();
  }
}

void KifuViewer::open(const osl::state::SimpleState& state,
		      const osl::stl::vector<osl::Move>& board_moves)
{
  osl::stl::vector<int> t(board_moves.size(), 1);
  open(state, board_moves, t);
}

void KifuViewer::open(KifuFile *file)
{
  QMutexLocker lk(&kifuFileMutex);
  clearState();
  initialState.copyFrom(osl::NumEffectState(file->getInitialState()));
  board->setState(initialState);
  list->setState(initialState, file->getMoves(), file->getTime(), file->getComments());
  filename = file->getFilename();
  new QListWidgetItem(QString::fromUtf8("先手: ") +
                      file->getPlayerName(osl::BLACK), players);
  new QListWidgetItem(QString::fromUtf8("後手: ") +
                      file->getPlayerName(osl::WHITE), players);
  for (size_t i = 0; i < file->getInitialComment().size(); ++i) {
    new QListWidgetItem(file->getInitialComment()[i], players);
  }
  comments = file->getComments();
  move_info = file->getSearchInfo();
  if (file != kifuFile.get())
  {
    kifuFile.reset(file);
  }
  openEvalGraph(false);
  emit stateReset();
}

void KifuViewer::open(const osl::state::SimpleState& state,
		      const osl::stl::vector<osl::Move>& board_moves,
		      const osl::stl::vector<int>& consumed_time)
{
  clearState();
  initialState.copyFrom(osl::NumEffectState(state));
  filename = "";
  board->setState(initialState);
  list->setState(initialState, board_moves, consumed_time, osl::vector<QString>());
  emit stateReset();
}


void KifuViewer::watchFile(bool enable)
{
  watchFileEnabled = enable;
}

osl::NumEffectState KifuViewer::getStateAndMovesToCurrent(osl::stl::vector<osl::Move> &m)
{
  for (int i = 0; i <= index; i++)
  {
    m.push_back(list->getMove(i));
  }
  for (int i = 0; i <= manualIndex; i++)
  {
    m.push_back(manualMoves[i]);
  }
  return initialState;
}

void KifuViewer::toInitialState()
{
  manualMoves.clear();
  index = -1;
  manualIndex = -1;
  updateState();
  emit indexUpdate(index);
}

void KifuViewer::toLastState()
{
  manualMoves.clear();
  manualIndex = -1;
  index = list->numMoves() - 1;
  updateState();
  emit indexUpdate(index);
}

void KifuViewer::updateIndex(int i)
{
  manualMoves.clear();
  manualIndex = -1;
  index = i;
  updateStateToIndex(i);
  emit indexUpdate(i);
}

QWidget *KifuViewer::moveGenerateDialog()
{
  MoveGeneratorDialog *dialog
    = (MoveGeneratorDialog *)BoardTabChild::moveGenerateDialog();
  if (dialog)
    connect(this, SIGNAL(statusChanged(const osl::state::SimpleState &,
				       const osl::stl::vector<osl::Move> &,
				       int, osl::Move)),
	    dialog, SLOT(setStatus(const osl::state::SimpleState &,
				   const osl::stl::vector<osl::Move> &,
				   int, osl::Move)));
  return dialog;
}

osl::Move KifuViewer::getNextMove()
{
  return list->getMove(index + 1);
}

void KifuViewer::analyze()
{
  osl::vector<osl::Move> moves;
  for (size_t i = 0; i < list->numMoves(); i++)
  {
    moves.push_back(list->getMove(i));
  }

  KifuAnalyzer *analyzer = new KifuAnalyzer(initialState, moves);
  analyzer->show();
}

const KifuFile *KifuViewer::getKifuFile()
{
  QMutexLocker lk(&kifuFileMutex);
  return kifuFile.get();
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
