;;; Copyright (C) 2011 Team GPS.
;;; 
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

(ns twitter_clj.engine_controler
  (:import (java.io InputStreamReader OutputStreamWriter))
  (:require [twitter_clj.common :as common]
            [twitter_clj.engine :as engine]
            [twitter_clj.ki2 :as ki2]
            [twitter_clj.snapshot :as snapshot]
            [twitter_clj.subprocess :as sub]
            [twitter_clj.twitter :as twitter]
            [clojure.string :as str]
            [clojure.contrib.command-line :as cmd]
            [clojure.contrib.duck-streams :as duck]
            [clojure.contrib.logging :as log]))

;; ==================================================
;; Global variables
;; ==================================================


;; ==================================================
;; Functions
;; ==================================================

(defn parse-line
  "Parse lines that gpsusi produces."
  [line]
  (let [r1 #"^info depth (\d+) .* score cp (-?\d+) .* pv (.*)$"]
    (if-let [m (re-find r1 line)]
      (let [[_ depth cp pv] m
            first-move (first (str/split pv #" "))]
        {:depth (Integer. depth) :cp (Integer. cp) :pv pv :first-move first-move})
      nil)))


(defn start-monitor
  "Start a monitor thread reading what gpsusi produces."
  [stdin]
  (log/debug "Starting a new monitor...")
  (future
    (reset! common/pv [])
    (loop [lines (line-seq stdin)]
      (if-let [line (first lines)]
        (if (or (re-find #"bestmove" line)
                (re-find #"warning stop ignored" line))
          (common/sort-pv @common/pv) ;; base
          (if-let [m (parse-line line)]
            (let [current-depth (or (:depth (first @common/pv)) 0)
                  depth (:depth m)]
              (if (< current-depth depth)
                (do
                  (reset! common/pv [m])
                  (recur (rest lines)))
                (do
                  (swap! common/pv conj m)
                  (recur (rest lines)))))
            (recur (rest lines))))))))


(defn get-nmove
  [line]
  {:pre  [(not-empty line)]
   :post [(pos? %)]}
  (let [m (re-find #"moves (.*)" line)
        _ (assert m)
        n (count (str/split (nth m 1) #" "))]
    n))


(defn interrupt-thread
  [thread]
  {:pre [(not (nil? thread))]}
  (.interrupt thread)
  (.join thread))
  

(defn stop-monitor
  "Stop a monitor (which is a future)."
  [stdout monitor]
  {:pre [(not (nil? monitor))]}
  (sub/write stdout "stop")
  (log/debug "waiting monitor")
  (log/debug (format "Monitor finished: %s" @monitor)))


(defn -main
  [& args]
  (cmd/with-command-line args
    "lein trampoline run [--dry-run]"
    [[dry-run?      "Do not tweet" false]
     [force-think   "Think at least n seconds for each move" 0]
     [force-update? "Update possile exisiting tweets" false]
     [gpsusi        "Path to gpsusi" "../../gpsshogi/bin/gpsusi"]
     remains]
    (swap! common/options assoc :dry-run dry-run?)
    (swap! common/options assoc :force-think (Integer. force-think))
    (swap! common/options assoc :force-update force-update?)
    (swap! common/options assoc :gpsusi gpsusi)
    (let [gpsusi-cmd (str (:gpsusi @common/options) " --extended-usi 1") ;; extended mode
          [proc stdin stdout stderr] (engine/start-engine gpsusi-cmd)]   ;; search engine
      (ki2/start-ki2-engine (:gpsusi @common/options)) ;; normal mode
      (twitter/post-version (:id-name @common/options))
      (twitter/post-title)
      (loop [lines (line-seq (java.io.BufferedReader. *in*))
             monitor nil
             snapshot-thread nil]
        (if (and (seq lines)
                 (not= "resign" (first lines)))
          (let [line (first lines)
                nmove (get-nmove line)]
            (log/info (format ">>> [%d] %s" nmove line))
            (when (and snapshot-thread
                       (pos? (:force-think @common/options)))
              (log/debug "wait for test")
              (Thread/sleep (* 1000 (:force-think @common/options))))
            (when snapshot-thread
              (interrupt-thread snapshot-thread))
            (when monitor
              (stop-monitor stdout monitor))
            ;; both snapshot-thread and monitor have finished.
            (if (and (twitter/moves-file-exists? nmove)
                     (not (:force-update @common/options)))
              (do
                (log/info "Found a twitter log file for this move. Skip it.")
                (recur (rest lines) nil nil))
              (do
                (log/debug "Sending the move to the engine...")
                (if-not (ki2/is-valid-position line)
                  (do
                    (log/warn (format "Read an invalid position. Skip it: %s" line))
                    (recur (rest lines) nil nil))
                  (do
                    (ki2/set-position line)
                    (sub/write stdout line)
                    (sub/write stdout "go infinite")
                    (recur (rest lines)
                           (start-monitor stdin)
                           (snapshot/start-snapshot-thread)))))))
          (do ; else
            (log/info "Finished reading.")
            (when snapshot-thread
              (interrupt-thread snapshot-thread))
            (when monitor
              (stop-monitor stdout monitor))
            nil)))
      (ki2/stop-ki2-engine)
      (engine/stop-engine proc stdin stdout stderr))
    (shutdown-agents)))

