/* $Id: linuxprocfs.cc,v 1.15 2002/03/10 18:37:10 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "transient.h"

#ifdef HAVELINUXPROCFS

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <asm/page.h>
#include <gtk/gtk.h>
#include "gps.h"
#include "polling.h"
#include "linuxprocfs.h"
#include "tstring.h"
#include "importglobals.h"

/* POSIX only for now, maybe add Linux specific later */

#define COUNTSIG 19

static char *signames[]={
  "SIGHUP", "SIGINT", "SIGQUIT","SIGILL", "SIGABRT","SIGFPE",
  "SIGKILL","SIGSEGV","SIGPIPE","SIGALRM","SIGTERM","SIGUSR1",
  "SIGUSR2","SIGCHLD","SIGCONT","SIGSTOP","SIGTSTP","SIGTTIN",
  "SIGTTOU"};

static gint sigvalues[]={SIGHUP,SIGINT,SIGQUIT,SIGILL,SIGABRT,SIGFPE,
			 SIGKILL,SIGSEGV,SIGPIPE,SIGALRM,SIGTERM,SIGUSR1,
			 SIGUSR2,SIGCHLD,SIGCONT,SIGSTOP,SIGTSTP,SIGTTIN,
			 SIGTTOU};

/* ********************************************************
 * 
 *
 *                 LinuxProcFsProcessListPoller
 *	      
 *
 * ********************************************************* */

LinuxProcFsProcessListPoller::LinuxProcFsProcessListPoller()
  : ProcessListPoller() {
  t=new TString(1024);
}

void LinuxProcFsProcessListPoller::poll() {
  static FILE *f,*g;
  static time_t btime;
  static int i,j;
  static int lastpid,mem_unit,yav;
  static unsigned long rtime,smem[2];
  static unsigned long ulv,dlv,rib;
  static double cpufrac;
  static time_t ptime;
  static char wantmore[1024],buffer[1024],buffercopy[1024],b2[128],b3[512],st[64];
  static char *p1,*p2;
  static DIR *slashproc;
  static struct dirent *dentry;
  static struct passwd *ident;
  static ProcessListItem *pli;
  static char fbuf[4096];

  /* reset current list */
  reset();

  /* get boot time and last pid */
  f=fopen("/proc/stat","r");
  if (f==NULL) return;
  setbuffer(f,fbuf,4096);
  while(1) {
    if (fgets(buffer,1024,f)==NULL)
      break;
    t->set(buffer);
    if (t->contains("btime ")!=NULL) {
      t->token(T_SNT);
      btime=gps_atoul(t->token(T_SNT));
    }
    if (t->contains("processes ")!=NULL) {
      t->token(T_SNT);
      lastpid=atoi(t->token(T_SNT));
      break;
    }
  } 
  fclose(f);

  /* open /proc dir */
  slashproc=opendir("/proc");
  if (slashproc==NULL) {
    fprintf(stderr,
	    "** gPS: unable to open /proc, cannot read process table\n");
    return;
  }

  dentry=readdir(slashproc);
  while(dentry!=NULL) {
    i=which_number(dentry->d_name); /* is process dir ? */
    if (i<0) {
      dentry=readdir(slashproc);
      continue;
    }
    
    snprintf(buffer,1024,"/proc/%d/stat",i);
    buffer[1023]=0;

    f=fopen(buffer,"r");
    if (f==NULL) {
      dentry=readdir(slashproc);
      continue;
    }
    setbuffer(f,fbuf,4096);
    fgets(wantmore,1023,f);
    fclose(f);

    pli=new ProcessListItem();

    /* get owner */
    snprintf(buffer,1024,"/proc/%d/status",i);
    buffer[1023]=0;
    g=fopen(buffer,"r");
    pli->setField(OWNER,"???");
    if (g!=NULL) {
      while(1) {
	if (fgets(buffer,1024,g)==NULL) {
           fclose(g);
           break;
        }
	t->set(buffer);
	if (t->contains("Uid:")==NULL)
	  continue;
	t->token(" \t");
	j=atoi(t->token(" \t"));
	fclose(g);
	ident=getpwuid(j);
	if (ident) {
	  pli->setField(OWNER,ident->pw_name);
	} else {
	  char tmpname[64];
	  snprintf(tmpname,64,"#%d",j);
	  tmpname[63]=0;
	  pli->setField(OWNER,tmpname);
	}
	break;
      } /* while 1 */
    } /* g!=NULL */


    /* PID */
    snprintf(b3,512,"%d",i);
    b3[511]=0;
    pli->setField(PID,b3);
    strcpy(buffer,wantmore);
    strcpy(buffercopy,buffer);

    /* HOST */
    pli->setField(MACHINE,get_local_hostname());

    /* NAME */
    p1=strchr(buffercopy,'(');
    p2=strrchr(buffercopy,')'); /* pattern matching, the hard way */

    if ((p1!=NULL)&&(p2!=NULL)) {
      memset(b2,0,128);
      memcpy(b2,p1+1,(p2-p1)-1);
      pli->setField(NAME,b2);
    } else
      pli->setField(NAME,"<null stat>");

    /* STATE (1/2) */
    strcpy(buffer,p2+1);
    t->set(buffer);
    strcpy(st,t->token(" \t\n"));

    for(j=0;j<10;j++)
      t->token(" \t\n");

    /* CPU */
    ulv= gps_atoul(t->token(T_SNT));
    ulv+=gps_atoul(t->token(T_SNT));
    dlv= gps_atoul(t->token(T_SNT));
    dlv+=gps_atoul(t->token(T_SNT));

    /* PRIORITY & NICE */
    rib=atoi(t->token(T_SNT));
    snprintf(b3,512,"%d",rib);
    b3[511]=0;
    pli->setField(PRIORITY,b3);
    rib=atoi(t->token(T_SNT));
    snprintf(b3,512,"%d",rib);
    b3[511]=0;
    pli->setField(NICE,b3);

    for(j=0;j<2;j++)
      t->token(T_SNT);

    rtime=gps_atoul(t->token(T_SNT));
    ptime=btime+(time_t)(rtime/100UL);

    strcpy(b3,ctime(&ptime));
    b2[0]=0;
    time_to_gps(ptime,b2); /* format conversion */

    pli->setField(START,b2);

    dlv=time(NULL)-ptime;
    if (dlv==0)
      dlv=1;
    cpufrac=((double)ulv)/((double)dlv);
    snprintf(b3,512,"%.1f",cpufrac);
    b3[511]=0;
    pli->setField(CPU,b3);

    /* MEM (VSIZE) */
    smem[0]=gps_atoul(t->token(T_SNT));
    mem_unit=0;
    while(smem[0]>(99UL<<10)) {
      ++mem_unit;
      smem[0]>>=10;
    }
    /* 500 used on purpose instead of 512, to leave room for the suffix */
    snprintf(b3,500,"%lu",smem[0]);
    b3[499]=0;
    strcat(b3,power_of_two_suffix(mem_unit));
    pli->setField(SIZE,b3);

    /* RSS */
    smem[1]=gps_atoul(t->token(T_SNT));

    /* STATE (2/2) */
    if (smem[1]==0)
      if (rib>0) /* NICED */
	strcat(st,"WN");
      else
	strcat(st,"W");
    else
      if (rib>0)
	strcat(st," N");

    pli->setField(STATE,st);

    smem[1]<<=PAGE_SHIFT;
    mem_unit=0;
    while(smem[1]>(99UL<<10)) {
      ++mem_unit;
      smem[1]>>=10;
    }
    /* 500 used on purpose to leave room for the suffix */
    snprintf(b3,500,"%lu",smem[1]);
    b3[499]=0;
    strcat(b3,power_of_two_suffix(mem_unit));
    pli->setField(RSS,b3);

    /* LONG NAMES */
    snprintf(buffer,1024,"/proc/%d/cmdline",i);
    buffer[1023]=0;
    f=fopen(buffer,"r");
    if (f!=NULL) {
      memset(buffer,0,1024);
      yav=fread(buffer,1,1023,f);
      fclose(f);
      for(int yayav=0;yayav<(yav-1);yayav++)
	if (buffer[yayav]==0)
	  buffer[yayav]=0x20;
      if (strlen(buffer)>0)
	pli->setField(LONGNAME,buffer);
    } /* if f != NULL */

    process_list=g_list_prepend(process_list,(gpointer)pli);
    dentry=readdir(slashproc); /* read next entry */
  } /* while dentry != NULL */    

  closedir(slashproc);
}

/* ********************************************************
 * 
 *
 *                 LinuxProcFsProcessDetailsPoller
 *	      
 *
 * ********************************************************* */

LinuxProcFsProcessDetailsPoller::LinuxProcFsProcessDetailsPoller()
  : ProcessDetailsPoller() {
  sdevs=NULL;
  init_device_list(); /* for TTY naming purposes */
}

void LinuxProcFsProcessDetailsPoller::poll(int whichpid) {
  FILE *f;
  char buffer[1024],buffercopy[1024],b2[128],b3[512];
  struct passwd *ident;
  time_t ptime;
  char *p1,*p2;
  int i,j;
  unsigned long la,rtime,btime,rssi;
  TString *t;

  reset();
  item=new ProcessItem();

  snprintf(b2,128,"%d",whichpid);
  b2[127]=0;
  item->setField(PID,b2);

  /* HOST */
  item->setField(MACHINE,get_local_hostname());

  snprintf(buffer,1024,"/proc/%d/stat",whichpid);
  buffer[1023]=0;
  f=fopen(buffer,"r");
  if (f==NULL) {
    reset();
    return;
  }
  fgets(buffer,1024,f);
  strcpy(buffercopy,buffer);
  fclose(f);

  t=new TString(1024);

  /* get boot time */
  f=fopen("/proc/stat","r");
  while(1) {
    if (fgets(buffer,1024,f)==NULL)
      break;
    t->set(buffer);
    if (t->contains("btime ")!=NULL) {
      t->token(T_SNT);
      btime=gps_atoul(t->token(T_SNT));
      break;
    }
  } 
  fclose(f);

  // user
  snprintf(b2,128,"/proc/%d/status",whichpid);
  b2[127]=0;

  strcpy(b3,"???");
  f=fopen(b2,"r");
  if (f!=NULL) {
    while(1) {
      if (fgets(b3,512,f)==NULL)
	break;
      if (strstr(b3,"Uid:")!=NULL)
	break;
    }
    fclose(f);
    t->set(b3);
    t->token(T_SNT);
    i=atoi(t->token(T_SNT));
    ident=getpwuid(i);
    if (ident) {
      item->setField(OWNER,ident->pw_name);
    } else {
      char tmpname[64];
      snprintf(tmpname,64,"#%d",i);
      item->setField(OWNER,tmpname);
    }
  }

  // cmdline
  snprintf(b2,128,"/proc/%d/cmdline",whichpid);
  b2[127]=0;
  f=fopen(b2,"r");
  if (f!=NULL) {
    memset(b3,0,512);
    i=fread(b3,1,512,f);
    fclose(f);
    for(j=0;j<(i-1);j++)
      if (b3[j]==0)
	b3[j]=0x20;
    t->set(b3);
    item->setField(LONGNAME,t->token("\n\t"));
  }

  // parse f into fields
  t->set(buffer);
  t->token(" \t\n"); // TAKE OFF PID (#1)

  // name (now fixed to get weird characters without mess)
  p1=strchr(buffercopy,'(');
  p2=strrchr(buffercopy,')'); /* pattern matching, the hard way */

  if ((p1!=NULL)&&(p2!=NULL)) {
    memset(b2,0,128);
    memcpy(b2,p1+1,(p2-p1)-1);
    strcpy(b3,b2);
    strcpy(buffer,p2+1);
    item->setField(NAME,b3);
  } else
    item->setField(NAME,"<null stat>");

  // state
  t->set(buffer);
  strcpy(b2,t->token(" \t\n")); // State (#3)
  item->setField(STATE,b2);

  // ppid
  strcpy(b2,t->token(" \t\n")); // PPID (#4)
  item->setField(PPID,b2);

  // pgid
  strcpy(b2,t->token(" \t\n")); // PGID (#5)
  item->setField(PGID,b2);

  // sid
  strcpy(b2,t->token(" \t\n")); // SID (#6)
  item->setField(SID,b2);

  // tty
  strcpy(b2,t->token(" \t\n")); // TTY (#7)
  i=atoi(b2);
  get_tty_name(i,b3);
  item->setField(TTY,b3);

  // tpgid
  strcpy(b2,t->token(" \t\n")); // TPGID (#8)
  item->setField(TPGID,b2);

  // flags
  strcpy(b2,t->token(" \t\n")); // FLAGS (#9)
  la=atol(b2);
  parse_process_flags(la,b3);
  item->setField(FLAGS,b3);

  // minflt
  strcpy(b2,t->token(" \t\n"));
  item->setField(MINFLT,b2);

  // cminflt
  strcpy(b2,t->token(" \t\n"));
  item->setField(CMINFLT,b2);

  // maxflt
  strcpy(b2,t->token(" \t\n"));
  item->setField(MAJFLT,b2);

  // cmaxflt
  strcpy(b2,t->token(" \t\n"));
  item->setField(CMAJFLT,b2);

  // utime
  strcpy(b2,t->token(" \t\n"));
  item->setField(USRJIFFIES,b2);

  // stime
  strcpy(b2,t->token(" \t\n"));
  item->setField(KRNJIFFIES,b2);

  // cutime
  strcpy(b2,t->token(" \t\n"));
  item->setField(CUSRJIFFIES,b2);

  // cstime
  strcpy(b2,t->token(" \t\n"));
  item->setField(CKRNJIFFIES,b2);

  // counter
  strcpy(b2,t->token(" \t\n"));
  item->setField(PRIORITY,b2);

  // nicety
  strcpy(b2,t->token(" \t\n"));
  item->setField(NICE,b2);

  t->token(" \t\n"); // timeout
  t->token(" \t\n"); // itrealvalue

  // start time
  strcpy(b2,t->token(" \t\n"));
  rtime=gps_atoul(b2);
  ptime=btime+(time_t)(rtime/100UL);
  strcpy(b3,ctime(&ptime));
  if (b3[strlen(b3)-1]=='\n')
    b3[strlen(b3)-1]=0;
  item->setField(START,b3);

  // vsize
  strcpy(b2,t->token(" \t\n"));
  item->setField(SIZE,b2);

  // RSS
  strcpy(b2,t->token(" \t\n"));
  rssi=gps_atoul(b2);
  rssi<<=PAGE_SHIFT;
  snprintf(b2,128,"%lu",rssi);
  b2[127]=0;
  item->setField(RSS,b2);

  // RLIM
  strcpy(b2,t->token(" \t\n"));
  item->setField(RSSLIM,b2);

  // signals
  for(i=0;i<5;i++)
    t->token(" \t\n");

  // pending signals
  strcpy(b2,t->token(" \t\n"));
  strcpy(b3,get_signal_string(atol(b2)));
  item->setField(SIGPENDING,b3);

  // blocked signals
  strcpy(b2,t->token(" \t\n"));
  strcpy(b3,get_signal_string(atol(b2)));
  item->setField(SIGBLOCKED,b3);

  // ignored signals
  strcpy(b2,t->token(" \t\n"));
  strcpy(b3,get_signal_string(atol(b2)));
  item->setField(SIGIGNORED,b3);

  // caught signals
  strcpy(b2,t->token(" \t\n"));
  strcpy(b3,get_signal_string(atol(b2)));
  item->setField(SIGCAUGHT,b3);
  delete t;
}

void LinuxProcFsProcessDetailsPoller::get_tty_name(int x,char *s) {
  dpair *p;
  GList *sp;
  int maj,min;

  if (x<=0) {
    strcpy(s,"none");
    return;
  }

  strcpy(s,"no tty info");
  if (sdevs==NULL)
    return;

  min=x&0xff;
  maj=x>>8;

  for(sp=sdevs;sp!=NULL;sp=g_list_next(sp)) {
    p=(dpair *)(sp->data);
    if (p->number==maj) {
      sprintf(s,"%s%d",p->name,min);
      break;
    }
  }
}

void LinuxProcFsProcessDetailsPoller::init_device_list() {
  FILE *f;
  char *a,*b;
  gint i,gotcd;
  dpair *d;
  TString *t;

  f=fopen("/proc/devices","r");
  if (f==NULL) {
    g_warning("Unable to read /proc/devices."\
	      "tty information will not be shown.\n");
    return;
  }
  t=new TString(1024);
  a=(char *)g_malloc0(1024);
  gotcd=FALSE;
  while(1) {
    if (fgets(a,1024,f)==NULL)
      break;    
    if (!gotcd)
      if (strstr(a,"haracter")==NULL)
	continue;
    gotcd=TRUE;
    t->set(a);
    b=t->token(T_SNT);
    if ((b==NULL)||(strlen(b)==0))
      break;
    if ((i=atoi(b))<=0)
      continue;
    d=new dpair();
    d->number=i;
    b=t->token(T_SNT);
    strcpy(d->name,b);
    sdevs=g_list_prepend(sdevs,(gpointer)d);
  }
  delete t;
  g_free(a);
  fclose(f);
}

void LinuxProcFsProcessDetailsPoller::parse_process_flags(unsigned long flags,
							  char *dest) {
  /* these macros taken from linux/sched.h of kernel 2.2.3 */
  gint i=0;
  dest[0]=0;

  if (flags&PF_ALIGNWARN) { strcat(dest,"ALIGNWARN"); ++i; }
  if (flags&PF_STARTING)  { if (i) strcat(dest,",");
                            strcat(dest,"STARTING"); ++i; }
  if (flags&PF_EXITING)   { if (i) strcat(dest,",");
                            strcat(dest,"EXITING"); ++i; }
  if (flags&PF_PTRACED)   { if (i) strcat(dest,",");
                            strcat(dest,"PTRACED"); ++i; }
  if (flags&PF_TRACESYS)  { if (i) strcat(dest,",");
                            strcat(dest,"TRACESYS"); ++i; }
  if (flags&PF_FORKNOEXEC){ if (i) strcat(dest,",");
                            strcat(dest,"FORKNOEXEC"); ++i; }
  if (flags&PF_SUPERPRIV) { if (i) strcat(dest,",");
                            strcat(dest,"SUPERPRIV"); ++i; }
  if (flags&PF_DUMPCORE)  { if (i) strcat(dest,",");
                            strcat(dest,"DUMPCORE"); ++i; }
  if (flags&PF_SIGNALED)  { if (i) strcat(dest,",");
                            strcat(dest,"SIGNALED"); ++i; }
  if (flags&PF_MEMALLOC)  { if (i) strcat(dest,",");
                            strcat(dest,"MEMALLOC"); ++i; }
  if (flags&PF_VFORK)     { if (i) strcat(dest,",");
                            strcat(dest,"VFORK"); ++i; }
  if (flags&PF_USEDFPU)   { if (i) strcat(dest,",");
                            strcat(dest,"USEDFPU"); ++i; }
  if (flags&PF_DTRACE)    { if (i) strcat(dest,",");
                            strcat(dest,"DTRACE"); ++i; }
  if (!i)
    strcpy(dest,"(no flags set)");
}

char *LinuxProcFsProcessDetailsPoller::get_signal_string(unsigned long sigs) {
  gint i,j,k,m;
  unsigned long cs;
  static char buffer[512];

  buffer[0]=0;
  k=sizeof(sigs)*8;
  cs=sigs;
  j=0;
  for(i=0;i<k;i++) {
    if (cs&0x01) {
      m=sigindex(i+1);
      if (m!=-1) {
	if (j>0)
	  strcat(buffer,",");
	strcat(buffer,signames[m]);
	j++;
      }
    }
    cs>>=1;
  }
  if (!j)
    strcpy(buffer,"(none)");
  return buffer;
}

gint LinuxProcFsProcessDetailsPoller::sigindex(gint val) {
  gint i;
  for(i=0;i<COUNTSIG;i++)
    if (sigvalues[i]==val)
      return i;
  return(-1);
}

/* LinuxProcFsSystemInfoProvider */

LinuxProcFsSystemInfoProvider::LinuxProcFsSystemInfoProvider() {
  int i;
  last_busy=-1;
  last_idle=-1;

  // find out number of CPUs
  countCpus();  
  // CpuCount=4;

  for(i=0;i<CpuCount;i++)
    lbusy[i]=lidle[i]=-1;
}

void LinuxProcFsSystemInfoProvider::update() {
  update_cpu();

  if (CpuCount==1)
    SMP_faker();

  /*
  CpuCount=4;
  cpus_load[0]=cpu_usage;
  cpus_load[1]=cpu_usage/2.0;
  cpus_load[2]=cpu_usage/3.0;
  cpus_load[3]=cpu_usage/2.5;
  cpu_usage=(cpus_load[0]+cpus_load[1]+cpus_load[2]+cpus_load[3])/4.0;
  */

  update_memory();
}

void LinuxProcFsSystemInfoProvider::update_cpu() {
  FILE *f;
  long idle,busy,da,db;
  double fa,fb;
  char buffer[1024];
  char *p;
  int i,curcpu;
  TString t(1024);

  f=fopen("/proc/stat","r");
  if (f==NULL) {
    cpu_usage=0.0;
    for(i=0;i<CpuCount;i++)
      cpus_load[i]=0.0;
    return;
  }

  for(curcpu=0;curcpu<64;) {
    p=fgets(buffer,1023,f);
    if (p==NULL) break;

    // cpu average
    if (strncmp(buffer,"cpu ",4)==0) {
      t.set(buffer);
      p=t.token(T_SNT);     
      busy=0;
      for(i=0;i<3;i++) {
	p=t.token(T_SNT);
	busy+=(p==NULL)?0:gps_atoul(p);
      }
      idle=gps_atoul(t.token(T_SNT));      
      if (last_busy==-1) {
	last_busy=busy;
	last_idle=idle;
	cpu_usage=0.0;
	continue;
      }
      if ((idle==last_idle)&&(busy==last_busy)) {
	last_busy=busy;
	last_idle=idle;
	cpu_usage=0.0;
	continue;
      }
      da=busy-last_busy;
      db=idle-last_idle;
      fa=(double)da;
      fb=(double)db;
      fa/=fa+fb;
      if (fa>1.0) fa=1.0;      
      last_busy=busy;
      last_idle=idle;
      cpu_usage=fa;    // <--- result in cpu_usage
      continue;
    } // cpu average


    // SMP cpu info
    if (strncmp(buffer,"cpu",3)==0) {
      t.set(buffer);
      p=t.token(T_SNT);
      busy=0;
      for(i=0;i<3;i++) {
	p=t.token(T_SNT);
	busy+=(p==NULL)?0:gps_atoul(p);
      }
      idle=gps_atoul(t.token(T_SNT));      
      if (lbusy[curcpu]==-1) {
	lbusy[curcpu]=busy;
	lidle[curcpu]=idle;
	cpus_load[curcpu]=0.0;
	++curcpu;
	continue;
      }
      if ((idle==lidle[curcpu])&&(busy==lbusy[curcpu])) {
	lbusy[curcpu]=busy;
	lidle[curcpu]=idle;
	cpus_load[curcpu]=0.0;
	++curcpu;
	continue;
      }
      da=busy-lbusy[curcpu];
      db=idle-lidle[curcpu];
      fa=(double)da;
      fb=(double)db;
      fa/=fa+fb;
      if (fa>1.0) fa=1.0;      
      lbusy[curcpu]=busy;
      lidle[curcpu]=idle;
      cpus_load[curcpu]=fa;
      ++curcpu;
      continue;
    } // strncmp(buffer,"cpu",3)
  } // for(curcpu...)

  fclose(f);
}

void LinuxProcFsSystemInfoProvider::update_memory() {
  FILE *f;
  char buffer[1024];
  char *p;
  TString *t;

  f=fopen("/proc/meminfo","r");

  if (f==NULL) {
    memory_used=swap_used=1;
    memory_total=swap_total=1;
    memory_free=swap_free=0;
    return;
  }

  /* MEMORY */
  do {
    p=fgets(buffer,1024,f);
    if (p==NULL)
      break;
  } while((strstr(buffer,"Mem:")==NULL)&&(p!=NULL));

  if (p==NULL) {
    memory_used=swap_used=1;
    memory_total=swap_total=1;
    memory_free=swap_free=0;
    return;
  }

  t=new TString(1024);
  t->set(buffer);
  t->token(T_SNT);

  memory_total=gps_atoul(t->token(T_SNT));
  memory_used=gps_atoul(t->token(T_SNT));
  t->token(T_SNT);
  t->token(T_SNT);
  memory_used-=gps_atoul(t->token(T_SNT)); // buffers
  memory_used-=gps_atoul(t->token(T_SNT)); // cached

  memory_free=memory_total-memory_used; // Ohhhhhhhh...

  /* SWAP */
  do {
    p=fgets(buffer,1024,f);
    if (p==NULL)
      break;
  } while((strstr(buffer,"Swap:")==NULL)&&(p!=NULL));
  fclose(f);

  if (p==NULL) {
    swap_used=1;
    swap_total=1;
    swap_free=0;
    delete t;
    return;
  }

  t->set(buffer);
  t->token(T_SNT);
  swap_total=gps_atoul(t->token(T_SNT));
  swap_used=gps_atoul(t->token(T_SNT));
  swap_free=swap_total-swap_used;

  // test with ulf's 4-GB-swap-box values
  /*
  swap_total=4194840576;
  swap_used=34021376;
  swap_free=swap_total-swap_used;
  */

  delete t;
}

void LinuxProcFsSystemInfoProvider::countCpus() {
  FILE *f;
  char z[256];
  CpuCount=1;

  f=fopen("/proc/cpuinfo","r");
  if (f==NULL) return;

  CpuCount=0;
  while(fgets(z,255,f))
    if (strlen(z)>=10)
      if (strncmp(z,"processor\t",10)==0)
	CpuCount++;
  fclose(f);
}

#endif /* HAVELINUXPROCFS */
