#!/usr/bin/env python
#
# Copyright 2006,2007,2009,2011 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street,
# Boston, MA 02110-1301, USA.
# 

from gnuradio import gr, gru, uhd, optfir, eng_notation, blks2, pager
from gnuradio.eng_option import eng_option
from optparse import OptionParser
import sys

class app_top_block(gr.top_block):
    def __init__(self, options, queue):
	gr.top_block.__init__(self, "usrp_flex_all")

        if options.from_file is not None:
            self.u = gr.file_source(gr.sizeof_gr_complex, options.from_file)
            if options.verbose:
                print "Reading samples from file", options.from_file
        else:
            # Set up USRP source
            self.u = uhd.usrp_source(device_addr=options.address, stream_args=uhd.stream_args('fc32'))

            # Grab 1 MHz of spectrum
            # (A UHD facility to get sample rate range and granularity would be useful)
            self.u.set_samp_rate(1e6)
            rate = self.u.get_samp_rate()
            if rate != 1e6:
                print "Unable to set required sample rate of 1 Msps (got %f)" % rate
                sys.exit(1)
                
            # Tune daughterboard
            r = self.u.set_center_freq(options.freq+options.calibration, 0)
            if not r:
                frange = self.u.get_freq_range()
                sys.stderr.write(("\nRequested frequency (%f) out or range [%f, %f]\n") % \
                                     (freq, frange.start(), frange.stop()))
                sys.exit(1)

            # if no gain was specified, use the mid-point in dB
            if options.rx_gain is None:
                grange = self.u.get_gain_range()
                options.rx_gain = float(grange.start()+grange.stop())/2.0
                print "\nNo gain specified."
                print "Setting gain to %f (from [%f, %f])" % \
                    (options.rx_gain, grange.start(), grange.stop())
        
            self.u.set_gain(options.rx_gain, 0)


        taps = gr.firdes.low_pass(1.0,
                                  1.0,
                                  1.0/40.0*0.4,
                                  1.0/40.0*0.1,
                                  gr.firdes.WIN_HANN)

        if options.verbose:
            print "Channel filter has", len(taps), "taps"

        bank = blks2.analysis_filterbank(40, taps)
        self.connect(self.u, bank)

        if options.log and options.from_file == None:
            src_sink = gr.file_sink(gr.sizeof_gr_complex, 'usrp.dat')
            self.connect(self.u, src_sink)

        for i in range(40):
	    if i < 20:
		freq = options.freq+i*25e3
	    else:
		freq = options.freq-0.5e6+(i-20)*25e3

	    self.connect((bank, i), pager.flex_demod(queue, freq, options.verbose, options.log))
	    if options.log:
		self.connect((bank, i), gr.file_sink(gr.sizeof_gr_complex, 'chan_'+'%3.3f'%(freq/1e6)+'.dat'))


def get_options():
    parser = OptionParser(option_class=eng_option)

    parser.add_option('-f', '--freq', type="eng_float", default=None,
                      help="Set receive frequency to FREQ [default=%default]",
                      metavar="FREQ")
    parser.add_option("-a", "--address", type="string", default="addr=192.168.10.2",
                      help="Address of UHD device, [default=%default]")
    parser.add_option("-A", "--antenna", type="string", default=None,
                      help="select Rx Antenna where appropriate")
    parser.add_option("", "--rx-gain", type="eng_float", default=None,
                      help="set receive gain in dB (default is midpoint)")
    parser.add_option("-c",   "--calibration", type="eng_float", default=0.0,
                      help="set frequency offset to Hz", metavar="Hz")
    parser.add_option("-v", "--verbose", action="store_true", default=False)
    parser.add_option("-l", "--log", action="store_true", default=False,
                      help="log flowgraph to files (LOTS of data)")
    parser.add_option("-F", "--from-file", default=None,
                      help="read samples from file instead of USRP")

    (options, args) = parser.parse_args()

    if len(args) > 0:
	print "Run 'usrp_flex_band.py -h' for options."
	sys.exit(1)

    if (options.freq is None):
        sys.stderr.write("You must specify -f FREQ or --freq FREQ\n")
        sys.exit(1)
        
    return (options, args)

    
if __name__ == "__main__":

    (options, args) = get_options()

    queue = gr.msg_queue()
    tb = app_top_block(options, queue)
    runner = pager.queue_runner(queue)

    try:
        tb.run()
    except KeyboardInterrupt:
        pass

    runner.end()


