/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
#include <config.h>
#include <stdlib.h>
#include <time.h>
#include <glib.h>

#include "om-utils.h"

static int
parse_date (const char *input, int year, int month, int day,
	    int hour, int minute, int second)
{
	time_t ts;
	struct tm *tm;
	static char buffer[81];

	ts = om_utils_parse_iso8601 (input);
	if (ts == -1) {
		g_printerr ("ERROR: could not parse '%s'\n", input);
		return 1;
	}
	tm = gmtime (&ts);
	if (tm->tm_year + 1900 != year ||
	    tm->tm_mon + 1 != month ||
	    tm->tm_mday != day ||
	    tm->tm_hour != hour ||
	    tm->tm_min != minute ||
	    tm->tm_sec != second) {
		strftime (buffer, sizeof(buffer) - 1, "%Y-%m-%d %H:%M:%S", tm);
		g_printerr ("ERROR: '%s' parsed incorrectly\n", input);
		g_printerr ("  Expected %04d-%02d-%02d %02d:%02d:%02d (UTC), "
			    "got %s\n", year, month, day, hour, minute, second,
			    buffer);
		return 1;
	}
	return 0;
}

static int
parse_failure (const char *input)
{
	time_t ts;
	struct tm *tm;
	static char buffer[81];

	ts = om_utils_parse_iso8601 (input);
	if (ts == -1) {
		return 0;
	}
	tm = gmtime (&ts);
	strftime (buffer, sizeof(buffer) - 1, "%Y-%m-%d %H:%M:%S", tm);
	g_printerr ("ERROR: '%s' parsed while being invalid\n", input);
	g_printerr ("  Got %s\n", buffer);
	return 1;
}

int
main (int argc, char **argv)
{
	int result = 0;

	/* ensure we are in a known time zone */
	putenv("TZ=Australia/Perth");
	tzset();

	/* UTC tests. */
	result |= parse_date ("19700101T000000Z", 1970, 1, 1, 0, 0, 0);

	/* Non-UTC tests.  These are interpreted in Perth local time. */
	result |= parse_date ("19800101T000008", 1979, 12, 31, 16, 0, 8);

	/* Daylight savings: */
	result |= parse_date ("20070101T090000", 2007, 1, 1, 0, 0, 0);
	result |= parse_date ("20070701T080000", 2007, 7, 1, 0, 0, 0);

	/* Failure cases */
	result |= parse_failure("2001");
	result |= parse_failure("200101");
	result |= parse_failure("20010101");
	result |= parse_failure("20010101T");
	result |= parse_failure("20010101T00");
	result |= parse_failure("20010101T0000");
	result |= parse_failure("20010101t000000"); /* lower case 't' */

	return result;
}
