/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-input.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-info-provider.h"

static gboolean 
get_provider_matches (GdmDevice *device)
{
        return (gdm_device_test_capability (device, "input.mouse") || 
                gdm_device_test_capability (device, "input.keyboard") ||
                gdm_device_test_capability (device, "input.keypad") ||
                gdm_device_test_capability (device, "input.touchpad"));
}

static char *
get_icon_name (GdmDevice *device)
{
        const char *str;

        str = "input-keyboard";
        if (gdm_device_test_capability (device, "input.mouse")) {
                str = "input-mouse";
        } else if (gdm_device_test_capability (device, "input.touchpad")) {
                str = "input-tablet";
        }

        return g_strdup (str);
}

static char *
get_short_name (GdmDevice *device)
{
        const char *str;
        str = _("Keyboard");
        if (gdm_device_test_capability (device, "input.keypad") && 
            !gdm_device_test_capability (device, "input.keyboard")) {
                str = _("Keypad");
        } else if (gdm_device_test_capability (device, "input.mouse")) {
                str = _("Pointing Device");
        } else if (gdm_device_test_capability (device, "input.touchpad")) {
                str = _("Touch pad");
        }
        return g_strdup (str);
}

static char *
get_long_name (GdmDevice *device)
{
        return get_short_name (device);
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup ("");
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "input.product"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if (value != NULL) {                                                 \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, value);                               \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        const char *product;
        const char *device_file;
        GSList *p = NULL;

        product = gdm_device_get_property_string (device, "input.product");
        device_file = gdm_device_get_property_string (device, "input.device");

        ADD_SUM (p, _("Model"), g_strdup (product));
        ADD_SUM (p, _("Device File"), g_strdup (device_file));

        return p;
}

GdmInfoProviderIface gdm_info_provider_input =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
};

