with Ada.Calendar;              use Ada.Calendar;
with Ada.Command_Line;          use Ada.Command_Line;
with Ada.Exceptions;            use Ada.Exceptions;
with Ada.Strings.Fixed;         use Ada.Strings.Fixed;
with Ada.Text_IO;               use Ada.Text_IO;

with GNAT.Directory_Operations; use GNAT.Directory_Operations;
with GNAT.OS_Lib;               use GNAT.OS_Lib;

with ALI_Parser;                use ALI_Parser;
with Basic_Types;
with CPP_Parser;                use CPP_Parser;
with Config;                    use Config;
with Doc_Utils;                 use Doc_Utils;
with Entities;                  use Entities;
with Entities.Debug;            use Entities.Debug;
with Entities.Queries;          use Entities.Queries;
with Language.Ada;              use Language.Ada;
with Language.C;                use Language.C;
with Language.Cpp;              use Language.Cpp;
with Language_Handlers;         use Language_Handlers;
with Prj;                       use Prj;
with Prj.Ext;                   use Prj.Ext;
with Projects;                  use Projects;
with Projects.Registry;         use Projects.Registry;
with String_Utils;              use String_Utils;
with Test_Parse_Support;        use Test_Parse_Support;
with Traces;                    use Traces;
with VFS;                       use VFS;

--  This scripts takes a single arguments in parameter:
--     test_parse <command_file>
--  The syntax of the command file is line-oriented: empty lines are ignored,
--  as well as lines starting with exactly "--".
--  Comment lines are duplicated to the output.
--  Other lines must start with one of the following commands, and contain a
--  valid syntax for that line. Very few error handling is made in the parsing
--  of this file, and an invalid file will likely crash this program.
--  Valid commands are:
--      #!test_parse
--          This line is allowed as the first line, and permits the execution
--          of test.cmd as if it was a shell executable. This is only for
--          convenience, and doesn't change the run test
--      GENERATE_DB
--          Generate the on-disk cross-references database for languages where
--          this applies
--      CALL_GRAPH
--          Compute the call graph for all parsed files
--      PARSE_ALL [sources] [+call_graph] [+timing]
--          Parse all source files in the project. If the keyword "sources" is
--          specified, the list of files to parse is computed from the list of
--          source files.
--          If the keyword "+call_graph" is present, then the call graph is
--          also computed. This is exclusive with the previous parameter.
--          If the keyword "+timing" is present, then the total time it took
--          to compute will be displayed
--      PARSE source_file
--          Forces an update of the LI information for source_file
--      PAUSE delay
--          Pause for delay seconds. This is useful if you need to mesure
--          for instance the amount of memory occupied by the application, or
--          to make sure the timestamps of some ALI file will be different
--      DECL name:file:line:column
--          Outputs the declaration for the entity,
--          as "DECL: name:file:line:col"
--      BODY name:file:line:column
--          Outputs all the bodies for the entity
--      REFS name:file:line:column [include_overriding]
--          Outputs all known references to the entity in the first part
--          If include_overriding is set to 1, overriding primitives are also
--          returned
--      DEP file
--          Print files that file depends on
--      DEP_ON file
--          Print files depending on file
--      PARAMS name:file:line:column
--          Print the list of parameters for that entity
--      FULL_NAME name:file:line:column
--          Output the fully qualified name for the entity
--      SCENARIO variable value
--          Change the value of one of the scenario variables
--      CALLERS name:file:line:column
--          Print the list of entities that call that entity
--      CALLS name:file:line:column
--          Print the list of entities that are called by that entity
--      ENTITIES file
--          List all entities referenced in file that start with prefix
--      DOC name:file:line:column
--          Print the documentation for the entity
--      DUMP [output_file]
--          Dump the contents of the cross-references information file.
--      DUMP_ENTITY name:file:line:column
--          Dump the description of a specific entity
--      PROJECT project_name
--          Load a specific project. This replaces the currently loaded project
--          Initially, a default project is used, that only uses "." for the
--          source and object directory.
--      TOUCH file
--          Will force a reparsing of the LI information for this file
--      SHELL cmd
--          Execute a simple shell command
--      RESET
--          Reset the entities database. This might take some while, and
--          doesn't use the fast algorithm
--      LIST_ENTITIES
--          List all the entities defined in the project. This can be used in
--          addition to RESET to check that the latter performed its work as
--          expected

procedure Test_Parse is
   Me : constant Debug_Handle := Create ("TEST");
   Registry       : aliased Project_Registry;
   Db             : Entities_Database;
   LI_Not_Found   : File_Error_Reporter;
   ALI, CPP_LI    : LI_Handler;
   Handler        : Language_Handler;
   Loaded_Project : String_Access := null;

   procedure Load_Project (Name : String);
   procedure Parse_Cmd_File (Name : String);
   procedure Parse_Decl_Line (Text : String);
   procedure Parse_Body_Line (Text : String);
   procedure Parse_Refs_Line (Text : String);
   procedure Parse_Callers_Line (Text : String);
   procedure Parse_Calls_Line (Text : String);
   procedure Parse_Params_Line (Text : String);
   procedure Parse_Dep_On_Line (Text : String);
   procedure Parse_Doc_Line (Text : String);
   procedure Parse_Dep_Line (Text : String);
   procedure Parse_Entities (Text : String);
   procedure Parse_Dump (Text : String);
   procedure Parse_Dump_Entity (Text : String);
   procedure Parse_All (Text : String);
   procedure Parse_Full_Name (Text : String);
   procedure Parse_Shell (Text : String);
   procedure Parse_Scenario (Text : String);
   procedure Parse_Reset (Text : String);
   procedure Parse_List_Entities (Text : String);
   procedure Pause (For_Delay : String);
   function Image (J : Integer) return String;
   function Image (Loc : File_Location) return String;
   function Image (Entity : Entity_Information) return String;
   function Image (S : Source_File) return String;
   procedure Parse_Entity
     (Text     : String;
      Index    : in out Natural;
      Name     : out String_Access;
      File     : out Virtual_File;
      Line     : out Natural;
      Column   : out Basic_Types.Visible_Column_Type;
      Has_Name : Boolean := True);

   procedure Compute_Predefined_Paths;
   --  Compute the predefined paths for the runtime

   function Create_Lang_Handler return Language_Handler;
   --  Create a language handler

   ------------------------------
   -- Compute_Predefined_Paths --
   ------------------------------

   procedure Compute_Predefined_Paths is
      Gnatls       : constant String := "gnatls";
      Gnatls_Args  : Argument_List_Access :=
        Argument_String_To_List (Gnatls & " -v");
      GNAT_Version : String_Access;

   begin
      Compute_Predefined_Paths
        (Registry,
         GNAT_Version,
         Gnatls_Args);

      Free (Gnatls_Args);
   end Compute_Predefined_Paths;

   -------------------------
   -- Create_Lang_Handler --
   -------------------------

   function Create_Lang_Handler return Language_Handler is
      Handler : Language_Handler;
   begin
      Create_Handler (Handler);
      Prj.Initialize (Get_Tree (Get_Root_Project (Registry)));

      Register_Language_Handler (Db, Handler);

      ALI := Create_ALI_Handler (Db, Registry);
      Register_Language (Handler, Ada_Lang, ALI);
      Register_Default_Language_Extension (Registry, "ada", ".ads", ".adb");

      CPP_LI := Create_CPP_Handler (Db, Registry);
      Register_Language (Handler, C_Lang, CPP_LI);
      Register_Default_Language_Extension (Registry, "c", ".h", ".c");

      Register_Language (Handler, Cpp_Lang, CPP_LI);
      Register_Default_Language_Extension (Registry, "c++", ".h", ".cc");

      Set_Registry (Handler, Registry'Unrestricted_Access);
      return Handler;
   end Create_Lang_Handler;

   ------------------
   -- Load_Project --
   ------------------

   procedure Load_Project (Name : String) is
      New_Project_Loaded : Boolean;
      Success            : Boolean;
   begin
      if Loaded_Project = null or else Name /= Loaded_Project.all then
         Reset (Db);
      end if;

      Load (Registry, Create (Name), null, New_Project_Loaded, Success);
      Free (Loaded_Project);
      Loaded_Project := new String'(Name);
      Recompute_View (Registry, null);
      Compute_Predefined_Paths;
      On_Project_View_Changed (CPP_LI);
   end Load_Project;

   -----------
   -- Image --
   -----------

   function Image (Entity : Entity_Information) return String is
   begin
      if Entity = null then
         return "<error>";
      else
         return Get_Name (Entity).all & ':'
                & Image (Get_Declaration_Of (Entity));
      end if;
   end Image;

   -----------
   -- Image --
   -----------

   function Image (J : Integer) return String is
      S : constant String := Integer'Image (J);
   begin
      if J < 0 then
         return S;
      else
         return S (S'First + 1 .. S'Last);
      end if;
   end Image;

   -----------
   -- Image --
   -----------

   function Image (S : Source_File) return String is
   begin
      return Base_Name (Get_Filename (S));
   end Image;

   -----------
   -- Image --
   -----------

   function Image (Loc : File_Location) return String is
   begin
      return Base_Name (Get_Filename (Get_File (Loc)))
        & ':' & Image (Get_Line (Loc)) & ':'
        & Image (Integer (Get_Column (Loc)));
   end Image;

   ------------------
   -- Parse_Entity --
   ------------------

   procedure Parse_Entity
     (Text     : String;
      Index    : in out Natural;
      Name     : out String_Access;
      File     : out Virtual_File;
      Line     : out Natural;
      Column   : out Basic_Types.Visible_Column_Type;
      Has_Name : Boolean := True)
   is
      Start : Natural := Index;
   begin
      if Has_Name then
         while Text (Index) /= ':' loop
            Index := Index + 1;
         end loop;

         Name := new String'(Text (Start .. Index - 1));

         Index := Index + 1;
         Start := Index;
      end if;

      while Text (Index) /= ':' loop
         Index := Index + 1;
      end loop;

      File := Create (Normalize_Pathname (Text (Start .. Index - 1)));

      Index := Index + 1;
      Start := Index;
      while Text (Index) /= ':' loop
         Index := Index + 1;
      end loop;

      Line := Integer'Value (Text (Start .. Index - 1));

      Index := Index + 1;
      Start := Index;
      while Index <= Text'Last and then Text (Index) /= ' ' loop
         Index := Index + 1;
      end loop;

      Column :=
        Basic_Types.Visible_Column_Type'Value (Text (Start .. Index - 1));
   end Parse_Entity;

   ---------------------
   -- Parse_Decl_Line --
   ---------------------

   procedure Parse_Decl_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);
      Put_Line ("DECL: " & Image (Entity) & ' ' & Status'Img);
   end Parse_Decl_Line;

   ---------------------
   -- Parse_Body_Line --
   ---------------------

   procedure Parse_Body_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
      Loc    : File_Location := No_File_Location;
      First_Loc : File_Location := No_File_Location;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      loop
         Find_Next_Body (Entity, Current_Location => Loc, Location => Loc);
         exit when Loc = First_Loc;

         if First_Loc = No_File_Location then
            First_Loc := Loc;
         end if;

         Put_Line ("BODY: " & Image (Loc) & ' ' & Status'Img);
      end loop;
   end Parse_Body_Line;

   ----------------
   -- Parse_Dump --
   ----------------

   procedure Parse_Dump (Text : String) is
      File : File_Type;
      procedure Output_File_Line (S : String);
      procedure Output_File (S : String);
      procedure Output_File (S : String) is
      begin
         Put (File, S);
      end Output_File;

      procedure Output_File_Line (S : String) is
      begin
         Put_Line (File, S);
      end Output_File_Line;

   begin
      if Text = "" then
         Set_Default_Output;
      else
         Create (File, Out_File, Text);
         Entities.Debug.Output_Line := Output_File_Line'Unrestricted_Access;
         Entities.Debug.Output := Output_File'Unrestricted_Access;
      end if;
      Dump (Db);
      Set_Default_Output;
   end Parse_Dump;

   -----------------------
   -- Parse_Dump_Entity --
   -----------------------

   procedure Parse_Dump_Entity (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("Couldn't find the entity " & Text);
      else
         Set_Default_Output;
         Dump (Entity, Full => True, Name => "");
      end if;
   end Parse_Dump_Entity;

   --------------------
   -- Parse_Scenario --
   --------------------

   procedure Parse_Scenario (Text : String) is
      Start : Integer := Text'First;
   begin
      while Text (Start) /= ' ' loop
         Start := Start + 1;
      end loop;
      Prj.Ext.Add (Text (Text'First .. Start - 1),
                   Text (Start + 1 .. Text'Last));
      Recompute_View (Registry, null);
      Compute_Predefined_Paths;
      On_Project_View_Changed (CPP_LI);
   end Parse_Scenario;

   ---------------------
   -- Parse_Refs_Line --
   ---------------------

   procedure Parse_Refs_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;

      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
      Ref     : Entity_Reference := No_Entity_Reference;
      Iter   : Entity_Reference_Iterator;
      Renamed : Entity_Information;
      Include_Overriding : Boolean;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      Index := Index + 1;
      Include_Overriding := Index <= Text'Last and then Text (Index) = '1';

      if Entity = null then
         Put_Line ("Couldn't find entity ");
      else
         Renamed := Renaming_Of (Entity);
         if Renamed /= null then
            Put_Line ("RENAMES: " & Image (Renamed));
         end if;

         Find_All_References
           (Iter,
            Entity                => Entity,
            File_Has_No_LI_Report => null,
            In_File               => null,
            Include_Overriding    => Include_Overriding);
         while not At_End (Iter) loop
            Ref := Get (Iter);
            if Ref /= No_Entity_Reference then
               Put_Line ("REF: " & Image (Get_Location (Ref)));
            end if;

            Next (Iter);
         end loop;

         Destroy (Iter);
      end if;
   end Parse_Refs_Line;

   -----------------------
   -- Parse_Dep_On_Line --
   -----------------------

   procedure Parse_Dep_On_Line (Text : String) is
      File : constant Virtual_File := Create (Normalize_Pathname (Text));
      Iter : Dependency_Iterator;
      Dep  : Source_File;
   begin
      Find_Ancestor_Dependencies
        (Iter,
         File           => Get_Or_Create (Db, File, Get_LI_Handler (Db, File)),
         File_Has_No_LI_Report => LI_Not_Found);

      while not At_End (Iter) loop
         Dep := Get (Iter);
         if Dep /= null then
            Put_Line ("DEP_ON (" & Base_Name (File) & "): " & Image (Dep)
                      & " Explicit=" & Is_Explicit (Iter)'Img);
         end if;

         Next (Iter);
      end loop;

      Destroy (Iter);
   end Parse_Dep_On_Line;

   --------------------
   -- Parse_Dep_Line --
   --------------------

   procedure Parse_Dep_Line (Text : String) is
      File : constant Virtual_File := Create (Normalize_Pathname (Text));
      Iter : File_Dependency_Iterator;
   begin
      Find_Dependencies
        (Iter,
         File           => Get_Or_Create (Db, File, Get_LI_Handler (Db, File)),
         File_Has_No_LI_Report => LI_Not_Found);

      while not At_End (Iter) loop
         Put_Line ("DEP (" & Base_Name (File) & "): " & Image (Get (Iter))
                   & " Explicit=" & Is_Explicit (Iter)'Img);
         Next (Iter);
      end loop;
   end Parse_Dep_Line;

   -----------------
   -- Parse_Reset --
   -----------------

   procedure Parse_Reset (Text : String) is
      pragma Unreferenced (Text);
      Files  : constant File_Array_Access := Get_Source_Files
        (Get_Root_Project (Registry), Recursive => True);
      Predefined : constant File_Array_Access :=
        Get_Predefined_Source_Files (Registry);
      Handler : LI_Handler;
      Source : Source_File;
   begin
      for F in Files'Range loop
         Source := Get_Or_Create
           (Db   => Db,
            File => Files (F),
            Handler => Get_LI_Handler (Db, Files (F)),
            Allow_Create => False);
         if Source /= null then
            Reset (Source);
         end if;
      end loop;

      for F in Predefined'Range loop
         Handler := Get_LI_Handler (Db, Predefined (F));
         if Handler /= null then
            Source := Get_Or_Create
              (Db           => Db,
               File         => Predefined (F),
               Handler      => Handler,
               Allow_Create => False);
            if Source /= null then
               Reset (Source);
            end if;
         end if;
      end loop;
   end Parse_Reset;

   -------------------------
   -- Parse_List_Entities --
   -------------------------

   procedure Parse_List_Entities (Text : String) is
      pragma Unreferenced (Text);
   begin
      Dump (Db, Full => False, Entities_Only => True);
      New_Line;
   end Parse_List_Entities;

   -----------------
   -- Parse_Shell --
   -----------------

   procedure Parse_Shell (Text : String) is
      First         : constant Natural := Text'First;
      Last          : constant Natural := Text'Last;
      Command_Start : Natural := First;
      Command_End   : Natural;
      Success       : Boolean;
      Command       : String_Access;
      Arguments     : Argument_List_Access;
   begin
      Put_Line ("EXECUTING " & Text);

      Skip_Blanks (Text, Command_Start);

      if Command_Start >= Last then
         return;
      end if;

      Command_End := Command_Start;
      Skip_To_Blank (Text, Command_End);
      Command_End := Command_End - 1;

      Command   := Locate_Exec_On_Path (Text (Command_Start .. Command_End));
      Arguments := Argument_String_To_List (Text (Command_End + 2 .. Last));

      if Command = null then
         Put_Line
           ("Cannot locate <" & Text (Command_Start .. Command_End)
            & "> on path");
         return;
      end if;

      if Arguments = null then
         Put_Line
           ("Cannot parse the arguments string <"
            & Text (Command_Start .. Command_End) & "<");
         return;
      end if;

      Spawn
        (Command.all,
         Arguments.all,
         Success);

      Free (Arguments);
      Free (Command);

   exception
      when E : others =>
         Put_Line ("Unexpected exception: " & Exception_Information (E));
   end Parse_Shell;

   --------------------
   -- Parse_Doc_Line --
   --------------------

   procedure Parse_Doc_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("DOC <unknown entity>");
      else
         Put_Line ("DOC " & Get_Name (Entity).all);
         Put_Line (Get_Documentation (Handler, Entity));
      end if;
   end Parse_Doc_Line;

   ----------------------
   -- Parse_Calls_Line --
   ----------------------

   procedure Parse_Calls_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
      Iter   : Calls_Iterator;
      Called : Entity_Information;
      Refs   : Entity_Reference_Iterator;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("CALLS <unknown entity>");
      else
         Put_Line ("CALLS " & Get_Name (Entity).all);

         Iter := Get_All_Called_Entities (Entity);
         while not At_End (Iter) loop
            Called := Get (Iter);

            Find_All_References
              (Iter     => Refs,
               Entity   => Called,
               In_Scope => Entity);

            Put ("  => " & Image (Called) & " (Refs: ");

            while not At_End (Refs) loop
               if Get (Refs) /= No_Entity_Reference then
                  Put (Image (Get_Location (Get (Refs))) & ' ');
               end if;
               Next (Refs);
            end loop;

            Destroy (Refs);

            Put_Line (")");

            Next (Iter);
         end loop;
      end if;

      Destroy (Iter);
   end Parse_Calls_Line;

   ---------------------
   -- Parse_Full_Name --
   ---------------------

   procedure Parse_Full_Name (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);
      if Entity = null then
         Put_Line ("ENTITY NOT FOUND " & Name.all & ' ' & Status'Img);
      else
         Put_Line ("FULL_NAME: " & Get_Full_Name (Entity));
      end if;
   end Parse_Full_Name;

   ------------------------
   -- Parse_Callers_Line --
   ------------------------

   procedure Parse_Callers_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
      Iter   : Entity_Reference_Iterator;
      Ref    : Entity_Reference;
      Caller : Entity_Information;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("ENTITY NOT FOUND " & Name.all & ' ' & Status'Img);
         return;
      end if;

      Put_Line ("CALLERS " & Image (Entity));

      Find_All_References
        (Iter                  => Iter,
         Entity                => Entity,
         File_Has_No_LI_Report => null);
      while not At_End (Iter) loop
         Ref := Get (Iter);
         if Ref /= No_Entity_Reference then
            Caller := Get_Caller (Ref);
            if Caller /= null then
               Put_Line ("  REF: " & Image (Get_Location (Ref))
                         & " => " & Image (Caller));
            end if;
         end if;

         Next (Iter);
      end loop;

      Destroy (Iter);
   end Parse_Callers_Line;

   --------------------
   -- Parse_Entities --
   --------------------

   procedure Parse_Entities (Text : String) is
      VF : constant Virtual_File :=
         Create (Base_Name (Text),
                 Registry,
                 Use_Object_Path => False);
      File : constant Source_File :=
        Get_Source_Info (Get_LI_Handler (Db, VF), VF);
      Iter : Entity_Iterator;
   begin
      Find_All_Entities_In_File
        (Iter   => Iter,
         File   => File);

      Put_Line ("ENTITIES " & Text);
      while not At_End (Iter) loop
         Put_Line ("  " & Image (Get (Iter)));
         Next (Iter);
      end loop;

      Destroy (Iter);
   end Parse_Entities;

   -----------------------
   -- Parse_Params_Line --
   -----------------------

   procedure Parse_Params_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line   : Natural;
      Column : Basic_Types.Visible_Column_Type;
      Entity, Param : Entity_Information;
      Iter   : Subprogram_Iterator;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      Iter := Get_Subprogram_Parameters (Entity, LI_Not_Found);
      loop
         Get (Iter, Param);
         exit when Param = null;

         Put_Line ("Param (" & Get_Name (Entity).all & "): "
                   & Image (Param) & " as " & Get_Type (Iter)'Img);

         Next (Iter);
      end loop;
   end Parse_Params_Line;

   ---------------
   -- Parse_All --
   ---------------

   procedure Parse_All (Text : String) is
      Parse_From_Sources : constant Boolean :=
        Index (Text, "sources") >= Text'First;
      Call_Graph         : constant Boolean :=
        Index (Text, "+call_graph") >= Text'First;
      Show_Timing        : constant Boolean :=
        Index (Text, "+timing") >= Text'First;
      Start  : Time := Clock;
      Count  : Natural := 0;
      Files  : File_Array_Access;
      Source : Source_File;
      pragma Unreferenced (Source);
   begin
      if Parse_From_Sources then
         Files := Get_Source_Files
           (Get_Root_Project (Registry), Recursive => True);
         Start := Clock;
         if Files /= null then
            for F in Files'Range loop
               Source := Get_Source_Info
                 (Get_LI_Handler (Db, Files (F)), Files (F));
               Count  := Count + 1;
            end loop;
         end if;

         if Show_Timing then
            Put_Line ("Time=" & Duration'Image (Clock - Start)
                      & " (" & Count'Img & " files)");
         end if;

      else
         Count := Parse_All_LI_Information
           (Handler   => ALI,
            Project   => Get_Root_Project (Registry),
            Recursive => True);
         Put_Line ("Time for Ada=" & Duration'Image (Clock - Start)
                   & " (" & Count'Img & " files)");
         Start := Clock;
         Count := Parse_All_LI_Information
           (Handler   => CPP_LI,
            Project   => Get_Root_Project (Registry),
            Recursive => True);

         if Show_Timing then
            Put_Line ("Time for C=" & Duration'Image (Clock - Start)
                      & " (" & Count'Img & " files)");
         end if;
      end if;

      if Call_Graph then
         Start := Clock;
         Compute_All_Call_Graphs (Db);

         if Show_Timing then
            Put_Line ("Time for Callgraph=" & Duration'Image (Clock - Start));
         end if;
      end if;
   end Parse_All;

   -----------
   -- Pause --
   -----------

   procedure Pause (For_Delay : String) is
      D : constant Duration := Duration'Value (For_Delay);
   begin
      delay D;
   end Pause;

   --------------------
   -- Parse_Cmd_File --
   --------------------

   procedure Parse_Cmd_File (Name : String) is
      File   : File_Type;
      Line   : String (1 .. 5096);
      Last   : Integer;
      VF     : Virtual_File;
      Source : Source_File;
   begin
      Open (File, In_File, Name);

      while not End_Of_File (File) loop
         Get_Line (File, Line, Last);

         Trace (Me, Line (Line'First .. Last));

         if Last < 1
           or else Line (Line'First .. Line'First + 1) = "--"
           or else Line (Line'First .. Line'First + 1) = "#!"
         then
            Put_Line (Line (Line'First .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "DECL "
         then
            Parse_Decl_Line (Line (Line'First + 5 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "BODY "
         then
            Parse_Body_Line (Line (Line'First + 5 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "REFS "
         then
            Parse_Refs_Line (Line (Line'First + 5 .. Last));

         elsif Last > 9
           and then Line (Line'First .. Line'First + 8) = "ENTITIES "
         then
            Parse_Entities (Line (Line'First + 9 .. Last));

         elsif Last > 6
           and then Line (Line'First .. Line'First + 5) = "PAUSE "
         then
            Pause (Line (Line'First + 6 .. Last));

         elsif Last >= 9
           and then Line (Line'First .. Line'First + 8) = "PARSE_ALL"
         then
            if Last >= Line'First + 10 then
               Parse_All (Line (Line'First + 10 .. Last));
            else
               Parse_All ("");
            end if;

         elsif Last > 6
           and then Line (Line'First .. Line'First + 5) = "PARSE "
         then
            VF := Create (Base_Name (Line (Line'First + 6 .. Last)),
                          Registry,
                          Use_Object_Path => False);
            Source := Get_Source_Info (Get_LI_Handler (Db, VF), VF);

         elsif Last = 11
           and then Line (Line'First .. Line'First + 10) = "GENERATE_DB"
         then
            --  ??? Should do this for all registered handlers
            declare
               LI_Iterator : LI_Handler_Iterator'Class :=
                 Generate_LI_For_Project
                   (CPP_LI, Handler,
                    Get_Root_Project (Registry), null, True); -- recursive
               Finished    : Boolean;
            begin
               loop
                  Continue (LI_Iterator, null, Finished);
                  exit when Finished;
                  delay 0.01;
               end loop;
               Destroy (LI_Iterator);
            end;

         elsif Last = 4
           and then Line (Line'First .. Line'First + 3) = "DUMP"
         then
            Parse_Dump ("");

         elsif Last >= 5
           and then Line (Line'First .. Line'First + 4) = "DUMP "
         then
            Parse_Dump (Line (Line'First + 5 .. Last));

         elsif Last = 5
           and then Line (Line'First .. Line'First + 4) = "RESET"
         then
            Parse_Reset (Line (Line'First + 5 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "SHELL"
         then
            Parse_Shell (Line (Line'First + 6 .. Last));

         elsif Last > 11
           and then Line (Line'First .. Line'First + 11) = "DUMP_ENTITY "
         then
            Parse_Dump_Entity (Line (Line'First + 12 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 5) = "TOUCH "
         then
            VF := Create (Base_Name (Line (Line'First + 6 .. Last)),
                          Registry,
                          Use_Object_Path => False);
            Source := Get_Or_Create (Db, VF, Get_LI_Handler (Db, VF));
            Set_Time_Stamp (Source,
                            Time_Of (Year  => Year_Number'First,
                                     Month => Month_Number'First,
                                     Day   => Day_Number'First));
            Reset (Source);

            if Get_LI (Source) /= null then
               Set_Time_Stamp
                 (Get_LI (Source),
                  Time_Of (Year  => Year_Number'First,
                           Month => Month_Number'First,
                           Day   => Day_Number'First));
            end if;

         elsif Line (Line'First .. Last) = "CALL_GRAPH" then
            Compute_All_Call_Graphs (Db);

         elsif Last > 6
           and then Line (Line'First .. Line'First + 6) = "DEP_ON "
         then
            Parse_Dep_On_Line (Line (Line'First + 7 .. Last));

         elsif Last > 3
           and then Line (Line'First .. Line'First + 3) = "DEP "
         then
            Parse_Dep_Line (Line (Line'First + 4 .. Last));

         elsif Last > 7
           and then Line (Line'First .. Line'First + 6) = "PARAMS "
         then
            Parse_Params_Line (Line (Line'First + 7 .. Last));

         elsif Last > 8
           and then Line (Line'First .. Line'First + 7) = "CALLERS "
         then
            Parse_Callers_Line (Line (Line'First + 8 .. Last));

         elsif Last > 10
           and then Line (Line'First .. Line'First + 9) = "FULL_NAME "
         then
            Parse_Full_Name (Line (Line'First + 10 .. Last));

         elsif Last > 9
           and  then Line (Line'First .. Line'First + 8) = "SCENARIO "
         then
            Parse_Scenario (Line (Line'First + 9 .. Last));

         elsif Last > 6
           and then Line (Line'First .. Line'First + 5) = "CALLS "
         then
            Parse_Calls_Line (Line (Line'First + 6 .. Last));

         elsif Last = 13
           and then Line (Line'First .. Line'First + 12) = "LIST_ENTITIES"
         then
            Parse_List_Entities (Line (Line'First + 13 .. Last));

         elsif Last > 4
           and then Line (Line'First .. Line'First + 3) = "DOC "
         then
            Parse_Doc_Line (Line (Line'First + 4 .. Last));

         elsif Last > 7
           and then Line (Line'First .. Line'First + 7) = "PROJECT "
         then
            Load_Project (Line (Line'First + 8 .. Last));

         else
            Put_Line ("Unknown command line: " & Line (Line'First .. Last));
         end if;
      end loop;

      Close (File);
   end Parse_Cmd_File;

begin
   Parse_Config_File (".gnatdebug");
   LI_Not_Found := new Test_LI_Not_Found_Record;

   Projects.Registry.Initialize;

   Db := Create (Registry'Unchecked_Access);

   Load_Empty_Project (Registry);
   Handler := Create_Lang_Handler;

   Recompute_View (Registry, null);
   Compute_Predefined_Paths;

   Set_Show_Timestamp (False);

   if Getenv ("GPS_ROOT").all = "" then
      declare
         Msg : constant String := Set_Executables (Config.Prefix, CPP_LI);
      begin
         if Msg /= "" then
            Put_Line ("Error: " & Msg);
            return;
         end if;
      end;
   else
      declare
         Msg : constant String :=
           Set_Executables (Getenv ("GPS_ROOT").all, CPP_LI);
      begin
         if Msg /= "" then
            Put_Line ("Error: " & Msg);
            return;
         end if;
      end;
   end if;

   if Argument_Count > 1 then
      Put_Line ("Usage: " & Command_Name & " <cmd_file>");
      return;
   elsif Argument_Count = 1 then
      Parse_Cmd_File (Argument (1));
   else
      Parse_Cmd_File ("test.cmd");
   end if;

   --  Force a proper reset. That helps detect inconsistencies in the xref
   --  database.
   Parse_Reset ("");

   --  Free the memory anyway
   Destroy (Db);
   Traces.Finalize;

end Test_Parse;
