#include <glib-object.h>

#include <gmyth/gmyth_backendinfo.h>
#include <gmyth/gmyth_remote_util.h>
#include <gmyth/gmyth_query.h>
#include <gmyth/gmyth_epg.h>
#include <gmyth/gmyth_common.h>

#include "common.h"

static          gboolean
test_recorder_availability(GMythBackendInfo * backend_info)
{
    GMythRecorder  *recorder;
    GMythSocket    *socket = gmyth_socket_new();

    if (gmyth_socket_connect_to_backend(socket,
                                        gmyth_backend_info_get_hostname
                                        (backend_info),
                                        gmyth_backend_info_get_port
                                        (backend_info), TRUE) == FALSE) {
        g_debug("Test recorder failed: Connection failed");
        return FALSE;
    }

    recorder = remote_request_next_free_recorder(socket, -1);
    gmyth_socket_close_connection(socket);

    if (socket != NULL)
        g_object_unref(socket);

    if (recorder == NULL) {
        g_debug("Recorder not available\n");
        return FALSE;
    }

    g_debug("Recorder found (num): %d", recorder->recorder_num);

    if (recorder != NULL)
        g_object_unref(recorder);

    return TRUE;
}

static          gboolean
test_recorder_setup(GMythBackendInfo * backend_info)
{
    GMythQuery     *query = gmyth_query_new();

    if (gmyth_query_connect_with_timeout(query, backend_info, 3) == TRUE) {
        g_debug("Mysql connection success");
        return TRUE;
    } else {
        g_debug("Mysql connection failed");
        return FALSE;
    }

    if (query != NULL)
        g_object_unref(query);

}

static          gboolean
test_recorder_check_channels(GMythBackendInfo * backend_info)
{
    GMythRecorder  *recorder;
    GMythSocket    *socket = gmyth_socket_new();
    GMythEPG       *epg = gmyth_epg_new();
    GList          *clist;
    gint            i,
                    length;

    // Gets the free recorder
    if (gmyth_socket_connect_to_backend(socket,
                                        gmyth_backend_info_get_hostname
                                        (backend_info),
                                        gmyth_backend_info_get_port
                                        (backend_info), TRUE) == FALSE) {
        g_debug("Test recorder failed: Connection failed");
        return FALSE;
    }

    recorder = remote_request_next_free_recorder(socket, -1);
    gmyth_socket_close_connection(socket);
    g_object_unref(socket);

    if (recorder == NULL) {
        g_debug("[%s] Recorder not available", __FUNCTION__);
        return FALSE;
    }
    // Connects the recorder socket
    gmyth_recorder_setup(recorder);

    // Gets the list of channels
    if (!gmyth_epg_connect(epg, backend_info)) {
        g_debug("%s: Not connected\n", __FUNCTION__);
        return FALSE;
    }

    length = gmyth_epg_get_channel_list(epg, &clist);
    gmyth_epg_disconnect(epg);
    g_object_unref(epg);

    g_print("==== Verifying the %d channels found in the EPG ====\n",
            length);
    for (i = 0; i < length; i++) {
        GMythChannelInfo *channel_info =
            (GMythChannelInfo *) g_list_nth_data(clist, i);
        gboolean        res;

        // Checks the channels
        res =
            gmyth_recorder_check_channel(recorder,
                                         channel_info->channel_ID);
        g_debug("Channel %d %s", channel_info->channel_ID,
                res ? "Found" : "Not found");
    }

    g_list_free(clist);

    if (recorder != NULL)
        g_object_unref(recorder);

}

int
main(int args, const char **argv)
{

    GMythBackendInfo *backend_info;
    g_type_init();
    g_thread_init(NULL);

    backend_info = gmyth_backend_info_new_with_uri(argv[1]);

    printf("******** Testing recorder availability ***********\n");
    test_recorder_availability(backend_info);

    printf
        ("******** Testing recorder check channels function ***********\n");
    test_recorder_check_channels(backend_info);

    fprintf(stdout, SYNC_STRING);
    fflush(NULL);
    getchar();

    if (backend_info != NULL)
        g_object_unref(backend_info);

    return (0);

}
