/*
 *
 *  $Id: icontextoestudio.cpp 4575 2012-01-20 14:21:44Z tovar $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 */
#include <wx/mstream.h>
#include <main/controllers/configurationcontroller.h>
#include <wx/filename.h>
#include <wx/xml/xml.h>
#include <wx/file.h>

#include <api/globals.h>
#include <api/ientorno.h>
#include <api/ivista.h>
#include <api/icontextoestudio.h>
#include <api/imodelointegracion.h>

#include <api/dicom/idicomizador.h>
#include <api/dicom/idicommanager.h>

#include <api/controllers/ipacscontroller.h>
#include <api/icontroladorhistorial.h>
#include <api/icontroladorherramientas.h>
#include <api/icontroladorlog.h>

#include <main/controllers/controladorcarga.h>
#include <main/controllers/controladorlog.h>
#include <export/tagsprivadoscomunes.h>



#undef __DEPRECATED
#include <vtkginkgoimageviewer.h>
#include <vtkExecutive.h>
#include <vtkImageData.h>
#include <vtkDataObject.h>
#include <vtkAlgorithmOutput.h>
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkUnsignedCharArray.h>
#include <vtkPointData.h>
#include <vtkCharArray.h>

#define _GINKGO_TRACE

GNC::GCS::IContextoEstudio::IContextoEstudio()
{
	GTRACE(">> IContextoEstudio::IContextoEstudio() " << this)

	Entorno             = NULL;
	Vista               = NULL;
	Viewer              = NULL;
	Modulo              = NULL;
	Ventana             = NULL;
	VentanaPadre        = NULL;

	Loader              = GNC::GCS::ControladorCarga::Instance()->NewLoader();

	renderConnection    = Loader->GetOutputPort();

	IndiceFicheroActivo = -1;
	IgnorarModificaciones = false;

	GTRACE("<< IContextoEstudio::IContextoEstudio() " << this)
}

GNC::GCS::IContextoEstudio::IContextoEstudio(const GNC::GCS::IContextoEstudio& o)
{
	GTRACE(">> IContextoEstudio::IContextoEstudio(IContextoEstudio& o) " << this)
	*this = o;
	GTRACE("<< IContextoEstudio::IContextoEstudio(IContextoEstudio& o) " << this)
}

GNC::GCS::IContextoEstudio::IContextoEstudio(const GNC::GCS::IContextoEstudio* o)
{
	GTRACE(">> IContextoEstudio::IContextoEstudio(IContextoEstudio* o) " << this)
	if (o != NULL) {
		*this = *o;
	}
	else {
		Entorno = NULL;
		Vista = NULL;
		Modulo = NULL;
		Ventana = NULL;
		VentanaPadre = NULL;
		Viewer       = NULL;
		Loader = GNC::GCS::ControladorCarga::Instance()->NewLoader();
	}
	GTRACE("<< IContextoEstudio::IContextoEstudio(IContextoEstudio* o) " << this)
}

GNC::GCS::IContextoEstudio::~IContextoEstudio()
{
	GTRACE(">> IContextoEstudio::~IContextoEstudio() " << this);
	GTRACE("<< IContextoEstudio::~IContextoEstudio() " << this);
}

GnkPtr<GNC::GCS::IContextoEstudioReferido> GNC::GCS::IContextoEstudio::NewRef(const GnkPtr<GNC::GCS::IContextoEstudio>& parent)
{
	GnkPtr<IContextoEstudioReferido> reference = new IContextoEstudioReferido();

	reference->Entorno               = parent->Entorno;
	reference->Vista                 = parent->Vista;
	reference->Modulo                = parent->Modulo;
	reference->Ventana               = parent->Ventana;
	reference->VentanaPadre          = parent->VentanaPadre;

	reference->IndiceFicheroActivo   = parent->IndiceFicheroActivo;

	reference->Ficheros              = parent->Ficheros;

	reference->ModeloIntegracion     = parent->ModeloIntegracion;

	reference->IgnorarModificaciones = parent->IgnorarModificaciones;

	reference->ModoFuncionamiento    = parent->ModoFuncionamiento;
	reference->UIDEstudioDiagnostico = parent->UIDEstudioDiagnostico;

	reference->ListaUIDsSerie        = parent->ListaUIDsSerie;

	reference->EstudioPadre          = parent;

	return reference;
}

GNC::GCS::IContextoEstudio& GNC::GCS::IContextoEstudio::operator = (const GNC::GCS::IContextoEstudio& o)
{
	Entorno               = o.Entorno;
	Vista                 = o.Vista;
	Modulo                = o.Modulo;
	Ventana               = o.Ventana;
	VentanaPadre          = o.VentanaPadre;

	Loader                = o.Loader;

	renderConnection      = Loader->GetOutputPort();

	IndiceFicheroActivo   = o.IndiceFicheroActivo;

	Ficheros              = o.Ficheros;

	ModeloIntegracion     = o.ModeloIntegracion;

	IgnorarModificaciones = o.IgnorarModificaciones;

	ModoFuncionamiento    = o.ModoFuncionamiento;
	UIDEstudioDiagnostico = o.UIDEstudioDiagnostico;

	ListaUIDsSerie.clear();
	for (std::list<std::string>::const_iterator it = o.ListaUIDsSerie.begin(); it != o.ListaUIDsSerie.end(); ++it) {
		ListaUIDsSerie.push_back((*it));
	}

	return *this;
}

bool GNC::GCS::IContextoEstudio::GetIgnorarModificaciones()
{
	return IgnorarModificaciones;
}

void GNC::GCS::IContextoEstudio::SetIgnorarModificaciones(bool ignorar)
{
	IgnorarModificaciones = ignorar;
}

vtkSmartPointer<vtkAlgorithmOutput> GNC::GCS::IContextoEstudio::GetLoaderOutputConnection()
{
	return vtkSmartPointer<vtkAlgorithmOutput>(Loader->GetOutputPort());
}

void GNC::GCS::IContextoEstudio::SetRendererInputConnection(const vtkSmartPointer<vtkAlgorithmOutput>& input)
{
	renderConnection = input;
}

void GNC::GCS::IContextoEstudio::SetViewer(const vtkSmartPointer<vtkGinkgoImageViewer>& viewer)
{
	Viewer = viewer;
	if (Viewer != NULL) {
		Viewer->Loader = Loader;
	}
}

void GNC::GCS::IContextoEstudio::SetIndiceActivo(int indice)
{
	TFicheroEstudio& fichero = (*Ficheros[indice]);

	IndiceFicheroActivo = indice;
	Loader->SetInput(fichero.RutaImagen);
	Loader->UpdateInformation();
	double spacing[3] = {1.0f, 1.0f, 1.0f};
	GetSpacingActiva(spacing[0], spacing[1], spacing[2]);
	Loader->SetOutputSpacing(spacing);
	double origin[3] = {0.0f, 0.0f, 0.0f};
	GetOriginActiva(origin[0], origin[1], origin[2]);
	Loader->SetOutputOrigin(origin);
	if(Viewer != NULL) {
		// FIXME: Comprobar si el pipeline esta conectado.
		Viewer->SetupPipeline();
	} else {
		//std::cerr << "No se ha inicializado el contexto correctamente (Viewer NULL) tal vez sea un estudio referido" << std::endl;
	}
}

void GNC::GCS::IContextoEstudio::InicializarContextoEstudio(std::vector<std::string>& rutas, const std::string uidEstudioDiagnostico, TModoFuncionamiento modoFuncionamiento)
{
	IndiceFicheroActivo = 0;
	UIDEstudioDiagnostico = uidEstudioDiagnostico;
	ModoFuncionamiento = modoFuncionamiento;


	Ficheros.clear();

	//se inicializan los vectores...

	for (std::vector<std::string>::const_iterator it = rutas.begin(); it != rutas.end(); it++)
	{
		Ficheros.push_back(new TFicheroEstudio());
		Ficheros.back()->RutaImagen = (*it);
	}
}

std::vector<std::string> GNC::GCS::IContextoEstudio::GetRutasImagenes()
{
	std::vector<std::string> resultado;
	for (TVectorFicherosEstudio::iterator it = Ficheros.begin(); it != Ficheros.end(); ++it)
	{
		resultado.push_back((*it)->RutaImagen);
	}

	return resultado;
}

std::vector<std::string> GNC::GCS::IContextoEstudio::GetRutasDiagnosticos()
{
	std::vector<std::string> resultado;
	for (TVectorFicherosEstudio::iterator it = Ficheros.begin(); it != Ficheros.end(); ++it)
	{
		if ((*it)->RutaDiagnostico != "") {
			resultado.push_back((*it)->RutaDiagnostico);
		}
	}

	return resultado;
}

std::list<std::string> GNC::GCS::IContextoEstudio::GetListaFicherosYDiagnosticos()
{
	std::list<std::string> resultado;
	for (TVectorFicherosEstudio::iterator it = Ficheros.begin(); it != Ficheros.end(); ++it)
	{
		resultado.push_back((*it)->RutaImagen);
		if ((*it)->RutaDiagnostico != "") {
			resultado.push_back((*it)->RutaDiagnostico);
		}
	}

	return resultado;
}

void GNC::GCS::IContextoEstudio::SetModeloIntegracion(GnkPtr<GIL::IModeloIntegracion> modeloIntegracion)
{
	ModeloIntegracion = modeloIntegracion;
}

GnkPtr<GIL::IModeloIntegracion>& GNC::GCS::IContextoEstudio::GetModeloIntegracion()
{
	return ModeloIntegracion;
}

const std::string& GNC::GCS::IContextoEstudio::GetRutaDeImagenActiva()
{
	return GetRutaDeImagen(IndiceFicheroActivo);
}

const std::string& GNC::GCS::IContextoEstudio::GetRutaDeImagen(const int indice)
{
	if (indice >= 0 && (unsigned int)indice < Ficheros.size()) {
		return Ficheros[indice]->RutaImagen;
	}
	else {
		throw GNC::GCS::VistaException("GetRutaDeImagen(): Indice fuera de rango");
	}
}

const std::string& GNC::GCS::IContextoEstudio::GetRutaDeDiagnosticoActivo()
{
	if (ModoFuncionamiento == TMF_NDiagnosticos) {
		if (IndiceFicheroActivo >= 0 && (unsigned int)IndiceFicheroActivo < Ficheros.size()) {
			return Ficheros[IndiceFicheroActivo]->RutaDiagnostico;
		}
		else {
			throw GNC::GCS::VistaException("GetRutaDeDiagnosticoActivo(): Indice fuera de rango");
		}
	} else {
		return Ficheros[0]->RutaDiagnostico;
	}
}

const std::string& GNC::GCS::IContextoEstudio::GetUIDEstudioDiagnostico()
{
	return UIDEstudioDiagnostico;
}

GnkPtr<GIL::DICOM::TipoMetaInfo> GNC::GCS::IContextoEstudio::GetMetaInfoDeImagenActiva()
{
	return GetMetaInfo(IndiceFicheroActivo);
}

GnkPtr<GIL::DICOM::TipoJerarquia> GNC::GCS::IContextoEstudio::GetTagsImagenDeImagenActiva()
{
	return GetTagsImagen(IndiceFicheroActivo);
}

GnkPtr<GIL::DICOM::TipoJerarquia> GNC::GCS::IContextoEstudio::GetTagsDiagnosticoDeImagenActiva()
{
	return GetTagsDiagnostico(IndiceFicheroActivo);
}

GnkPtr<GIL::DICOM::TipoPrivateTags> GNC::GCS::IContextoEstudio::GetTagsPrivadosDeImagenActiva()
{
	return GetTagsPrivados(IndiceFicheroActivo);
}

GnkPtr<GIL::DICOM::TipoMetaInfo> GNC::GCS::IContextoEstudio::GetMetaInfo(const int indice)
{
	if(indice < 0 || indice >= (int)Ficheros.size())
		return NULL;
	
	if(!Ficheros[indice]->MetaInfo.IsValid()) {
		CargarMetaInfo(indice);
	}
	return Ficheros[indice]->MetaInfo;
}

GnkPtr<GIL::DICOM::TipoJerarquia> GNC::GCS::IContextoEstudio::GetTagsImagen(const int indice)
{
	if(indice < 0 || indice >= (int)Ficheros.size())
		return NULL;

	if(!Ficheros[indice]->TagsImagen.IsValid()) {
		CargarTagsImagen(indice);
	}
	return Ficheros[indice]->TagsImagen;
}

GnkPtr<GIL::DICOM::TipoJerarquia> GNC::GCS::IContextoEstudio::GetTagsDiagnostico(int indice)
{
	if (ModoFuncionamiento == TMF_UNDiagnostico) {
		indice = 0;
	}

	if(indice < 0 || indice >= (int)Ficheros.size())
		return NULL;

	if(!Ficheros[indice]->TagsDiagnostico.IsValid()) {
		CargarTagsImagen(indice);
	}

	return Ficheros[indice]->TagsDiagnostico;
}

GnkPtr<GIL::DICOM::TipoPrivateTags> GNC::GCS::IContextoEstudio::GetTagsPrivados(int indice)
{
	if (ModoFuncionamiento == TMF_UNDiagnostico) {
		indice = 0;
	}

	if(indice < 0 || indice >= (int)Ficheros.size())
		return NULL;

	if(!Ficheros[indice]->TagsPrivados.IsValid()) {
		CargarTagsImagen(indice);
	}
	return Ficheros[indice]->TagsPrivados;
}

int GNC::GCS::IContextoEstudio::GetNumeroCortes()
{
	return Ficheros.size();
}

bool GNC::GCS::IContextoEstudio::GetSpacingActiva(double* spacing)
{
	return GNC::GCS::IContextoEstudio::GetSpacing(IndiceFicheroActivo, spacing[0], spacing[1] , spacing[2]);
}

bool GNC::GCS::IContextoEstudio::GetSpacingActiva(double& x, double& y, double& z)
{
	return GNC::GCS::IContextoEstudio::GetSpacing(IndiceFicheroActivo, x, y ,z);
}

void GNC::GCS::IContextoEstudio::RecalibrarImagenActiva(double spacing[3], double origin[3])
{
	RecalibrarImagen(IndiceFicheroActivo, spacing, origin);
}

void GNC::GCS::IContextoEstudio::RecalibrarImagen(const int indice, double spacing[3], double origin[3])
{
	if (indice == IndiceFicheroActivo) {
		Loader->SetOutputOrigin(origin);
		Loader->SetOutputSpacing(spacing);

		Viewer->SetupPipeline();
	}

	SetSpacing(indice, spacing[0], spacing[1], spacing[2]);
}

bool GNC::GCS::IContextoEstudio::GetSpacing(const int indice, double& x, double& y, double& z)
{
	bool hasSpacing = false;
	std::string spacing;
	char c;
	x=0.0f;
	y=0.0f;
	z=0.0f;
	GnkPtr<GIL::DICOM::TipoJerarquia> tagsDiagnostico = GetTagsDiagnostico(indice);
	if (tagsDiagnostico.IsValid()) {
		if(tagsDiagnostico->getTag("0028|0030",spacing)) {
			std::istringstream issl(spacing);
			issl >> x;
			if(!issl.eof()){
				issl>>c;//la barra
				issl>>y;
			}
			if(!issl.eof()){
				issl>>c;//la barra
				issl>>z;
			}
			hasSpacing = true;
		}
	}
	if (x < std::numeric_limits<double>::epsilon()) {
		x = 1.0f;
		hasSpacing = false;
	}
	if (y < std::numeric_limits<double>::epsilon()) {
		y = 1.0f;
		hasSpacing = false;
	}
	if (z < std::numeric_limits<double>::epsilon()) {
		z = 1.0f;
	}
	return hasSpacing;
}

void GNC::GCS::IContextoEstudio::SetSpacing(const int indice, const double x, const double y, const double z)
{
	std::ostringstream os;
	os << x << "\\" << y << "\\" <<z;
	std::string nuevoSpacing(os.str());

	GnkPtr<GIL::DICOM::TipoJerarquia> tagsDiagnosticoActivo = GetTagsDiagnostico(indice);
	if (tagsDiagnosticoActivo.IsValid()) {
		std::string spacingTmp;
		tagsDiagnosticoActivo->getTag("0028|0030",spacingTmp);
		if(nuevoSpacing != spacingTmp) {
			tagsDiagnosticoActivo->tags[std::string("0028|0030")] = nuevoSpacing;
			SetModificado(indice);
		}
	}
}

void GNC::GCS::IContextoEstudio::GetOriginActiva(double* origin)
{
	GetOrigin(IndiceFicheroActivo, origin[0], origin[1], origin[2]);
}

void GNC::GCS::IContextoEstudio::GetOriginActiva(double& x, double& y, double& z)
{
	GetOrigin(IndiceFicheroActivo, x, y ,z);
}

void GNC::GCS::IContextoEstudio::GetOrigin(const int indice, double& x, double& y, double& z)
{
	std::string origin;
	char c;
	x = y = z =0.0f;
	GnkPtr<GIL::DICOM::TipoJerarquia> tagsImagen = GetTagsImagen(indice);
	if (tagsImagen.IsValid()) {
		if(tagsImagen->getTag("0020|0032",origin)) {
			std::istringstream issl(origin);
			issl >> x;
			if(!issl.eof()){
				issl>>c;//la barra
				issl>>y;
			}
			if(!issl.eof()){
				issl>>c;//la barra
				issl>>z;
			}
		}
	}
}

void GNC::GCS::IContextoEstudio::GetDimensionsImagenActiva(int* dims)
{
	GetDimensionsImagen(IndiceFicheroActivo, dims[0], dims[1], dims[2]);
}

void GNC::GCS::IContextoEstudio::GetDimensionsImagenActiva(int& x, int& y, int& z)
{
	GetDimensionsImagen(IndiceFicheroActivo, x, y, z);
}

void GNC::GCS::IContextoEstudio::GetDimensionsImagen(const int indice, int& x, int& y, int& z)
{
	//z = Ficheros.size();
	x = y = z = 0;
	GnkPtr<GIL::DICOM::TipoJerarquia> tagsImagenOriginal = GetTagsImagen(indice);
	if (tagsImagenOriginal.IsValid()) {
		std::string tag;
		if (tagsImagenOriginal->getTag("0028|0010",tag) )
		{
			std::istringstream issl(tag);
			issl >> y;
		}
		if (tagsImagenOriginal->getTag("0028|0011",tag))
		{
			std::istringstream issl(tag);
			issl >> x;
		}

		if (tagsImagenOriginal->getTag("0028|0008",tag))
		{
			std::istringstream issl(tag);
			issl >> z;
		}
		else {
			z = 1;
		}
	}
}

int GNC::GCS::IContextoEstudio::GetTSizeActiva()
{
	unsigned int dimensions[3] = {0,0,0};
	Loader->GetOutputDimensions(dimensions);
	return dimensions[2];
}


const std::string GNC::GCS::IContextoEstudio::GetNombreMedico()
{
	std::string nombre ("");
	if(ModeloIntegracion.IsValid()) {
		std::ostringstream ostr;
		ostr<<ModeloIntegracion->Medico.apellido1 << ' ' << ModeloIntegracion->Medico.apellido2 << '^' << ModeloIntegracion->Medico.nombre;
		nombre = ostr.str();
	} else {
		GNC::GCS::ConfigurationController::Instance()->readStringUser("/GinkgoCore/Estacion","NombreMedico",nombre);
	}
	return nombre;
}


const std::string GNC::GCS::IContextoEstudio::GetNombreCentro()
{
	std::string nombre ("");
	if(ModeloIntegracion.IsValid()) {
		nombre = ModeloIntegracion->Medico.nombreCentro;
	} else {
		GNC::GCS::ConfigurationController::Instance()->readStringGeneral("/GinkgoCore/Estacion","CentroNombre",nombre);
	}
	return nombre;
}

bool GNC::GCS::IContextoEstudio::GetTagImagenActiva(const std::string& tag, std::string& valor)
{
	GnkPtr<GIL::DICOM::TipoJerarquia> pointer = GetTagsImagenDeImagenActiva();
	if (pointer.IsValid()) {
		return pointer->getTag(tag,valor);
	} else {
		return false;
	}
}

bool GNC::GCS::IContextoEstudio::GetTagDiagnosticoActivo(const std::string& tag, std::string& valor)
{
	GnkPtr<GIL::DICOM::TipoJerarquia> pointer = GetTagsDiagnosticoDeImagenActiva();
	if (pointer.IsValid()) {
		return pointer->getTag(tag,valor);
	} else {
		return false;
	}
}

int GNC::GCS::IContextoEstudio::GetIndicePath(const std::string& path)
{
	for (int i = 0; i < (int)Ficheros.size(); ++i)
	{
		if (Ficheros[i]->RutaImagen == path)
		{
			return i;
		}
	}
	return -1;
}

GNC::GCS::IContextoEstudio::TModoFuncionamiento GNC::GCS::IContextoEstudio::GetModoFuncionamiento()
{
	return ModoFuncionamiento;
}


bool GNC::GCS::IContextoEstudio::EstaModificadoActivo()
{
	if(IndiceFicheroActivo < 0 || IndiceFicheroActivo >= (int)Ficheros.size())
		return false;

	else return Ficheros[IndiceFicheroActivo]->Modificado;
}

bool GNC::GCS::IContextoEstudio::EstaModificado()
{
	if(IndiceFicheroActivo<0)
		return false;

	for (TVectorFicherosEstudio::iterator it = Ficheros.begin(); it != Ficheros.end(); ++it)
	{
		if((*it)->Modificado)
		{
			return true;
		}
	}
	return false;
}

void GNC::GCS::IContextoEstudio::SetModificadoFicheroActivo()
{
	SetModificado(IndiceFicheroActivo);
}

void GNC::GCS::IContextoEstudio::SetModificado(const int indice)
{
	if (IgnorarModificaciones)
		return;
	if(indice<0 || indice >= (int)Ficheros.size())
		return;

	Ficheros[indice]->Modificado = true;
}

bool GNC::GCS::IContextoEstudio::Guardar()
{
	if (ModoFuncionamiento == TMF_NDiagnosticos) {
		return GuardarNDiagnosticos();
	} else {
		return GuardarUnDiagnostico();
	}
}

bool GNC::GCS::IContextoEstudio::GuardarUnDiagnostico()
{
	bool correcto = true;
	//primero se guardan los widgets
	if (!EstaModificado())
		return true;

	GnkPtr<GIL::DICOM::TipoJerarquia> pTagDiagnostico = GetTagsDiagnostico(0);
	GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados = GetTagsPrivados(0);
	GnkPtr<GIL::DICOM::TipoJerarquia> pTagsImagen = GetTagsImagen(0);

	if (!pTagDiagnostico.IsValid() || !pTagsPrivados.IsValid() || !pTagsImagen.IsValid())
	{
		LOG_ERROR("IContextoEstudio", "Error al obtener los tags del estudio");
		return false;
	}

	//primero se guardan widgets..
	GuardarWidgets(0);

	//se vuelven a poner estos tags...
	pTagDiagnostico->tags[std::string("0008|0021")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha de la serie
	pTagDiagnostico->tags[std::string("0008|0031")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora de la serie
	pTagDiagnostico->tags[std::string("0008|0022")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha del imagen
	pTagDiagnostico->tags[std::string("0008|0032")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora del imagen

	//institucion
	pTagDiagnostico->tags[std::string("0008|0080")] = GetNombreCentro();

	//nombre del medico responsable de la institucion
	pTagDiagnostico->tags[std::string("0008|0090")] = GetNombreMedico();

	if(!wxFile::Exists( FROMPATH(Ficheros[0]->RutaDiagnostico) ))
	{
		//se crea en un directorio temporal y luego se mueve al "dicomdir"
		std::string stdDirPathTmp = Entorno->CrearDirectorioTemporal();

		wxString pathTmp = (FROMPATH(stdDirPathTmp) + wxFileName::GetPathSeparator() << rand()) + wxT(".dcm");
		std::string stdPathTmp(TOPATH(pathTmp));

		GIL::DICOM::TipoJerarquia baseSR;
		SetTagsDiagnostico( baseSR, pTagDiagnostico);

		//se hace que apunte al los ficheros modificados...
		{
			std::string sopClassUID;
			std::string sopInstanceUID;
			for (int i = 0; i < (int)Ficheros.size(); ++i)
			{
				if (!Ficheros[i]->Modificado)
					continue;
				GnkPtr<GIL::DICOM::TipoJerarquia> tagsImagen = GetTagsImagen(i);
				if(tagsImagen->getTag(std::string("0008|0016"),sopClassUID) && tagsImagen->getTag(std::string("0008|0018"),sopInstanceUID)) {
					baseSR.AddReference(sopClassUID,sopInstanceUID);
				}
				Ficheros[i]->Modificado = false;
			}
		}

		GIL::DICOM::IDICOMImg2DCM* pDicomizador = Entorno->GetPACSController()->CrearInstanciaDeImportacion();
		std::list<GnkPtr<GIL::DICOM::TipoPrivateTags> > listaTagsPrivados;
		listaTagsPrivados.push_back(pTagsPrivados);

		correcto = correcto && CallbackPreCrearDiagnostico(0, &baseSR, listaTagsPrivados);	//se deja a las clases hijas que indtroduzcan los tags que deseen

		pDicomizador->CrearSRDoc(stdPathTmp,baseSR,listaTagsPrivados);
		Entorno->GetPACSController()->LiberarInstanciaDeImportacion(pDicomizador);

		//se le el uid de serie, estudio y paciente para copiarlo al dicomdir
		GIL::DICOM::IDICOMManager* pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
		pDICOMManager->CargarFichero(stdPathTmp, *pTagDiagnostico);
		Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);

		//si el uid del estudio de diagnosticos no esta establecido, se establece
		if(UIDEstudioDiagnostico == std::string("")) {
			pTagDiagnostico->getTag(std::string("0020|000d"),UIDEstudioDiagnostico);
		}

		//el uid de la serie, paciente e imagen (diagnostico)...
		std::string uidSerie, uidImagen, uidPaciente;
		pTagDiagnostico->getTag(std::string("0010|0020"),uidPaciente);
		pTagDiagnostico->getTag(std::string("0020|000e"),uidSerie);
		pTagDiagnostico->getTag(std::string("0008|0018"),uidImagen);

		Entorno->GetPACSController()->GetRutaImagen(uidPaciente,UIDEstudioDiagnostico,uidSerie,uidImagen,Ficheros[0]->RutaDiagnostico);

		if (!wxCopyFile(pathTmp,FROMPATH( Ficheros[0]->RutaDiagnostico ))) {
			LOG_ERROR("IContextoEstudio", "error copying file " << stdPathTmp << " TO " << Ficheros[0]->RutaDiagnostico);
			correcto = false;
		}

		wxRemoveFile(pathTmp);
		#if defined(_WINDOWS)
		wxRmDir( FROMPATH(stdDirPathTmp) );
		#else
		wxRmDir( stdDirPathTmp.c_str() );
		#endif
	} else {
		{
			//se hace que apunte a los ficheros modificados
			std::string sopClassUID;
			std::string sopInstanceUID;
			for (int i = 0; i < (int)Ficheros.size(); ++i)
			{
				if (!Ficheros[i]->Modificado)
					continue;
				GnkPtr<GIL::DICOM::TipoJerarquia> tagsImagen = GetTagsImagen(i);
				if(tagsImagen->getTag(std::string("0008|0016"),sopClassUID) && tagsImagen->getTag(std::string("0008|0018"),sopInstanceUID)) {
					pTagDiagnostico->AddReference(sopClassUID,sopInstanceUID);
				}
				Ficheros[i]->Modificado = false;
			}
		}

		GIL::DICOM::IDICOMManager*	pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
		GIL::DICOM::TipoJerarquia base;
		pDICOMManager->CargarFichero(Ficheros[0]->RutaDiagnostico, base);

		//se actualiza el fichero y se guarda
		pDICOMManager->ActualizarJerarquia(*pTagDiagnostico);
		pDICOMManager->ActualizarTagsPrivados(*pTagsPrivados);
		correcto = correcto && CallbackPreGuardarDiagnostico(0,pDICOMManager); //se ofrece la posibilidad de insertar mas cosas
		correcto = correcto && pDICOMManager->AlmacenarFichero(Ficheros[0]->RutaDiagnostico);
		//se libera el dicommanager
		Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);
	}

	return correcto;
}

bool GNC::GCS::IContextoEstudio::GuardarNDiagnosticos()
{
	bool correcto = true;
	for (int i = 0; i < (int)Ficheros.size(); ++i)
	{
		if ( !Ficheros[i]->Modificado )
			continue;

		GnkPtr<GIL::DICOM::TipoJerarquia> pTagDiagnostico = GetTagsDiagnostico(i);
		GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados = GetTagsPrivados(i);
		GnkPtr<GIL::DICOM::TipoJerarquia> pTagsImagen = GetTagsImagen(i);

		if (!pTagDiagnostico.IsValid() || !pTagsPrivados.IsValid() || !pTagsImagen.IsValid())
		{
			LOG_ERROR("IContextoEstudio", "Error al obtener los tags del estudio");
			return false;
		}

		//primero de todo se guardan los widgets...
		GuardarWidgets(i);

		//despues el spacing

		//se vuelven a poner estos tags...
		pTagDiagnostico->tags[std::string("0008|0021")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha de la serie
		pTagDiagnostico->tags[std::string("0008|0031")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora de la serie
		pTagDiagnostico->tags[std::string("0008|0022")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha del imagen
		pTagDiagnostico->tags[std::string("0008|0032")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora del imagen

		//institucion
		std::string strTmp;
		if(ModeloIntegracion.IsValid()) {
			pTagDiagnostico->tags[std::string("0008|0080")] = ModeloIntegracion->Medico.nombreCentro;
		} else {
			GNC::GCS::ConfigurationController::Instance()->readStringGeneral("/GinkgoCore/Estacion","CentroNombre",strTmp);
			if(strTmp.size() != 0){
				pTagDiagnostico->tags[std::string("0008|0080")] = strTmp;
			}
		}

		//nombre del medico responsable de la institucion
		pTagDiagnostico->tags[std::string("0008|0090")] = GetNombreMedico();

		if(!wxFile::Exists( FROMPATH(Ficheros[i]->RutaDiagnostico) ))
		{
			//se crea en un directorio temporal y luego se mueve al "dicomdir"
			std::string stdDirPathTmp = Entorno->CrearDirectorioTemporal();

			wxString pathTmp = (FROMPATH(stdDirPathTmp) + wxFileName::GetPathSeparator() << rand()) + wxT(".dcm");
			std::string stdPathTmp(TOPATH(pathTmp));

			GIL::DICOM::TipoJerarquia baseSR;

			SetTagsDiagnostico( baseSR, pTagDiagnostico);


			//se hace que apunte al fichero original
			std::string sopClassUID;
			std::string sopInstanceUID;

			if(pTagsImagen->getTag(std::string("0008|0016"),sopClassUID) && pTagsImagen->getTag(std::string("0008|0018"),sopInstanceUID)) {
				baseSR.AddReference(sopClassUID,sopInstanceUID);
			}

			GIL::DICOM::IDICOMImg2DCM* pDicomizador = Entorno->GetPACSController()->CrearInstanciaDeImportacion();
			std::list<GnkPtr<GIL::DICOM::TipoPrivateTags> > listaTagsPrivados;
			listaTagsPrivados.push_back(pTagsPrivados);

			correcto = correcto && CallbackPreCrearDiagnostico(i, &baseSR, listaTagsPrivados);	//se deja a las clases hijas que indtroduzcan los tags que deseen

			pDicomizador->CrearSRDoc(stdPathTmp,baseSR,listaTagsPrivados);
			Entorno->GetPACSController()->LiberarInstanciaDeImportacion(pDicomizador);

			//se le el uid de serie, estudio y paciente para copiarlo al dicomdir
			GIL::DICOM::IDICOMManager* pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
			pDICOMManager->CargarFichero(stdPathTmp, *pTagDiagnostico);
			Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);

			//si el uid del estudio de diagnosticos no esta establecido, se establece
			if(UIDEstudioDiagnostico == std::string("")) {
				pTagDiagnostico->getTag(std::string("0020|000d"),UIDEstudioDiagnostico);
			}

			//el uid de la serie, paciente e imagen (diagnostico)...
			std::string uidSerie, uidImagen, uidPaciente;
			pTagDiagnostico->getTag(std::string("0010|0020"),uidPaciente);
			pTagDiagnostico->getTag(std::string("0020|000e"),uidSerie);
			pTagDiagnostico->getTag(std::string("0008|0018"),uidImagen);

			Entorno->GetPACSController()->GetRutaImagen(uidPaciente,UIDEstudioDiagnostico,uidSerie,uidImagen,Ficheros[i]->RutaDiagnostico);

			if (!wxCopyFile(pathTmp,FROMPATH( Ficheros[i]->RutaDiagnostico ))) {
				LOG_ERROR("IContextoEstudio", "error copying file " << stdPathTmp << " TO " << Ficheros[0]->RutaDiagnostico);
				correcto = false;
			}

			wxRemoveFile(pathTmp);
			#if defined(_WINDOWS)
			wxRmDir( FROMPATH(stdDirPathTmp) );
			#else
			wxRmDir( stdDirPathTmp.c_str() );
			#endif
		} else {
			GIL::DICOM::IDICOMManager*	pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
			GIL::DICOM::TipoJerarquia base;
			pDICOMManager->CargarFichero(Ficheros[i]->RutaDiagnostico, base);

			//se actualiza el fichero y se guarda
			pDICOMManager->ActualizarJerarquia(*pTagDiagnostico);
			pDICOMManager->ActualizarTagsPrivados(*pTagsPrivados);
			correcto = correcto && CallbackPreGuardarDiagnostico(i,pDICOMManager); //se ofrece la posibilidad de insertar mas cosas
			correcto = correcto && pDICOMManager->AlmacenarFichero(Ficheros[i]->RutaDiagnostico);
			//se libera el dicommanager
			Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);
		}

		Ficheros[i]->Modificado = false;
	}
	return correcto;
}

void GNC::GCS::IContextoEstudio::CreateLoader()
{
	Loader              = GNC::GCS::ControladorCarga::Instance()->NewLoader();

	renderConnection    = Loader->GetOutputPort();
}

void GNC::GCS::IContextoEstudio::SetTagsDiagnostico( GIL::DICOM::TipoJerarquia& baseSR, GnkPtr<GIL::DICOM::TipoJerarquia> pTagDiagnostico)
{
	//todos los tags del paciente...
	for(GIL::DICOM::ListaTags::iterator it = pTagDiagnostico->tags.begin(); it!= pTagDiagnostico->tags.end(); it++){
		//paciente, estudio
		if((*it).first.substr(0,4) == "0010"){
			baseSR.tags[(*it).first] = (*it).second;
		}
	}

	//spacing
	if (pTagDiagnostico->tags["0028|0030"] != "") {
		baseSR.tags["0028|0030"] = pTagDiagnostico->tags["0028|0030"];
	}

	baseSR.tags[std::string("0008|0070")] = std::string("Metaemotion S.L.");

	baseSR.tags[std::string("0008|1090")] = std::string("Ginkgo APrimaria");

	if(UIDEstudioDiagnostico != std::string("")){
		baseSR.tags[std::string("0020|000d")] = UIDEstudioDiagnostico;		//uid estudio diagnostico
	}

	//TODO ESTO HAY QUE ELIMINARLO... PORQUE ESTA PREPARADO PARA APRIMARIA
	baseSR.tags[std::string("0018|1030")] = "METAEMOTION GINKGO RX";				//uid importador que crea la serie
	baseSR.tags[std::string("0008|0021")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha de la serie
	baseSR.tags[std::string("0008|0031")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora de la serie
	baseSR.tags[std::string("0008|0020")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha de la estudio
	baseSR.tags[std::string("0008|0030")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora de la estudio
	baseSR.tags[std::string("0008|0022")] = wxDateTime::Now().Format(wxT("%Y%m%d")).ToUTF8(); //fecha de la imagen
	baseSR.tags[std::string("0008|0032")] = wxDateTime::Now().Format(wxT("%H%M%S")).ToUTF8(); //hora de la imagen
	baseSR.tags[std::string("0008|0080")] = pTagDiagnostico->tags["0008|0080"];//centro
	baseSR.tags[std::string("0008|0090")] = pTagDiagnostico->tags["0008|0090"];//medico
	baseSR.tags[std::string("0008|0050")] = pTagDiagnostico->tags["0008|0050"];//accession number
}

void GNC::GCS::IContextoEstudio::GuardarWidgets(int indice)
{
	GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados = GetTagsPrivados(indice);
	if (!pTagsPrivados.IsValid())
		return;

	wxXmlDocument documento;
	wxXmlNode* raiz = NULL;
	if (ModoFuncionamiento == TMF_UNDiagnostico) {
		//se carga el antiguo xml de widgets
		unsigned char tagWidgetsGinkgo=TAG_WIDGETS;

		GIL::DICOM::TagPrivadoUndefined* pTagWidgets = pTagsPrivados->GetTagUndefined(tagWidgetsGinkgo);
		if(pTagWidgets != NULL){
			wxMemoryInputStream input(pTagWidgets->GetValor(),pTagWidgets->GetSize());
			documento.Load(input,wxT("UTF 8"));
		}

		if(documento.IsOk()){
			raiz = documento.GetRoot();
		} else {
			raiz = new wxXmlNode(NULL,wxXML_ELEMENT_NODE, wxT("ginkgo"));
			documento.SetFileEncoding(wxT("UTF-8"));
			documento.SetRoot(raiz);
		}

		GNC::GCS::IControladorHerramientas* cH = Entorno->GetControladorHerramientas();
		GNC::GCS::IControladorHistorial* pHistorial = Entorno->GetControladorHistorial();
		std::string sopInstanceUID;
		for (int i = 0; i < (int)Ficheros.size(); ++i)
		{
			if (!Ficheros[i]->Modificado)
				continue;

			sopInstanceUID = pHistorial->GetSopInstanceUID(Ficheros[i]->RutaImagen);

			cH->Serializar(Vista,i,raiz,GetNombreMedico(), sopInstanceUID);
		}
	} else {
		//si el funcionamiento es de n diagnosticos se guardan los widgets que hay
		raiz = new wxXmlNode(NULL,wxXML_ELEMENT_NODE, wxT("ginkgo"));
		documento.SetFileEncoding(wxT("UTF-8"));
		documento.SetRoot(raiz);

		//se lee el sopinstance uid
		GNC::GCS::IControladorHistorial* pHistorial = Entorno->GetControladorHistorial();
		std::string sopInstanceUID = pHistorial->GetSopInstanceUID(Ficheros[indice]->RutaImagen);

		GNC::GCS::IControladorHerramientas* cH = Entorno->GetControladorHerramientas();
		cH->Serializar(Vista,indice,raiz,GetNombreMedico(), sopInstanceUID);
	}

	unsigned char tagWidgetsGinkgo=TAG_WIDGETS;

	wxMemoryOutputStream out;
	documento.Save(out);
	bool addCaracter =false;
	wxFileOffset longitud = out.GetLength();
	if(longitud%2!=0){
		longitud++;
		addCaracter = true;
	}
	unsigned char* vectorWidgets = new unsigned char[longitud];
	if(addCaracter){
		out.CopyTo(vectorWidgets,longitud-1);
		vectorWidgets[longitud-1] = ' ';
	}else{
		out.CopyTo(vectorWidgets,longitud);
	}

	pTagsPrivados->SetTag(tagWidgetsGinkgo, vectorWidgets,longitud);
	delete []vectorWidgets;
}

bool GNC::GCS::IContextoEstudio::IsKeyImageActive()
{
	return IsKeyImage(IndiceFicheroActivo);
}

bool GNC::GCS::IContextoEstudio::IsKeyImage(int indice)
{
	GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados = GetTagsPrivados(indice);
	if (!pTagsPrivados.IsValid())
		return false;

	if (ModoFuncionamiento == TMF_UNDiagnostico) {
		indice = 0;
	}
	std::string isKey;
	if (pTagsPrivados->GetTag(TAG_KEY_IMAGE, isKey)) {
		return isKey.compare("YES") == 0;
	} else {
		return false;
	}
}

void GNC::GCS::IContextoEstudio::CargarWidgetsActivo(bool force)
{
	CargarWidgets(IndiceFicheroActivo, force);
}

void GNC::GCS::IContextoEstudio::CargarWidgets(int indice, bool force) {
	GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados = GetTagsPrivados(indice);
	if (!pTagsPrivados.IsValid())
		return;

	if (ModoFuncionamiento == TMF_UNDiagnostico) {
		indice = 0;
	}

	if( !Ficheros[indice]->WidgetsCargados || force) {
		unsigned char tagWidgetsGinkgo=TAG_WIDGETS;

		wxXmlDocument documento;
		GIL::DICOM::TagPrivadoUndefined* pTagWidgets = pTagsPrivados->GetTagUndefined(tagWidgetsGinkgo);
		if(pTagWidgets != NULL){
			wxMemoryInputStream input(pTagWidgets->GetValor(),pTagWidgets->GetSize());
			documento.Load(input,wxT("UTF 8"));
		}

		if(documento.IsOk()){
			GNC::GCS::IControladorHerramientas* cH = Entorno->GetControladorHerramientas();
			GNC::GCS::IControladorHistorial* pHistorial = Entorno->GetControladorHistorial();
			std::string sopInstanceUID;

			if (ModoFuncionamiento == TMF_UNDiagnostico) {
				for (int i = 0; i < (int)Ficheros.size(); i++)
				{
					sopInstanceUID = pHistorial->GetSopInstanceUID(Ficheros[i]->RutaImagen);
					cH->Deserializar(Vista,i,documento.GetRoot(), sopInstanceUID);
				}
			} else {
				sopInstanceUID = pHistorial->GetSopInstanceUID(Ficheros[indice]->RutaImagen);
				cH->Deserializar(Vista,indice,documento.GetRoot(), sopInstanceUID);
			}
		}

		Ficheros[indice]->WidgetsCargados = true;
	}
}



void GNC::GCS::IContextoEstudio::CargarMetaInfo(const int indice)
{
	Lock("CargarMetaInfo");
	//tags imagenes
	if (!Ficheros[indice]->MetaInfo.IsValid()) {
		Ficheros[indice]->MetaInfo = new GIL::DICOM::TipoMetaInfo();
		
		GIL::DICOM::IDICOMManager*	pDICOMManager;
		pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
		pDICOMManager->CargarMetaInfo(Ficheros[indice]->RutaImagen, *Ficheros[indice]->MetaInfo);
		Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);
	}
	UnLock("CargarMetaInfo");
}

void GNC::GCS::IContextoEstudio::CargarTagsImagen(const int indice)
{
	Lock("CargarTags");
	int indiceDiagnostico = indice;
	if (ModoFuncionamiento == TMF_UNDiagnostico)
	{
		indiceDiagnostico = 0;
	}

	if (!Ficheros[indiceDiagnostico]->TagsDiagnostico.IsValid() && !Ficheros[indiceDiagnostico]->TagsPrivados.IsValid()) {
		Ficheros[indice]->TagsImagen = new GIL::DICOM::TipoJerarquia();
		Ficheros[indiceDiagnostico]->TagsDiagnostico = new GIL::DICOM::TipoJerarquia();
		//el 100 es el uid de los tags comunes...
		Ficheros[indiceDiagnostico]->TagsPrivados = new GIL::DICOM::TipoPrivateTags(UID_TAGS_PRIVADOS_COMUNES);

		GIL::DICOM::IDICOMManager*	pDICOMManager;
		pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
		pDICOMManager->CargarFichero(Ficheros[indice]->RutaImagen, *Ficheros[indice]->TagsImagen);

		//se busca el fichero de diagnostico...
		GNC::GCS::IControladorHistorial::ModeloDCM modelo;
		if (ModoFuncionamiento == TMF_UNDiagnostico) {
			//se busca el diagnostico que apunte a cualquier fichero de la serie
			std::string uidSerie;
			Ficheros[indice]->TagsImagen->getTag("0020|000e",uidSerie);
			modelo = Entorno->GetControladorHistorial()->GetDiagnosticoSerie(uidSerie,UIDEstudioDiagnostico);
		} else {
			//se  busca el diagnostico que apunte a este fichero
			modelo = Entorno->GetControladorHistorial()->GetDiagnostico(Ficheros[indice]->RutaImagen,UIDEstudioDiagnostico);
		}
		std::string pathAbsoluto = Entorno->GetPathAbsolutoFichero(modelo.m_pathRelativo);
		if(wxFileExists(FROMPATH(pathAbsoluto)))
		{
			//leemos los tags privados del diagnostico
			Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);

			Ficheros[indiceDiagnostico]->RutaDiagnostico = pathAbsoluto;

			pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
			pDICOMManager->CargarFichero(Ficheros[indiceDiagnostico]->RutaDiagnostico, *Ficheros[indiceDiagnostico]->TagsDiagnostico);

			//si el uid del estudio de diagnosticos no esta establecido, se establece
			if(UIDEstudioDiagnostico == std::string("")) {
				Ficheros[indiceDiagnostico]->TagsDiagnostico->getTag(std::string("0020|000d"),UIDEstudioDiagnostico);
			}

			pDICOMManager->CargarTagsPrivados(*Ficheros[indiceDiagnostico]->TagsPrivados);
			CallbackCargarTagsImagen(indice, pDICOMManager);
			Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);
		} else {
			//leemos los tags privados de la imagen...
			pDICOMManager->CargarTagsPrivados(*Ficheros[indiceDiagnostico]->TagsPrivados);
			CallbackCargarTagsImagen(indice, pDICOMManager);

			(*Ficheros[indiceDiagnostico]->TagsDiagnostico) = (*Ficheros[indice]->TagsImagen);
			Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);
		}
	}

	//tags imagenes
	if (!Ficheros[indice]->TagsImagen.IsValid()) {
		Ficheros[indice]->TagsImagen = new GIL::DICOM::TipoJerarquia();

		GIL::DICOM::IDICOMManager*	pDICOMManager;
		pDICOMManager= Entorno->GetPACSController()->CrearInstanciaDeDICOMManager();
		pDICOMManager->CargarFichero(Ficheros[indice]->RutaImagen, *Ficheros[indice]->TagsImagen);
		Entorno->GetPACSController()->LiberarInstanciaDeDICOMManager(pDICOMManager);
	}


	UnLock("CargarTags");
}

//inicializa el mapa de valoracion con los datos del tag privado, si no existe el tag privado lo crea y lo inserta
void GNC::GCS::IContextoEstudio::CrearMapaValoracionUnsignedChar(vtkSmartPointer<vtkImageData>& pMapa, GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados, const unsigned char tag, const int indice, unsigned char valorInicial)
{
	Lock("CrearMapaValoracion");
	//se crea la imagen
	if (pMapa == NULL) {
		pMapa = vtkSmartPointer<vtkImageData>::New();

		double spacing[3] = {1.0f,1.0f,1.0f};
		double origin[3] = {0.0f,0.0f,0.0f};
		int dimensions[3] = {0.0f,0.0f,0.0f};
		GetSpacing(indice,spacing[0],spacing[1],spacing[2]);
		GetOrigin(indice,origin[0],origin[1],origin[2]);
		GetDimensionsImagen(indice,dimensions[0],dimensions[1], dimensions[2]);

		//se inicializan valores de dimensions, scalartype...
		pMapa->SetDimensions(dimensions[0],dimensions[1],1);
		pMapa->SetScalarTypeToUnsignedChar();
		pMapa->SetNumberOfScalarComponents(1);
		pMapa->SetSpacing(spacing[0],spacing[1],spacing[2]);
		pMapa->SetOrigin(origin[0],origin[1],origin[2]);

		GIL::DICOM::TagPrivadoUndefined* pTagPrivado = pTagsPrivados->GetTagUndefined(tag);
		if(pTagPrivado == NULL) {
			const unsigned int size = dimensions[0] * dimensions[1];
			pTagPrivado = pTagsPrivados->NewTagUndefined(tag,size);
			//se hace un fill 0
			const unsigned char* fin = size + pTagPrivado->GetValor();
			for( unsigned char* pData = pTagPrivado->GetValor(); pData < fin; ++pData)
			{
				(*pData) = valorInicial;
			}
		}

		vtkSmartPointer<vtkUnsignedCharArray> dataImagen = vtkSmartPointer<vtkUnsignedCharArray>::New();
		dataImagen->SetNumberOfComponents(1);
		unsigned int size = pMapa->GetDimensions()[0] * pMapa->GetDimensions()[1];
		dataImagen->SetArray(pTagPrivado->GetValor(),size,1);

		pMapa->GetPointData()->SetScalars(dataImagen);
	}
	UnLock("CrearMapaValoracion");
}

void GNC::GCS::IContextoEstudio::CrearMapaValoracionChar(vtkSmartPointer<vtkImageData>& pMapa, GnkPtr<GIL::DICOM::TipoPrivateTags> pTagsPrivados, const unsigned char tag, const int indice, char valorInicial)
{
	Lock("CrearMapaValoracion");
	//se crea la imagen
	if (pMapa == NULL) {
		pMapa = vtkSmartPointer<vtkImageData>::New();

		double spacing[3] = {1.0f,1.0f,1.0f};
		double origin[3] = {0.0f,0.0f,0.0f};
		int dimensions[3] = {0.0f,0.0f,0.0f};
		GetSpacing(indice,spacing[0],spacing[1],spacing[2]);
		GetOrigin(indice,origin[0],origin[1],origin[2]);
		GetDimensionsImagen(indice,dimensions[0],dimensions[1], dimensions[2]);

		//se inicializan valores de dimensions, scalartype...
		pMapa->SetDimensions(dimensions[0],dimensions[1],1);
		pMapa->SetScalarTypeToChar();
		pMapa->SetNumberOfScalarComponents(1);
		pMapa->SetSpacing(spacing[0],spacing[1],spacing[2]);
		pMapa->SetOrigin(origin[0],origin[1],origin[2]);

		GIL::DICOM::TagPrivadoUndefined* pTagPrivado = pTagsPrivados->GetTagUndefined(tag);
		if(pTagPrivado == NULL) {
			const unsigned int size = dimensions[0] * dimensions[1];
			pTagPrivado = pTagsPrivados->NewTagUndefined(tag,size);
			//se hace un fill 0
			const char* fin = size + (char*)pTagPrivado->GetValor();
			for( char* pData = (char*)pTagPrivado->GetValor(); pData < fin; ++pData)
			{
				(*pData) = valorInicial;
			}
		}

		vtkSmartPointer<vtkCharArray> dataImagen = vtkSmartPointer<vtkCharArray>::New();
		dataImagen->SetNumberOfComponents(1);
		unsigned int size = pMapa->GetDimensions()[0] * pMapa->GetDimensions()[1];
		dataImagen->SetArray((char*)pTagPrivado->GetValor(),size,1);

		pMapa->GetPointData()->SetScalars(dataImagen);
	}
	UnLock("CrearMapaValoracion");
}

GNC::GCS::IContextoEstudioReferido::IContextoEstudioReferido()
{
	GTRACE(">> IContextoEstudio::IContextoEstudioReferido() " << this);
	GTRACE("<< IContextoEstudio::IContextoEstudioReferido() " << this);
}

GNC::GCS::IContextoEstudioReferido::~IContextoEstudioReferido()
{
	GTRACE(">> IContextoEstudio::~IContextoEstudioReferido() " << this);
	GTRACE("<< IContextoEstudio::~IContextoEstudioReferido() " << this);
}

void GNC::GCS::IContextoEstudioReferido::UnRefViewer()
{
	Viewer = NULL;
}
