/*
** 1998-05-25 -	This module deals with formatting directory pane entries for display.
**		That's all it does, but since that's a fairly involved thing to do
**		(we aim for plenty of flexibility here), it deserves a module of its
**		own.
** 1999-05-15 -	Added the utility dpf_set_default_format() funtion, for cfg_dirpane.
** 1999-05-20 -	Adapted for the new ultra-flexible & dynamic styles subsystem. Now
**		does three hash-lookups rather than three vector-dittos per row. :)
*/

#include "gentoo.h"

#include <time.h>

#include "dpformat.h"
#include "userinfo.h"
#include "iconutil.h"
#include "sizeutil.h"
#include "strutil.h"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	DPContent	id;
	const gchar	*name_full;		/* Long, descriptive name. Multiple words, mixed caps. */
	const gchar	*name_short;		/* Short mnemonic name, no whitespace, all lower case. */
	const gchar	*column_title;		/* Single word, with caps. Handy for pane column titles. */
} ContentInfo;

/* This table is used for the various dpf_get_content_XXX() functions below. It's sorted
** in rising content ID order, but we typically ignore that and search it sequentially
** every time. Just keep those searches out from the inner loops, and things should be fine.
*/
static ContentInfo content_info[] = {
	{ DPC_NAME,		N_("Name"),			"name",		N_("Name") },
	{ DPC_SIZE,		N_("Size"),			"size",		N_("Size") },
	{ DPC_BLOCKS,		N_("Blocks"),			"blocks",	N_("Blocks") },
	{ DPC_BLOCKSIZE,	N_("Block Size"),		"blocksize",	N_("BSize") },
	{ DPC_MODENUM,		N_("Mode, numerical"),		"modenum",	N_("Mode") },
	{ DPC_MODESTR,		N_("Mode, string"),		"modestr",	N_("Mode") },
	{ DPC_NLINK,		N_("Number of links"),		"nlink",	N_("Nlink") },
	{ DPC_UIDNUM,		N_("Owner ID"),			"uid",		N_("Uid") },
	{ DPC_UIDSTR,		N_("Owner Name"),		"uname",	N_("Uname") },
	{ DPC_GIDNUM,		N_("Group ID"),			"gid",		N_("Gid") },
	{ DPC_GIDSTR,		N_("Group Name"),		"gname",	N_("Gname") },
	{ DPC_DEVICE,		N_("Device Number"),		"device",	N_("Device") },
	{ DPC_DEVMAJ,		N_("Device Number, major"),	"devmaj",	N_("DevMaj") },
	{ DPC_DEVMIN,		N_("Device Number, minor"),	"devmin",	N_("DevMin") },
	{ DPC_ATIME,		N_("Date of last Access"),	"atime",	N_("Accessed") },
	{ DPC_MTIME,		N_("Date of last Modification"),"mtime",	N_("Modified") },
	{ DPC_CTIME,		N_("Date of Creation"),		"ctime",	N_("Created") },
	{ DPC_TYPE,		N_("Type Name"),		"type",		N_("Type") },
	{ DPC_ICON,		N_("Icon"),			"icon",		N_("I") }
	};

/* ----------------------------------------------------------------------------------------- */

/* 2000-07-02 -	Initialize the dpformat module. Mainly concerned with some I18N stuff. */
void dpf_initialize(void)
{
	guint	i;

	for(i = 0; i < sizeof content_info / sizeof content_info[0]; i++)
		content_info[i].column_title = _(content_info[i].column_title);
}

/* 1999-05-15 -	Get the "official" content name for column content number <content>. */
const gchar * dpf_get_content_name(DPContent content)
{
	guint	i;

	for(i = 0; i < sizeof content_info / sizeof content_info[0]; i++)
	{
		if(content_info[i].id == content)
			return _(content_info[i].name_full);
	}
	return NULL;
}

/* 1999-05-15 -	Map a string to a DPContent integer. Returns an illegal index if <name> is bad. */
DPContent dpf_get_content_from_name(const gchar *name)
{
	guint	i;

	for(i = 0; i < sizeof content_info / sizeof content_info[0]; i++)
	{
		if(strcmp(content_info[i].name_full, name) == 0)	/* Compatibility: try it in English first. */
			return content_info[i].id;
		else if(strcmp(_(content_info[i].name_full), name) == 0)
			return content_info[i].id;
	}
	return DPC_NUM_TYPES;
}

/* 2002-06-16 -	Get mnemonic (short, single-word, non-translated) name for content. */
const gchar * dpf_get_content_mnemonic(DPContent content)
{
	guint i;

	for(i = 0; i < sizeof content_info / sizeof *content_info; i++)
	{
		if(content_info[i].id == content)
			return content_info[i].name_short;
	}
	return NULL;
}

/* 1999-11-14 -	Map a short, single word mnemonic content name to an integer content identifier.
**		Returns an illegal identifier if the name is unknown.
*/
DPContent dpf_get_content_from_mnemonic(const gchar *name)
{
	guint	i;

	for(i = 0; i < sizeof content_info / sizeof content_info[0]; i++)
	{
		if(strcmp(content_info[i].name_short, name) == 0)
			return (DPContent) content_info[i].id;
	}
	return DPC_NUM_TYPES;
}

/* 1999-05-15 -	Get a recommended column title for a column showing <content>. */
const gchar * dpf_get_content_title(DPContent content)
{
	guint	i;

	for(i = 0; i < sizeof content_info / sizeof content_info[0]; i++)
	{
		if(content_info[i].id == content)
			return content_info[i].column_title;
	}
	return NULL;
}

/* ----------------------------------------------------------------------------------------- */

/* 2000-04-16 -	Build and return a menu containing all the content type names. */
GtkWidget * dpf_get_content_menu(GtkSignalFunc sigfunc, gpointer user)
{
	GtkWidget	*menu, *item;
	guint		i;

	menu = gtk_menu_new();
	for(i = 0; i < sizeof content_info / sizeof content_info[0]; i++)
	{
		item = gtk_menu_item_new_with_label(_(content_info[i].name_full));
		gtk_object_set_user_data(GTK_OBJECT(item), GUINT_TO_POINTER(content_info[i].id));
		gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(sigfunc), user);
		gtk_menu_append(GTK_MENU(menu), item);
	}
	return menu;
}

/* ----------------------------------------------------------------------------------------- */

void dpf_set_default_format(DpCFmt *fmt, DPContent content)
{
	stu_strncpy(fmt->title, dpf_get_content_title(content), sizeof fmt->title);
	fmt->content = content;
	fmt->just  = GTK_JUSTIFY_LEFT;
	fmt->width = 50;

	switch(fmt->content)
	{
		case DPC_NAME:
			fmt->extra.name.show_type = FALSE;
			fmt->width = 250;
			break;
		case DPC_SIZE:
			fmt->extra.size.type = DPSIZE_BYTES;
			fmt->extra.size.digits = 0;
			fmt->extra.size.ticks  = 0;
			fmt->extra.size.tick   = ',';
			g_snprintf(fmt->extra.size.dformat, sizeof fmt->extra.size.dformat, "%%#.%uf", fmt->extra.size.digits);
			fmt->extra.size.dir_show_fs_size = TRUE;
			fmt->just = GTK_JUSTIFY_RIGHT;
			break;
		case DPC_BLOCKS:
			stu_strncpy(fmt->extra.blocks.format, "%lu", sizeof fmt->extra.blocks.format);
			break;
		case DPC_BLOCKSIZE:
			stu_strncpy(fmt->extra.blocksize.format, "%lu", sizeof fmt->extra.blocksize.format);
			fmt->just = GTK_JUSTIFY_RIGHT;
			break;
		case DPC_MODENUM:
			stu_strncpy(fmt->extra.mode.format, "%o", sizeof fmt->extra.mode.format);
			break;
		case DPC_NLINK:
			stu_strncpy(fmt->extra.nlink.format, "%u", sizeof fmt->extra.nlink.format);
			fmt->width = 75;
			break;
		case DPC_UIDNUM:
			stu_strncpy(fmt->extra.uidnum.format, "%u", sizeof fmt->extra.uidnum.format);
			fmt->width = 75;
			break;
		case DPC_GIDNUM:
			stu_strncpy(fmt->extra.gidnum.format, "%u", sizeof fmt->extra.gidnum.format);
			fmt->width = 75;
			break;
		case DPC_DEVICE:
			stu_strncpy(fmt->extra.device.format, "%u", sizeof fmt->extra.device.format);
			fmt->width = 75;
			break;
		case DPC_DEVMAJ:
			stu_strncpy(fmt->extra.devmaj.format, "%u", sizeof fmt->extra.devmaj.format);
			fmt->width = 75;
			break;
		case DPC_DEVMIN:
			stu_strncpy(fmt->extra.devmin.format, "%u", sizeof fmt->extra.devmin.format);
			fmt->width = 75;
			break;
		case DPC_ATIME:
			stu_strncpy(fmt->extra.a_time.format, "%Y-%m-%d %H:%M:%S", sizeof fmt->extra.a_time.format);
			fmt->width = 184;
			break;
		case DPC_CTIME:
			stu_strncpy(fmt->extra.c_time.format, "%Y-%m-%d %H:%M:%S", sizeof fmt->extra.c_time.format);
			fmt->width = 184;
			break;
		case DPC_MTIME:
			stu_strncpy(fmt->extra.m_time.format, "%Y-%m-%d %H:%M:%S", sizeof fmt->extra.c_time.format);
			fmt->width = 184;
			break;
		case DPC_TYPE:
			break;
		case DPC_ICON:
			fmt->width = 16;
			fmt->just = GTK_JUSTIFY_CENTER;
			break;
		default:
			;
	}
}

/* 1998-10-15 -	Moved this old code out of the main module, since it really didn't belong
**		in there.
** 1999-05-15 -	Thanks to the slightly more general routine above, this could be simplified. Great.
** 1999-05-16 -	Now also initializes the default colors.
*/
void dpf_init_defaults(CfgInfo *cfg)
{
	DPFormat	*fmt = &cfg->dp_format[0];
	guint		i;

	for(i = 0; i < 15; i++)
		dpf_set_default_format(&fmt->format[i], (DPContent) i);
	fmt->num_columns = i;

	fmt->sort.content = DPC_NAME;
	fmt->sort.invert  = FALSE;
	fmt->sort.mode    = DPS_DIRS_FIRST;

	strcpy(fmt->def_path, ".");
	fmt->path_above = FALSE;
	fmt->hide_allowed = TRUE;
	fmt->scrollbar_always = FALSE;
	fmt->ctrl_mode = CTM_SET;

	cfg->dp_format[1] = cfg->dp_format[0];

	strcpy(cfg->dp_format[1].def_path, ".");

	cfg->dp_colors.color[DPCOL_PANE_SELECTED].red	= 0xEFBE;
	cfg->dp_colors.color[DPCOL_PANE_SELECTED].green	= 0xEBAD;
	cfg->dp_colors.color[DPCOL_PANE_SELECTED].blue	= 0xFFFF;
	cfg->dp_colors.color[DPCOL_PANE_SELECTED].pixel	= 0UL;

	cfg->dp_colors.color[DPCOL_FOCUS_UNSELECTED].red   = 0xAEBA;
	cfg->dp_colors.color[DPCOL_FOCUS_UNSELECTED].green = 0x0000;
	cfg->dp_colors.color[DPCOL_FOCUS_UNSELECTED].blue  = 0xF7DE;
	cfg->dp_colors.color[DPCOL_FOCUS_UNSELECTED].pixel = 0UL;

	cfg->dp_colors.color[DPCOL_FOCUS_SELECTED].red	   = 0xAEBA;
	cfg->dp_colors.color[DPCOL_FOCUS_SELECTED].green   = 0x0000;
	cfg->dp_colors.color[DPCOL_FOCUS_SELECTED].blue	   = 0x9E79;
	cfg->dp_colors.color[DPCOL_FOCUS_SELECTED].pixel   = 0UL;

	cfg->dp_paning.orientation = DPORIENT_HORIZ;
	cfg->dp_paning.mode	   = DPSPLIT_FREE;
	cfg->dp_paning.value	   = 0.0;

	cfg->dp_history.select = TRUE;
	cfg->dp_history.save   = TRUE;	/* Slightly out of place, but hey. */
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-22 -	Format a name with additional type information in it (slash for directories,
**		asterisk for executables, et cetera). Think ls -F.
*/
static void format_typed_name(gchar *out, const gchar *name, mode_t mode)
{
	if(S_ISLNK(mode))
		sprintf(out, "%s@", name);
	else if(S_ISDIR(mode))
		sprintf(out, "%s/", name);
	else if(S_ISREG(mode) && (mode & S_IXUSR))
		sprintf(out, "%s*", name);
	else if(S_ISFIFO(mode))
		sprintf(out, "%s|", name);
	else if(S_ISSOCK(mode))
		sprintf(out, "%s=", name);
	else
		strcpy(out, name);
}

/* ----------------------------------------------------------------------------------------- */

/* 2002-03-27 -	A new "size" content formatter. This one is a lot more intelligent and feature-rich
**		than the old one, and even incorporates the "IQ Size" content type, too. Way cleaner.
*/
static void put_size(gchar **out, const DC_Size *dcs, const DirRow *row)
{
	gchar	*uptr, *unit = NULL;

	if(!(S_ISDIR(DP_ROW_LSTAT(row).st_mode) && dcs->dir_show_fs_size) && !DP_ROW_FLAGS_CHK(row, DPRF_HAS_SIZE))
		return;
	if(dcs->type == DPSIZE_NATIVE)			/* Native gets it done right away, in a single line. */
		g_snprintf(*out, 64, "%Lu", (guint64) DP_ROW_LSTAT(row).st_size);
	else
	{
		gdouble	size = DP_ROW_LSTAT(row).st_size;

		if(dcs->type == DPSIZE_KILOBYTES)
			size /= 1024.0;
		else if(dcs->type == DPSIZE_MEGABYTES)
			size /= 1024.0 * 1024.0;
		else if(dcs->type == DPSIZE_GIGABYTES)
			size /= 1024.0 * 1024.0 * 1024.0;
		else if(dcs->type == DPSIZE_SMART)	/* Let's do this inline, for speed. */
		{
			if(size < (1 << 10))
				unit = _("B");
			else if(size < (1 << 20))
			{
				size /= 1024.0;
				unit = _("KB");
			}
			else if(size < (1 << 30))
			{
				size /= 1024.0 * 1024.0;
				unit = _("MB");
			}
			else
			{
				size /= 1024.0 * 1024.0 * 1024.0;
				unit = _("GB");
			}
		}
		if(dcs->digits && (size - (guint64) size) != 0.0)
			uptr = *out + 32 + sprintf(*out + 32, dcs->dformat, size - (guint64) size);
		else
			*(uptr = *out + 33) = '\0';
		if(dcs->ticks)
			*out = stu_tickify(*out + 32 + 1, size, dcs->tick);
		else
			*out = stu_tickify(*out + 32 + 1, size, '\0');
		if(unit)
			strcpy(uptr, unit);
	}
}

/* 1998-05-23 -	Format column <column> of given directory line into <out>.
** 1998-05-31 -	Added support for showing the device number (and also to split it into its major and minor
**		components, of course).
** 1998-06-21 -	Added the <dp> argument, since it is required to find the formatting info nowadays.
**		Implemented framework for supporting column-specific configuration. Excellent place to stuff
**		the formatting strings used for date columns, for example.
** 1998-06-24 -	By direct logical extension, I made all numerical columns' formatting configurable.
** 1998-08-10 -	Made this routine a bit more complex, and added it's friend the post-insertion
**		dpf_post_format_column() function (below).
** 2002-04-28 -	Added one level of indirection to the <out> argument, for new size formatting.
*/
void dpf_format_column(MainInfo *min, const DirPane *dp, const DirRow *row, guint column, gchar **out)
{
	const DPFormat	*fmt;
	const DpCFmt	*col_fmt;
	const gchar	*name;
	struct tm	*tm;

	fmt = &min->cfg.dp_format[dp->index];
	if(column >= fmt->num_columns)
		return;

	**out = '\0';
	col_fmt = &fmt->format[column];

	switch(col_fmt->content)
	{
		case DPC_NAME:
			if(col_fmt->extra.name.show_type)
				format_typed_name(*out, DP_ROW_NAME(row), DP_ROW_LSTAT(row).st_mode);
			else
				sprintf(*out, "%s", DP_ROW_NAME(row));
			if(S_ISLNK(DP_ROW_LSTAT(row).st_mode) && col_fmt->extra.name.show_linkname && DP_ROW_LINKNAME(row)[0])
				sprintf(strchr(*out, '\0'), " -> %s", DP_ROW_LINKNAME(row));
			break;
		case DPC_SIZE:
			put_size(out, &col_fmt->extra.size, row);
			break;
		case DPC_BLOCKS:
			sprintf(*out, col_fmt->extra.blocks.format, DP_ROW_LSTAT(row).st_blocks);
			break;
		case DPC_BLOCKSIZE:
			sprintf(*out, col_fmt->extra.blocksize.format, DP_ROW_LSTAT(row).st_blocks * dp->dir.fs.stat.f_bsize);
			break;
		case DPC_MODENUM:
			sprintf(*out, col_fmt->extra.mode.format, DP_ROW_LSTAT(row).st_mode);
			break;
		case DPC_MODESTR:
			stu_mode_to_text(*out, 32, DP_ROW_LSTAT(row).st_mode);
			break;
		case DPC_NLINK:
			sprintf(*out, col_fmt->extra.nlink.format, DP_ROW_LSTAT(row).st_nlink);
			break;

		case DPC_UIDNUM:
			sprintf(*out, col_fmt->extra.uidnum.format, DP_ROW_LSTAT(row).st_uid);
			break;
		case DPC_UIDSTR:
			if((name = usr_lookup_uname(DP_ROW_LSTAT(row).st_uid)) != NULL)
				strcpy(*out, name);
			else
				sprintf(*out, "%d", (gint) DP_ROW_LSTAT(row).st_uid);
			break;
		case DPC_GIDNUM:
			sprintf(*out, col_fmt->extra.gidnum.format, DP_ROW_LSTAT(row).st_gid);
			break;
		case DPC_GIDSTR:
			if((name = usr_lookup_gname(DP_ROW_LSTAT(row).st_gid)) != NULL)
				strcpy(*out, name);
			else
				sprintf(*out, "%d", (gint) DP_ROW_LSTAT(row).st_gid);
			break;

		case DPC_DEVICE:
			sprintf(*out, col_fmt->extra.device.format, DP_ROW_LSTAT(row).st_rdev);
			break;
		case DPC_DEVMAJ:
			sprintf(*out, col_fmt->extra.devmaj.format, DP_ROW_LSTAT(row).st_rdev >> 8);
				break;
		case DPC_DEVMIN:
			sprintf(*out, col_fmt->extra.devmin.format, DP_ROW_LSTAT(row).st_rdev & 0xFF);
			break;

		case DPC_ATIME:
			if((tm = localtime(&DP_ROW_LSTAT(row).st_atime)) != NULL)
				strftime(*out, 32, col_fmt->extra.a_time.format, tm);
			break;
		case DPC_MTIME:
			if((tm = localtime(&DP_ROW_LSTAT(row).st_mtime)) != NULL)
				strftime(*out, 32, col_fmt->extra.m_time.format, tm);
			break;
		case DPC_CTIME:
			if((tm = localtime(&DP_ROW_LSTAT(row).st_ctime)) != NULL)
				strftime(*out, 32, col_fmt->extra.c_time.format, tm);
			break;
		case DPC_TYPE:
			if(DP_ROW_TYPE(row) != NULL)
				sprintf(*out, "%s", DP_ROW_TYPE(row)->name);
			else
				sprintf(*out, "(none)");
			break;
		case DPC_ICON:
			break;
		case DPC_NUM_TYPES:
			break;
	}
}

/* 1998-08-11 -	Format the given row, once it has actually been added to the dirpane. Modified
**		from a routine created yesterday. Takes into account the selection-state for
**		the row, and sets the colors accordingly. This is called when the user does a
**		selection/unselection, too.
** 1998-09-19 -	Something in the clist color setting seems to clear errno sometimes (!), which
**		really isn't acceptable. So now we just save and restore errno. Stupid.
** 1999-03-04 -	No longer cares about the old "selected" flag, since it's no longer there.
*/
void dpf_post_format_row(MainInfo *min, DirPane *dp, gint row)
{
	DPFormat	*fmt;
	const gchar	*iname;
	guint		col;
	gint		oe = errno;
	Style		*stl;
	GdkPixmap	*ipix;
	GdkBitmap	*imsk;

	fmt = &min->cfg.dp_format[dp->index];
	stl = DP_ROW_TYPE(&dp->dir.row[row])->style;
	if(stl == NULL)
	{
		g_warning("DpFormat: No style for \"%s\"", DP_ROW_NAME(&dp->dir.row[row]));
		return;
	}
	iname = stl_style_property_get_icon(stl, SPN_ICON_UNSEL);
	for(col = 0; col < min->cfg.dp_format[dp->index].num_columns; col++)
	{
		if(fmt->format[col].content == DPC_ICON)
		{
			if((ipix = ico_icon_get(min, iname, &imsk)) != NULL)
				gtk_clist_set_pixmap(GTK_CLIST(dp->list), row, col, ipix, imsk);
			else
				gtk_clist_set_text(GTK_CLIST(dp->list), row, col, "");
		}
	}
	dpf_format_colors_row(dp, row);
	errno = oe;
}

/* 1999-05-10 -	Set colors of <row> in <dp> to the correct style's colors. */
void dpf_format_colors_row(DirPane *dp, gint row)
{
	Style			*stl;
	register GdkColor	*col;

	stl = DP_ROW_TYPE(&dp->dir.row[row])->style;

	if((col = (GdkColor *) stl_style_property_get_color(stl, SPN_COL_UNSEL_BG)) != NULL)
		gtk_clist_set_background(GTK_CLIST(dp->list), row, col);
	gtk_clist_set_foreground(GTK_CLIST(dp->list), row, (GdkColor *) stl_style_property_get_color(stl, SPN_COL_UNSEL_FG));
}
