/**
 * @file meta-account.c Meta-Account API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "meta-account.h"
#include "meta-presence.h"
#include "filter.h"

static void
_galagod_dbus_accounts_emit_account_removed(GalagoAccount *account)
{
	GalagoService *service;
	DBusMessage *message;
	DBusMessageIter iter;

	g_return_if_fail(account != NULL);

	service = galago_account_get_service(account);

	message = dbus_message_new_signal(
		galago_object_get_dbus_path(GALAGO_OBJECT(service)),
		GALAGO_DBUS_SERVICE_INTERFACE,
		"AccountRemoved");

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(account));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

GalagodMetaAccount *
galagod_meta_account_new(GalagodMetaService *meta_service,
						 GalagodMetaPerson *meta_person,
						 const char *username)
{
	GalagodMetaAccount *meta_account;
	char *obj_path;

	g_assert(meta_person != NULL);
	g_return_val_if_fail(meta_service != NULL, NULL);
	g_return_val_if_fail(meta_person  != NULL, NULL);
	g_return_val_if_fail(username     != NULL, NULL);

	meta_account = g_new0(GalagodMetaAccount, 1);

	meta_account->account = galago_service_create_account(
		galagod_meta_service_get_service(meta_service),
		galagod_meta_person_get_person(meta_person),
		username);

	obj_path = galagod_account_generate_path(meta_account->account);
	galago_object_set_dbus_path(GALAGO_OBJECT(meta_account->account),
								obj_path);
	g_free(obj_path);

	g_object_set_data(G_OBJECT(meta_account->account),
					  "meta-account", meta_account);

	galagod_account_add_filter(meta_account->account);

	galagod_account_list_add_meta_account(
		galagod_meta_service_get_account_list(meta_service),
		meta_account);

	return meta_account;
}

void
galagod_meta_account_destroy(GalagodMetaAccount *meta_account)
{
	GalagoService *service;
	GalagodMetaService *meta_service;

	g_return_if_fail(meta_account != NULL);

	_galagod_dbus_accounts_emit_account_removed(
		galagod_meta_account_get_account(meta_account));

	service = galago_account_get_service(meta_account->account);
	meta_service = GALAGOD_META_SERVICE(service);

	while (meta_account->accounts != NULL)
	{
		galagod_meta_account_remove_account(meta_account,
			(GalagoAccount *)meta_account->accounts->data);
	}

	galagod_account_list_remove_meta_account(
		galagod_meta_service_get_account_list(meta_service), meta_account);
	galagod_account_remove_filter(meta_account->account);
	g_object_unref(meta_account->account);

	g_free(meta_account);
}

GalagoAccount *
galagod_meta_account_get_account(const GalagodMetaAccount *meta_account)
{
	g_return_val_if_fail(meta_account != NULL, NULL);

	return meta_account->account;
}

GList *
galagod_meta_account_get_accounts(const GalagodMetaAccount *meta_account)
{
	g_return_val_if_fail(meta_account != NULL, NULL);

	return meta_account->accounts;
}

void
galagod_meta_account_add_account(GalagodMetaAccount *meta_account,
								 GalagoAccount *account)
{
	g_return_if_fail(meta_account != NULL);
	g_return_if_fail(account      != NULL);

	meta_account->accounts = g_list_append(meta_account->accounts, account);

	g_object_set_data(G_OBJECT(account), "meta-account", meta_account);

	if (galago_account_is_display_name_set(account))
	{
		galago_account_set_display_name(meta_account->account,
			galago_account_get_display_name(account));
	}

	common_meta_object_merge_attributes(GALAGO_OBJECT(meta_account->account),
										GALAGO_OBJECT(account),
										meta_account->accounts);

	galagod_account_add_filter(account);
}

void
galagod_meta_account_remove_account(GalagodMetaAccount *meta_account,
									GalagoAccount *account)
{
	GalagoPresence *presence;

	g_return_if_fail(meta_account != NULL);
	g_return_if_fail(account      != NULL);

	galagod_account_remove_filter(account);

	g_object_set_data(G_OBJECT(account), "meta-account", NULL);

	meta_account->accounts = g_list_remove(meta_account->accounts,
												account);

	presence = galago_account_get_presence(account, FALSE);

	if (presence != NULL)
	{
		GalagodMetaPresence *meta_presence = GALAGOD_META_PRESENCE(presence);

		if (meta_presence != NULL)
		{
			galagod_meta_presence_remove_presence(meta_presence, presence);

			if (galagod_meta_presence_get_presences(meta_presence) == NULL)
				galagod_meta_presence_destroy(meta_presence);
		}
	}
}
