(**************************************************************************)
(*                                                                        *)
(*  This file is part of Frama-C.                                         *)
(*                                                                        *)
(*  Copyright (C) 2007-2011                                               *)
(*    CEA (Commissariat  l'nergie atomique et aux nergies              *)
(*         alternatives)                                                  *)
(*                                                                        *)
(*  you can redistribute it and/or modify it under the terms of the GNU   *)
(*  Lesser General Public License as published by the Free Software       *)
(*  Foundation, version 2.1.                                              *)
(*                                                                        *)
(*  It is distributed in the hope that it will be useful,                 *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU Lesser General Public License for more details.                   *)
(*                                                                        *)
(*  See the GNU Lesser General Public License version 2.1                 *)
(*  for more details (enclosed in the file licenses/LGPLv2.1).            *)
(*                                                                        *)
(**************************************************************************)

(** State Dependency Graph.
    @since Carbon-20101201 *)

(** {2 Signatures} *)

(** Sub-signature of [Graph.Sig.G] *)
module type G = sig

  (** @since Carbon-20101201 *)
  module V: Graph.Sig.VERTEX with type t = State.t and type label = State.t

  (** @since Carbon-20101201 *)
  module E: Graph.Sig.EDGE with type vertex = State.t
                           and type t = State.t * State.t
  type t
  (** @since Carbon-20101201 *)

  val iter_vertex: (V.t -> unit) -> t -> unit
  (** @since Carbon-20101201 *)

  val fold_vertex: (V.t -> 'a -> 'a) -> t -> 'a -> 'a
  (** @since Carbon-20101201 *)

  val iter_edges_e : (E.t -> unit) -> t -> unit
  (** @since Carbon-20101201 *)

  val fold_edges : (V.t -> V.t -> 'a -> 'a) -> t -> 'a -> 'a
  (** @since Carbon-20101201 *)

  val fold_edges_e : (E.t -> 'a -> 'a) -> t -> 'a -> 'a
  (** @since Carbon-20101201 *)

  val iter_succ : (V.t -> unit) -> t -> V.t -> unit
  (** @since Carbon-20101201 *)

  val fold_succ: (V.t -> 'a -> 'a) -> t -> V.t -> 'a -> 'a
  (** @since Carbon-20101201 *)

  val fold_pred: (V.t -> 'a -> 'a) -> t -> V.t -> 'a -> 'a
  (** @since Carbon-20101201 *)

  val mem_vertex: t -> V.t -> bool
  (** @since Carbon-20101201 *)

  val mem_edge_e: t -> E.t -> bool
  (** @since Carbon-20101201 *)

  val in_degree : t -> V.t -> int
  (** @since Carbon-20101201 *)

  val out_degree : t -> V.t -> int
  (** @since Carbon-20101201 *)

  val nb_vertex: t -> int
(** @since Carbon-20101201 *)

end

(** Signature of a State Dependency Graph.
    It is compatible with the signature of OcamlGraph imperative graph
    [Graph.Sig.I].
    @since Carbon-20101201 *)
module type S = sig

  module G: G
  val graph: G.t

  val add_dependencies: from:State.t -> State.t list -> unit
  (** Add an edge in [graph] from the state [from] to each state of the list.
      @since Carbon-20101201 *)

  val add_codependencies: onto:State.t -> State.t list -> unit
(** Add an edge in [graph] from each state of the list to the state [onto].
    @since Carbon-20101201 *)

end

(** Signature required by [Graph.GraphViZ.Dot]. See the OcamlGraph's
    documentation for additional details.
    @since Carbon-20101201 *)
module type Attributes = sig
  open Graph.Graphviz
  val graph_attributes: 'a -> DotAttributes.graph list
  val default_vertex_attributes: 'a -> DotAttributes.vertex list
  val vertex_name : State.t -> string
  val vertex_attributes: State.t -> DotAttributes.vertex list
  val default_edge_attributes: 'a -> DotAttributes.edge list
  val edge_attributes: State.t * State.t -> DotAttributes.edge list
  val get_subgraph : State.t -> DotAttributes.subgraph option
end

(** The Dynamic State Dependency Graph.
    This module contains both the static states and the dynamic ones, with all
    their dependencies.
    Static states are built at Frama-C initialisation time and are never
    deleted. Dynamic states are mainly generated by dashtables: a dynamic state
    is generated each time a new binding is added on a dashtable and deleted
    each time a binding is removed from a dashtable.
    @since Carbon-20101201 *)
module Dynamic: sig
  include S
  module Attributes: Attributes
  module Dot(A: Attributes) : sig val dump: string -> unit end
  val dump: string -> unit
end

(** The Static State Dependency Graph.
    This graph only contains the static states and the dependencies between
    them. Thus that is a subgraph of the Dynamic State Dependency Graph.
    @since Carbon-20101201 *)
module Static: sig
  include S with module G.V = Dynamic.G.V
  include State.Local
  val add_state: State.t -> State.t list -> unit
(** Add a new static state.
    @since Carbon-20101201 *)
end

module Static_datatype: Datatype.S with type t = Static.t

module Remove_useless_states
  (G:Graph.Sig.P)(X:sig val kind: G.V.t -> State.kind end) :
sig
  val get: G.t -> G.t
end

(* ************************************************************************** *)
(** {2 Internals}

    All this stuff should not be used outside of the Project library. *)
(* ************************************************************************** *)

val add_state_like_the_others: State.t list -> State.t -> unit
(** [add_state_like_others l s] adds [s] to the same graph that each state of
    [l]. If states of [l] belong to different graphs or if [l] is empty, then
    adds [s] to the best graph as possible.
    @since Carbon-20101201 *)

module Vertices: State.Local
module Vertices_datatype: Datatype.S with type t = Vertices.t

(** Build a new dynamic graph, as part of the whole Dynamic State Dependency
    Graph.
    @since Carbon-20101201 *)
module Make_dynamic(T: sig val name: string end) : sig
  include State.Local
  module Datatype: Datatype.S with type t = t
  val add_state: State.t -> State.t list -> unit
  val remove_state: State.t -> unit
  val real_clear: unit -> unit
end

(*
Local Variables:
compile-command: "make -C ../.."
End:
*)
