#!/usr/bin/perl -w

# F*EX GUI upload client
#
# Author: Ulli Horlacher <framstag@rus.uni-stuttgart.de>
#
# Contribs:
#	Beate Herrmann <beate@fex.rus.uni-stuttgart.de> (artwork)
#
# Copyright: GNU General Public License

use Tk;
use Tk::FileSelect;
use Tk::ProgressBar;
use MIME::Base64;
use Getopt::Std;
use Encode;
use Config;
use Socket;
use IO::Handle;
use IO::Socket::INET;
# use IO::Socket::SSL;		# not available for windows
use Digest::MD5 qw(md5_hex);	# encrypted ID / SID 

# require Tk::ErrorDialog;

my $useragent = "schwuppdiwupp";

# callback function for Tk error messages, alternative to Tk::ErrorDialog
sub Tk::Error {
  my ($widget,$error,@locations) = @_;
  my $E = MainWindow->new();
  $E->title('schwuppdiwupp ERROR');
  $E->raise;
  $error =~ s/$0\s*:\s*//;
  $E->Label(-text => $error)->pack();
  $E->Button(-text => 'OK', -command => [$E => 'withdraw'])->pack();
}

# Windows?  
if ($Config{osname} =~ /^mswin/i) {
  $windoof = $Config{osname};
  $ENV{HOME} = $ENV{USERPROFILE};
  $fexhome = $ENV{FEXHOME} || $ENV{HOME}.'\\fex';
  $tmpdir = $ENV{FEXTMP} || $ENV{TMP} || "$fexhome\\tmp";
  $cf = $fexhome .'\\config';
} 
# else we assume UNIX
else {
  $0 =~ s:.*/::;
  $fexhome = $ENV{FEXHOME} || $ENV{HOME}.'/.fex';
  $tmpdir = $ENV{FEXTMP} || "$fexhome/tmp";
  $cf = $fexhome .'/config';
}

# inquire default character set
# cannot use "use I18N::Langinfo" because of no windows support!
eval {
  require I18N::Langinfo;
  I18N::Langinfo->import(qw(langinfo CODESET));
  $CTYPE = langinfo(CODESET());
};
$CTYPE = 'ISO-8859-1' unless $CTYPE;

unless (-d $fexhome) {
  mkdir $fexhome,0700 || die "$0: cannot create directory $fexhome - $!\n";
}
  
$opt_v = 0;
getopts('v');

$file   = shift || '';
$to     = shift || '';
$server = '';
$user   = '';
$id     = '';

# default placements
# -fill both -expand 1
@pl = qw(-side left);
@pr = qw(-side right);
@pb = qw(-side bottom);
@pt = qw(-side top);
@relief = qw(-relief sunken -width 80);

$T = MainWindow->new();

$T->title('F*EX schwuppdiwupp');
$T->update;

# display cool logo
$T->Label(-image => $T->Photo(-data => &logo))->pack(@pt);

$f = $T->Frame->pack(@pt);
$f->Button(-text => 'preferences', -command => \&config)->pack(@pl);
$f->Button(-text => 'about',       -command => \&about )->pack(@pr);

# default frame
$f = $T->Frame->pack(-fill => 'both');

$to_e  = $f->Entry(@relief);
$label = $f->Label(-text => 'Recipient(s): ', -anchor => 'e', -justify => 'right');

$row = 0;
Tk::grid($label, -row => $row, -column => 0, -sticky => 'e');
Tk::grid($to_e,  -row => $row, -column => 1, -sticky => 'ew');
$f->gridRowconfigure(1, -weight => 1);  

$row++;
$file_e = $f->Entry(@relief);
$label  = $f->Label(-text => 'File: ', -anchor => 'e', -justify => 'right');
$browse = $f->Button(-text => "Browse ...",
  		     -command => sub { fsb($T,$file_e,$info)});
Tk::grid($label,  -row => $row, -column => 0, -sticky => 'e');
Tk::grid($file_e, -row => $row, -column => 1, -sticky => 'ew');
Tk::grid($browse, -row => $row, -column => 2, -sticky => 'ew');

$row++;
$info = $f->Label(-relief => 'flat', -width => 80, -anchor => 'w');
Tk::grid($info,   -row => $row, -column => 1, -sticky => 'ew');

$row++;
$comment_e = $f->Entry(@relief);
$label = $f->Label(-text => 'Comment: ', -anchor => 'e', -justify => 'right');
Tk::grid($label,     -row => $row, -column => 0, -sticky => 'e');
Tk::grid($comment_e, -row => $row, -column => 1, -sticky => 'ew');

$to_e->insert(0,"$to");
$file_e->insert(0,$file);

$to_e->focus;

$f = $T->Frame->pack(@pb);
$send = $f->Button(-text => 'send', -command => sub { sendfile($to_e,$file_e,$comment_e) });
$quit = $f->Button(-text => 'quit', -command => sub { $T->destroy; kill 15,$$ });
$send->pack(@pl);
$quit->pack(@pr);

# workaround for wrong window size displaying
$T->configure(-width => 600);

$^W = 0;
MainLoop;
exit;


# file selection browser
sub fsb {
  my ($W,$entry,$label) = @_;
  my ($file,$size,$date);
  local $^W = 1;
  
  my $fs = $W->FileSelect(-directory => '.', -width => 50);
  $fs->title('schwuppdiwupp file selection');
  $file = $fs->Show;
# print "$file\n"; warn $file;
  $file = locale($file);
# $file = encode_utf8($file);
  if ($file and -f $file) {
    # warn sprintf("utf8-flag for >$file< : %d\n",Encode::is_utf8($file));
    $entry->delete(0,'end');
    $entry->insert(0,"$file");
    $entry->xview('end');
    $size = isosize($file);
    $date = isodate((stat $file)[9]);
    $label->configure(-text => "$size    $date");
  }
}


# return date and time in ISO format
sub isodate {
  my @d = localtime shift;
  return sprintf('%d-%02d-%02d %02d:%02d:%02d',
                 $d[5]+1900,$d[4]+1,$d[3],$d[2],$d[1],$d[0]);
}


# return file size in ISO format
sub isosize {
  my $size = -s shift || 0;

  return $size < 2048      ? "$size Bytes":
         $size < 2048*1024 ? int($size/1024).' kB':
                             int($size/1024/1024).' MB';
}

# window for setting configuration data
sub config {
  my $row = 0;
  my $C = $T->Toplevel;
  my $f = $C->Frame->pack(-fill => 'both');
  my ($label,$example);
  
  $C->title('schwuppdiwupp configuration');
  $C->raise;
  
  &read_config;
  
  my $server_e = $f->Entry(@relief);
  $server_e->insert(0,$server);
  $label   = $f->Label(-text => 'F*EX server: ', -anchor => 'e', -justify => 'right');
  $example = $f->Label(-text => 'example: fex.rus.uni-stuttgart.de');
  Tk::grid($label,    -row => $row, -column => 0, -sticky => 'e');
  Tk::grid($server_e, -row => $row, -column => 1, -sticky => 'ew');
#  $row++;
#  Tk::grid($example,  -row => $row, -column => 1, -sticky => 'w');
  
  $f->gridRowconfigure(1, -weight => 1);  

  $row++;
  my $user_e = $f->Entry(@relief);
  $user_e->insert(0,"$user");
  $label   = $f->Label(-text => 'F*EX user e-mail: ', -anchor => 'e', -justify => 'right');
  $example = $f->Label(-text => 'example: framstag@rus.uni-stuttgart.de');
  Tk::grid($label,  -row => $row, -column => 0, -sticky => 'e');
  Tk::grid($user_e, -row => $row, -column => 1, -sticky => 'ew');
#  $row++;
#  Tk::grid($example,  -row => $row, -column => 1, -sticky => 'w');

  $row++;
  my $id_e = $f->Entry(-relief => 'sunken', -width => 16);
  $id_e->insert(0,"$id");
  $label   = $f->Label(-text => 'F*EX auth-ID: ', -anchor => 'e', -justify => 'right');
  $example = $f->Label(-text => 'example: xgHfeQ');
  Tk::grid($label, -row => $row, -column => 0, -sticky => 'e');
  Tk::grid($id_e,  -row => $row, -column => 1, -sticky => 'w');
#  $row++;
#  Tk::grid($example,  -row => $row, -column => 1, -sticky => 'w');

  $server_e->focus;
  
  $f = $C->Frame->pack(@pb);
  $f->Button(-text => 'save', 
             -command => sub { save_config($C,$server_e,$user_e,$id_e) }
  )->pack(@pl);
  $f->Button(-text => 'quit', 
             -command => [$C => 'withdraw']
  )->pack(@pr);
}


# window for programm information
sub about {
  my $A = $T->Toplevel;
  
  $A->title('about schwuppdiwupp');
  $A->raise;
  
  $A->Label( 
    -text => 'schwuppdiwupp is a F*EX client'
  )->pack(@pt);
  $A->Label( 
    -text => 'Author: Ulli Horlacher <framstag@rus.uni-stuttgart.de>'
  )->pack(@pt);
  $A->Label( 
    -text => 'Artwork: Beate Herrmann <beate@fex.rus.uni-stuttgart.de>'
  )->pack(@pt);
  $A->Label( 
    -text => 'F*EX home page: http://fex.rus.uni-stuttgart.de'
  )->pack(@pt);
  $A->Button(-text => 'quit', -command => [$A => 'withdraw'])->pack(@pt);
}


sub save_config {
  my $W      = $_[0];
  my $server = $_[1]->get;
  my $user   = $_[2]->get;
  my $id     = $_[3]->get;
  my $error;
  
  $server =~ s/\s//g;
  $user   =~ s/\s//g;
  $id     =~ s/\s//g;
  
  return unless ($server and $user and $id);

  if (open $cf,'>',$cf) {
    printf {$cf} "server = %s\n",$server;
    printf {$cf} "user = %s\n",$user;
    printf {$cf} "ID = %s\n",$id;
    close $cf;
    $W->withdraw;
  } else {
    $error = "cannot write $cf : $!";
    $T->messageBox(-message => $error, -type => 'Ok');
  }
}


sub read_config {
  if (open $cf,$cf) {
    while (<$cf>) {
      if (/^\s*server\s*=\s(\S+)/i) { $server = $1 }
      if (/^\s*user\s*=\s(\S+)/i)   { $user   = $1 }
      if (/^\s*id\s*=\s(\S+)/i)     { $id     = $1 }
    }
    close $cf;
  }
}


# check all data and send the file
sub sendfile {
  my $to	= $_[0]->get;
  my $file	= $_[1]->get;
  my $comment	= $_[2]->get;
  my $msg;

  &read_config;

  # request config if data is missing
  unless ($server and $user and $id) {
    &config;
    return;
  }

  # do nothing if no recipient or file is given
  return unless ($to and $file);

#  $T->messageBox(-message => "schwuppdiwupp!", -type => 'Ok');
  
  my @r = formdatapost($server,{
	from	=> $user,
	id	=> $id,
	to	=> $to,
	file	=> $file,
	comment	=> $comment,
  });
  
}


# send the selected file by HTTP POST
sub formdatapost {
  my $server = shift;
  my $pv = shift; # post variables hash reference
  my ($from,$to,$file,$comment);
  my ($boundary,$filename,$length,$v,$buf,$fsize,$resume,$seek);
  my ($iaddr,$port);
  my (@hh,@r);
  my ($B,$b,$t);
  my ($t0,$t1,$t2,$tt,$tp);
  my $bs = 65536; # blocksize for reading and sending file
  my $progress = 0;
  my ($pb);
  local $_;
  
  $port = 80;
  $port = 443 if $server =~ s{https://}{};
  
  $server =~ s{http://}{};
  $server =~ s:(/.*)::;
  
  $port = $1 if $server =~ s/:(\d+)//;

  $file = $pv->{file};

  # HTTP needs UTF strings
  $filename = encode_utf8($file);
  
  # strip off path from file name
  if ($windoof) { 
    $filename =~ s/^[a-z]://i;
    $filename =~ s:.*\\:: 
  }
  $filename =~ s:.*/::;

  # file name in locale character set
  $file = locale($file);
  $fsize = -s $file;
  
  $seek = 0;
  $resume = '';

  tcpconnect($server,$port);
  
  # use encrypted session ID methode if not SSL
  if ($port != 443) {
    $pv->{id} = query_sid($server,$port,$pv->{id});
  }

  # already (partially) transfered?
  if ($file and (-s $file||0) > $bs*10) {
    $seek = query_file($server,$port,$pv->{to},$pv->{from},$pv->{id},$filename);
    if ($fsize == $seek) {
      return "$file has been already transfered\n";
    } elsif ($seek and $seek < $fsize) {
      $resume = "(resuming at byte $seek)";
    } elsif ($fsize <= $seek) {
      $seek = 0;
    }
  }

  $boundary = randstring(48);
  $pv->{seek} = $seek;
  
  # send HTTP POST variables
  foreach $v (qw(from to id keep autodelete comment seek)) {
    if ($pv->{$v}) {
      push @hh,"--$boundary";
      push @hh,"Content-Disposition: form-data; name=\"".uc($v)."\"";
      push @hh,"";
      push @hh,encode_utf8($pv->{$v});
    }
  }
  
  # at last, the file attachment
  push @hh,"--$boundary";
  push @hh,"Content-Disposition: form-data; name=\"FILE\"; filename=\"$filename\"";
  push @hh,"Content-Type: application/octet-stream";
  push @hh,"";
  push @hh,"";
  push @hh,"--$boundary--";

  # length of POST 
  $length = length(join('',@hh)) + scalar(@hh)*2 + $fsize - $seek;

  # add the HTTP header (in front)
  unshift @hh, "POST /fup HTTP/1.1",
               "Host: $server:$port",
               "Content-Length: $length",
               "User-Agent: $useragent",
               "Content-Type: multipart/form-data; boundary=$boundary",
               "";

  if ($opt_v) {
    $hh[-2] = '(file content)' if $file;
    print "$_\n" foreach (@hh);
  }
  
  $SIG{PIPE} = \&sigpipehandler;

  # delete last 2 dummy lines
  pop @hh;
  pop @hh;
  
  # send prepared posting
  print {$SH} $_,"\r\n" foreach (@hh);
  
  $t0 = $t2 = time;
  $t1 = 0;

  open $file,$file or die "$0: cannot read $file - $!\n";
  binmode $file;
  seek $file,$seek,0;
  print "sending $file to $to $resume\n" if $opt_v;
  
  $S = $T->Toplevel;
  $S->title('schwuppdiwupp upload status');
  my @pl = qw(-side top -expand yes -pady 2 -anchor w);
  $S->Label(-text => $file)->pack(@pl);
  $S->Label(-text => isodate((stat $file)[9]))->pack(@pl);
  $S->Label(-text => isosize($file))->pack(@pl);
  $pb = $S->ProgressBar(
    -length => 600,
    -padx => 2,
    -pady => 2,
#   -variable => \$progress,
    -blocks => 0,
  );
  $pb->pack(-side => 'top');
    
  $B = 0;
  $SH->autoflush(0);
  
  # read and send the file in chunks
  while ($b = read $file,$buf,$bs) {
    print {$SH} $buf;
    $B += $b;
    $t2 = time;
    if ($t2 != $t1) {
      $progress = int(($B+$seek)/$fsize*100);
      $pb->value($progress);
      $pb->update;
      if ($opt_v) {
        if ($fsize>2097152) {
          printf "transfered: %d MB (%d%%)\r",($B+$seek)/1048576,$progress;
        } else {
          printf "transfered: %d kB (%d%%)\r",($B+$seek)/1024,$progress
        }
      }
      $t1 = $t2;
    }
  }
  
  close $file;
  
  $progress = 100;
  $pb->value($progress);
  $tt = ($t2-$t0)||1;
  $tp = int($B/1024/$tt);
  if ($opt_v) {
    if ($seek) {
      if ($fsize>2097152) {
        printf "transfered: %d MB in %d s with %d kB/s, total %d MB\n",
          int($B/1048576),$tt,$tp,int($fsize/1048576);
      } else {
        printf "transfered: %d kB in %d s with %d kB/s, total %d kB\n",
          int($B/1024),$tt,$tp,int($fsize/1024);
      }
    } else {
      if ($fsize>2097152) {
        printf "transfered: %d MB in %d s with %d kB/s\n",$B/1048576,$tt,$tp;
      } else {
        printf "transfered: %d kB in %d s with %d kB/s\n",$B/1024,$tt,$tp;
      }
    }
  }
  $SH->autoflush(1);
  print {$SH} "\r\n--$boundary--\r\n";

  # read answer from server, try it 3 times
  for (1..3) { 
    sleep 1; 
    @r = <$SH>; 
    last if @r; 
  }
  @r = ("HTTP/0.0 000 No response from server\r\n") unless @r;
  close $SH;
  grep { s/\r//; print "\t$_" } @r if $opt_v;
  
  if ($r[0] =~ /^HTTP.[\d.]+ \d+ (.+)/) { $msg = $1 }
  else                                  { $msg = 'unknown error' }
  
#  $T->messageBox(-message => $msg, -type => 'Ok');
  $S->Label(
    -text => "$tp kB/s", 
    -relief => 'flat',
  )->pack(@pl);
  $S->Button(
    -text => "Server reply: $msg", 
    -command => [$S => 'withdraw'],
  )->pack(@pt);

}

# set up tcp/ip connection
sub tcpconnect {
  my ($server,$port) = @_;
  
  if ($port == 443) {
    eval "use IO::Socket::SSL";
    $SH = IO::Socket::SSL->new(
      PeerAddr => $server,
      PeerPort => $port,
      Proto    => 'tcp',
    );
  } else {
    $SH = IO::Socket::INET->new(
      PeerAddr => $server,
      PeerPort => $port,
      Proto    => 'tcp',
    );
  }
  
  die "cannot connect $server:$port - $!\n" unless $SH;
  binmode $SH;
}

# ask server if it has the file already (partially)
sub query_file {
  my ($server,$port,$to,$from,$id,$filename) = @_;
  my $seek = 0;
  local $_;
  
  $SH->autoflush(1);
  $filename =~ s/([^_=:,;<>()+.\w\-])/'%'.uc(unpack("H2",$1))/ge;
  print {$SH} "HEAD /fop/$to/$from/$filename??&ID=$id HTTP/1.1\r\n\r\n";
  
  $_ = <$SH>;
  unless (defined $_ and /\w/) { 
    die "$0: no response from server\n";
  }
  
  unless (/^HTTP.* 200/) {
    s:HTTP/[\d\. ]+::;
    die "$0: server response: $_" unless /no such file/;
  }
  
  while (<$SH>) { 
    last if /^\r?\n/;
    if (/^Content-Length:\s+(\d+)/i) {
      $seek = $1;
    }
  }
  
  unless ($id =~ /^MD5H:/) {
    close $SH;
    tcpconnect($server,$port);
  }
    
  return $seek;
}

# request session ID and generate encrypted auth ID
sub query_sid {
  my ($server,$port,$id) = @_;
  my ($sid,$req);
  local $_;

  $sid = $id;
  $SH->autoflush(1);
  $req = "GET SID HTTP/1.1";
  print "$req\n" if $opt_v;
  print {$SH} "$req\r\n",
              "Host: $server:$port\r\n",
              "User-Agent: $useragent\r\n",
              "\r\n";
  
  $_ = <$SH>;
  
  unless (defined $_ and /\w/) { 
    print "\n" if $opt_v;
    die "$0: no response from server\n";
  }
  
  s/\r//;
  print "\t$_" if $opt_v;
  
  if (/^HTTP.* 201 (.+)/) {
    $sid = 'MD5H:'.md5_hex($id.$1);
    while (<$SH>) { 
      s/\r//;
      print "\t$_" if $opt_v;
      last if /^\n/;
    }
  } else {
    # no SID support from F*EX-Server (old version?)
    close $SH;
    tcpconnect($server,$port);
  }
  return $sid;
}


sub randstring {
  my $n = shift;
  my @rc = ('A'..'Z','a'..'z',0..9 ); 
  my $rn = @rc; 
  my $rs;
  
  for (1..$n) { $rs .= $rc[int(rand($rn))] };
  return $rs;
}


# broken pipe signal handler
sub sigpipehandler { 
  local $_ = '';
  
  local $SIG{ALRM} = sub { };
  alarm(1);
  @_ = <$SH>;
  if (@_  and $_[0] =~ /^HTTP.* \d+ (.*)/) {
    $T->messageBox(-message => "server error: $1", -type => 'Ok');
  } else {
    $T->messageBox(-message => "got SIGPIPE (F*EX server died?)", -type => 'Ok');
  }
}


# recode name into local charater set
sub locale {
  my $string = shift;

  if ($windoof) {
    return encode($CTYPE,$string);
  }
  
  if ($CTYPE) {
    if ($CTYPE =~ /UTF-?8/i) {
      return $string;
    } elsif (grep { $CTYPE =~ /^$_$/i } Encode->encodings()) {
      return encode($CTYPE,$string);
    } else {
      return encode('ISO-8859-1',$string);
    }
  }
  
  return $string;
}


# the F*EX logo
sub logo {
  return <<EOD;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EOD
}
