/*
 * $Id: cpu_intel_80386_interpreter.c,v 1.18 2009-10-26 10:54:02 sand Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-main.h"

#include "cpu_intel_80386_interpreter.h"

struct cpssp {
	/* Config */

	/* Ports */
	struct sig_host_bus_main *host_bus_main;

	/* Signals */

	/* State */
	int state_power;
	int state_n_reset;
	int state_n_init;

	/* Processes */
	struct process process;
};

static void
cpu_mr(void * _cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	*valp = -1;
	if (sig_host_bus_mr(cpssp->host_bus_main, cpssp, 0, addr, bs, valp) != 0) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				SIG_HOST_BUS_MR, addr);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus_main, cpssp,
				bs, valp);
	}
}

static void
cpu_mw(void * _cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (sig_host_bus_mw(cpssp->host_bus_main, cpssp, 0, addr, bs, val) != 0) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				SIG_HOST_BUS_MW, addr);
		/* delay... */
		sig_host_bus_write_data(cpssp->host_bus_main, cpssp,
				bs, val);
	}
}

static void
cpu_ior(void * _cpssp, uint32_t port, unsigned int bs, uint32_t *valp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	*valp = -1;
	if (sig_host_bus_ior(cpssp->host_bus_main, cpssp, port, bs, valp) != 0) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				SIG_HOST_BUS_IOR, port);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus_main, cpssp,
				bs, valp);
	}
}

static void
cpu_iow(void * _cpssp, uint32_t port, unsigned int bs, uint32_t val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (sig_host_bus_iow(cpssp->host_bus_main, cpssp, port, bs, val) != 0) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				SIG_HOST_BUS_IOW, port);
		/* delay... */
		sig_host_bus_write_data(cpssp->host_bus_main, cpssp,
				bs, val);
	}
}

uint8_t
irq_get(void * _cpssp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;
	uint8_t vector;
	int ret;

	ret = sig_host_bus_inta_addr(cpssp->host_bus_main, cpssp);

	assert(!ret);

	ret = sig_host_bus_inta_data(cpssp->host_bus_main, cpssp, &vector);

	assert(!ret);

	return vector;
}

#define INTERPRETER 1
#include "chip_intel_80386_interpreter.c"
#undef INTERPRETER

void __attribute__((__noreturn__))
step(void *_cpssp)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	sched_to_scheduler();
again:	;
	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		if (! cpssp->state_power) {

			cpssp->process.inst_cnt = cpssp->process.inst_limit;

		} else if (cpssp->state_n_reset != 3 || cpssp->state_n_init != 3) {

			reset();
			cpssp->state_n_reset |= 2;
			cpssp->state_n_init  |= 2;
			cpssp->process.inst_cnt = cpssp->process.inst_limit;

		} else {
			handle_instruction();
			cpssp->process.inst_cnt += 1;
		}
	}

        sched_to_scheduler();

        goto again;
}

static void
power_set(void *_cpssp, unsigned int val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	cpssp->state_power = val;
}

static void
n_reset_set(void *_cpssp, unsigned int n_val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (n_val) {
		/* Reset gone. */
		cpssp->state_n_reset |= 1;
	} else {
		/* New reset. */
		cpssp->state_n_reset = 0;
	}
}

static void
n_init_set(void *_cpssp, unsigned int n_val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (n_val) {
		/* Init gone. */
		cpssp->state_n_init |= 1;
	} else {
		/* New init. */
		cpssp->state_n_init = 0;
	}
}

static void
_irq_set(void *_cpssp, unsigned int val)
{
#if 0
	struct cpssp * cpssp = (struct cpssp *) _cpssp;
#endif
	irq_set(val);
}

static void
_nmi_set(void *_cpssp, unsigned int val)
{
#if 0
	struct cpssp * cpssp = (struct cpssp *) _cpssp;
#endif
	nmi_set(val);
}

void *
cpu_intel_80386_interpreter_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_host_bus *port_conn
)
{
	static const struct sig_boolean_or_funcs irq_func = {
		.set = _irq_set,
	};
	static const struct sig_boolean_or_funcs nmi_func = {
		.set = _nmi_set,
	};
	static const struct sig_boolean_funcs power_funcs = {
		.set = power_set,
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = n_reset_set,
	};
	static const struct sig_boolean_funcs n_init_funcs = {
		.set = n_init_set,
	};

	struct cpssp * cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	memset(cpssp, 0, sizeof(*cpssp));

	/* Running a 20MHz CPU - FIXME */
	cpssp->process.inst_hz = 20*1000*1000;

	cpssp->host_bus_main = port_conn->main;

	sig_boolean_or_connect_in(port_conn->lint0, cpssp, &irq_func);
	sig_boolean_or_connect_in(port_conn->lint1, cpssp, &nmi_func);
	sig_boolean_connect_in(port_conn->power, cpssp, &power_funcs);
	sig_boolean_connect_in(port_conn->n_reset, cpssp, &n_reset_funcs);
	sig_boolean_connect_in(port_conn->n_init, cpssp, &n_init_funcs);

	interpreter_init(cpssp);

	sched_process_init(&cpssp->process, step, cpssp);

	return cpssp;
}

void
cpu_intel_80386_interpreter_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
