/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-subapp.c,v 1.10 2005/02/01 02:32:01 hoa Exp $
 */

#include "etpan-subapp.h"
#include "etpan-subapp-types.h"
#include "etpan-app-subapp.h"
#include <stdlib.h>
#include <string.h>
#include "etpan-errors.h"
#include "etpan-app.h"

void etpan_subapp_idle(struct etpan_subapp * app)
{
  if (app->driver->idle)
    app->driver->idle(app);
}

void etpan_subapp_set_fd(struct etpan_subapp * app, fd_set * fds,
    int * pmaxfd)
{
  if (app->driver->set_fd)
    app->driver->set_fd(app, fds, pmaxfd);
}

void etpan_subapp_handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  if (app->driver->handle_fd)
    app->driver->handle_fd(app, fds);
}

void etpan_subapp_handle_key(struct etpan_subapp * app, int key)
{
  if (app->driver->handle_key)
    app->driver->handle_key(app, key);
}

void etpan_subapp_update_display_size(struct etpan_subapp * app)
{
  app->display_left = app->left;
  app->display_top = app->top;
  app->display_width = app->width;
  app->display_height = app->height;
  
  if (app->title != NULL) {
    app->display_top ++;
    app->display_height --;
  }
}

void etpan_subapp_handle_resize(struct etpan_subapp * app)
{
  if (app->driver->handle_resize)
    app->driver->handle_resize(app);

  app->left = (int) (app->relative_left *
      (double) app->app->width / 100. + 0.5);
  app->top = (int) (app->relative_top *
      (double) app->app->height / 100. + 0.5);
  app->width = (int) (app->relative_width *
      (double) app->app->width / 100. + 0.5);
  app->height = (int) (app->relative_height *
      (double) app->app->height / 100. + 0.5);
  
  etpan_subapp_update_display_size(app);
}

void etpan_subapp_set_relative_coord(struct etpan_subapp * app)
{
  app->relative_left = (double) app->left * 100. /
    (double) app->app->width;
  app->relative_top = (double) app->top * 100. /
    (double) app->app->height;
  app->relative_width = (double) app->width * 100. /
    (double) app->app->width;
  app->relative_height = (double) app->height * 100. /
    (double) app->app->height;
}

void etpan_subapp_display(struct etpan_subapp * app)
{
  WINDOW * w;
  int i;
  
  if (app->app->buf_len == 0) {
    return;
  }
  
  etpan_subapp_display_init(app);
  
  for(i = 0 ; i < app->width ; i ++)
    app->app->fill[i] = ' ';
  app->app->fill[app->width] = '\0';
  
  if (app->app->current_subapp == app)
    attron(app->app->sel_title_attr);
  else
    attron(app->app->title_attr);
  
  /* are thread running */
  if (carray_count(app->thread_list) != 0)
    snprintf(app->app->output, app->display_width + 1,
        " %s ...%s", app->title, app->app->fill);
  else
    snprintf(app->app->output, app->display_width + 1,
        " %s%s", app->title, app->app->fill);
  mvaddstr(app->top, app->left, app->app->output);
  
  if (app->app->current_subapp == app)
    attroff(app->app->sel_title_attr);
  else
    attroff(app->app->title_attr);
  
  w = newwin(app->display_height, app->display_width,
      app->display_top, app->display_left);
  if (w == NULL)
    return;
  
  app->cursor_x = 0;
  app->cursor_y = 0;
  
  if (app->driver->display)
    app->driver->display(app, w);
  
  overwrite(w, stdscr);
  
  delwin(w);
  
  move(app->display_top + app->cursor_y,
      app->display_left + app->cursor_x);
}

int etpan_subapp_set_title(struct etpan_subapp * app, char * title)
{
  char * dup_title;
  
  if (title != NULL) {
    dup_title = strdup(title);
    if (dup_title == NULL)
      return ERROR_MEMORY;
  }
  else
    dup_title = NULL;
  
  if (app->title != NULL)
    free(app->title);
  
  app->title = dup_title;
  
  etpan_subapp_update_display_size(app);
  
  return NO_ERROR;
}

void etpan_subapp_set_color(struct etpan_subapp * app)
{
  if (app->driver->set_color)
    app->driver->set_color(app);
}

void etpan_subapp_free(struct etpan_subapp * subapp)
{
  etpan_subapp_leave(subapp, NULL);
  if (subapp->driver->done)
    subapp->driver->done(subapp);
  etpan_app_remove_subapp(subapp->app, subapp);
  
  carray_free(subapp->thread_list);
  if (subapp->title != NULL)
    free(subapp->title);
  free(subapp);
}

struct etpan_subapp * etpan_subapp_new(struct etpan_app * app,
    struct etpan_subapp_driver * driver)
{
  struct etpan_subapp * subapp;
  int r;

  subapp = malloc(sizeof(* subapp));
  if (subapp == NULL)
    goto err;
  
  subapp->driver = driver;
  subapp->app = app;
  subapp->data = NULL;
  subapp->enabled = 0;
  subapp->display_init = 0;
  subapp->relative_width = 0.;
  subapp->relative_height = 0.;
  subapp->relative_top = 0.;
  subapp->relative_left = 0.;
  subapp->display_width = 0;
  subapp->display_height = 0;
  subapp->display_left = 0;
  subapp->display_top = 0;
  subapp->width = 0;
  subapp->height = 0;
  subapp->top = 0;
  subapp->left = 0;
  subapp->parent = 0;
  subapp->show_cursor = 0;
  subapp->title = NULL;
  subapp->cursor_x = 0;
  subapp->cursor_y = 0;
  subapp->thread_list = carray_new(16);
  if (subapp->thread_list == NULL)
    goto free;
  
  r = etpan_app_add_subapp(app, subapp);
  if (r != NO_ERROR)
    goto free_array;

  if (subapp->driver->init) {
    r = subapp->driver->init(subapp);
    if (r != NO_ERROR)
      goto remove;
  }

  return subapp;
  
 remove:
  etpan_app_remove_subapp(app, subapp);
 free_array:
  carray_free(subapp->thread_list);
 free:
  free(subapp);
 err:
  return NULL;
}

void etpan_subapp_enter(struct etpan_subapp * app,
    struct etpan_subapp * old_app)
{
  if (app->driver->enter)
    app->driver->enter(app, old_app);
  app->enabled = 1;
}

void etpan_subapp_leave(struct etpan_subapp * app,
    struct etpan_subapp * new_app)
{
  if (app->driver->leave)
    app->driver->leave(app, new_app);
  app->enabled = 0;
  app->parent = NULL;
}

int etpan_subapp_display_init(struct etpan_subapp * app)
{
  int r;
  
  if (app->display_init)
    return NO_ERROR;
  
  r = ERROR_NOT_IMPLEMENTED;
  
  if (app->driver->display_init) {
    r = app->driver->display_init(app);
    etpan_subapp_update_display_size(app);
  }

  if (r != NO_ERROR)
    return r;
  
  app->display_init = 1;
  
  return NO_ERROR;
}

void etpan_subapp_display_done(struct etpan_subapp * app)
{
  if (app->driver->display_done)
    app->driver->display_done(app);
}

void etpan_subapp_set_parent(struct etpan_subapp * app,
    struct etpan_subapp * parent_app)
{
  app->parent = parent_app;
}

struct etpan_subapp * etpan_subapp_get_parent(struct etpan_subapp * app)
{
  return app->parent;
}

int etpan_subapp_get_idle_delay(struct etpan_subapp * app)
{
  if (app->driver->get_idle_delay)
    return app->driver->get_idle_delay(app);
  
  return -1;
}

int etpan_subapp_get_idle_udelay(struct etpan_subapp * app,
    struct timeval * delay)
{
  if (app->driver->get_idle_udelay)
    return app->driver->get_idle_udelay(app, delay);
  
  return -1;
}
