/*
 * Sample LDAP adress book client to test LDAP backend
 * $Id: etpan-ldap-test.c,v 1.2 2003/12/24 11:52:07 g_roualland Exp $
 *
 * Sample usage : etpan-ldap-test -h server -b "base dn" name
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <libetpan/carray.h>
#include "etpan-cfg-abook.h"
#include "etpan-cfg-abook-types.h"
#include "etpan-ldap-types.h"
#include "etpan-ldap.h"

#ifndef USE_LDAP
#error "LDAP support needed"
#endif

void optstring(char **str, char *optarg) {
  if (*str != NULL)
    free(*str);
  *str = strdup(optarg);
}

void optint(int *val, char *optarg) {
  *val = atoi(optarg);
}

void usage(char *prog) {
  fprintf(stderr, "usage: %s [-b base] [-D binddn] [-w bindpw] [-h host] [-p port]\n"
	  "      [-P version] [-f filter] [-z sizelimit] [-t timeout] [-a attr=name] key1 key2 ...\n",
	  prog);
  exit(EXIT_FAILURE);
}

static char *_errors[] = { "Success", "System error", "Auth error", "LDAP error", "Partial results", "Internal error" };

static char *ldap_error(unsigned int ret) {
  if (ret < sizeof(_errors))
    return _errors[ret];
  return "Unknown error";
}

int main(int argc, char **argv)
{
  struct etpan_ldap_config ldapcfg;
  struct etpan_ldap_session *ld;
  struct etpan_abook_entry *abook_entry;
  extern char *optarg;
  extern int optind, opterr, optopt;
  carray *result;
  int ret;
  unsigned int i;
    
  memset(&ldapcfg, 0, sizeof(ldapcfg));
  ldapcfg.port = ETPAN_DEFAULT_LDAP_PORT;
  ldapcfg.version = ETPAN_DEFAULT_LDAP_VERSION;
  ldapcfg.filter = strdup(ETPAN_DEFAULT_LDAP_FILTER);
  ldapcfg.hostname = strdup("localhost");
  
  while ((ret = getopt(argc, argv, "h:D:w:p:P:b:z:f:a:t:")) != EOF) {
    switch (ret) {
    case 'h':
      optstring(&(ldapcfg.hostname), optarg);
      break;
    case 'D':
      optstring(&(ldapcfg.binddn), optarg);
      break;
    case 'w':
      optstring(&(ldapcfg.bindpw), optarg);
      break;
    case 'p':
      optint((int *) &(ldapcfg.port), optarg);
      break;
    case 'P':
      optint(&(ldapcfg.version), optarg);
      break;
    case 'b':
      optstring(&(ldapcfg.base), optarg);
      break;
    case 'z':
      optint(&(ldapcfg.sizelimit), optarg);
      break;
    case 'f':
      optstring(&(ldapcfg.filter), optarg);
      break;
    case 't':
      optint(&(ldapcfg.timeout), optarg);
      break;
    case 'a':
      if (!strncasecmp(optarg, "mail=", 5))
	optstring(&(ldapcfg.attrs.mail), optarg+5);
      else if (!strncasecmp(optarg, "cn=", 3))
	optstring(&(ldapcfg.attrs.cn), optarg+3);
      else if (!strncasecmp(optarg, "sn=", 2))
	optstring(&(ldapcfg.attrs.sn), optarg+3);
      else if (!strncasecmp(optarg, "givenname=", 10))
	optstring(&(ldapcfg.attrs.givenname), optarg+10);
      else {
	fprintf(stderr, "%s: bad argument for -a: %s\n", argv[0], optarg);
	usage(argv[0]);
      }
      break;
    default:
      usage(argv[0]);
    }
  }

  if (optind >= argc)
    usage(argv[0]);

  if ((ld = etpan_ldap_session_new(&ldapcfg)) == NULL) {
    fprintf(stderr, "can't create session");
    return EXIT_FAILURE;
  }

  if ((ret = etpan_ldap_connect(ld))) {
    fprintf(stderr, "can't connect to ldap server: %s: %s\n",
	    ldap_error(ret), etpan_ldap_error(ld));
    return EXIT_FAILURE;
  }

  for (; optind < argc; optind++) {
    printf("%s: ", argv[optind]);
    ret = etpan_ldap_search(ld, argv[optind], &result);
    printf("%s: ", ldap_error(ret));
    if (ret != ETPAN_LDAP_NO_ERROR && ret != ETPAN_LDAP_ERROR_PARTIAL) {
      printf("%s\n", etpan_ldap_error(ld));
      continue;
    }

    printf("%d results.\n", result != NULL ? carray_count(result) : 0);
    if (result == NULL)
      continue;

    for (i = 0; i < carray_count(result); i++) {
      abook_entry = (struct etpan_abook_entry *) carray_get(result, i);
      if (abook_entry == NULL)
	continue;
      if (abook_entry->name != NULL) {
	printf("  %s <%s>\n", abook_entry->name, abook_entry->addr);
	free(abook_entry->name);
      } else
	printf("  %s\n", abook_entry->addr);
      free(abook_entry->addr);
      free(abook_entry);
    }
    carray_free(result);
  }
  
  etpan_ldap_session_free(ld);
  return EXIT_SUCCESS;
}
