/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-input-common.c,v 1.8 2004/08/24 01:11:59 hoa Exp $
 */

#include "etpan-input-common.h"
#include <ncurses.h>
#include <string.h>
#include <stdlib.h>
#include "etpan-errors.h"
#include "etpan-subapp.h"
#include "etpan-app-subapp.h"
#include "etpan-app.h"

void etpan_input_common_handle_key(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state,
    int key)
{
  switch (key) {
  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_INPUT_COMMON_CANCEL, state->upcall_data);
    break;
      
  case KEY_CTRL('U'):
    state->count = 0;
    state->pos = 0;
    break;

  case '\n':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_INPUT_COMMON_VALID, state->upcall_data);
    break;

  case KEY_HOME:
  case KEY_CTRL('A'):
    state->pos = 0;
    break;

  case KEY_END:
  case KEY_CTRL('E'):
    state->pos = state->count;
    break;

  case KEY_LEFT:
    if (state->pos > 0)
      state->pos --;
    break;

  case KEY_RIGHT:
    if (state->pos < state->count)
      state->pos ++;
    break;
      
  case KEY_BACKSPACE:
  case 127:
  case 8:
    if (state->pos > 0) {
      memmove(state->data + state->pos - 1, state->data + state->pos,
          state->count - state->pos);
      state->pos --;
      state->count --;
    }
    break;

  case KEY_DC:
  case KEY_CTRL('D'):
    if (state->pos < state->count) {
      memmove(state->data + state->pos, state->data + state->pos + 1,
          state->count - state->pos - 1);
      state->count --;
    }
    break;
      
  default:
    if ((key >= 32) && (key <= 255)) {
      if (state->count + 1 < state->size) {
        memmove(state->data + state->pos + 1, state->data + state->pos,
            state->count - state->pos);
        state->data[state->pos] = key;
        state->count ++;
        state->pos ++;
      }
    }
    else {
      ETPAN_APP_LOG((app->app, "key : %i", key));
    }
    break;
  }
}

void
etpan_input_common_handle_resize(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state)
{
  app->left = 0;
  app->top = app->app->height - 1;
  app->width = app->app->width;
  app->height = 1;
  etpan_subapp_set_relative_coord(app);
}

void etpan_input_common_set_color(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state)
{
  etpan_app_set_color(app->app, "main", &state->main_attr, A_NORMAL);
}

int etpan_input_common_init(struct etpan_input_common_app_state * state)
{
  state->prompt = NULL;
  state->prompt_len = 0;
  state->count = 0;
  state->left = 0;
  state->pos = 0;
  state->data = NULL;
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  return NO_ERROR;
}

void etpan_input_common_flush(struct etpan_input_common_app_state * state)
{
  if (state->prompt != NULL)
     free(state->prompt);
  state->prompt = NULL;
  state->prompt_len = 0;
  if (state->data != NULL)
    free(state->data);
  state->data = NULL;
  state->size = 0;
  state->left = 0;
  state->pos = 0;
  state->upcall = NULL;
  state->upcall_data = NULL;
  state->flags = 0;
}

int etpan_input_common_set(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state,
    char * prompt, size_t size,
    char * default_str,
    int flags,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * data)
{
  etpan_input_common_flush(state);
  
  state = app->data;
  
  /* to get real input size */
  size ++;

  state->prompt = strdup(prompt);
  if (state->prompt == NULL)
    goto err;
  
  state->prompt_len = strlen(prompt);
  state->data = malloc(size);
  if (state->data == NULL)
    goto free_prompt;
  
  state->size = size;
  state->count = 0;
  state->left = 0;
  state->pos = 0;
  
  if (default_str != NULL) {
    strncpy(state->data, default_str, state->size);
    state->data[state->size - 1] = 0;
    state->count = strlen(state->data);
    state->pos = state->count;
  }
  
  state->upcall = upcall;
  state->upcall_data = data;
  state->flags = flags;
  
  return NO_ERROR;

 free_prompt:
  free(prompt);
 err:
  return ERROR_MEMORY;
}

void
etpan_input_common_done(struct etpan_input_common_app_state * state)
{
  etpan_input_common_flush(state);
}


static void update_view(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state)
{
  if (state->pos < state->left) {
    if (state->pos > (size_t) app->display_width / 2)
      state->left = state->pos - app->display_width / 2;
    else
      state->left = 0;
  }
  
  if (state->pos > state->left + app->display_width - state->prompt_len - 1)
    state->left = state->pos - app->display_width + state->prompt_len + 1 +
      app->display_width / 2;
  
  state->data[state->count] = '\0';
}

int etpan_input_common_display(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state, WINDOW * w)
{
  char * output;
  char * fill;
  
  update_view(app, state);
  
  output = app->app->output;
  fill = app->app->fill;
  
  if ((state->flags & ETPAN_INPUT_COMMON_HIDDEN) != 0)
    snprintf(output, app->display_width + 1, "%s%s", state->prompt, fill);
  else
    snprintf(output, app->display_width + 1, "%s%s%s", state->prompt,
        state->data + state->left, fill);
  
  wattron(w, state->main_attr);
  mvwaddstr(w, app->display_height - 1, 0, output);
  wattroff(w, state->main_attr);
  
  app->cursor_y = app->display_height - 1;
  if ((state->flags & ETPAN_INPUT_COMMON_HIDDEN) != 0) {
    app->cursor_x = state->prompt_len;
  }
  else
    app->cursor_x = state->prompt_len + state->pos - state->left;
  
  return NO_ERROR;
}

char * etpan_input_common_get_value(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state)
{
  return state->data;
}

void etpan_input_common_leave(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state,
    struct etpan_subapp * new_app)
{
  etpan_input_common_flush(state);
}

void etpan_input_common_set_value(struct etpan_subapp * app,
    struct etpan_input_common_app_state * state, char * value)
{
  size_t len;
  
  len = strlen(value);
  
  strncpy(state->data, value, state->size);
  if (state->size - 1 < len)
    len = state->size - 1;
  state->data[len] = 0;
  
  state->count = len;
  state->pos = state->count;
}
