/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-account-common.c,v 1.5 2004/11/12 12:13:26 nyoxi Exp $
 */

#include "etpan-account-common.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-cfg-account.h"
#include "etpan-cfg-common.h"
#include "etpan-mime-tools.h"
#include <stdlib.h>

struct account_ref_info {
  int ref;
};

struct account_ref_info * get_account_ref_info(chash * account_ref,
    struct etpan_account_info * account)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = &account;
  key.len = sizeof(account);
  
  r = chash_get(account_ref, &key, &value);
  if (r < 0)
    return NULL;
  else
    return value.data;
}


void etpan_account_common_display(struct etpan_subapp * app,
    struct etpan_account_common_state * state, WINDOW * w)
{
  clistiter * cur;
  unsigned int percent;
  unsigned int list_lines;
  char * output;
  char * fill;
  unsigned int y;
  unsigned int i;
  char * buffer;

  list_lines = app->display_height - 1;

  buffer = app->app->buffer;
  output = app->app->output;
  fill = app->app->fill;
  
  /* update window */
  
  if (state->index > state->count - 1)
    state->index = state->count - 1;
  
  if (state->index < state->first)
    state->first = state->index;
  if (state->index - state->first + 1 > (unsigned int) list_lines)
    state->first = state->index - list_lines + 1;

  cur = clist_nth(state->config->account_config->list, state->first);
  
  /* start to display */
  
  i = state->first;
  wattron(w, state->main_attr);
  
  y = 0;
  while (y < list_lines) {
    struct etpan_account_info * account;
    struct account_ref_info * info;
    int ref;
    
    if (cur == NULL)
      break;
    
    account = cur->data;
    
    ref = 0;
    info = get_account_ref_info(state->account_ref, account);
    if (info != NULL)
      ref = info->ref;
    
    * buffer = '\0';
    if (account->name != NULL) {
      char * decoded_display_name;
      
      decoded_display_name = etpan_decode_mime_header(app->app,
          account->name);
      if (decoded_display_name != NULL) {
        if (account->addr != NULL)
          snprintf(buffer, app->display_width, "%s: %s <%s> (%i)",
              account->id, decoded_display_name, account->addr, ref);
        else
          snprintf(buffer, app->display_width, "%s: %s <> (%i)",
              account->id, decoded_display_name, ref);
        free(decoded_display_name);
      }
    }
    else {
      if (account->addr != NULL)
        snprintf(buffer, app->display_width, "%s: <%s> (%i)",
            account->id, account->addr, ref);
      else
        snprintf(buffer, app->display_width, "%s: <> (%i)",
            account->id, ref);
    }
    if (account == state->config->account_config->default_account)
      snprintf(output, app->display_width, "%s (*)%s", buffer, fill);
    else
      snprintf(output, app->display_width, "%s%s", buffer, fill);
    
    if (i == state->index) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    mvwprintw(w, y, 0, "%s", output);
    if (i == state->index) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    y ++;
    i ++;
    
    cur = clist_next(cur);
  }
  
  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }

  wattroff(w, state->main_attr);
  
  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->count == 0)
    percent = 0;
  else if (state->count == 1)
	percent = 100;
  else
    percent = state->index * 100 / (state->count-1);
  
  snprintf(output, app->display_width + 1,
      " %3i %% | %i accounts | [enter]: select  ^G: cancel%s",
      percent, state->count, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
}

int etpan_account_common_init(struct etpan_account_common_state * state)
{
  state->account_ref = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (state->account_ref == NULL)
    goto err;
  
  state->config = NULL;
  state->count = 0;
  state->index = 0;
  state->first = 0;

  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;

  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

void etpan_account_common_flush(struct etpan_account_common_state * state)
{
  chashiter * cur;
  
  state->config = NULL;
  
  for(cur = chash_begin(state->account_ref) ; cur != NULL ;
      cur = chash_next(state->account_ref, cur)) {
    chashdatum value;
    struct account_ref_info * info;
    
    chash_value(cur, &value);
    info = value.data;
    free(info);
  }
  chash_clear(state->account_ref);
}

static void
recursive_folder_count_ref(struct etpan_account_common_state * state,
    struct mailfolder * folder)
{
  unsigned int i;
  int r;
  chashdatum key;
  chashdatum value;
  struct etpan_account_info * account;
  
  account = etpan_vfolder_get_account(state->config->vfolder_config,
      state->config->account_config, folder);
  if (state->config->account_config->default_account == account)
    account = NULL;
  
  if (account != NULL) {
    struct account_ref_info * info;
    
    info = get_account_ref_info(state->account_ref, account);
    if (info == NULL) {
      info = malloc(sizeof(* info));
      if (info == NULL) {
        /* error */
        return;
      }
      
      info->ref = 0;
      
      key.data = &account;
      key.len = sizeof(account);
      value.data = info;
      value.len = 0;
      r = chash_set(state->account_ref, &key, &value, NULL);
      if (r < 0) {
        free(info);
        info = NULL;
      }
    }
    if (info != NULL) {
      info->ref ++;
    }
  }
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, i);
    recursive_folder_count_ref(state, child);
  }
}

int etpan_account_common_set(struct etpan_account_common_state * state,
    struct etpan_app_config * config,
    struct etpan_account_info * account)
{
  clistiter * cur;
  unsigned int i;
  
  etpan_account_common_flush(state);
  
  state->config = config;
  
  state->count = clist_count(state->config->account_config->list);
  
  if (state->config->vfolder_config != NULL) {
    if (state->config->vfolder_config->root != NULL)
      recursive_folder_count_ref(state,
          state->config->vfolder_config->root);
  }
  
  i = 0;
  for(cur = clist_begin(state->config->account_config->list) ;
      cur != NULL ; cur = clist_next(cur)) {
    if (account == clist_content(cur)) {
      state->index = i;
      break;
    }
    
    i ++;
  }
  
  return NO_ERROR;
}

void etpan_account_common_handle_key(struct etpan_subapp * app,
    struct etpan_account_common_state * state,
    int key)
{
  unsigned int list_lines;
  
  list_lines = app->display_height - 1;
  
  switch (key) {
  case 'k':
  case KEY_UP:
    if (state->index > 0)
      state->index --;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->index < state->count - 1)
      state->index ++;
    break;

  case KEY_NPAGE:
    if (state->index + list_lines - 1 < state->count)
      state->index += list_lines - 1;
    else
      state->index = state->count - 1;
    break;
    
  case KEY_PPAGE:
    if (state->index >= list_lines - 1)
      state->index -= list_lines - 1;
    else
      state->index = 0;
    break;
    
  case KEY_HOME:
    state->index = 0;
    break;
    
  case KEY_END:
    state->index = state->count - 1;
    break;
  }
}

void etpan_account_common_set_color(struct etpan_subapp * app,
    struct etpan_account_common_state * state)
{
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

void etpan_account_common_done(struct etpan_account_common_state * state)
{
  chash_free(state->account_ref);
}


struct etpan_account_info *
etpan_account_common_get_account(struct etpan_account_common_state * state)
{
  clistiter * cur;
  struct etpan_account_info * account;

  cur = clist_nth(state->config->account_config->list, state->index);
  if (cur == NULL)
    return NULL;
  
  account = clist_content(cur);
  
  return account;
}

int etpan_account_common_get_account_ref_count(struct
    etpan_account_common_state * state,
    struct etpan_account_info * account)
{
  struct account_ref_info * info;
  
  info = get_account_ref_info(state->account_ref, account);
  if (info == NULL)
    return 0;
  
  return info->ref;
}
