/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////

#include <mpi.h>
#include <boost/version.hpp>
#include <boost/python.hpp>
#include "Python/esys/lsm/RotThermalParticlePy.h"
#include <stdexcept>
namespace esys
{
  namespace lsm
  {
    RotThermalParticlePy::RotThermalParticlePy() : CRotThermParticle()
    {
    }

    RotThermalParticlePy::RotThermalParticlePy(int id, const Vec3Py &posn, double radius, double mass)
      : CRotThermParticle(radius, mass, posn, Vec3(), Vec3(), id, true)
    {
    }
    
    RotThermalParticlePy::RotThermalParticlePy(const CRotThermParticle &p) : CRotThermParticle(p)
    {
    }

    RotThermalParticlePy::RotThermalParticlePy(const RotThermalParticlePy &p) : CRotThermParticle(p)
    {
    }

    Vec3Py RotThermalParticlePy::getPosn() const
    {
      return Vec3Py(getPos());
    }

    void RotThermalParticlePy::setPosn(const Vec3Py &posn)
    {
      setPos(posn);
    }

    Vec3Py RotThermalParticlePy::getInitialPosn() const
    {
      return Vec3Py(getInitPos());
    }

    Vec3Py RotThermalParticlePy::getLinearVelocity() const
    {
      return Vec3Py(getVel());
    }

    void RotThermalParticlePy::setLinearVelocity(const Vec3Py &vel)
    {
      return setVel(vel);
    }

    Vec3Py RotThermalParticlePy::getLinearForce() const
    {
      return Vec3Py(m_force);
    }

    void RotThermalParticlePy::setLinearForce(const Vec3Py &force)
    {
      setForce(force);
    }

    Vec3Py RotThermalParticlePy::getLinearAcceleration() const
    {
      return Vec3Py(getForce()*getInvMass());
    }

    void RotThermalParticlePy::setLinearAcceleration(const Vec3Py &accel)
    {
      setForce(accel*getMass());
    }

    Vec3Py RotThermalParticlePy::getAngularVelocity() const
    {
      return Vec3Py(getAngVel());
    }

    void RotThermalParticlePy::setAngularVelocity(const Vec3Py &vel)
    {
      setAngVel(vel);
    }

    Vec3Py RotThermalParticlePy::getAngularAcceleration() const
    {
      return getMoment()*getInvInertRot();
    }

    void RotThermalParticlePy::setAngularAcceleration(const Vec3Py &accel)
    {
      setMoment(accel*getInertRot());
    }

    Vec3Py RotThermalParticlePy::getAngularForce() const
    {
      return Vec3Py(m_moment);
    }

    void RotThermalParticlePy::setAngularForce(const Vec3Py &moment)
    {
      setMoment(moment);
    }

    QuaternionPy RotThermalParticlePy::getOrientation() const
    {
      return QuaternionPy(getQuat());
    }

    void RotThermalParticlePy::setOrientation(const QuaternionPy &quat)
    {
      setQuat(quat);
    }

    using boost::python::extract;

    boost::python::tuple
    RotThermalParticlePy::PickleSuite::getstate(boost::python::object pcObj)
    {
      const RotThermalParticlePy &p = extract<const RotThermalParticlePy &>(pcObj);
      boost::python::list l;

      l.append(p.getTag());
      l.append(p.getID());
      l.append(p.getPosn());
      l.append(p.getInitialPosn());
      l.append(p.getLinearVelocity());
      l.append(p.getLinearForce());
      l.append(p.getRad());
      l.append(p.getMass());
      l.append(p.getInertRot());
      l.append(p.getAngularVelocity());
      l.append(p.getAngularForce());
      l.append(p.getOrientation());

      l.append(p.getEquilibRadius());
      l.append(p.getEquilibTemperature());
      l.append(p.getCp());
      l.append(p.getThermExpansion0());
      l.append(p.getThermExpansion1());
      l.append(p.getThermExpansion2());
      l.append(p.getTemperature());

      return boost::python::make_tuple(pcObj.attr("__dict__"), l);
    }

    void
    RotThermalParticlePy::PickleSuite::setstate(
      boost::python::object pcObj,
      boost::python::tuple state
    )
    {
      // restore the object's __dict__
      boost::python::dict d =
        boost::python::extract<boost::python::dict>(
          pcObj.attr("__dict__")
        )();
      d.update(state[0]);

      RotThermalParticlePy &p = extract<RotThermalParticlePy &>(pcObj);
      boost::python::list l = extract<boost::python::list>(state[1]);
      int i = -1;
      p.setTag(extract<int>(l[++i]));
      p.setID(extract<int>(l[++i]));
      p.setPosn(extract<const Vec3Py &>(l[++i]));
      p.setInitPos(extract<const Vec3Py &>(l[++i]));
      p.setLinearVelocity(extract<const Vec3Py &>(l[++i]));
      p.setLinearForce(extract<const Vec3Py &>(l[++i]));
      p.setRad(extract<double>(l[++i]));
      p.setMass(extract<double>(l[++i]));
      p.setInertRot(extract<double>(l[++i]));
      p.setAngularVelocity(extract<const Vec3Py &>(l[++i]));
      p.setAngularForce(extract<const Vec3Py &>(l[++i]));
      p.setOrientation(extract<const QuaternionPy &>(l[++i]));

      p.setEquilibRadius(extract<double>(l[++i]));
      p.setEquilibTemperature(extract<double>(l[++i]));
      p.setCp(extract<double>(l[++i]));
      p.setThermExpansion0(extract<double>(l[++i]));
      p.setThermExpansion1(extract<double>(l[++i]));
      p.setThermExpansion2(extract<double>(l[++i]));
      p.setTemperature(extract<double>(l[++i]));
    }

    bool RotThermalParticlePy::PickleSuite::getstate_manages_dict()
    {
      return true;
    }

    using boost::python::arg;
    void exportRotThermalParticle()
    {
      // Check that Boost 1.34.0 or higher is being used.
      // If so, disable auto-generation of C++ signatures for Epydoc
      // (which stumbles over indentation in the auto-generated strings).
      #if ((BOOST_VERSION / 100000 >= 1) \
          && (BOOST_VERSION / 100 % 1000 >= 34)) \
          || (BOOST_VERSION / 100000 >= 2)
        boost::python::docstring_options no_autogen(true,false);
      #endif

      boost::python::class_<RotThermalParticlePy>(
        "RotThermalSphere",
        "EXPERIMENTAL.\n"
        "Rotational sphere with thermal properties.\n"
      )
        .def(boost::python::init<>())
        .def(boost::python::init<const RotThermalParticlePy &>())
        .def(boost::python::init<int,const Vec3Py &, double, double>(
          (
            arg("id"),
            arg("posn"),
            arg("radius"),
            arg("mass")
          ),
          "Construct a rotational spherical particle.\n"
          "@type id: int\n"
          "@kwarg id: Unique identifier for particle.\n"
          "@type posn: L{Vec3<esys.lsm.util.FoundationPy.Vec3>}\n"
          "@kwarg posn: Initial position of particle, centre-point of sphere.\n"
          "@type radius: float\n"
          "@kwarg radius: The radius of the sphere.\n"
          "@type mass: float\n"
          "@kwarg mass: Mass of particle.\n"
        ))
        .def("getId",                   &RotThermalParticlePy::getID)
        .def("getTag",                  &RotThermalParticlePy::getTag)
        .def("setTag",                  &RotThermalParticlePy::setTag)
        .def("getPosn",                 &RotThermalParticlePy::getPosn)
        .def("setPosn",                 &RotThermalParticlePy::setPosn)
        .def("getInitialPosn",          &RotThermalParticlePy::getInitialPosn)
        .def("getLinearVelocity",       &RotThermalParticlePy::getLinearVelocity)
        .def("setLinearVelocity",       &RotThermalParticlePy::setLinearVelocity)
        .def("getLinearAcceleration",   &RotThermalParticlePy::getLinearAcceleration)
        .def("setLinearAcceleration",   &RotThermalParticlePy::setLinearAcceleration)
        .def("getAngularVelocity",      &RotThermalParticlePy::getAngularVelocity)
        .def("setAngularVelocity",      &RotThermalParticlePy::setAngularVelocity)
        .def("getAngularAcceleration",  &RotThermalParticlePy::getAngularAcceleration)
        .def("setAngularAcceleration",  &RotThermalParticlePy::setAngularAcceleration)
        .def("getOrientation",          &RotThermalParticlePy::getOrientation)
        .def("setOrientation",          &RotThermalParticlePy::setOrientation)
        .def("getRadius",               &RotThermalParticlePy::getRad)
        .def("getRad",                  &RotThermalParticlePy::getRad)
        .def("getCenter",               &RotThermalParticlePy::getPosn)
        .def("getCentre",               &RotThermalParticlePy::getPosn)
        .def("getMass",                 &RotThermalParticlePy::getMass)
        .def("getEquilibRadius",        &RotThermalParticlePy::getEquilibRadius)
        .def("setEquilibRadius",        &RotThermalParticlePy::setEquilibRadius)
        .def("getEquilibTemperature",   &RotThermalParticlePy::getEquilibTemperature)
        .def("setEquilibTemperature",   &RotThermalParticlePy::setEquilibTemperature)
        .def("getTemperature",          &RotThermalParticlePy::getTemperature)
        .def("setTemperature",          &RotThermalParticlePy::setTemperature)
        .def("getCp",                   &RotThermalParticlePy::getCp)
        .def("setCp",                   &RotThermalParticlePy::setCp)
        .def("getExpansionCoeff0",      &RotThermalParticlePy::getThermExpansion0)
        .def("setExpansionCoeff0",      &RotThermalParticlePy::setThermExpansion0)
        .def("getExpansionCoeff1",      &RotThermalParticlePy::getThermExpansion1)
        .def("setExpansionCoeff1",      &RotThermalParticlePy::setThermExpansion1)
        .def("getExpansionCoeff2",      &RotThermalParticlePy::getThermExpansion2)
        .def("setExpansionCoeff2",      &RotThermalParticlePy::setThermExpansion2)
        ;
    }
  }
}
