# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core import common
from elisa.core.utils.misc import is_hildon_desktop_running
from raval_widgets.finite_circular_list import FiniteCircularList

import pgm
from pgm.graph.group import Group
from pgm.graph.image import Image
from pgm.utils.image import cairo_gradient

class ListItem(Group):

    def __init__(self):
        Group.__init__(self)
        
        # preview drawable
        self.image = Image()
        self.add(self.image, True)
        self.image.bg_a = 0
        self.image.layout = pgm.IMAGE_SCALED
        self.image.alignment = pgm.IMAGE_BOTTOM
        self.image.visible = True

        # reflection drawable
        self.reflection = Image()
        self.add(self.reflection)
        flip_matrix = pgm.mat4x4_new_predefined(pgm.MAT4X4_FLIP_VERTICAL)
        self.reflection.mapping_matrix = flip_matrix
        self.reflection.bg_a = 0
        self.reflection.layout = pgm.IMAGE_SCALED
        self.reflection.opacity = 150
        self.reflection.y += self.image.height*0.9
        self.reflection.alignment = pgm.IMAGE_TOP
        self.reflection.visible = True


plugin_registry = common.application.plugin_registry
BaseListView = plugin_registry.get_component_class('raval:list_view')

class MainMenuView(BaseListView, FiniteCircularList):
    
    def clean(self):
        super(MainMenuView, self).clean()
        self.canvas = None

    def create_widgets(self):
        super(MainMenuView, self).create_widgets()

        canvas = self.frontend.context.viewport_handle.get_canvas()
        self.width = canvas.width * 0.9
        self.height = canvas.height
        self.x = (canvas.width - self.width)/2.0
        self.y = (canvas.height - self.height)/2.0 + self.height * 0.1
        
        if is_hildon_desktop_running():
            self._create_buttons()
        
    def _create_buttons(self):
        #Hack for moblin 
        
        icon_size = (0.4, 0.4)
        icon_opacity = 80
        canvas = self.frontend.context.viewport_handle.get_canvas()
        
        icon = self.frontend.theme.get_media('moblin_home_icon')
        self.home_image = Image()
        self.home_image.bg_color = (0, 0, 0, 0)
        self.home_image.set_from_file(icon)
        self.home_image.alignment = pgm.IMAGE_TOP_LEFT
        self.home_image.position = (-self.x, -self.y, 0.0)
        self.home_image.size = icon_size
        self.home_image.visible = True
        self.home_image.opacity = icon_opacity
        self.home_image.connect('clicked', self._home_clicked_callback)
        
        icon = self.frontend.theme.get_media('moblin_close_icon')
        self.close_image = Image()
        self.close_image.bg_color = (0, 0, 0, 0)
        self.close_image.set_from_file(icon)
        self.close_image.alignment = pgm.IMAGE_TOP_RIGHT
        self.close_image.position = (canvas.width - icon_size[0] - self.x, -self.y, 0.0)
        self.close_image.size = icon_size
        self.close_image.visible = True
        self.close_image.opacity = icon_opacity
        self.close_image.connect('clicked', self._close_clicked_callback)
        
        self.add(self.home_image) 
        self.add(self.close_image) 
    
    def _home_clicked_callback(self, drawable, x, y, z, button, time):
        self.frontend.context.reduce_window()
        
    def _close_clicked_callback(self, drawable, x, y, z, button, time):
        common.application.stop()
        

    def create_item(self, model):
        list_item = ListItem()
        self.load_from_theme(model.theme_icon, list_item.image)
        icon = self.frontend.theme.get_media(model.theme_icon)
        cairo_gradient(icon, list_item.reflection, 0.4)

        return list_item

    def do_selected_item_changed(self, index):
        self.controller.current_index = index

    def do_child_clicked(self, index):
        old_selected = self.selected_item
        super(FiniteCircularList, self).do_child_clicked(index)

        if old_selected == self.selected_item:
            return self.controller.activate_item(self.frontend)
