/* Copyright (C) 2001, 2004, 2009 and 2010 Chris Vine


 The following code declares classes to read from and write to
 Unix file descriptors.

 The whole work comprised in files fdstream.h and fdstream.tcc is
 distributed by Chris Vine under the GNU Lesser General Public License
 as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file or uses macros or
inline functions (of any length) should by reason only of that
instantiation or use be subject to the restrictions of use in the GNU
Lesser General Public License.  With that in mind, the words "and
macros, inline functions and instantiations of templates (of any
length)" shall be treated as substituted for the words "and small
macros and small inline functions (ten lines or less in length)" in
the fourth paragraph of section 5 of that licence.  This does not
affect any other reason why object code may be subject to the
restrictions in that licence (nor for the avoidance of doubt does it
affect the application of section 2 of that licence to modifications
of the source code in this file).

The attach(), close() and xsputn() methods added by Chris Vine, 2001.
All the classes were rewritten, and also provided in template form for
wide characters, by Chris Vine 2004.
*/

/**
 * @defgroup fdstreams fdstreams
 *
 * \#include <c++-gtk-utils/fdstream.h>
 *
 * The c++-gtk-utils library contains classes providing streambuffers
 * and stream objects for unix file descriptors..  By default, like
 * the fstream::fstream(int fd) and fstream::attach(int fd) extensions
 * in libstdc++-v2, the destructors of these classes close the file
 * descriptors concerned, which helps exception safety (the attach()
 * method will also close any previous file descriptor).  If this
 * behaviour is not wanted, pass `false' as the second argument of
 * fdostream/fdistream constructor or of the attach() method.  This
 * will enable the same file descriptor to be used successively for,
 * say, reading and writing, or to be shared between fdistream and
 * fdostream objects (but take great care if doing the latter, since
 * the file pointers can easily get out of sync).
 *
 * The streambuf classes provide a block read and write in xsgetn()
 * and xsputn().  They operate (after appropriately vacating and
 * resetting the buffers) by doing a block read and write by calling
 * Unix read() and write() and are very efficient for large block
 * reads (those significantly exceeding the buffer size).  If users
 * want all reads and writes to go through the buffers, by using
 * std::basic_streambuf<charT, Traits>::xsputn() and
 * std::basic_streambuf<charT, Traits>::xsgetn() then the symbol
 * FDSTREAM_USE_STD_N_READ_WRITE can be defined for the purpose.
 * (libstdc++-3 provides efficient inbuilt versions of these
 * std::basic_streambuf functions for block reads not significantly
 * larger than the buffer size.)
 *
 * 4 characters are stored and available for putback.  However, if the
 * symbol FDSTREAM_USE_STD_N_READ_WRITE is not defined, then a call to
 * fdinbuf::xsgetn() with a request for less than 4 characters will
 * result in less than 4 characters available for putback (if xsgetn()
 * obtains some characters but less than 4, only the number of
 * characters returned by xsgetn() is guaranteed to be available for
 * putback).
 * 
 * @note 1.  With the wide stream classes, there is a direct byte for
 * byte read() and write() of wide characters into bytes on the
 * relevant target, without taking on board the inefficiencies of
 * endian adjustment for platform independence.  The wfdostream and
 * wfdistream classes are therefore suitable for use with temporary
 * files, pipes, fifos, unix domain sockets and network sockets on
 * localhost, but not for sockets communicating with other computers
 * over a network, nor files which may be distributed to and read by
 * other computers with different endianness.  Where wide characters
 * are to be exported to other machines, it is best to convert to and
 * from UTF-8 with Utf8::uniwide_from_utf8(), Utf8::uniwide_to_utf8(),
 * Utf8::wide_from_utf8() or Utf8::wide_to_utf8(), and to use
 * fdostream/fdistream with the converted text.
 * 
 * @note 2.  Users cannot (except by derivation) use the virtual
 * protected methods of the streambuffer classes, including xsgetn()
 * and xsputn().  Instead, if they want direct access to the
 * streambuffer other than through the
 * fdostream/wfdostream/fdistream/wfdistream methods, they should use
 * the public forwarding functions provided by std::streambuf base
 * class.
*/

#ifndef CGU_FDSTREAM_H
#define CGU_FDSTREAM_H

// see above for what this does
//#define FDSTREAM_USE_STD_N_READ_WRITE 1

#include <unistd.h>
#include <errno.h>
#include <istream>
#include <ostream>
#include <streambuf>

#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

/*
The following convenience typedefs appear at the end of this file:
typedef basic_fdinbuf<char> fdinbuf;
typedef basic_fdoutbuf<char> fdoutbuf;
typedef basic_fdistream<char> fdistream;
typedef basic_fdostream<char> fdostream;
typedef basic_fdinbuf<wchar_t> wfdinbuf;
typedef basic_fdoutbuf<wchar_t> wfdoutbuf;
typedef basic_fdistream<wchar_t> wfdistream;
typedef basic_fdostream<wchar_t> wfdostream;
*/


/**
 * @headerfile fdstream.h c++-gtk-utils/fdstream.h
 * @brief Output stream buffer for unix file descriptors
 * @sa fdstreams
 * @ingroup fdstreams
 *
 * This class provides an output stream buffer for unix file
 * descriptors.  It does the buffering for the basic_fdostream stream
 * class.
 */
template <class charT , class Traits = std::char_traits<charT> >
class basic_fdoutbuf: public std::basic_streambuf<charT, Traits> {

public:
  typedef charT char_type;
  typedef Traits traits_type;
  typedef typename traits_type::int_type int_type;

private:
  int fd;    // file descriptor
  bool manage;

  static const int buf_size = 1024;   // size of the data write buffer
  char_type buffer[buf_size];
  int flush_buffer();
protected:
  virtual int sync();
  virtual int_type overflow(int_type);
#ifndef FDSTREAM_USE_STD_N_READ_WRITE
  virtual std::streamsize xsputn(const char_type*, std::streamsize);
#endif
public:
 /**
  * @param fd_ The file descriptor to be attached to the
  *            streambuffer, or -1 to attach it latter with the
  *            attach() method.
  *
  * @param manage_ Whether the streambuffer should manage the file
  *                descriptor (that is, close it in its destructor or
  *                when a new file descriptor is attached).
  */
  basic_fdoutbuf(int fd_ = -1, bool manage_ = true);
  virtual ~basic_fdoutbuf();

 /**
  * Attach a new file descriptor to the streambuffer (and close any
  * file descriptor at present managed by it).
  *
  * @param fd_ The new file descriptor to be attached to the
  *            streambuffer.
  *
  * @param manage_ Whether the streambuffer should manage the new file
  *                descriptor (that is, close it in its destructor or
  *                when a further file descriptor is attached).
  */
  void attach_fd(int fd_, bool manage_ = true);

 /**
  * Close the file descriptor at present attached to the streambuffer
  * (if any).
  */
  void close_fd();

 /**
  * Get the file descriptor at present attached to the streambuffer (if
  * any).
  * @return The file descriptor at present attached to the
  * streambuffer, or -1 if none has been attached
  */
  int get_fd() const {return fd;}

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/**
 * @headerfile fdstream.h c++-gtk-utils/fdstream.h
 * @brief Output stream for unix file descriptors
 * @sa fdstreams
 * @ingroup fdstreams
 *
 * This class provides standard ostream services for unix file
 * descriptors.
 */
template <class charT , class Traits = std::char_traits<charT> >
class basic_fdostream: public std::basic_ostream<charT, Traits> {
  basic_fdoutbuf<charT, Traits> buf;
public:
 /**
  * @param fd The file descriptor to be attached to the stream object.
  *
  * @param manage Whether the stream should manage the file descriptor
  *               (that is, close it in its destructor or when a new
  *               file descriptor is attached).
  */
  basic_fdostream(int fd, bool manage = true): std::basic_ostream<charT, Traits>(0),
                                                buf(fd, manage) { // pass the descriptor at construction
    rdbuf(&buf);
  }

  /**
   * With this constructor, the file descriptor must be attached later
   * with the attach method.
   */
  basic_fdostream(): std::basic_ostream<charT, Traits>(0) {       // attach the descriptor later
    rdbuf(&buf);
  }

 /**
  * Attach a new file descriptor to the stream object (and close any file
  * descriptor at present managed by it).
  *
  * @param fd The new file descriptor to be attached to the stream
  *           object.
  *
  * @param manage Whether the stream object should manage the new file
  *               descriptor (that is, close it in its destructor or
  *               when a further file descriptor is attached).
  */
  void attach(int fd, bool manage = true) {buf.attach_fd(fd, manage);}

 /**
  * Close the file descriptor at present attached to the stream object
  * (if any).
  */
  void close() {buf.close_fd();}

 /**
  * Get the file descriptor at present attached to the stream object
  * (if any).
  * @return The file descriptor at present attached to the
  * stream object, or -1 if none has been attached
  */
  int filedesc() const {return buf.get_fd();}

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};


/**
 * @headerfile fdstream.h c++-gtk-utils/fdstream.h
 * @brief Input stream buffer for unix file descriptors
 * @sa fdstreams
 * @ingroup fdstreams
 *
 * This class provides an input stream buffer for unix file
 * descriptors.  It does the buffering for the basic_fdistream stream
 * class.
 */
template <class charT , class Traits = std::char_traits<charT> >
class basic_fdinbuf : public std::basic_streambuf<charT, Traits> {

public:
  typedef charT char_type;
  typedef Traits traits_type;
  typedef typename traits_type::int_type int_type;

private:
  int fd;    // file descriptor
  bool manage;

  static const int putback_size = 4;     // size of putback area
  static const int buf_size = 1024;      // size of the data buffer
  char_type buffer[buf_size + putback_size];  // data buffer
  void reset();
protected:
  virtual int_type underflow();
#ifndef FDSTREAM_USE_STD_N_READ_WRITE
  virtual std::streamsize xsgetn(char_type*, std::streamsize);
#endif
public:
 /**
  * @param fd_ The file descriptor to be attached to the
  *            streambuffer, or -1 to attach it latter with the
  *            attach() method.
  *
  * @param manage_ Whether the streambuffer should manage the file
  *                descriptor (that is, close it in its destructor or
  *                when a new file descriptor is attached).
  */
  basic_fdinbuf(int fd_ = -1, bool manage_ = true);
  virtual ~basic_fdinbuf();

 /**
  * Attach a new file descriptor to the streambuffer (and close any
  * file descriptor at present managed by it).
  *
  * @param fd_ The new file descriptor to be attached to the
  *            streambuffer.
  *
  * @param manage_ Whether the streambuffer should manage the new file
  *                descriptor (that is, close it in its destructor or
  *                when a further file descriptor is attached).
  */
  void attach_fd(int fd_, bool manage_ = true);

 /**
  * Close the file descriptor at present attached to the streambuffer
  * (if any).
  */
  void close_fd();

 /**
  * Get the file descriptor at present attached to the streambuffer (if
  * any).
  * @return The file descriptor at present attached to the
  * streambuffer, or -1 if none has been attached
  */
  int get_fd() const {return fd;}

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/**
 * @headerfile fdstream.h c++-gtk-utils/fdstream.h
 * @brief Input stream for unix file descriptors
 * @sa fdstreams
 * @ingroup fdstreams
 *
 * This class provides standard istream services for unix file
 * descriptors.
 */
template <class charT , class Traits = std::char_traits<charT> >
class basic_fdistream : public std::basic_istream<charT, Traits> {
  basic_fdinbuf<charT , Traits> buf;
public:
 /**
  * @param fd The file descriptor to be attached to the stream object.
  *
  * @param manage Whether the stream should manage the file descriptor
  *               (that is, close it in its destructor or when a new
  *               file descriptor is attached).
  */
  basic_fdistream (int fd, bool manage = true) : std::basic_istream<charT, Traits>(0),
                                                 buf(fd, manage) { // pass the descriptor at construction
    rdbuf(&buf);
  }

  /**
   * With this constructor, the file descriptor must be attached later
   * with the attach method.
   */
  basic_fdistream () : std::basic_istream<charT, Traits>(0) {      // attach the descriptor later
    rdbuf(&buf);
  }

 /**
  * Attach a new file descriptor to the stream object (and close any file
  * descriptor at present managed by it).
  *
  * @param fd The new file descriptor to be attached to the stream
  *           object.
  *
  * @param manage Whether the stream object should manage the new file
  *               descriptor (that is, close it in its destructor or
  *               when a further file descriptor is attached).
  */
  void attach(int fd, bool manage = true) {buf.attach_fd(fd, manage);}

 /**
  * Close the file descriptor at present attached to the stream object
  * (if any).
  */
  void close() {buf.close_fd();}

 /**
  * Get the file descriptor at present attached to the stream object
  * (if any).
  * @return The file descriptor at present attached to the
  * stream object, or -1 if none has been attached
  */
  int filedesc() const {return buf.get_fd();}

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/**
 * @defgroup fdstreams fdstreams
 */
/**
 * @typedef fdinbuf.
 * @brief Input stream buffer for file descriptors for char type
 * @ingroup fdstreams
 */
typedef basic_fdinbuf<char> fdinbuf;

/**
 * @typedef fdoutbuf.
 * @brief Output stream buffer for file descriptors for char type
 * @ingroup fdstreams
 */
typedef basic_fdoutbuf<char> fdoutbuf;

/**
 * @typedef fdistream.
 * @brief Input stream for file descriptors for char type
 * @anchor fdistreamAnchor
 * @ingroup fdstreams
 */
typedef basic_fdistream<char> fdistream;

/**
 * @typedef fdostream.
 * @brief Output stream for file descriptors for char type
 * @anchor fdostreamAnchor
 * @ingroup fdstreams
 */
typedef basic_fdostream<char> fdostream;

/**
 * @typedef wfdinbuf.
 * @brief Input stream buffer for file descriptors for wchar_t type
 * @ingroup fdstreams
 */
typedef basic_fdinbuf<wchar_t> wfdinbuf;

/**
 * @typedef wfdoutbuf.
 * @brief Output stream buffer for file descriptors for wchar_t type
 * @ingroup fdstreams
 */
typedef basic_fdoutbuf<wchar_t> wfdoutbuf;

/**
 * @typedef wfdistream.
 * @brief Input stream for file descriptors for wchar_t type
 * @anchor wfdistreamAnchor
 * @ingroup fdstreams
 */
typedef basic_fdistream<wchar_t> wfdistream;

/**
 * @typedef wfdostream.
 * @brief Output stream for file descriptors for wchar_t type
 * @anchor wfdostreamAnchor
 * @ingroup fdstreams
 */
typedef basic_fdostream<wchar_t> wfdostream;

} // namespace Cgu

#include <c++-gtk-utils/fdstream.tcc>

#endif /*CGU_FDSTREAM_H*/
