#ifndef _ECS_CHAMP_H_
#define _ECS_CHAMP_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associées à la structure `ecs_champ_t' décrivant un champ
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/


/*============================================================================
 *                         Déclaration de la structure
 *============================================================================*/

typedef struct _ecs_champ_t  ecs_champ_t;


/*============================================================================
 *                         Définitions d'énumérations
 *============================================================================*/

typedef enum {

  ECS_CHAMP_NUL = -1,
  ECS_CHAMP_DEF,
  ECS_CHAMP_ATT,
  ECS_CHAMP_FIN

} ECS_CHAMP_E;


#define ECS_CHAMP_DEB ECS_CHAMP_DEF


typedef enum {

  ECS_CHAMP_STATUT_INDEFINI     = -1,
  ECS_CHAMP_STATUT_REF_ELT,            /* Champ de numéros d'éléments         */
  ECS_CHAMP_STATUT_HERITABLE,          /* Champ qui peut être hérité          */

} ECS_CHAMP_STATUT_E;


/*============================================================================
 *                         Définition de macros
 *============================================================================*/

/* Noms des champs */
/*-----------------*/

/*  Champs principaux */

#define ECS_CHAMP_NOM_DEFINIT                                  "definition"


/*  Champs "attribut" */

#define ECS_CHAMP_NOM_CONNECT                                "connectivite"
#define ECS_CHAMP_NOM_CONNECT_INVERSE                "connectivite inverse"
#define ECS_CHAMP_NOM_COULEUR                                     "couleur"
#define ECS_CHAMP_NOM_FILIATION                                 "filiation"
#define ECS_CHAMP_NOM_FAMILLE                                     "famille"
#define ECS_CHAMP_NOM_GROUPE                                       "groupe"
#define ECS_CHAMP_NOM_FAC_PERIO            "faces initiales -> periodiques"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui crée une structure `ecs_champ_t'
 *
 *  La structure devient propriétaire des tableaux tab_pos et tab_val
 *   fournis en argument.
 *
 *   nbr_elt   : Nombre d'éléments à remplir
 *   pas_pos   : Pas des positions  si REGLE
 *   tab_pos   : Positions du champ si non REGLE
 *   tab_val   : Valeurs du champ
 *   typ_val_e : Type des valeurs du champ
 *   nom       : Nom du champ
 *   descr     : Pointeur sur le descripteur
 *   statut_e  : Statut dans une transformation
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__cree
(
       size_t                      nbr_elt,
       size_t                      pas_pos,
       ecs_size_t           *const tab_pos,
       void                 *const tab_val,
       ecs_type_t                  typ_val_e,
 const char                 *const nom,
       ecs_descr_t          *const descr,
       ECS_CHAMP_STATUT_E          statut_e
);


/*----------------------------------------------------------------------------
 *  Fonction libérant une structure `ecs_champ_t' donnée en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__detruit
(
 ecs_champ_t * this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_champ_t' donnée
 *   sur le flux décrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_champ__imprime
(
 const ecs_champ_t  *const this_champ, /* --> Structure à imprimer           */
       size_t              imp_col,
       size_t              nbr_imp,
       bft_file_t   *const fic_imp      /* --> Descripteur fic. d'impression  */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_champ_t'
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_taille
(
 const ecs_champ_t *const this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un champ entièrement réalloué
 *   dont le contenu est copié à partir du champ donné
 *
 *  Le membre donnant le lien sur un champ suivant `l_champ_sui'
 *   n'est pas copié et est mis à `NULL'
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__copie
(
 ecs_champ_t *const champ_init
);


/*----------------------------------------------------------------------------
 *  Fonction qui créé une structure `ecs_champ_t'
 *   à partir d'un tableau `tab_elt' contenant les valeurs du champ.
 *
 *  Les valeurs du champ doivent être des valeurs entières.
 *  Si un élément n'a pas de valeur associée, la valeur correspondante
 *   dans `tab_elt' est `0'
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__transforme_tableau
(                                              /* <-- Pointeur créé           */
       size_t                      nbr_elt,    /* --> Nbr.éléments à remplir  */
 const ecs_int_t            *const tab_elt,    /* --> Valeurs du champ        */
 const char                 *const nom,        /* --> Nom du champ            */
       ecs_descr_t          *const descr,      /* --> Ptr. sur descripteur    */
       ECS_CHAMP_STATUT_E          statut_e    /* --> Statut pour transform.  */
);


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de positions d'un champ donné
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_pos_nbr
(
 const ecs_champ_t *const this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction renvoyant le nombre de valeurs associées à un champ donné
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_val_nbr
(
 const ecs_champ_t  *const this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de descripteurs d'un champ donné
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_descr_nbr
(
 const ecs_champ_t *const this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction retournant le type des valeurs d'un champ donné
 *----------------------------------------------------------------------------*/

ecs_type_t ecs_champ__ret_val_typ
(
 const ecs_champ_t *const this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction renvoyant un pointeur sur le tableau des positions d'une
 *   structure `ecs_champ_t' donnée.
 *
 *  Si les positions correspondent à une REGLE, le tableau est alloué et
 *   construit. S'il est déjà défini, on renvoie simplement son pointeur.
 *   Pour libérer le tableau lorsqu'il correspond à une REGLE (sans le
 *   libérer si ce n'est pas le cas), on appele ecs_champ__libere_pos_tab().
 *----------------------------------------------------------------------------*/

ecs_size_t * ecs_champ__ret_pos_tab
(
 ecs_champ_t  *const this_champ
);


/*----------------------------------------------------------------------------
 *  Fonction libérant un pointeur sur le tableau des positions d'une
 *   structure `ecs_champ_t' donnée.
 *
 *  Si les positions correspondent à une REGLE, le tableau est libéré.
 *   Sinon, il est conservé par la structure ecs_champ_t.
 *----------------------------------------------------------------------------*/

void ecs_champ__libere_pos_tab
(
 const ecs_champ_t  *const this_champ,
       ecs_size_t   *      pos_tab
);


/*----------------------------------------------------------------------------
 *  Fonction qui concatène dans un champ récepteur donné,
 *   un champ à concaténer donné
 *
 *  Les 2 champs à concaténer sont supposés avoir le même nom
 *
 *  La concaténation de 2 champs consiste à concaténer :
 *  - les tables des positions des 2 champs ;
 *  - les tables des valeurs   des 2 champs ;
 *  - les listes chaînées des descripteurs des 2 champs
 *    (nécessaire uniquement pour des champs de type "attribut")
 *
 *  Les autres membres de la structure du champ récepteur ne sont pas modifiés
 *----------------------------------------------------------------------------*/

void ecs_champ__concatene
(
 ecs_champ_t *const champ_recept,
 ecs_champ_t *const champ_concat
);


/*----------------------------------------------------------------------------
 *  Fonction qui prolonge un champ récepteur donné
 *
 *  Il s'agit en fait de concaténer le champ avec un champ vide. Seule la
 *  table des positions est modifiée. Les autres membres de la structure du
 *  champ récepteur ne sont pas modifiés.
 *----------------------------------------------------------------------------*/

void ecs_champ__prolonge
(
 ecs_champ_t *const champ_recept,
 size_t             nbr_elt_prec,
 size_t             nbr_elt_suiv
);


/*----------------------------------------------------------------------------
 *  Fonction qui convertit, si possible,
 *   le tableau des positions d'un champ en REGLE
 *----------------------------------------------------------------------------*/

void ecs_champ__pos_en_regle
(
 ecs_champ_t  *this_champ    /* --> Structure contenant le champ             */
);


#endif /* _ECS_CHAMP_H_ */
