/*============================================================================
 *  Programme principal de l'Enveloppe du Code_Saturne
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

#include "ecs_config.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <locale.h>
#include <stdlib.h>
#include <string.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' système ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>
#include <bft_mem.h>
#include <bft_mem_usage.h>
#include <bft_printf.h>
#include <bft_timer.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_comm.h"
#include "ecs_def.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Pre-Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post.h"

#include "ecs_pre.h"
#include "ecs_pre_meta.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_entmail.h"

#include "ecs_maillage.h"
#include "ecs_maillage_post.h"
#include "ecs_maillage_ncs.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_cmd.h"


/*----------------------------------------------------------------------------
 *  Fichier `include' du paquetage courant associé au fichier courant
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' privés du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_cmd_priv.h"


/*----------------------------------------------------------------------------
 *  Variables globales statiques
 *----------------------------------------------------------------------------*/


const char nom_fic_dump[] = "preprocessor_dump.txt";


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction affichant la taille théorique d'un maillage
 *----------------------------------------------------------------------------*/

static void
_aff_taille_maillage(const ecs_maillage_t *maillage)
{
  size_t       pr_size;
  double       f_size;
  int          nb_div;
  static char  unit_prefix[] = {' ', 'K', 'M', 'G'};

  f_size = ecs_maillage__ret_taille(maillage);

  for (nb_div = 0; f_size > 1024. && nb_div < 3; nb_div++)
    f_size /= 1024.;

  bft_printf(_("  Theoretical mesh size:       %15.3f %cb\n"),
             f_size, unit_prefix[nb_div]);

  f_size = bft_mem_size_current();

  for (nb_div = 1; f_size > 1024.0 && nb_div < 3; nb_div++)
    f_size /= 1024.;

  bft_printf(_("  Theoretical current memory:  %15.3f %cb\n"),
             f_size, unit_prefix[nb_div]);

  f_size = bft_mem_size_max();

  for (nb_div = 1; f_size > 1024.0 && nb_div < 3; nb_div++)
    f_size /= 1024.;

  bft_printf(_("  Theoretical peak memory:     %15.3f %cb\n"),
             f_size, unit_prefix[nb_div]);

  pr_size = bft_mem_usage_max_pr_size();

  if (pr_size > 0) {

    f_size = pr_size;

    for (nb_div = 1; f_size > 1024.0 && nb_div < 3; nb_div++)
      f_size /= 1024.;

    bft_printf(_("  Total memory used:           %15.3f %cb\n"),
               f_size, unit_prefix[nb_div]);

  }

}


/*----------------------------------------------------------------------------
 *  Fonction qui lit les maillages sur fichiers
 *
 *  La fonction renvoie le maillage concaténé
 *----------------------------------------------------------------------------*/

static ecs_maillage_t *
_lit_maillage(const ecs_cmd_t *cmd)
{
  size_t           ific;

  ecs_maillage_t * maillage = NULL;
  ecs_maillage_t * maillage_lu = NULL;

  char           * chaine;

  const char *const titre = "Apres lecture du fichier de maillage ";


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  for (ific = 0; ific < cmd->liste_fic_maillage.nbr; ific++) {

    /* Chargement de la structure */

    if (cmd->liste_fmt_maillage[ific] == ECS_PRE_FORMAT_META)

      maillage_lu
        = ecs_pre_meta__lit_maillage(cmd->liste_fic_maillage.val[ific]);

    else

      maillage_lu = ecs_pre__lit_maillage(cmd->liste_fic_maillage.val[ific],
                                          cmd->liste_fmt_maillage[ific],
                                          cmd->liste_num_maillage[ific],
                                          cmd->liste_grp_maillage[ific*4    ],
                                          cmd->liste_grp_maillage[ific*4 + 1],
                                          cmd->liste_grp_maillage[ific*4 + 2],
                                          cmd->liste_grp_maillage[ific*4 + 3]);

    /* Affichage des infos maillage */

    BFT_MALLOC(chaine,
               strlen(titre) + strlen(cmd->liste_fic_maillage.val[ific]) + 1,
               char);

    strcpy(chaine, titre);
    strcat(chaine, cmd->liste_fic_maillage.val[ific]);


    if (cmd->nbr_dump > 0) {

      ecs_maillage__imprime(maillage_lu,
                            nom_fic_dump,
                            cmd->nbr_dump,
                            chaine);

    }


    if (ific == 0)
      maillage = maillage_lu;

    else
      ecs_maillage__concatene_nodal(maillage, maillage_lu);


    BFT_FREE(chaine);


  } /* Fin : boucle sur les fichiers de maillage à lire */


  if (cmd->liste_fic_maillage.nbr != 1 && cmd->nbr_dump > 0)
    ecs_maillage__imprime(maillage,
                          nom_fic_dump,
                          cmd->nbr_dump,
                          "After concatenation of meshes from files");


  if (cmd->liste_fic_maillage.nbr == 1)
    bft_printf (_("\nDone reading mesh"
                  "\n-----------------\n"));
  else
    bft_printf (_("\nDone reading and concatenating meshes"
                  "\n-------------------------------------\n"));


  _aff_taille_maillage(maillage);


  /* Retour du maillage                                        */
  /*  construit à partir de la concaténation des maillages lus */


  return maillage;
}


/*----------------------------------------------------------------------------
 *  Fonction qui prépare un cas de post traitement si nécessaire
 *----------------------------------------------------------------------------*/

static ecs_post_t *
_init_post(const ecs_cmd_t *cmd)
{

  ecs_post_t     * cas_post;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  cas_post = NULL;


  if (cmd->post_ens != NULL
#if defined(HAVE_CGNS)
      || cmd->post_cgns != NULL
#endif
#if defined(HAVE_MED)
      || cmd->post_med != NULL
#endif
      )
    cas_post = ecs_post__cree_cas(cmd->nom_cas);


  if (cmd->post_ens != NULL) {

    cas_post->post_ens = true;

    cas_post->opt_ens.no_poly     = cmd->post_ens->no_poly;
    cas_post->opt_ens.single_part = cmd->post_ens->simple;
    cas_post->opt_ens.text        = cmd->post_ens->text;
    cas_post->opt_ens.big_endian  = cmd->post_ens->big_endian;

    cas_post->opt_ens.ecr_type[ECS_POST_TYPE_VOLUME]
      = cmd->post_ens->volume;
    cas_post->opt_ens.ecr_type[ECS_POST_TYPE_BORD]
      = cmd->post_ens->bord;
    cas_post->opt_ens.ecr_type[ECS_POST_TYPE_INFO]
      = cmd->post_ens->info;

  }

#if defined(HAVE_CGNS)

  if (cmd->post_cgns != NULL) {

    cas_post->post_cgns = true;

    cas_post->opt_cgns.no_poly = cmd->post_cgns->no_poly;

    cas_post->opt_cgns.ecr_type[ECS_POST_TYPE_VOLUME]
      = cmd->post_cgns->volume;
    cas_post->opt_cgns.ecr_type[ECS_POST_TYPE_BORD]
      = cmd->post_cgns->bord;
    cas_post->opt_cgns.ecr_type[ECS_POST_TYPE_INFO]
      = cmd->post_cgns->info;

  }

#endif /* HAVE_CGNS */

#if defined(HAVE_MED)

  if (cmd->post_med != NULL) {

    cas_post->post_med = true;

    cas_post->opt_med.no_poly = cmd->post_med->no_poly;
    cas_post->opt_med.color_to_group = cmd->post_med->color_to_group;

    cas_post->opt_med.ecr_type[ECS_POST_TYPE_VOLUME]
      = cmd->post_med->volume;
    cas_post->opt_med.ecr_type[ECS_POST_TYPE_BORD]
      = cmd->post_med->bord;
    cas_post->opt_med.ecr_type[ECS_POST_TYPE_INFO]
      = cmd->post_med->info;

  }

#endif /* HAVE_MED */

  return cas_post;
}


/*----------------------------------------------------------------------------
 *  Fonction fusionnant les entités selon leur filiation éventuelle
 *----------------------------------------------------------------------------*/

static void
_recolle_vis(ecs_maillage_t   *maillage,
             const ecs_cmd_t  *cmd,
             ecs_post_t       *cas_post)
{

  ecs_int_t      retour;
  ecs_tab_int_t  liste_fac_mod;
  ecs_tab_int_t  liste_fac_new;
  ecs_tab_int_t  liste_fac_err;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  liste_fac_mod.nbr = 0;
  liste_fac_mod.val = NULL;
  liste_fac_new.nbr = 0;
  liste_fac_new.val = NULL;
  liste_fac_err.nbr = 0;
  liste_fac_err.val = NULL;


  retour = ecs_maillage__recolle_vis(maillage,
                                     &liste_fac_new,
                                     &liste_fac_mod,
                                     &liste_fac_err);

  if (retour > 0) {

    if (cmd->nbr_dump > 0)
      ecs_maillage__imprime(maillage,
                            nom_fic_dump,
                            cmd->nbr_dump,
                            "Mesh joined based on associated data");


    if (cas_post != NULL) {

      if (liste_fac_new.nbr > 0) {

        ecs_maillage_post__ecr_fac_liste(_(ECS_NOM_FACES_NEW_RC_AUTO),
                                         maillage,
                                         liste_fac_new,
                                         false,
                                         ECS_POST_TYPE_INFO,
                                         cas_post);

        BFT_FREE(liste_fac_new.val);

      }

      if (liste_fac_mod.nbr > 0) {

        ecs_maillage_post__ecr_fac_liste(_(ECS_NOM_FACES_MOD_RC_AUTO),
                                         maillage,
                                         liste_fac_mod,
                                         false,
                                         ECS_POST_TYPE_INFO,
                                         cas_post);

        BFT_FREE(liste_fac_mod.val);

      }

    }
    else {

      BFT_FREE(liste_fac_new.val);
      BFT_FREE(liste_fac_mod.val);

    }

    if (cas_post != NULL && liste_fac_err.nbr > 0) {

      ecs_maillage_post__ecr_fac_liste(_(ECS_NOM_FACES_ERR_RC_AUTO),
                                       maillage,
                                       liste_fac_err,
                                       false,
                                       ECS_POST_TYPE_ERREUR,
                                       cas_post);

      BFT_FREE(liste_fac_err.val);

    }

  }
}


/*----------------------------------------------------------------------------
 *  Impression de faces internes correspondant à un critère de sélection donné
 *----------------------------------------------------------------------------*/

static void
_post_faces_int(ecs_maillage_t    *maillage,
                ecs_post_t        *cas_post,
                const ecs_cmd_t   *cmd)
{

  ecs_select_fac_t  * select_fac;

  ecs_tab_int_t       liste_fac_interne;
  ecs_tab_int_t       liste_fac_select;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(maillage != NULL);


  if (cmd->cmd_select_fac_int != NULL) {

    liste_fac_interne = ecs_maillage__liste_fac_int(maillage);

    select_fac = cmd->cmd_select_fac_int;

    bft_printf(_("\n\nSelection of internal faces bearing colors or "
                 "groups:\n"
                 "--------------------------------------------------"
                 "---\n\n"));

    liste_fac_select
      = ecs_maillage__selectionne_fam(maillage,
                                      ECS_ENTMAIL_FAC,
                                      &liste_fac_interne,
                                      select_fac->liste_couleur_fac,
                                      select_fac->liste_groupe_fac,
                                      select_fac->inv_selection);


    /* Sortie pour post-traitement */

    if (cas_post != NULL) {

      ecs_maillage_post__ecr_fac_liste(_(ECS_NOM_FACES_SELECT),
                                       maillage,
                                       liste_fac_select,
                                       true,
                                       ECS_POST_TYPE_INFO,
                                       cas_post);

    }

    liste_fac_select.nbr = 0;
    BFT_FREE(liste_fac_select.val);

  }

  BFT_FREE(liste_fac_interne.val);

}


/*----------------------------------------------------------------------------
 *  Impression de cellules (ou faces pour une maillage de peau uniquement)
 *   correspondant à un critère de sélection donné
 *----------------------------------------------------------------------------*/

static void
_post_ele_liste(ecs_maillage_t       *maillage,
                const ecs_tab_int_t   liste_filtre,
                const char           *nom,
                ecs_post_type_t       type_post,
                ecs_post_t           *cas_post)
{
  ecs_maillage_t  *maillage_post;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  maillage_post
    = ecs_maillage__extrait(maillage,
                            (   ecs_maillage__ret_entmail_max(maillage)
                             == ECS_ENTMAIL_CEL ?
                                ECS_ENTMAIL_CEL : ECS_ENTMAIL_FAC),
                            &liste_filtre,
                            false);

  ecs_maillage_post__ecr(nom,
                         maillage_post,
                         (   ecs_maillage__ret_entmail_max(maillage_post)
                          == ECS_ENTMAIL_CEL ?
                             ECS_ENTMAIL_CEL : ECS_ENTMAIL_FAC),
                         type_post,
                         cas_post);

  maillage_post = ecs_maillage__detruit(maillage_post);

}


/*----------------------------------------------------------------------------
 *  Renvoi de la liste des paramètres de périodicité disponibles
 *----------------------------------------------------------------------------*/

static ecs_param_perio_t *
_liste_param_perio(const ecs_cmd_t  *cmd)
{
  ecs_int_t              ind_rc;
  ecs_int_t              cpt_per = 0;

  ecs_param_perio_t    * liste_param_perio = NULL;

  if (cmd->nbr_cmd_perio > 0) {

    BFT_MALLOC(liste_param_perio, cmd->nbr_cmd_perio, ecs_param_perio_t);

    for (ind_rc = 0; ind_rc < cmd->nbr_cmd_rc; ind_rc++) {

      if (cmd->liste_cmd_rc[ind_rc]->param_rc.param_perio != NULL) {
        liste_param_perio[cpt_per]
          = *(cmd->liste_cmd_rc[ind_rc]->param_rc.param_perio);
        cpt_per += 1;
      }

    }

    assert(cpt_per == cmd->nbr_cmd_perio);

  }

  return liste_param_perio;

}


/*----------------------------------------------------------------------------
 *  Affiche les temps d'exécution
 *----------------------------------------------------------------------------*/

static void
_chrono_total(void)
{
  double  utime;
  double  stime;
  double  time_cpu;
  double  time_tot;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf(_("\n\nTime and memory summary\n"
               "-----------------------\n\n")) ;


  bft_timer_cpu_times(&utime, &stime) ;

  if (utime > 0. || stime > 0.)
    time_cpu = utime + stime;

  else
    time_cpu = bft_timer_cpu_time() ;


  /* (heure de fin d'execution) - (heure de debut d'execution) */


  if (utime > 0. || stime > 0.) {

    bft_printf("  ") ;
    ecs_print_padded_str(_("User CPU time                       (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)utime);

    bft_printf("  ") ;
    ecs_print_padded_str(_("System CPU time                     (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)stime);

  }

  else if (time_cpu > 0.) {

    bft_printf("  ") ;
    ecs_print_padded_str(_("Total CPU time                      (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)time_cpu);

  }


  /* Durée d'exécution  */

  time_tot = bft_timer_wtime();

  if (time_tot > 0.) {

    bft_printf("  ") ;
    ecs_print_padded_str(_("Total time                          (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)time_tot);

    if (time_cpu > 0.) {

      bft_printf("  ") ;
      ecs_print_padded_str(_("Total CPU time / Total time              "),
                           ECS_LNG_AFF_STR) ;
      bft_printf(" : %*.*f\n",
                 ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
                 (float)(time_cpu/time_tot));

    }

  }
}

/*============================================================================
 *                             Fonction principale
 *============================================================================*/

int
main(int    argc ,
     char * argv[])
{
  ecs_cmd_t         * cmd;
  ecs_int_t           ind;

  /* Communication avec le noyau */

  bool                passe_verif;

  ecs_int_t           val_ret;

  bft_file_t        * fic_imp;

  ecs_maillage_t    * maillage;

  ecs_post_t        * cas_post;

  ecs_tab_int_t       liste_fac_mod;
  ecs_tab_int_t       liste_fac_new;
  ecs_tab_int_t       liste_fac_err;
  ecs_tab_int_t       liste_cel_err;
  ecs_tab_int_t       liste_cel_correct;

  ecs_param_perio_t * liste_param_perio;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  ecs_init_gestion_erreur();

#if defined(ECS_ARCH_Linux)

  /*
    setlocale() doit être appellé sous certaines distributions de Linux
    pour que les accents apparaissent correctement dans les chaînes de
    caractères.
  */

  if (getenv("LANG") != NULL)
     setlocale(LC_ALL,"");
  else
     setlocale(LC_ALL,"C");
  setlocale(LC_NUMERIC,"C");

#endif

#if defined(ENABLE_NLS)
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
#endif

  /* Initialisation comptage et gestion mémoire */

  bft_mem_usage_init();

  bft_mem_init(getenv("CS_PREPROCESS_MEM_LOG"));


  /* On vérifie que les dimensions de certains types sont valables */

  if (sizeof(ecs_int_32_t) != 4)
    bft_error(__FILE__, __LINE__, 0,
              _("This program was built with type \"ecs_int_32_t\"\n"
                "of length %d bytes instead of 4. (porting error)\n"),
              sizeof(ecs_int_32_t));

  if (sizeof(ecs_real_32_t) != 4)
    bft_error(__FILE__, __LINE__, 0,
              _("This program was built with type \"ecs_real_32_t\"\n"
                "of length %d bytes instead of 4. (porting error)\n"),
              sizeof(ecs_real_32_t));

#if defined(DEBUG_TYPES)
printf("sizeof(ecs_int_t)  = %" ECS_FORMAT_size_t "\n", sizeof(ecs_int_t));
printf("sizeof(ecs_real_t) = %" ECS_FORMAT_size_t "\n", sizeof(ecs_real_t));
printf("sizeof(int)        = %" ECS_FORMAT_size_t "\n", sizeof(int));
printf("sizeof(long)       = %" ECS_FORMAT_size_t "\n", sizeof(long));
printf("sizeof(short)      = %" ECS_FORMAT_size_t "\n", sizeof(short));
printf("sizeof(size_t)     = %" ECS_FORMAT_size_t "\n", sizeof(size_t));
printf("sizeof(float)      = %" ECS_FORMAT_size_t "\n", sizeof(float));
printf("sizeof(double)     = %" ECS_FORMAT_size_t "\n", sizeof(double));
#endif


  /* Lecture de la ligne de commande */

  cmd = ecs_cmd__lit_arg(argc, argv);


  /*=========================================================*/
  /* Début du traitement en fonction de la ligne de commande */
  /*=========================================================*/

  if (cmd->nbr_dump > 0) {
    /* Pour écraser le fichier d'impression d'une précédente execution */
    fic_imp = bft_file_open(nom_fic_dump,
                            BFT_FILE_MODE_WRITE, BFT_FILE_TYPE_TEXT);
    bft_printf("  %s %s\n", _("Creating file:"), nom_fic_dump);
    bft_file_free(fic_imp);
  }


  /*==========================================================================*/
  /* Lecture des fichiers d'entrée contenant les maillages                    */
  /*==========================================================================*/

  maillage = _lit_maillage(cmd);


  ecs_maillage__calc_coo_ext(maillage);


  /*==========================================================================*/
  /* Tri des types géométriques pour post-traitement Ensight                  */
  /*==========================================================================*/

  ecs_maillage__trie_typ_geo(maillage);


  /*==========================================================================*/
  /* Préparation du Post-traitement                                           */
  /*==========================================================================*/

  cas_post = _init_post(cmd);


  /*==========================================================================*/
  /* Vérification et correction éventuelle de l'orientation des éléments      */
  /*==========================================================================*/

  liste_cel_err.nbr     = 0;
  liste_cel_err.val     = NULL;
  liste_cel_correct.nbr = 0;
  liste_cel_correct.val = NULL;

  ecs_maillage__orient_nodal(maillage,
                             cas_post != NULL ? &liste_cel_err     : NULL,
                             cas_post != NULL ? &liste_cel_correct : NULL,
                             cmd->correct_orient);


  /*==========================================================================*/
  /* Écriture de la connectivité nodale sur fichier pour Post-traitement      */
  /*==========================================================================*/

  if (cas_post != NULL) {

    ecs_maillage_post__ecr(_(ECS_MAILLAGE_NOM_PRINCIPAL),
                           maillage,
                           (   ecs_maillage__ret_entmail_max(maillage)
                            == ECS_ENTMAIL_CEL
                            ?  ECS_ENTMAIL_CEL : ECS_ENTMAIL_FAC),
                           ECS_POST_TYPE_VOLUME,
                           cas_post);

    if (liste_cel_err.nbr > 0)
      _post_ele_liste(maillage,
                      liste_cel_err,
                      _(ECS_MAILLAGE_NOM_ORIENT_ERR),
                      ECS_POST_TYPE_ERREUR,
                      cas_post);
    if (liste_cel_correct.nbr > 0)
      _post_ele_liste(maillage,
                      liste_cel_correct,
                      _(ECS_MAILLAGE_NOM_ORIENT_CORRECT),
                      ECS_POST_TYPE_INFO,
                      cas_post);

  }


  /* On libère les listes des éléments avec problème d'orientation */

  if (liste_cel_err.nbr > 0) {
    BFT_FREE(liste_cel_err.val);
    liste_cel_err.nbr = 0;
  }
  if (liste_cel_correct.nbr > 0) {
    BFT_FREE(liste_cel_correct.val);
    liste_cel_correct.nbr = 0;
  }


  /*==========================================================================*/
  /* Création de la table de connectivité descendante                         */
  /*==========================================================================*/

  /* Passage en connectivité descendante */

  ecs_maillage__connect_descend(maillage);


  if (cmd->nbr_dump > 0)
    ecs_maillage__imprime(maillage,
                          nom_fic_dump,
                          cmd->nbr_dump,
                          "Descending connectivity");


  bft_printf (_("\nEnd of conversion to descending connectivity"
                "\n--------------------------------------------\n"));

  _aff_taille_maillage(maillage);


  /* Nettoyage éventuel */

  ecs_maillage__nettoie_descend(maillage);


  /* Fusion connectivité éventuelle */

  _recolle_vis(maillage, cmd, cas_post);


  /*========================================================================*/
  /* Recollement des faces non conformes                                    */
  /*========================================================================*/

  /* Cas non périodique : on recolle tout avant de passer au périodique */
  /*--------------------------------------------------------------------*/

  for (ind = 0; ind < cmd->nbr_cmd_rc; ind++) {

    if (cmd->liste_cmd_rc[ind]->param_rc.param_perio == NULL) {

      liste_fac_mod.nbr = 0;
      liste_fac_mod.val = NULL;
      liste_fac_new.nbr = 0;
      liste_fac_new.val = NULL;
      liste_fac_err.nbr = 0;
      liste_fac_err.val = NULL;


      val_ret = ecs_maillage__recolle(maillage,
                                      cmd->liste_cmd_rc[ind]->select_fac_rc,
                                      &liste_fac_new,
                                      &liste_fac_mod,
                                      &liste_fac_err,
                                      cmd->liste_cmd_rc[ind]->param_rc);


      if (val_ret != 0) {

        if (cmd->nbr_dump > 0)
          ecs_maillage__imprime(maillage,
                                nom_fic_dump,
                                cmd->nbr_dump,
                                "Joined mesh");


        if (cas_post != NULL) {

          char             *nom_rc = NULL;


          BFT_MALLOC(nom_rc, strlen(_(ECS_NOM_FACES_NEW_RC)) + 5 + 1, char);

          sprintf(nom_rc, "%s %4d", _(ECS_NOM_FACES_NEW_RC),
                  cmd->liste_cmd_rc[ind]->param_rc.num_rc);

          ecs_maillage_post__ecr_fac_liste(nom_rc,
                                           maillage,
                                           liste_fac_new,
                                           false,
                                           ECS_POST_TYPE_INFO,
                                           cas_post);

          BFT_FREE(nom_rc);
          BFT_FREE(liste_fac_new.val);


          BFT_MALLOC(nom_rc, strlen(_(ECS_NOM_FACES_MOD_RC)) + 5 + 1, char);

          sprintf(nom_rc, "%s %4d", _(ECS_NOM_FACES_MOD_RC),
                  cmd->liste_cmd_rc[ind]->param_rc.num_rc);

          ecs_maillage_post__ecr_fac_liste(nom_rc,
                                           maillage,
                                           liste_fac_mod,
                                           false,
                                           ECS_POST_TYPE_INFO,
                                           cas_post);

          BFT_FREE(nom_rc);
          BFT_FREE(liste_fac_mod.val);

        }
        else {

          BFT_FREE(liste_fac_new.val);
          BFT_FREE(liste_fac_mod.val);

        }


        /*
          En cas d'erreurs lors du recollement, on affiche les faces
          posant problème dès que l'on a un cas EnSight, indépendamment
          du mode vérification
        */

        if (cas_post != NULL && liste_fac_err.nbr > 0) {

          char             *nom_rc = NULL;

          BFT_MALLOC(nom_rc, strlen(_(ECS_NOM_FACES_ERR_RC)) + 5 + 1, char);

          sprintf(nom_rc, "%s %4d", _(ECS_NOM_FACES_ERR_RC),
                  cmd->liste_cmd_rc[ind]->param_rc.num_rc);

          ecs_maillage_post__ecr_fac_liste(nom_rc,
                                           maillage,
                                           liste_fac_err,
                                           false,
                                           ECS_POST_TYPE_ERREUR,
                                           cas_post);

          BFT_FREE(nom_rc);

          BFT_MALLOC(nom_rc, strlen(_(ECS_NOM_CEL_ERR_RC)) + 5 + 1, char);

          sprintf(nom_rc, "%s_%d", _( ECS_NOM_CEL_ERR_RC),
                  cmd->liste_cmd_rc[ind]->param_rc.num_rc);

          liste_cel_err = ecs_maillage__liste_cel_fac(maillage,
                                                      liste_fac_err);

          ecs_maillage_post__ecr_tab_liste(nom_rc,
                                           maillage,
                                           liste_cel_err,
                                           ECS_ENTMAIL_CEL,
                                           cas_post);

          BFT_FREE(liste_cel_err.val);
          BFT_FREE(nom_rc);

        }

        BFT_FREE(liste_fac_err.val);

      }

    }

  }


  /* Cas périodique : */
  /*------------------*/

  for (ind = 0; ind < cmd->nbr_cmd_rc; ind++) {

    if (cmd->liste_cmd_rc[ind]->param_rc.param_perio != NULL) {

      liste_fac_mod.nbr = 0;
      liste_fac_mod.val = NULL;
      liste_fac_new.nbr = 0;
      liste_fac_new.val = NULL;
      liste_fac_err.nbr = 0;
      liste_fac_err.val = NULL;


      val_ret = ecs_maillage__recolle(maillage,
                                      cmd->liste_cmd_rc[ind]->select_fac_rc,
                                      &liste_fac_new,
                                      &liste_fac_mod,
                                      &liste_fac_err,
                                      cmd->liste_cmd_rc[ind]->param_rc);


      if (val_ret != 0) {

        if (cmd->nbr_dump > 0)
          ecs_maillage__imprime(maillage,
                                nom_fic_dump,
                                cmd->nbr_dump,
                                "Duplicated and joined periodic mesh");


        if (cas_post != NULL) {

          char *nom_perio = NULL;

          BFT_MALLOC(nom_perio, strlen(_(ECS_NOM_FACES_PERIO)) + 5 + 1, char);

          sprintf(nom_perio, "%s %4d", _(ECS_NOM_FACES_PERIO),
                  (cmd->liste_cmd_rc[ind]->param_rc.param_perio)->num_perio);


          ecs_maillage_post__ecr_fac_liste(nom_perio,
                                           maillage,
                                           liste_fac_new,
                                           false,
                                           ECS_POST_TYPE_INFO,
                                           cas_post);

          BFT_FREE(nom_perio);


          BFT_MALLOC(nom_perio,
                     strlen(_(ECS_NOM_FACES_MOD_PERIO)) + 5 + 1,
                     char);

          sprintf(nom_perio, "%s %4d", _(ECS_NOM_FACES_MOD_PERIO),
                  (cmd->liste_cmd_rc[ind]->param_rc.param_perio)->num_perio);


          ecs_maillage_post__ecr_fac_liste(nom_perio,
                                           maillage,
                                           liste_fac_mod,
                                           false,
                                           ECS_POST_TYPE_INFO,
                                           cas_post);


          BFT_FREE(nom_perio);

        }

        BFT_FREE(liste_fac_mod.val);
        BFT_FREE(liste_fac_new.val);


        /*
          En cas d'erreurs sur la périodicité, on affiche les faces
          posant problème dès que l'on a un cas EnSight, indépendamment
          du mode vérification
        */

        if (cas_post != NULL && liste_fac_err.nbr > 0) {

          char *nom_perio = NULL;

          BFT_MALLOC(nom_perio, strlen(_(ECS_NOM_CEL_ERR_PERIO)) + 5 + 1, char);

          sprintf(nom_perio, "%s_%d", _(ECS_NOM_CEL_ERR_PERIO),
                  (cmd->liste_cmd_rc[ind]->param_rc.param_perio)->num_perio);

          liste_cel_err = ecs_maillage__liste_cel_fac(maillage,
                                                      liste_fac_err);

          ecs_maillage_post__ecr_tab_liste(nom_perio,
                                           maillage,
                                           liste_cel_err,
                                           ECS_ENTMAIL_CEL,
                                           cas_post);

          BFT_FREE(liste_cel_err.val);
          BFT_FREE(nom_perio);

          BFT_MALLOC(nom_perio,
                     strlen(_(ECS_NOM_FACES_ERR_PERIO)) + 5 + 1,
                     char);

          sprintf(nom_perio, "%s %4d", _(ECS_NOM_FACES_ERR_PERIO),
                  (cmd->liste_cmd_rc[ind]->param_rc.param_perio)->num_perio);


          ecs_maillage_post__ecr_fac_liste(nom_perio,
                                           maillage,
                                           liste_fac_err,
                                           false,
                                           ECS_POST_TYPE_ERREUR,
                                           cas_post);

          BFT_FREE(nom_perio);

        }

        BFT_FREE(liste_fac_err.val);

      }

    }

  }


  /*========================================================================*/
  /* Passages des couleurs et groupes aux familles                          */
  /* (se fait dès que le maillage est "figé")                               */
  /*========================================================================*/

  bft_printf(_("\n\n"
               "Defining families\n"
               "-----------------\n\n"));

  ecs_maillage__cree_famille(maillage,
                             ECS_ENTMAIL_FAC,
                             ECS_ENTMAIL_CEL);

  ecs_maillage__supprime_attributs(maillage);


  /*========================================================================*/
  /* Sélections utilisateur                                                 */
  /*========================================================================*/

  if (cmd->cmd_select_fac_int != NULL) {

    _post_faces_int(maillage, cas_post, cmd);

  }


  /*========================================================================*/
  /* Vérification du maillage                                               */
  /*========================================================================*/

  if (ecs_maillage__ret_entmail_max(maillage) == ECS_ENTMAIL_CEL) {

    passe_verif = ecs_maillage__verif(maillage,
                                      cmd->nbr_dump,
                                      nom_fic_dump,
                                      cas_post);

    if (passe_verif == false) {

      ecs_warn();
      bft_printf(_("The mesh has face -> cell connectivity errors.\n"
                   "We can go no further."));

    }

  }
  else {

    passe_verif = false;

    ecs_warn();
    bft_printf(_("The mesh does not contain volume elements.\n"
                 "We can go no further."));

  }


  /* Tous les post-traitements sont terminés à ce stade */

  ecs_post__detruit_cas(cas_post);


  /*========================================================================*/
  /* Si l'on ne peut créer de structure pour le Noyau, on termine           */
  /*========================================================================*/

  if (passe_verif == false)
    goto Etape_fin;


  /*========================================================================*/
  /* Envoi des informations pour le noyau                                   */
  /*========================================================================*/

  if (cmd->nbr_dump > 0)
    ecs_maillage__imprime(maillage,
                          nom_fic_dump,
                          cmd->nbr_dump,
                          "Before output for Kernel");


  liste_param_perio = _liste_param_perio(cmd);

  ecs_maillage_ncs__ecr(cmd->sim_comm,
                        maillage,
                        liste_param_perio);

  BFT_FREE(liste_param_perio);


  /*==========================================================================*/
  /* Libération en mémoire des structures de maillage                         */
  /*==========================================================================*/

 Etape_fin:

  maillage = ecs_maillage__detruit(maillage);


  /* Libérations */
  /*=============*/

  ecs_cmd__detruit(cmd);

  _chrono_total();

  /* Bilan mémoire */

  {
    int    nb_div;
    double f_size;
    size_t pr_size;
    static char  unit_prefix[] = {'K', 'M', 'G'};

    bft_printf(_("\nMemory use summary:\n\n"));

    pr_size = bft_mem_usage_max_pr_size();

    if (pr_size > 0) {
      f_size = pr_size;
      for (nb_div = 0; f_size > 1024. && nb_div < 2; nb_div++)
        f_size /= 1024.;
      bft_printf(_("  Total memory used:                         %15.3f %cb\n"),
                 f_size, unit_prefix[nb_div]);
    }

    f_size = bft_mem_size_max();
    for (nb_div = 0; f_size > 1024. && nb_div < 2; nb_div++)
      f_size /= 1024.;
    bft_printf(_("  Theoretical instrumented dynamic memory:   %15.3f %cb\n"),
               f_size, unit_prefix[nb_div]);

  }


  bft_mem_end();
  bft_mem_usage_end();

  /*---------------------------------------------------------*/
  /* Impression d'un message de fin normale du préprocesseur */
  /*---------------------------------------------------------*/

  bft_printf(_("\n\n"
               "  .-----------------------.\n"
               "  |                       |\n"
               "  |  Preprocessor finish  |\n"
               "  |                       |\n"
               "  `-----------------------'\n"
               "\n\n"));


  if (passe_verif == false)
    return EXIT_FAILURE;

  else
    return EXIT_SUCCESS;

}


/*============================================================================*/
