/*********************************************************************************************************
DVR, Digital Video Recorder - a tool to record movies (audio/video), using realtime compression

It uses libavifile (see http://divx.euro.ru) and some code from kwintv (see wenk@mathematik.uni-kl.de)

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General
Public License as published by the Free Software Foundation, etc.
This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY, etc.

copyright(C) february 2001, Pierre Hbert (pierre.hebert@netcourrier.com)
*********************************************************************************************************/
#include <unistd.h>
#include <stdlib.h>

#include <fstream>
#include <iostream>
#include <list>
#include <utility>

#include <qpushbutton.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <qgroupbox.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qimage.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qasciidict.h>
#include <qtabwidget.h>
#include <qprogressbar.h>
#include <qfiledialog.h>

#include <avifile/avm_fourcc.h>

#include "v4l.h"
#include "QDvrGUIImpl.h"
#include "QCodecConfigImpl.h"
#include "QTVConfigImpl.h"
#include "QAudioConfigImpl.h"
#include "QDeviceSelection.h"
#include "qv4lwindow.h"
#include "dvr.h"
#include "vidmode.h"
#include "conf_analysis_yy.h"

//SDL_Surface *QDvrGUIImpl::sdl_video_display=NULL;
//QV4LWindow *QDvrGUIImpl::qt_video_display=NULL;

// default values of parameters


struct PredefinedSize QDvrGUIImpl::predefined_sizes[]={
	{"160X120", 160, 120},
	{"192X144", 192, 144},
	{"320x240", 320, 240},
	{"352x288", 352, 288},
	{"384x288", 384, 288},
	{"400x300", 400, 300},
	{"512x384", 512, 384},
	{"640x480", 640, 480},
	{"768x576", 768, 576}
};


QDvrGUIImpl::QDvrGUIImpl(QApplication *a, QWidget* parent,  const char* name, WFlags fl )
    : QDvrGUI( parent, name, fl ),
  app(a),
  qt_video_display(NULL),
	gui_video_width(384),
	gui_video_height(288),
	gui_video_top_margin(0),
	gui_video_bottom_margin(0),
	gui_video_codec_name("FFMPEG DivX5"),
	gui_sound_recording_enabled(false),
	gui_sound_sample_size(16),
	gui_sound_frequency(44100),
	gui_sound_channels(1),
	gui_sound_format(WAVE_FORMAT_MPEGLAYER3),
	gui_sound_byterate(8000), // byte not bit. beware of confusion. here 8000o/s=64kbit/s (CD mono)
	gui_video_channel(0),
	gui_video_norm(0),
	gui_sound_device("/dev/dsp"),
	gui_video_device("/dev/video"),
	current_channel(""),
	config_file_name((const char*)(QDir::homeDirPath()+"/.dvrrc")),
	predefined_size(2)
{
	for(unsigned int f=0; f<(sizeof(predefined_sizes)/sizeof(*predefined_sizes)); f++) {
		qcb_predefined_size->insertItem(predefined_sizes[f].label);
	}

	loadConfigFile();

	my_dvr=new DVR();
	
  QDeviceSelection *device_dialog=new QDeviceSelection;
  vector<V4LDevice> devices=my_dvr->getV4l()->availableDevicesList();
  if(devices.size()==0) {
    QMessageBox::critical(this, trUtf8("DVR : error"), trUtf8("Sorry, no video device was found during startup."));
    exit(0);
  }
  
  for(unsigned int i=0; i<devices.size(); i++) {
    QString d=devices[i].name.c_str();
    d+=" ("+devices[i].file_name+")";
    device_dialog->qcb_device->insertItem(d);
    if(gui_video_device==devices[i].file_name) {
      device_dialog->qcb_device->setCurrentItem(i);
    }
  }
  device_dialog->qle_device->setText(gui_video_device.c_str());
  
  if(device_dialog->exec()==QDialog::Accepted) {
    // combo box
    gui_video_device=devices[device_dialog->qcb_device->currentItem()].file_name.c_str();
  } else {
    // line edit
    gui_video_device=(const char*)device_dialog->qle_device->text();
  }
  
  delete device_dialog;
  
	my_dvr->setParameter("video_channel", gui_video_channel);
	my_dvr->setParameter("video_norm", gui_video_norm);
	my_dvr->setParameter("sound_device", gui_sound_device);
	my_dvr->setParameter("video_device", gui_video_device);

	if(!my_dvr->initialize()) {
		errorProcessing();
		exit(0);
	}

  my_dvr->getV4l()->setCaptureSize(gui_video_width, gui_video_height);

	qtvconfig->setV4L(my_dvr->getV4l());
	qtvconfig->setChannelList(&channels, current_channel.c_str());
	connect(qtvconfig, SIGNAL(parameterChanged()), this, SLOT(setVideoParameters()));

	qcodecconfig->setVideoCodecName(gui_video_codec_name);
	connect(qcodecconfig, SIGNAL(parameterChanged()), this, SLOT(setCodecParameters()));

	qaudioconfig->setSoundRecordingEnabled(gui_sound_recording_enabled);
	qaudioconfig->setChannels(gui_sound_channels);
	qaudioconfig->setFrequency(gui_sound_frequency);
	qaudioconfig->setSampleSize(gui_sound_sample_size);
	qaudioconfig->setFormat(gui_sound_format);
	qaudioconfig->setByterate(gui_sound_byterate);
	qaudioconfig->setDevice(gui_sound_device.c_str());
	connect(qaudioconfig, SIGNAL(parameterChanged()), this, SLOT(setAudioParameters()));



	int minx, miny, maxx, maxy;
	my_dvr->getV4l()->getMinSize(minx, miny);
	my_dvr->getV4l()->getMaxSize(maxx, maxy);

	qsb_width->setValue(gui_video_width);
	qsb_height->setValue(gui_video_height);
	qsb_width->setMinValue(minx); qsb_width->setMaxValue(maxx);
	qsb_height->setMinValue(miny); qsb_height->setMaxValue(maxy);
  

	qsb_top_margin->setValue(gui_video_top_margin);
	qsb_top_margin->setMaxValue(gui_video_height);
	qsb_bottom_margin->setValue(gui_video_bottom_margin);
	qsb_bottom_margin->setMaxValue(gui_video_height);

	qtt_is_record_done=new QTimer(this);
	connect(qtt_is_record_done, SIGNAL(timeout()), this, SLOT(isRecordDone()) );

	qtt_start_time=new QTimer(this);
	qtt_start_time->start(900);
	connect(qtt_start_time, SIGNAL(timeout()), this, SLOT(verifyStartTime()) );

	show();			

	setPreviewMode();
}

/*
 *  Destroys the object and frees any allocated resources
 */
QDvrGUIImpl::~QDvrGUIImpl()
{
	restoreOriginalVidMode();

	delete my_dvr;

	saveConfigFile();
}

void QDvrGUIImpl::errorProcessing() {
	QMessageBox::information(this, trUtf8("DVR : error"), my_dvr->errorMessage().c_str());
	cerr<<trUtf8("DVR error : ")<<my_dvr->errorMessage()<<endl;
}

void QDvrGUIImpl::startRecording() {	
	if(QFile::exists(qle_filename->text())) {
		switch( QMessageBox::information( this, trUtf8("DVR"), trUtf8("A file named ")+qle_filename->text()+trUtf8(" already exists.\n Overwrite it ?"), trUtf8("Overwrite"), trUtf8("Cancel"), QString::null, 0, 1 )) {
			case 0:	//overwrite
				break;
			case 1: //cancel
				return;
		}
	}


	qtt_start_time->stop();	
	my_dvr->setParameter("video_width", gui_video_width);
	my_dvr->setParameter("video_height", gui_video_height);
	//my_dvr->setParameter("video_bpp", qcb_image_bpp->currentText().toInt());
	my_dvr->setParameter("video_top_margin", gui_video_top_margin);
	my_dvr->setParameter("video_bottom_margin", gui_video_bottom_margin);
	my_dvr->setParameter("video_codec_name", gui_video_codec_name);
	my_dvr->setParameter("video_frame_rate", double(qsb_video_frame_rate->value()));
  if(qcb_different_frame_rate->isChecked()) {
    my_dvr->setParameter("capture_frame_rate", double(qsb_capture_frame_rate->value()));
  } else {
    my_dvr->setParameter("capture_frame_rate", 0.0);
  }

	my_dvr->setParameter("sound_recording_enabled", gui_sound_recording_enabled);
	my_dvr->setParameter("sound_sample_size", gui_sound_sample_size);
	my_dvr->setParameter("sound_frequency", gui_sound_frequency);
	my_dvr->setParameter("sound_channels", gui_sound_channels);
	my_dvr->setParameter("sound_format", gui_sound_format);	
	my_dvr->setParameter("sound_byterate", gui_sound_byterate);

	my_dvr->setParameter("file_name", (const char*)qle_filename->text());
	my_dvr->setParameter("file_segment_size", qsb_segmented_file->value());
	
	if(qcb_duration_enabled->isChecked()) {
		my_dvr->setParameter("max_recording_time", qsb_duration_hour->value()*3600+qsb_duration_min->value()*60+qsb_duration_sec->value());
	} else {
		my_dvr->setParameter("max_recording_time", 0);
	}

	if(qcb_preview->isChecked()) {
    if(qcb_preview_during_capture->isChecked()) {
      // stop preview : it will be re-enabled in record mode after dvr has started
      qt_video_display->stopVideoPreview();
    } else {
      enableVideoPreview(false);
    }
  }

  // always allow external frame reading
  my_dvr->setParameter("allow_external_frame_reading", 1);
    
	tw_configuration->setCurrentPage(4);
	
	qpb_stop->setEnabled(true);
	qpb_start->setEnabled(false);
	qgb_video_parameters->setEnabled(false);

	qlb_status->setText(trUtf8("Recording..."));	
	if(!my_dvr->startRecording()) {
		postRecordingProcessing();
		return;
	}

	if(qcb_preview->isChecked() && qcb_preview_during_capture->isChecked()) {
    qt_video_display->setRecordMode(true);
    qt_video_display->startVideoPreview();
  }


  qtt_is_record_done->start(500);
	
	pthread_create(&t_video_monitoring, NULL, QDvrGUIImpl::video_monitoring, this);	
}

void QDvrGUIImpl::stopRecording() {
	my_dvr->stopRecording();
}

void QDvrGUIImpl::postRecordingProcessing() {
	pthread_join(t_video_monitoring, NULL);
	qlb_status->setText(trUtf8("Ready."));	
	
	qpb_stop->setEnabled(false);
	qpb_start->setEnabled(true);
	qgb_video_parameters->setEnabled(true);

	qtt_start_time->start(900);
	
	if(!my_dvr->isGood()) {
		errorProcessing();
	}
  
	if(qcb_preview->isChecked() && qcb_preview_during_capture->isChecked()) {
    qt_video_display->stopVideoPreview();
    qt_video_display->setRecordMode(false);
    qt_video_display->startVideoPreview();
  } else if(qcb_preview->isChecked() && !qcb_preview_during_capture->isChecked()) {
    enableVideoPreview(true);
  }
}


void QDvrGUIImpl::closeEvent(QCloseEvent *e) {
	if(my_dvr->getStatus()!=DVR::DVR_READY) {
		switch(QMessageBox::information(NULL, trUtf8("confirmation"), trUtf8("Recording in progress, quit anyway ?"), trUtf8("Yes"), trUtf8("No"), NULL, 0,1)) {
			case 0:
        enableVideoPreview(false);
				stopRecording();
        while(my_dvr->getStatus()!=DVR::DVR_READY) {
          app->processEvents();	
        }
				e->accept();
				break;
			case 1:
				e->ignore();
				break;
		}
	} else {
    enableVideoPreview(false);
		e->accept();
	}
}

void QDvrGUIImpl::loadConfigFile() {
	FILE *f=fopen(config_file_name.c_str(), "r");
  if(!f) {
  	cerr<<trUtf8("Unable to open configuration file : ")<<config_file_name<<trUtf8(". A new default file will be created")<<endl;
    saveConfigFile();
	  f=fopen(config_file_name.c_str(), "r");
    if(!f) {
  	  cerr<<trUtf8("Unable to open configuration file after trying to create it ! Check permissions and disk space.")<<endl;
      return;
    }
	}
	
	
  yyin=f;
	
	qcb_duration_enabled->setChecked(false);
	qcb_start_time_enabled->setChecked(false);
	
	enum { SECTION_UNDEFINED, SECTION_RECORDING, SECTION_GUI, SECTION_CHANNELS };
	bool b=false;
	int current_section=SECTION_UNDEFINED;
	while(yylex() != 0) {
		switch(token_type) {
			case TOKEN_SECTION :
				if(string(section_name)=="Recording") current_section=SECTION_RECORDING;
				else if(string(section_name)=="GUI") current_section=SECTION_GUI;
				else if(string(section_name)=="Channels") current_section=SECTION_CHANNELS;
				else { 	current_section=SECTION_UNDEFINED;
								cerr<<trUtf8("unrecognized section in config file (")<<config_file_name<<") : "<<string(section_name)<<endl;
							}
				break;
				
			case TOKEN_PARAMETER :
				switch(current_section) {
					case SECTION_RECORDING :
						if(string(parameter_name)=="video_width") gui_video_width=atoi(parameter_value);
						else if(string(parameter_name)=="video_height") gui_video_height=atoi(parameter_value);
						//else if(string(parameter_name)=="video_bpp") qcb_image_bpp->setCurrentItem(atoi(parameter_value)==16?0:1);
						else if(string(parameter_name)=="video_top_margin") gui_video_top_margin=atoi(parameter_value);
						else if(string(parameter_name)=="video_bottom_margin") gui_video_bottom_margin=atoi(parameter_value);
						else if(string(parameter_name)=="video_codec_name") gui_video_codec_name=parameter_value;
						else if(string(parameter_name)=="video_frame_rate") qsb_video_frame_rate->setValue(atoi(parameter_value));
						else if(string(parameter_name)=="capture_frame_rate") {
              int v=atoi(parameter_value);
              if(v!=0) {
                qsb_capture_frame_rate->setValue(v);
                qcb_different_frame_rate->setChecked(true);
                qsb_capture_frame_rate->setEnabled(true);
              } else {
                qcb_different_frame_rate->setChecked(false);
                qsb_capture_frame_rate->setEnabled(false);
              }
            }
						else if(string(parameter_name)=="video_channel") gui_video_channel=atoi(parameter_value);
						else if(string(parameter_name)=="video_norm") gui_video_norm=atoi(parameter_value);
						else if(string(parameter_name)=="video_device") gui_video_device=parameter_value;
						else if(string(parameter_name)=="sound_recording_enabled") gui_sound_recording_enabled=atoi(parameter_value);
						else if(string(parameter_name)=="sound_sample_size") gui_sound_sample_size=atoi(parameter_value);
						else if(string(parameter_name)=="sound_frequency") gui_sound_frequency=atoi(parameter_value);
						else if(string(parameter_name)=="sound_channels") gui_sound_channels=atoi(parameter_value);
						else if(string(parameter_name)=="sound_format") gui_sound_format=atoi(parameter_value);
						else if(string(parameter_name)=="sound_byterate") gui_sound_byterate=atoi(parameter_value);
						else if(string(parameter_name)=="sound_device") gui_sound_device=parameter_value;
						else if(string(parameter_name)=="file_segment_size") qsb_segmented_file->setValue(atoi(parameter_value));
						else if(string(parameter_name)=="file_name") qle_filename->setText(parameter_value);
						else if(string(parameter_name)=="max_recording_time") {
							qcb_duration_enabled->setChecked(true);
							int v=atoi(parameter_value);
							qsb_duration_hour->setValue(v/3600);
							qsb_duration_min->setValue((v%3600)/60);
							qsb_duration_sec->setValue((v%3600)%60);
							}
						else cerr<<trUtf8("unrecognized parameter in config file (")<<config_file_name<<") : "<<string(parameter_name)<<endl;					
						break;
						
					case SECTION_GUI :
						if(string(parameter_name)=="start_hour") { b=true; qsb_start_hour->setValue(atoi(parameter_value)); }
						else if(string(parameter_name)=="start_min") { b=true; qsb_start_min->setValue(atoi(parameter_value)); }
						else if(string(parameter_name)=="start_sec") { b=true; qsb_start_sec->setValue(atoi(parameter_value)); }
						else if(string(parameter_name)=="predefined_size") qcb_predefined_size->setCurrentItem(atoi(parameter_value));
						else if(string(parameter_name)=="preview") qcb_preview->setChecked(atoi(parameter_value));
						else if(string(parameter_name)=="preview_during_capture") qcb_preview_during_capture->setChecked(atoi(parameter_value));
						else if(string(parameter_name)=="fullscreen_preview") qcb_fullscreen_preview->setChecked(atoi(parameter_value));
						else cerr<<"unrecognized parameter in config file ("<<config_file_name<<") : "<<string(parameter_name)<<endl;					
												
						if(b) { qcb_start_time_enabled->setChecked(true); b=false; }
						break;
						
					case SECTION_CHANNELS :
						if(string(parameter_name)=="current_channel") current_channel=parameter_value;
						else if(string(parameter_name)=="frequency") {
							QString *s=new QString(name_index);
							channels.append(new pair<QString*, unsigned long>(s, atoi(parameter_value)));
						}		
						else cerr<<trUtf8("unrecognized parameter in config file (")<<config_file_name<<") : "<<string(parameter_name)<<endl;					
												
						break;
				}
				break;
		}
	}

	delete f;
}

void QDvrGUIImpl::saveConfigFile() {
	QFile f(config_file_name.c_str());

	if(!f.open(IO_WriteOnly | IO_Truncate)) {
		cout<<trUtf8("unable to open configuration file ")<<f.name()<<trUtf8(" for writing.")<<endl;
		return;
	}

	QTextStream ts(&f);

	ts<<"[Recording]"<<endl;
	ts<<"video_width="<<gui_video_width<<endl;
	ts<<"video_height="<<gui_video_height<<endl;
	//ts<<"video_bpp="<<qcb_image_bpp->currentText()<<endl;
	ts<<"video_top_margin="<<qsb_top_margin->value()<<endl;
	ts<<"video_bottom_margin="<<qsb_bottom_margin->value()<<endl;
	ts<<"video_codec_name="<<gui_video_codec_name.c_str()<<endl;
	ts<<"video_frame_rate="<<qsb_video_frame_rate->value()<<endl;
  if(qcb_different_frame_rate->isChecked()) {
    ts<<"capture_frame_rate="<<qsb_capture_frame_rate->value()<<endl;
  } else {
    ts<<"capture_frame_rate=0"<<endl;
  }
	ts<<"video_channel="<<gui_video_channel<<endl;
	ts<<"video_norm="<<gui_video_norm<<endl;
	ts<<"video_device="<<gui_video_device.c_str()<<endl;
	ts<<"sound_recording_enabled="<<(gui_sound_recording_enabled?1:0)<<endl;
	ts<<"sound_sample_size="<<gui_sound_sample_size<<endl;
	ts<<"sound_frequency="<<gui_sound_frequency<<endl;
	ts<<"sound_channels="<<gui_sound_channels<<endl;
	ts<<"sound_format="<<gui_sound_format<<endl;
	ts<<"sound_byterate="<<gui_sound_byterate<<endl;
	ts<<"sound_device="<<gui_sound_device.c_str()<<endl;
	ts<<"file_segment_size="<<qsb_segmented_file->value()<<endl;	
	ts<<"file_name="<<qle_filename->text()<<endl;
	if(qcb_duration_enabled->isChecked()) {
		ts<<"max_recording_time="<<qsb_duration_hour->value()*3600+qsb_duration_min->value()*60+qsb_duration_sec->value()<<endl;
	}
	
	ts<<"[GUI]"<<endl;
	ts<<"predefined_size="<<qcb_predefined_size->currentItem()<<endl;
	ts<<"preview="<<(qcb_preview->isChecked()?1:0)<<endl;		
	ts<<"preview_during_capture="<<(qcb_preview_during_capture->isChecked()?1:0)<<endl;		
	ts<<"fullscreen_preview="<<(qcb_fullscreen_preview->isChecked()?1:0)<<endl;		
	if(qcb_start_time_enabled->isChecked()) {
		ts<<"start_hour="<<qsb_start_hour->value()<<endl;
		ts<<"start_min="<<qsb_start_min->value()<<endl;
		ts<<"start_sec="<<qsb_start_sec->value()<<endl;
	}
		
	ts<<"[Channels]"<<endl;
	if(current_channel!="") ts<<"current_channel="<<current_channel.c_str()<<endl;
	
	pair<QString*, unsigned long> *e;
	for(e=channels.first(); e!=0; e=channels.next()) {
		ts<<"frequency["<<*(e->first)<<"]="<<e->second<<endl;
	}
	
	f.close();
}

void QDvrGUIImpl::isRecordDone() {
  if(my_dvr->getStatus()!=DVR::DVR_RECORDING) {
    qtt_is_record_done->stop();
    my_dvr->stopRecording();
    qlb_status->setText(trUtf8("Flushing audio and video queues, please wait..."));

    while(my_dvr->getStatus()!=DVR::DVR_READY) {
      app->processEvents();	
    }
	
    postRecordingProcessing();
  }
}

void QDvrGUIImpl::verifyStartTime() {
	QTime ct=QTime::currentTime(), st(qsb_start_hour->value(), qsb_start_min->value(), qsb_start_sec->value());

	// the millisecond part is fixed to 0
	ct.setHMS(ct.hour(), ct.minute(), ct.second());

	if(st==ct && qcb_start_time_enabled->isChecked()) {
		qtt_start_time->stop();
		startRecording();
	}
}

void QDvrGUIImpl::changeTopMargin(int m) {
	gui_video_top_margin=m;
}

void QDvrGUIImpl::changeBottomMargin(int m) {
	gui_video_bottom_margin=m;
}

void QDvrGUIImpl::setVideoFrameWidth(int w) {
  if(gui_video_width!=w) {
  	gui_video_width=w;
    if(qt_video_display) {
      qt_video_display->stopVideoPreview();
      my_dvr->getV4l()->setCaptureSize(gui_video_width, gui_video_height);
      qt_video_display->resize(gui_video_width, gui_video_height);
      // no need to restart video preview because resize does it
      //qt_video_display->startVideoPreview();
    }
  }
}

void QDvrGUIImpl::setVideoFrameHeight(int h) {
  if(gui_video_height!=h) {
    gui_video_height=h;
    if(gui_video_bottom_margin+gui_video_top_margin>h) {
      qsb_bottom_margin->setValue(0);
      qsb_top_margin->setValue(0);
    }
    qsb_bottom_margin->setMaxValue(h);
    qsb_top_margin->setMaxValue(h);
    if(qt_video_display) {
      qt_video_display->stopVideoPreview();
      my_dvr->getV4l()->setCaptureSize(gui_video_width, gui_video_height);
      qt_video_display->resize(gui_video_width, gui_video_height);
      // no need to restart video preview because resize does it
      //qt_video_display->startVideoPreview();
    }
  }
}

void QDvrGUIImpl::setTopMargin(int m) {
	if(m+gui_video_bottom_margin<gui_video_height) {
		gui_video_top_margin=m;
	}
}

void QDvrGUIImpl::setBottomMargin(int m) {
	if(m+gui_video_top_margin<gui_video_height) {
		gui_video_bottom_margin=m;
	}
}

void QDvrGUIImpl::setVideoFrameSize(const QString &s) {
	for(unsigned int f=0; f<(sizeof(predefined_sizes)/sizeof(*predefined_sizes)); f++) {
		if(QString::compare(predefined_sizes[f].label,s)==0) {
      int w=predefined_sizes[f].width;
      int h=predefined_sizes[f].height;
      if(gui_video_height!=w || gui_video_height!=h) {
        gui_video_width=w;
        gui_video_height=h;
        qsb_width->setValue(gui_video_width);
        qsb_height->setValue(gui_video_height);
        if(qt_video_display) {
          qt_video_display->stopVideoPreview();
          my_dvr->getV4l()->setCaptureSize(w, h);
          qt_video_display->resize(w, h);
          // no need to restart video preview because resize does it
          //qt_video_display->startVideoPreview();
        }
      }
			return;
		}
	}
}

void QDvrGUIImpl::setCodecParameters() {
	gui_video_codec_name=qcodecconfig->videoCodecName();
}

void QDvrGUIImpl::setAudioParameters() {
	gui_sound_recording_enabled=qaudioconfig->getSoundRecordingEnabled();
	gui_sound_channels=qaudioconfig->getChannels();
	gui_sound_frequency=qaudioconfig->getFrequency();
	gui_sound_sample_size=qaudioconfig->getSampleSize();
	gui_sound_format=qaudioconfig->getFormat();
	gui_sound_byterate=qaudioconfig->getByterate();
	gui_sound_device=(const char*)qaudioconfig->getDevice();
}

void QDvrGUIImpl::setVideoParameters() {
	gui_video_channel=qtvconfig->getChannel();
	gui_video_norm=qtvconfig->getNorm();
	my_dvr->getV4l()->videoChannels()[my_dvr->getV4l()->currentVideoChannel()].norm=gui_video_norm;
	my_dvr->getV4l()->setCurrentVideoChannel(gui_video_channel);

}

void QDvrGUIImpl::browseFile() {
	QString filename=QFileDialog::getSaveFileName( qle_filename->text(), "avi file (*.avi)", this );
	if(!filename.isNull()) {
		qle_filename->setText(filename);
	}
}

void QDvrGUIImpl::changeTimeEnabling() {
	qsb_start_hour->setEnabled(qcb_start_time_enabled->isChecked());
	qsb_start_min->setEnabled(qcb_start_time_enabled->isChecked());
	qsb_start_sec->setEnabled(qcb_start_time_enabled->isChecked());		
	
	qsb_duration_hour->setEnabled(qcb_duration_enabled->isChecked());
	qsb_duration_min->setEnabled(qcb_duration_enabled->isChecked());
	qsb_duration_sec->setEnabled(qcb_duration_enabled->isChecked());
}
	

int QDvrGUIImpl::cpuLoad() {
	static QTime old_time;
	static int old_jiffies=0;
	int user, system;

	FILE *file=fopen("/proc/stat", "r");
	if(file==NULL) return -1;
	QTime new_time=QTime::currentTime();
	fscanf(file, "%*s %d %*s %d", &user, &system);
	fclose(file);
	int new_jiffies=user+system;

	if(old_jiffies==0) {
		old_time=new_time;	old_jiffies=new_jiffies;
		return 0;
	}

	int load=(new_jiffies-old_jiffies)*1000/old_time.msecsTo(new_time);
	old_time=new_time;
	old_jiffies=new_jiffies;

	return load;
}

void QDvrGUIImpl::setPreviewMode() {
  if(qcb_preview->isChecked()) {
    qcb_preview_during_capture->setEnabled(true);
    qcb_fullscreen_preview->setEnabled(true);
  } else {
    qcb_preview_during_capture->setEnabled(false);
    qcb_fullscreen_preview->setEnabled(false);
  }

  if(qcb_preview->isChecked() && my_dvr->getStatus()==DVR::DVR_READY) {
    enableVideoPreview(true);
  } else if(qcb_preview->isChecked() && qcb_preview_during_capture->isChecked() && my_dvr->getStatus()==DVR::DVR_RECORDING) {
    enableVideoPreview(true);
  } else if(!qcb_preview->isChecked()) {
    enableVideoPreview(false);
  } else if(!qcb_preview_during_capture->isChecked() && my_dvr->getStatus()==DVR::DVR_RECORDING) {
    enableVideoPreview(false);
  }

	if(qcb_fullscreen_preview->isChecked()) {
		unsigned int x, y;
		unsigned int w, h;
		my_dvr->getV4l()->getWindowPosition(x, y);
		my_dvr->getV4l()->getWindowSize(w, h);
		setVidMode(x, y, w, h);
    /* TODO : display a message if no fullscreen mode has been found */
	} else {
		restoreOriginalVidMode();
	}
}

void QDvrGUIImpl::changeVideoFrameRate() {
  if(qcb_different_frame_rate->isChecked()) {
    qsb_capture_frame_rate->setEnabled(true);
  } else {
    qsb_capture_frame_rate->setEnabled(false);
  }
}

void QDvrGUIImpl::enableVideoPreview(bool on) {
  if(!qt_video_display && on) {
    qt_video_display=new QV4LWindow(my_dvr);
    if(my_dvr->getStatus()==DVR::DVR_RECORDING) {
      qt_video_display->setRecordMode(true);
    }
    qt_video_display->show();
    //qt_video_display->startVideoPreview();
  }
  if(qt_video_display && !on) {
    delete qt_video_display;
    qt_video_display=NULL;
  }
}
