"""This module provides functionality for form assembly in Python,
corresponding to the C++ assembly and PDE classes.

The C++ :py:class:`assemble <dolfin.cpp.assemble>` function
(renamed to cpp_assemble) is wrapped with an additional
preprocessing step where code is generated using the
FFC JIT compiler.

The C++ PDE classes are reimplemented in Python since the C++ classes
rely on the dolfin::Form class which is not used on the Python side."""

# Copyright (C) 2007-2008 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Martin Sandve Alnaes, 2008.
# Modified by Johan Hake, 2008-2009.
# Modified by Garth N. Wells, 2008-2009.
#
# First added:  2007-08-15
# Last changed: 2010-11-04

__all__ = ["assemble", "assemble_system"]

import types

# UFL modules
from ufl import Cell

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp

# Local imports
from dolfin.fem.form import *

# Cache for tensors
_tensor_cache = {}

# JIT assembler
def assemble(form,
             tensor=None,
             mesh=None,
             coefficients=None,
             function_spaces=None,
             cell_domains=None,
             exterior_facet_domains=None,
             interior_facet_domains=None,
             reset_sparsity=None,
             add_values=None,
             backend=None,
             form_compiler_parameters=None):
    """
    Assemble the given form and return the corresponding tensor.

    *Arguments*
        Depending on the input form, which may be a functional, linear
        form, bilinear form or higher rank form, a scalar value, a vector,
        a matrix or a higher rank tensor is returned.

    In the simplest case, no additional arguments are needed. However,
    additional arguments may and must in some cases be provided as
    outlined below.

    The ``form`` can be either an FFC form or a precompiled UFC
    form. If a precompiled or 'pure' UFC form is given, then
    ``coefficients`` and ``function_spaces`` have to be provided
    too. The coefficient functions should be provided as a 'dict'
    using the FFC functions as keys. The function spaces should be
    provided either as a list where the number of function spaces must
    correspond to the number of basis functions in the form, or as a
    single argument, implying that the same FunctionSpace is used for
    all test/trial spaces.

    If the form defines integrals over different subdomains,
    :py:class:`MeshFunctions <dolfin.cpp.MeshFunction>` over the
    corresponding topological entities defining the subdomains can be
    provided. An instance of a :py:class:`SubDomain
    <dolfin.cpp.SubDomain>` can also be passed for each subdomain.

    The implementation of the returned tensor is determined by the
    default linear algebra backend. This can be overridden by
    specifying a different backend.

    Each call to assemble() will create a new tensor. If the
    ``tensor`` argument is provided, this will be used instead. If
    ``reset_sparsity`` is set to True, the provided tensor will not be
    reset to zero before assembling (adding) values to the tensor.

    Specific form compiler parameters can be provided by the
    ``form_compiler_parameters`` argument. Form compiler parameters
    can also be controlled using the global parameters stored in
    parameters["form_compiler"].

    *Examples of usage*
        The standard stiffness matrix ``A`` and a load vector ``b``
        can be assembled as follows:

        .. code-block:: python

            A = assemble(inner(grad(u),grad(v))*dx)
            b = assemble(f*v*dx)

        It is possible to explicitly prescribe the domains over which
        integrals wll be evaluated using the arguments
        ``cell_domains``, ``exterior_facet_domains`` and
        ``interior_facet_domains``. For instance, using a mesh
        function marking parts of the boundary:

        .. code-block:: python

            # MeshFunction marking boundary parts
            boundary_parts = MeshFunction("uint", mesh, mesh.topology().dim()-1)

            # Sample variational forms
            a = inner(grad(u), grad(v))*dx + p*u*v*ds(0)
            L = f*v*dx - g*v*ds(1) + p*q*v*ds(0)

            A = assemble(a, exterior_facet_domains=boundary_parts)
            b = assemble(L, exterior_facet_domains=boundary_parts)

        To ensure that the assembled matrix has the right type, one may use
        the ``tensor`` argument:

        .. code-block:: python

            A = PETScMatrix()
            assemble(a, tensor=A)

        The form ``a`` is now assembled into the PETScMatrix ``A``.

    """

    # Extract common cell from mesh (may be missing in form definition)
    common_cell = None
    if mesh is not None:
        dim = mesh.topology().dim()
        common_cell = Cell({1: "interval", 2: "triangle", 3: "tetrahedron"}[dim])

    # Wrap form
    dolfin_form = Form(form,
                       function_spaces=function_spaces,
                       coefficients=coefficients,
                       form_compiler_parameters=form_compiler_parameters,
                       common_cell=common_cell)

    # Set mesh if specified (important for functionals without a function spaces)
    if mesh is not None:
        dolfin_form.set_mesh(mesh)

    # Create tensor
    (tensor, reset_sparsity) = _create_tensor(form, dolfin_form.rank(), backend, tensor, reset_sparsity)

    # Set default value for reset_sparsity if not specified
    if reset_sparsity is None:
        reset_sparsity = True

    # Set default value for add_values if not specified
    if add_values is None:
        add_values = False

    # Extract domains
    cell_domains, exterior_facet_domains, interior_facet_domains = \
        _extract_domains(dolfin_form.mesh(),
                         cell_domains,
                         exterior_facet_domains,
                         interior_facet_domains)

    # Assemble tensor from compiled form
    cpp.assemble(tensor,
                 dolfin_form,
                 cell_domains,
                 exterior_facet_domains,
                 interior_facet_domains,
                 reset_sparsity,
                 add_values)

    # Convert to float for scalars
    if dolfin_form.rank() == 0:
        tensor = tensor.getval()

    # Return value
    return tensor

# JIT system assembler
def assemble_system(A_form,
                    b_form,
                    bcs=None,
                    x0=None,
                    A_coefficients=None,
                    b_coefficients=None,
                    A_function_spaces=None,
                    b_function_spaces=None,
                    cell_domains=None,
                    exterior_facet_domains=None,
                    interior_facet_domains=None,
                    reset_sparsity=None,
                    add_values=None,
                    A_tensor=None,
                    b_tensor=None,
                    backend=None,
                    form_compiler_parameters=None):
    """
    Assemble form(s) and apply any given boundary conditions in a
    symmetric fashion and return tensor(s).

    The standard application of boundary conditions does not
    necessarily preserve the symmetry of the assembled matrix. In
    order to perserve symmetry in a system of equations with boundary
    conditions, one may use the alternative assemble_system instead of
    multiple calls to :py:func:`assemble
    <dolfin.fem.assembling.assemble>`.

    *Examples of usage*

       For instance, the statements

       .. code-block:: python

           A = assemble(a)
           b = assemble(L)
           bc.apply(A, b)

       can alternatively be carried out by

       .. code-block:: python

           A, b = assemble_system(a, L, bc)

       The statement above is valid even if ``bc`` is a list of
       :py:class:`DirichletBC <dolfin.fem.bcs.DirichletBC>`
       instances. For more info and options, see :py:func:`assemble
       <dolfin.fem.assembling.assemble>`.

    """
    subdomains = { "cell": cell_domains,
                   "exterior_facet": exterior_facet_domains,
                   "interior_facet": interior_facet_domains, }
    A_dolfin_form = Form(A_form, A_function_spaces, A_coefficients, subdomains, form_compiler_parameters)
    b_dolfin_form = Form(b_form, b_function_spaces, b_coefficients, subdomains, form_compiler_parameters)

    (A_tensor, reset_sparsity) = _create_tensor(A_form, A_dolfin_form.rank(), backend, A_tensor, reset_sparsity)
    (b_tensor, reset_sparsity) = _create_tensor(b_form, b_dolfin_form.rank(), backend, b_tensor, reset_sparsity)

    # Set default value for reset_sparsity if not specified
    if reset_sparsity is None:
        reset_sparsity = True

    # Set default value for add_values if not specified
    if add_values is None:
        add_values = False


    # Extract domains
    cell_domains, exterior_facet_domains, interior_facet_domains = \
                  _extract_domains(A_dolfin_form.mesh(),
                                   cell_domains,
                                   exterior_facet_domains,
                                   interior_facet_domains)

    # Check bcs
    if not isinstance(bcs,(types.NoneType,list,cpp.DirichletBC)):
        raise TypeError, "expected a 'list', or a 'DirichletBC' as bcs argument"
    if bcs is None:
        bcs = []
    elif isinstance(bcs,cpp.DirichletBC):
        bcs = [bcs]

    cpp.assemble_system(A_tensor,
                        b_tensor,
                        A_dolfin_form,
                        b_dolfin_form,
                        bcs,
                        cell_domains,
                        exterior_facet_domains,
                        interior_facet_domains,
                        x0,
                        reset_sparsity)

    return A_tensor, b_tensor

def _create_tensor(form, rank, backend, tensor, reset_sparsity):
    "Create tensor for form"

    # Check backend argument
    if (not backend is None) and (not isinstance(backend,cpp.LinearAlgebraFactory)):
        raise TypeError, "Provide a LinearAlgebraFactory as 'backend'"

    # Check if tensor is supplied by user
    if tensor is not None:
        return (tensor, reset_sparsity)

    # Check that reset_sparsity pattern is not False when not providing tensor
    if reset_sparsity == False:
        raise TypeError, "expected a tensor, when 'reset_sparsity' is 'False'"

    # Decide if we should reset the tensor
    use_cache = cpp.parameters["optimize_use_tensor_cache"] or \
                cpp.parameters["optimize"]
    if use_cache and reset_sparsity is None:
        reset_sparsity = not form in _tensor_cache

    # Check if we should use the cache
    if use_cache and form in _tensor_cache:
        return (_tensor_cache[form], reset_sparsity)

    # Create tensor
    if rank == 0:
        tensor = cpp.Scalar()
    elif rank == 1:
        if backend: tensor = backend.create_vector()
        else:       tensor = cpp.Vector()
    elif rank == 2:
        if backend: tensor = backend.create_matrix()
        else:       tensor = cpp.Matrix()
    else:
        raise RuntimeError, "Unable to create tensors of rank %d." % rank

    # Store in cache
    if use_cache:
        _tensor_cache[form] = tensor

    return (tensor, reset_sparsity)


def _extract_domains(mesh,
                     cell_domains,
                     exterior_facet_domains,
                     interior_facet_domains):

    def check_domain_type(domain,domain_type):
        if not isinstance(domain,(cpp.SubDomain,cpp.MeshFunctionUInt,types.NoneType)):
            raise TypeError, "expected a 'SubDomain', 'MeshFunction' of 'UInt' or 'None', for the '%s'"%domain_type

    def build_mf(subdomain, dim):
        " Builds a MeshFunction from a SubDomain"
        mf = cpp.MeshFunction("uint", mesh, dim)
        mf.set_all(1)
        subdomain.mark(mf,0)
        return mf

    # Type check of input
    check_domain_type(cell_domains,"cell_domains")
    check_domain_type(exterior_facet_domains,"exterior_facet_domains")
    check_domain_type(interior_facet_domains,"interior_facet_domains")

    # The cell dimension
    cell_dim = mesh.topology().dim()

    # Get cell_domains
    if isinstance(cell_domains, cpp.SubDomain):
        cell_domains = build_mf(cell_domains, cell_dim)

    # Get exterior_facet_domains (may be stored as part of the mesh)
    if exterior_facet_domains is None:
        exterior_facet_domains = mesh.data().mesh_function("exterior facet domains")
    elif isinstance(exterior_facet_domains, cpp.SubDomain):
        exterior_facet_domains = build_mf(exterior_facet_domains, cell_dim-1)

    # Get interior_facet_domains
    if isinstance(interior_facet_domains, cpp.SubDomain):
        interior_facet_domains = build_mf(interior_facet_domains, cell_dim-1)

    return cell_domains, exterior_facet_domains, interior_facet_domains
