"""This module provides a Python layer on top of the C++
Adaptive*VariationalSolver classes"""

# Copyright (C) 2011 Marie E. Rognes
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2011-06-27
# Last changed: 2011-06-27

__all__ = ["AdaptiveLinearVariationalSolver",
           "AdaptiveNonlinearVariationalSolver",
           "generate_error_control",
           "generate_error_control_forms"]

import dolfin.cpp as cpp

from dolfin.fem.form import Form
from dolfin.fem.solving import LinearVariationalProblem
from dolfin.fem.solving import NonlinearVariationalProblem

from dolfin.fem.errorcontrolgenerator import DOLFINErrorControlGenerator

class AdaptiveLinearVariationalSolver(cpp.AdaptiveLinearVariationalSolver):

    # Reuse doc-string
    __doc__ = cpp.AdaptiveLinearVariationalSolver.__doc__

    def __init__(self, problem):
        """
        Create AdaptiveLinearVariationalSolver

        *Arguments*

            problem (:py:class:`LinearVariationalProblem <dolfin.fem.solving.LinearVariationalProblem>`)

        """

        # Store problem
        self.problem = problem

        # Initialize C++ base class
        cpp.AdaptiveLinearVariationalSolver.__init__(self, problem)

    def solve(self, tol, goal):
        """
        Solve such that the estimated error in the functional 'goal'
        is less than the given tolerance 'tol'

        *Arguments*

            tol (float)

                The error tolerance

            goal (:py:class:`Form <dolfin.fem.form.Form>`)

                The goal functional

        """

        # Generate error control object
        ec = generate_error_control(self.problem, goal)

        # Compile goal functional separately
        p = self.problem.form_compiler_parameters
        M = Form(goal, form_compiler_parameters=p)

        # Call cpp.AdaptiveLinearVariationalSolver.solve with ec
        cpp.AdaptiveLinearVariationalSolver.solve(self, tol, M, ec)

class AdaptiveNonlinearVariationalSolver(cpp.AdaptiveNonlinearVariationalSolver):

    # Reuse doc-string
    __doc__ = cpp.AdaptiveNonlinearVariationalSolver.__doc__

    def __init__(self, problem):
        """
        Create AdaptiveLinearVariationalSolver

        *Arguments*

            problem (:py:class:`NonlinearVariationalProblem <dolfin.fem.solving.NonlinearVariationalProblem>`)

        """

        # Store problem
        self.problem = problem

        # Initialize C++ base class
        cpp.AdaptiveNonlinearVariationalSolver.__init__(self, problem)

    def solve(self, tol, goal):
        """
        Solve such that the estimated error in the functional 'goal'
        is less than the given tolerance 'tol'.

        *Arguments*

            tol (float)

                The error tolerance

            goal (:py:class:`Form <dolfin.fem.form.Form>`)

                The goal functional

        """

        # Generate error control object
        ec = generate_error_control(self.problem, goal)

        # Compile goal functional separately
        p = self.problem.form_compiler_parameters
        M = Form(goal, form_compiler_parameters=p)

        # Call cpp.AdaptiveNonlinearVariationlSolver.solve with ec
        cpp.AdaptiveNonlinearVariationalSolver.solve(self, tol, M, ec)

def generate_error_control(problem, goal):
    """
    Create suitable ErrorControl object from problem and the goal

    *Arguments*

        problem (:py:class:`LinearVariationalProblem <dolfin.fem.solving.LinearVariationalProblem>` or :py:class:`NonlinearVariationalProblem <dolfin.fem.solving.NonlinearVariationalProblem>`)

            The (primal) problem

        goal (:py:class:`Form <dolfin.fem.form.Form>`)

            The goal functional

    *Returns*

        :py:class:`ErrorControl <dolfin.cpp.ErrorControl>`

    """

    # Generate UFL forms to be used for error control
    (ufl_forms, is_linear) = generate_error_control_forms(problem, goal)

    # Compile generated forms
    p = problem.form_compiler_parameters
    forms = [Form(form, form_compiler_parameters=p) for form in ufl_forms]

    # Create cpp.ErrorControl object
    forms += [is_linear]  # NOTE: Lingering design inconsistency.
    ec = cpp.ErrorControl(*forms)

    # Return generated ErrorControl
    return ec

def generate_error_control_forms(problem, goal):
    """
    Create UFL forms required for initializing an ErrorControl object

    *Arguments*

        problem (:py:class:`LinearVariationalProblem <dolfin.fem.solving.LinearVariationalProblem>` or :py:class:`NonlinearVariationalProblem <dolfin.fem.solving.NonlinearVariationalProblem>`)

            The (primal) problem

        goal (:py:class:`Form <dolfin.fem.form.Form>`)

            The goal functional

    *Returns*

        (tuple of forms, bool)

    """

    msg = "Generating forms required for error control. This can take time..."
    cpp.info(msg)

    # Extract primal forms from problem
    is_linear = True
    if isinstance(problem, LinearVariationalProblem):
        primal = (problem.a_ufl, problem.L_ufl)
    elif isinstance(problem, NonlinearVariationalProblem):
        is_linear = False
        primal = problem.F_ufl
    else:
        cpp.error("Unknown problem type %s" % str(problem))

    # Extract unknown Function from problem
    u = problem.u_ufl

    # Get DOLFIN's error control generator to generate all forms
    generator = DOLFINErrorControlGenerator(primal, goal, u)
    forms = generator.generate_all_error_control_forms()

    return (forms, is_linear)
