"""This module provides the class VariationalProblem in Python. This
needs special handling and we cannot use the SWIG wrapper directly
since we need to call the JIT compiler."""

__author__ = "Anders Logg (logg@simula.no)"
__date__ = "2007-08-15 -- 2009-12-11"
__copyright__ = "Copyright (C) 2007-2008 Anders Logg"
__license__  = "GNU LGPL Version 2.1"

# Modified by Johan Hake, 2009

__all__ = ["VariationalProblem"]

import types

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin
import dolfin.cpp as cpp

# Local imports
from dolfin.fem.form import *
from dolfin.function.function import *

class VariationalProblem(cpp.VariationalProblem):

    # Reuse docstring from cpp.VariationalProblem
    __doc__ = cpp.VariationalProblem.__doc__

    def __init__(self, a, L,
                 bcs=None,
                 cell_domains=None,
                 exterior_facet_domains=None,
                 interior_facet_domains=None,
                 nonlinear=False,
                 form_compiler_parameters={}):
        "Define a variational problem."

        # Wrap forms
        self.a = Form(a, form_compiler_parameters=form_compiler_parameters)
        self.L = Form(L, form_compiler_parameters=form_compiler_parameters)

        # Check bcs
        if not isinstance(bcs, (types.NoneType, list, cpp.BoundaryCondition)):
            raise TypeError, "expected a 'list' or a 'BoundaryCondition' as bcs argument"
        if bcs is None:
            self.bcs = []
        elif isinstance(bcs, cpp.BoundaryCondition):
            self.bcs = [bcs]
        else:
            self.bcs = bcs

        # Initialize base class
        cpp.VariationalProblem.__init__(self, self.a, self.L, self.bcs, cell_domains, exterior_facet_domains, interior_facet_domains, nonlinear)

    def solve(self, u=None):
        "Solve variational problem and return solution."

        # Extract trial space and create function for solution
        if u is None:
            if not len(self.a.function_spaces) == 2:
                dolfin.error("Unable to extract trial space for solution of variational problem, is 'a' bilinear?")
            V = self.a.function_spaces[1]
            u = Function(V)

        # Solve
        cpp.VariationalProblem.solve(self, u)

        # Return solution
        return u
