# Copyright (C) 2008-2009 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN.  If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Johan Hake, 2008.
#
# First added:  2008-12-02
# Last changed: 2010-08-16

__all__ = ["Constant"]

# Import UFL and SWIG-generated extension module (DOLFIN C++)
import ufl
import dolfin.cpp as cpp
import numpy

class Constant(ufl.Coefficient, cpp.Constant):

    def __init__(self, value):
        """Create constant-valued function with given value.

        The value may be either a single scalar value, or a tuple/list of
        values for vector-valued functions, or nested lists or a numpy array for
        tensor-valued functions."""

        array = numpy.array(value)
        dim = len(array.shape)
        floats = map(float, array.flat)

        # Create UFL element and initialize constant
        if dim == 0:
            self._ufl_element = ufl.FiniteElement("Discontinuous Lagrange", None, 0)
            cpp.Constant.__init__(self, floats[0])
        elif dim == 1:
            self._ufl_element = ufl.VectorElement("Discontinuous Lagrange", None, 0, len(floats))
            cpp.Constant.__init__(self, floats)
        else:
            self._ufl_element = ufl.TensorElement("Discontinuous Lagrange", None, 0, shape=array.shape)
            cpp.Constant.__init__(self, list(array.shape), floats)

        # Initialize base classes
        ufl.Coefficient.__init__(self, self._ufl_element)
