/* ****************************************************************************
 *
 * Copyright (c) Microsoft Corporation. 
 *
 * This source code is subject to terms and conditions of the Microsoft Public License. A 
 * copy of the license can be found in the License.html file at the root of this distribution. If 
 * you cannot locate the  Microsoft Public License, please send an email to 
 * dlr@microsoft.com. By using this source code in any fashion, you are agreeing to be bound 
 * by the terms of the Microsoft Public License.
 *
 * You must not remove this notice, or any other, from this software.
 *
 *
 * ***************************************************************************/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Reflection;
using Microsoft.Scripting.Utils;

namespace Microsoft.Scripting.Runtime {

    internal struct TypeName {
        private readonly string _namespace;
        private readonly string _typeName;

        internal TypeName(Type type) {
            Debug.Assert(!ReflectionUtils.IsNested(type));
            _namespace = type.Namespace;
            _typeName = type.Name;
        }

        internal TypeName(string nameSpace, string typeName) {
            _namespace = nameSpace;
            _typeName = typeName;
        }

        internal string Namespace { get { return _namespace; } }
        internal string Name { get { return _typeName; } }

        public override int GetHashCode() {
            int hash = 13 << 20;
            if (_namespace != null) hash ^= _namespace.GetHashCode();
            if (_typeName != null) hash ^= _typeName.GetHashCode();
            return hash;
        }

        public override bool Equals(object obj) {
            if (!(obj is TypeName)) {
                return false;
            }
            TypeName tn = (TypeName)obj;
            return tn._namespace == _namespace && tn._typeName == _typeName;
        }

        public static bool operator ==(TypeName a, TypeName b) {
            return a._namespace == b._namespace && a._typeName == b._typeName;
        }

        public static bool operator !=(TypeName a, TypeName b) {
            return !(a == b);
        }
    }

    // TODO: Only used by ComObjectWityTypeInfo. Remove when gone!
    internal static class AssemblyTypeNames {
        public static IEnumerable<TypeName> GetTypeNames(Assembly assem, bool includePrivateTypes) {
#if !SILVERLIGHT
            switch (assem.FullName) {
                case "mscorlib, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089": return Get_mscorlib_TypeNames();
                case "System, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089": return Get_System_TypeNames();
                case "System.Xml, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089": return Get_SystemXml_TypeNames();
                case "System.Drawing, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a": return Get_SystemDrawing_TypeNames();
                case "System.Windows.Forms, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089": return Get_SystemWindowsForms_TypeNames();
            }
#endif

            Type[] types = LoadTypesFromAssembly(assem, includePrivateTypes);

            return GetTypeNames(types);
        }

        static IEnumerable<TypeName> GetTypeNames(Type[] types) {
            foreach (Type t in types) {
                if (ReflectionUtils.IsNested(t)) continue;
                TypeName typeName = new TypeName(t);
                yield return typeName;
            }
        }

        // Note that the result can contain null references.
        private static Type[] GetAllTypesFromAssembly(Assembly asm) {
#if SILVERLIGHT // ReflectionTypeLoadException
            try {
                return asm.GetTypes();
            } catch (Exception) {
                return Type.EmptyTypes;
            }
#else
            try {
                return asm.GetTypes();
            } catch (ReflectionTypeLoadException rtlException) {
                return rtlException.Types;
            }
#endif
        }

        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Design", "CA1031:DoNotCatchGeneralExceptionTypes")]
        public static Type[] LoadTypesFromAssembly(Assembly assembly, bool includePrivateTypes) {
            ContractUtils.RequiresNotNull(assembly, "assembly");

            if (includePrivateTypes) {
                return ArrayUtils.FindAll(GetAllTypesFromAssembly(assembly), (type) => type != null);
            }

            try {
                return assembly.GetExportedTypes();
            } catch (NotSupportedException) {
                // GetExportedTypes does not work with dynamic assemblies
            } catch (Exception) {
                // Some type loads may cause exceptions. Unfortunately, there is no way to ask GetExportedTypes
                // for just the list of types that we successfully loaded.
            }

            return ArrayUtils.FindAll(GetAllTypesFromAssembly(assembly), (type) => type != null && type.IsPublic);
        }

#if !SILVERLIGHT
        static IEnumerable<TypeName> GetTypeNames(string[] namespaces, string[][] types, TypeName[] orcasTypes) {
            Debug.Assert(namespaces.Length == types.Length);

            for (int i = 0; i < namespaces.Length; i++) {
                for (int j = 0; j < types[i].Length; j++) {
                    TypeName typeName = new TypeName(namespaces[i], types[i][j]);
                    yield return typeName;
                }
            }

            if (IsOrcas) {
                foreach(TypeName orcasType in orcasTypes) {
                    yield return orcasType;
                }
            }
        }

        static bool IsOrcas {
            get {
                Type t = typeof(object).Assembly.GetType("System.DateTimeOffset", false);
                return t != null;
            }
        }

        #region Generated Well-known assembly type names

        // *** BEGIN GENERATED CODE ***

        static IEnumerable<TypeName> Get_mscorlib_TypeNames() {
            // mscorlib, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089
            // Total number of types = 1264

            string [] namespaces = new string[47] {
                "Microsoft.Win32",
                "Microsoft.Win32.SafeHandles",
                "System",
                "System.Collections",
                "System.Collections.Generic",
                "System.Collections.ObjectModel",
                "System.Configuration.Assemblies",
                "System.Deployment.Internal",
                "System.Diagnostics",
                "System.Diagnostics.CodeAnalysis",
                "System.Diagnostics.SymbolStore",
                "System.Globalization",
                "System.IO",
                "System.IO.IsolatedStorage",
                "System.Reflection",
                "System.Reflection.Emit",
                "System.Resources",
                "System.Runtime",
                "System.Runtime.CompilerServices",
                "System.Runtime.ConstrainedExecution",
                "System.Runtime.Hosting",
                "System.Runtime.InteropServices",
                "System.Runtime.InteropServices.ComTypes",
                "System.Runtime.InteropServices.Expando",
                "System.Runtime.Remoting",
                "System.Runtime.Remoting.Activation",
                "System.Runtime.Remoting.Channels",
                "System.Runtime.Remoting.Contexts",
                "System.Runtime.Remoting.Lifetime",
                "System.Runtime.Remoting.Messaging",
                "System.Runtime.Remoting.Metadata",
                "System.Runtime.Remoting.Metadata.W3cXsd2001",
                "System.Runtime.Remoting.Proxies",
                "System.Runtime.Remoting.Services",
                "System.Runtime.Serialization",
                "System.Runtime.Serialization.Formatters",
                "System.Runtime.Serialization.Formatters.Binary",
                "System.Runtime.Versioning",
                "System.Security",
                "System.Security.AccessControl",
                "System.Security.Cryptography",
                "System.Security.Cryptography.X509Certificates",
                "System.Security.Permissions",
                "System.Security.Policy",
                "System.Security.Principal",
                "System.Text",
                "System.Threading",
            };

            string [][] types = new string[47][] {

                new string[6] { // Microsoft.Win32
                    "Registry",
                    "RegistryHive",
                    "RegistryKey",
                    "RegistryKeyPermissionCheck",
                    "RegistryValueKind",
                    "RegistryValueOptions",
                },

                new string[6] { // Microsoft.Win32.SafeHandles
                    "CriticalHandleMinusOneIsInvalid",
                    "CriticalHandleZeroOrMinusOneIsInvalid",
                    "SafeFileHandle",
                    "SafeHandleMinusOneIsInvalid",
                    "SafeHandleZeroOrMinusOneIsInvalid",
                    "SafeWaitHandle",
                },

                new string[171] { // System
                    "AccessViolationException",
                    "Action`1",
                    "ActivationContext",
                    "Activator",
                    "AppDomain",
                    "AppDomainInitializer",
                    "AppDomainManager",
                    "AppDomainManagerInitializationOptions",
                    "AppDomainSetup",
                    "AppDomainUnloadedException",
                    "ApplicationException",
                    "ApplicationId",
                    "ApplicationIdentity",
                    "ArgIterator",
                    "ArgumentException",
                    "ArgumentNullException",
                    "ArgumentOutOfRangeException",
                    "ArithmeticException",
                    "Array",
                    "ArraySegment`1",
                    "ArrayTypeMismatchException",
                    "AssemblyLoadEventArgs",
                    "AssemblyLoadEventHandler",
                    "AsyncCallback",
                    "Attribute",
                    "AttributeTargets",
                    "AttributeUsageAttribute",
                    "BadImageFormatException",
                    "Base64FormattingOptions",
                    "BitConverter",
                    "Boolean",
                    "Buffer",
                    "Byte",
                    "CLSCompliantAttribute",
                    "CannotUnloadAppDomainException",
                    "Char",
                    "CharEnumerator",
                    "Comparison`1",
                    "Console",
                    "ConsoleCancelEventArgs",
                    "ConsoleCancelEventHandler",
                    "ConsoleColor",
                    "ConsoleKey",
                    "ConsoleKeyInfo",
                    "ConsoleModifiers",
                    "ConsoleSpecialKey",
                    "ContextBoundObject",
                    "ContextMarshalException",
                    "ContextStaticAttribute",
                    "Convert",
                    "Converter`2",
                    "CrossAppDomainDelegate",
                    "DBNull",
                    "DataMisalignedException",
                    "DateTime",
                    "DateTimeKind",
                    "DayOfWeek",
                    "Decimal",
                    "Delegate",
                    "DivideByZeroException",
                    "DllNotFoundException",
                    "Double",
                    "DuplicateWaitObjectException",
                    "EntryPointNotFoundException",
                    "Enum",
                    "Environment",
                    "EnvironmentVariableTarget",
                    "EventArgs",
                    "EventHandler",
                    "EventHandler`1",
                    "Exception",
                    "ExecutionEngineException",
                    "FieldAccessException",
                    "FlagsAttribute",
                    "FormatException",
                    "GC",
                    "Guid",
                    "IAppDomainSetup",
                    "IAsyncResult",
                    "ICloneable",
                    "IComparable",
                    "IComparable`1",
                    "IConvertible",
                    "ICustomFormatter",
                    "IDisposable",
                    "IEquatable`1",
                    "IFormatProvider",
                    "IFormattable",
                    "IServiceProvider",
                    "IndexOutOfRangeException",
                    "InsufficientMemoryException",
                    "Int16",
                    "Int32",
                    "Int64",
                    "IntPtr",
                    "InvalidCastException",
                    "InvalidOperationException",
                    "InvalidProgramException",
                    "LoaderOptimization",
                    "LoaderOptimizationAttribute",
                    "LocalDataStoreSlot",
                    "MTAThreadAttribute",
                    "MarshalByRefObject",
                    "Math",
                    "MemberAccessException",
                    "MethodAccessException",
                    "MidpointRounding",
                    "MissingFieldException",
                    "MissingMemberException",
                    "MissingMethodException",
                    "ModuleHandle",
                    "MulticastDelegate",
                    "MulticastNotSupportedException",
                    "NonSerializedAttribute",
                    "NotFiniteNumberException",
                    "NotImplementedException",
                    "NotSupportedException",
                    "NullReferenceException",
                    "Nullable",
                    "Nullable`1",
                    "Object",
                    "ObjectDisposedException",
                    "ObsoleteAttribute",
                    "OperatingSystem",
                    "OperationCanceledException",
                    "OutOfMemoryException",
                    "OverflowException",
                    "ParamArrayAttribute",
                    "PlatformID",
                    "PlatformNotSupportedException",
                    "Predicate`1",
                    "Random",
                    "RankException",
                    "ResolveEventArgs",
                    "ResolveEventHandler",
                    "RuntimeArgumentHandle",
                    "RuntimeFieldHandle",
                    "RuntimeMethodHandle",
                    "RuntimeTypeHandle",
                    "SByte",
                    "STAThreadAttribute",
                    "SerializableAttribute",
                    "Single",
                    "StackOverflowException",
                    "String",
                    "StringComparer",
                    "StringComparison",
                    "StringSplitOptions",
                    "SystemException",
                    "ThreadStaticAttribute",
                    "TimeSpan",
                    "TimeZone",
                    "TimeoutException",
                    "Type",
                    "TypeCode",
                    "TypeInitializationException",
                    "TypeLoadException",
                    "TypeUnloadedException",
                    "TypedReference",
                    "UInt16",
                    "UInt32",
                    "UInt64",
                    "UIntPtr",
                    "UnauthorizedAccessException",
                    "UnhandledExceptionEventArgs",
                    "UnhandledExceptionEventHandler",
                    "ValueType",
                    "Version",
                    "Void",
                    "WeakReference",
                    "_AppDomain",
                },

                new string[22] { // System.Collections
                    "ArrayList",
                    "BitArray",
                    "CaseInsensitiveComparer",
                    "CaseInsensitiveHashCodeProvider",
                    "CollectionBase",
                    "Comparer",
                    "DictionaryBase",
                    "DictionaryEntry",
                    "Hashtable",
                    "ICollection",
                    "IComparer",
                    "IDictionary",
                    "IDictionaryEnumerator",
                    "IEnumerable",
                    "IEnumerator",
                    "IEqualityComparer",
                    "IHashCodeProvider",
                    "IList",
                    "Queue",
                    "ReadOnlyCollectionBase",
                    "SortedList",
                    "Stack",
                },

                new string[13] { // System.Collections.Generic
                    "Comparer`1",
                    "Dictionary`2",
                    "EqualityComparer`1",
                    "ICollection`1",
                    "IComparer`1",
                    "IDictionary`2",
                    "IEnumerable`1",
                    "IEnumerator`1",
                    "IEqualityComparer`1",
                    "IList`1",
                    "KeyNotFoundException",
                    "KeyValuePair`2",
                    "List`1",
                },

                new string[3] { // System.Collections.ObjectModel
                    "Collection`1",
                    "KeyedCollection`2",
                    "ReadOnlyCollection`1",
                },

                new string[3] { // System.Configuration.Assemblies
                    "AssemblyHash",
                    "AssemblyHashAlgorithm",
                    "AssemblyVersionCompatibility",
                },

                new string[2] { // System.Deployment.Internal
                    "InternalActivationContextHelper",
                    "InternalApplicationIdentityHelper",
                },

                new string[14] { // System.Diagnostics
                    "ConditionalAttribute",
                    "DebuggableAttribute",
                    "Debugger",
                    "DebuggerBrowsableAttribute",
                    "DebuggerBrowsableState",
                    "DebuggerDisplayAttribute",
                    "DebuggerHiddenAttribute",
                    "DebuggerNonUserCodeAttribute",
                    "DebuggerStepThroughAttribute",
                    "DebuggerStepperBoundaryAttribute",
                    "DebuggerTypeProxyAttribute",
                    "DebuggerVisualizerAttribute",
                    "StackFrame",
                    "StackTrace",
                },

                new string[1] { // System.Diagnostics.CodeAnalysis
                    "SuppressMessageAttribute",
                },

                new string[15] { // System.Diagnostics.SymbolStore
                    "ISymbolBinder",
                    "ISymbolBinder1",
                    "ISymbolDocument",
                    "ISymbolDocumentWriter",
                    "ISymbolMethod",
                    "ISymbolNamespace",
                    "ISymbolReader",
                    "ISymbolScope",
                    "ISymbolVariable",
                    "ISymbolWriter",
                    "SymAddressKind",
                    "SymDocumentType",
                    "SymLanguageType",
                    "SymLanguageVendor",
                    "SymbolToken",
                },

                new string[37] { // System.Globalization
                    "Calendar",
                    "CalendarAlgorithmType",
                    "CalendarWeekRule",
                    "CharUnicodeInfo",
                    "ChineseLunisolarCalendar",
                    "CompareInfo",
                    "CompareOptions",
                    "CultureInfo",
                    "CultureTypes",
                    "DateTimeFormatInfo",
                    "DateTimeStyles",
                    "DaylightTime",
                    "DigitShapes",
                    "EastAsianLunisolarCalendar",
                    "GregorianCalendar",
                    "GregorianCalendarTypes",
                    "HebrewCalendar",
                    "HijriCalendar",
                    "IdnMapping",
                    "JapaneseCalendar",
                    "JapaneseLunisolarCalendar",
                    "JulianCalendar",
                    "KoreanCalendar",
                    "KoreanLunisolarCalendar",
                    "NumberFormatInfo",
                    "NumberStyles",
                    "PersianCalendar",
                    "RegionInfo",
                    "SortKey",
                    "StringInfo",
                    "TaiwanCalendar",
                    "TaiwanLunisolarCalendar",
                    "TextElementEnumerator",
                    "TextInfo",
                    "ThaiBuddhistCalendar",
                    "UmAlQuraCalendar",
                    "UnicodeCategory",
                },

                new string[35] { // System.IO
                    "BinaryReader",
                    "BinaryWriter",
                    "BufferedStream",
                    "Directory",
                    "DirectoryInfo",
                    "DirectoryNotFoundException",
                    "DriveInfo",
                    "DriveNotFoundException",
                    "DriveType",
                    "EndOfStreamException",
                    "File",
                    "FileAccess",
                    "FileAttributes",
                    "FileInfo",
                    "FileLoadException",
                    "FileMode",
                    "FileNotFoundException",
                    "FileOptions",
                    "FileShare",
                    "FileStream",
                    "FileSystemInfo",
                    "IOException",
                    "MemoryStream",
                    "Path",
                    "PathTooLongException",
                    "SearchOption",
                    "SeekOrigin",
                    "Stream",
                    "StreamReader",
                    "StreamWriter",
                    "StringReader",
                    "StringWriter",
                    "TextReader",
                    "TextWriter",
                    "UnmanagedMemoryStream",
                },

                new string[6] { // System.IO.IsolatedStorage
                    "INormalizeForIsolatedStorage",
                    "IsolatedStorage",
                    "IsolatedStorageException",
                    "IsolatedStorageFile",
                    "IsolatedStorageFileStream",
                    "IsolatedStorageScope",
                },

                new string[76] { // System.Reflection
                    "AmbiguousMatchException",
                    "Assembly",
                    "AssemblyAlgorithmIdAttribute",
                    "AssemblyCompanyAttribute",
                    "AssemblyConfigurationAttribute",
                    "AssemblyCopyrightAttribute",
                    "AssemblyCultureAttribute",
                    "AssemblyDefaultAliasAttribute",
                    "AssemblyDelaySignAttribute",
                    "AssemblyDescriptionAttribute",
                    "AssemblyFileVersionAttribute",
                    "AssemblyFlagsAttribute",
                    "AssemblyInformationalVersionAttribute",
                    "AssemblyKeyFileAttribute",
                    "AssemblyKeyNameAttribute",
                    "AssemblyName",
                    "AssemblyNameFlags",
                    "AssemblyNameProxy",
                    "AssemblyProductAttribute",
                    "AssemblyTitleAttribute",
                    "AssemblyTrademarkAttribute",
                    "AssemblyVersionAttribute",
                    "Binder",
                    "BindingFlags",
                    "CallingConventions",
                    "ConstructorInfo",
                    "CustomAttributeData",
                    "CustomAttributeFormatException",
                    "CustomAttributeNamedArgument",
                    "CustomAttributeTypedArgument",
                    "DefaultMemberAttribute",
                    "EventAttributes",
                    "EventInfo",
                    "ExceptionHandlingClause",
                    "ExceptionHandlingClauseOptions",
                    "FieldAttributes",
                    "FieldInfo",
                    "GenericParameterAttributes",
                    "ICustomAttributeProvider",
                    "IReflect",
                    "ImageFileMachine",
                    "InterfaceMapping",
                    "InvalidFilterCriteriaException",
                    "LocalVariableInfo",
                    "ManifestResourceInfo",
                    "MemberFilter",
                    "MemberInfo",
                    "MemberTypes",
                    "MethodAttributes",
                    "MethodBase",
                    "MethodBody",
                    "MethodImplAttributes",
                    "MethodInfo",
                    "Missing",
                    "Module",
                    "ModuleResolveEventHandler",
                    "ObfuscateAssemblyAttribute",
                    "ObfuscationAttribute",
                    "ParameterAttributes",
                    "ParameterInfo",
                    "ParameterModifier",
                    "Pointer",
                    "PortableExecutableKinds",
                    "ProcessorArchitecture",
                    "PropertyAttributes",
                    "PropertyInfo",
                    "ReflectionTypeLoadException",
                    "ResourceAttributes",
                    "ResourceLocation",
                    "StrongNameKeyPair",
                    "TargetException",
                    "TargetInvocationException",
                    "TargetParameterCountException",
                    "TypeAttributes",
                    "TypeDelegator",
                    "TypeFilter",
                },

                new string[37] { // System.Reflection.Emit
                    "AssemblyBuilder",
                    "AssemblyBuilderAccess",
                    "ConstructorBuilder",
                    "CustomAttributeBuilder",
                    "DynamicILInfo",
                    "DynamicMethod",
                    "EnumBuilder",
                    "EventBuilder",
                    "EventToken",
                    "FieldBuilder",
                    "FieldToken",
                    "FlowControl",
                    "GenericTypeParameterBuilder",
                    "ILGenerator",
                    "Label",
                    "LocalBuilder",
                    "MethodBuilder",
                    "MethodRental",
                    "MethodToken",
                    "ModuleBuilder",
                    "OpCode",
                    "OpCodeType",
                    "OpCodes",
                    "OperandType",
                    "PEFileKinds",
                    "PackingSize",
                    "ParameterBuilder",
                    "ParameterToken",
                    "PropertyBuilder",
                    "PropertyToken",
                    "SignatureHelper",
                    "SignatureToken",
                    "StackBehaviour",
                    "StringToken",
                    "TypeBuilder",
                    "TypeToken",
                    "UnmanagedMarshal",
                },

                new string[11] { // System.Resources
                    "IResourceReader",
                    "IResourceWriter",
                    "MissingManifestResourceException",
                    "MissingSatelliteAssemblyException",
                    "NeutralResourcesLanguageAttribute",
                    "ResourceManager",
                    "ResourceReader",
                    "ResourceSet",
                    "ResourceWriter",
                    "SatelliteContractVersionAttribute",
                    "UltimateResourceFallbackLocation",
                },

                new string[2] { // System.Runtime
                    "GCSettings",
                    "MemoryFailPoint",
                },

                new string[50] { // System.Runtime.CompilerServices
                    "AccessedThroughPropertyAttribute",
                    "CallConvCdecl",
                    "CallConvFastcall",
                    "CallConvStdcall",
                    "CallConvThiscall",
                    "CompilationRelaxations",
                    "CompilationRelaxationsAttribute",
                    "CompilerGeneratedAttribute",
                    "CompilerGlobalScopeAttribute",
                    "CompilerMarshalOverride",
                    "CustomConstantAttribute",
                    "DateTimeConstantAttribute",
                    "DecimalConstantAttribute",
                    "DefaultDependencyAttribute",
                    "DependencyAttribute",
                    "DiscardableAttribute",
                    "FixedAddressValueTypeAttribute",
                    "FixedBufferAttribute",
                    "HasCopySemanticsAttribute",
                    "IDispatchConstantAttribute",
                    "IUnknownConstantAttribute",
                    "IndexerNameAttribute",
                    "InternalsVisibleToAttribute",
                    "IsBoxed",
                    "IsByValue",
                    "IsConst",
                    "IsCopyConstructed",
                    "IsExplicitlyDereferenced",
                    "IsImplicitlyDereferenced",
                    "IsJitIntrinsic",
                    "IsLong",
                    "IsPinned",
                    "IsSignUnspecifiedByte",
                    "IsUdtReturn",
                    "IsVolatile",
                    "LoadHint",
                    "MethodCodeType",
                    "MethodImplAttribute",
                    "MethodImplOptions",
                    "NativeCppClassAttribute",
                    "RequiredAttributeAttribute",
                    "RuntimeCompatibilityAttribute",
                    "RuntimeHelpers",
                    "RuntimeWrappedException",
                    "ScopelessEnumAttribute",
                    "SpecialNameAttribute",
                    "StringFreezingAttribute",
                    "SuppressIldasmAttribute",
                    "TypeForwardedToAttribute",
                    "UnsafeValueTypeAttribute",
                },

                new string[5] { // System.Runtime.ConstrainedExecution
                    "Cer",
                    "Consistency",
                    "CriticalFinalizerObject",
                    "PrePrepareMethodAttribute",
                    "ReliabilityContractAttribute",
                },

                new string[2] { // System.Runtime.Hosting
                    "ActivationArguments",
                    "ApplicationActivator",
                },

                new string[165] { // System.Runtime.InteropServices
                    "ArrayWithOffset",
                    "AssemblyRegistrationFlags",
                    "AutomationProxyAttribute",
                    "BINDPTR",
                    "BIND_OPTS",
                    "BStrWrapper",
                    "BestFitMappingAttribute",
                    "CALLCONV",
                    "COMException",
                    "CONNECTDATA",
                    "CallingConvention",
                    "CharSet",
                    "ClassInterfaceAttribute",
                    "ClassInterfaceType",
                    "CoClassAttribute",
                    "ComAliasNameAttribute",
                    "ComCompatibleVersionAttribute",
                    "ComConversionLossAttribute",
                    "ComDefaultInterfaceAttribute",
                    "ComEventInterfaceAttribute",
                    "ComImportAttribute",
                    "ComInterfaceType",
                    "ComMemberType",
                    "ComRegisterFunctionAttribute",
                    "ComSourceInterfacesAttribute",
                    "ComUnregisterFunctionAttribute",
                    "ComVisibleAttribute",
                    "CriticalHandle",
                    "CurrencyWrapper",
                    "DESCKIND",
                    "DISPPARAMS",
                    "DefaultCharSetAttribute",
                    "DispIdAttribute",
                    "DispatchWrapper",
                    "DllImportAttribute",
                    "ELEMDESC",
                    "EXCEPINFO",
                    "ErrorWrapper",
                    "ExporterEventKind",
                    "ExtensibleClassFactory",
                    "ExternalException",
                    "FILETIME",
                    "FUNCDESC",
                    "FUNCFLAGS",
                    "FUNCKIND",
                    "FieldOffsetAttribute",
                    "GCHandle",
                    "GCHandleType",
                    "GuidAttribute",
                    "HandleRef",
                    "ICustomAdapter",
                    "ICustomFactory",
                    "ICustomMarshaler",
                    "IDLDESC",
                    "IDLFLAG",
                    "IDispatchImplAttribute",
                    "IDispatchImplType",
                    "IMPLTYPEFLAGS",
                    "INVOKEKIND",
                    "IRegistrationServices",
                    "ITypeLibConverter",
                    "ITypeLibExporterNameProvider",
                    "ITypeLibExporterNotifySink",
                    "ITypeLibImporterNotifySink",
                    "ImportedFromTypeLibAttribute",
                    "ImporterEventKind",
                    "InAttribute",
                    "InterfaceTypeAttribute",
                    "InvalidComObjectException",
                    "InvalidOleVariantTypeException",
                    "LCIDConversionAttribute",
                    "LIBFLAGS",
                    "LayoutKind",
                    "Marshal",
                    "MarshalAsAttribute",
                    "MarshalDirectiveException",
                    "ObjectCreationDelegate",
                    "OptionalAttribute",
                    "OutAttribute",
                    "PARAMDESC",
                    "PARAMFLAG",
                    "PreserveSigAttribute",
                    "PrimaryInteropAssemblyAttribute",
                    "ProgIdAttribute",
                    "RegistrationClassContext",
                    "RegistrationConnectionType",
                    "RegistrationServices",
                    "RuntimeEnvironment",
                    "SEHException",
                    "STATSTG",
                    "SYSKIND",
                    "SafeArrayRankMismatchException",
                    "SafeArrayTypeMismatchException",
                    "SafeHandle",
                    "SetWin32ContextInIDispatchAttribute",
                    "StructLayoutAttribute",
                    "TYPEATTR",
                    "TYPEDESC",
                    "TYPEFLAGS",
                    "TYPEKIND",
                    "TYPELIBATTR",
                    "TypeLibConverter",
                    "TypeLibExporterFlags",
                    "TypeLibFuncAttribute",
                    "TypeLibFuncFlags",
                    "TypeLibImportClassAttribute",
                    "TypeLibImporterFlags",
                    "TypeLibTypeAttribute",
                    "TypeLibTypeFlags",
                    "TypeLibVarAttribute",
                    "TypeLibVarFlags",
                    "TypeLibVersionAttribute",
                    "UCOMIBindCtx",
                    "UCOMIConnectionPoint",
                    "UCOMIConnectionPointContainer",
                    "UCOMIEnumConnectionPoints",
                    "UCOMIEnumConnections",
                    "UCOMIEnumMoniker",
                    "UCOMIEnumString",
                    "UCOMIEnumVARIANT",
                    "UCOMIMoniker",
                    "UCOMIPersistFile",
                    "UCOMIRunningObjectTable",
                    "UCOMIStream",
                    "UCOMITypeComp",
                    "UCOMITypeInfo",
                    "UCOMITypeLib",
                    "UnknownWrapper",
                    "UnmanagedFunctionPointerAttribute",
                    "UnmanagedType",
                    "VARDESC",
                    "VARFLAGS",
                    "VarEnum",
                    "VariantWrapper",
                    "_Activator",
                    "_Assembly",
                    "_AssemblyBuilder",
                    "_AssemblyName",
                    "_Attribute",
                    "_ConstructorBuilder",
                    "_ConstructorInfo",
                    "_CustomAttributeBuilder",
                    "_EnumBuilder",
                    "_EventBuilder",
                    "_EventInfo",
                    "_Exception",
                    "_FieldBuilder",
                    "_FieldInfo",
                    "_ILGenerator",
                    "_LocalBuilder",
                    "_MemberInfo",
                    "_MethodBase",
                    "_MethodBuilder",
                    "_MethodInfo",
                    "_MethodRental",
                    "_Module",
                    "_ModuleBuilder",
                    "_ParameterBuilder",
                    "_ParameterInfo",
                    "_PropertyBuilder",
                    "_PropertyInfo",
                    "_SignatureHelper",
                    "_Thread",
                    "_Type",
                    "_TypeBuilder",
                },

                new string[46] { // System.Runtime.InteropServices.ComTypes
                    "BINDPTR",
                    "BIND_OPTS",
                    "CALLCONV",
                    "CONNECTDATA",
                    "DESCKIND",
                    "DISPPARAMS",
                    "ELEMDESC",
                    "EXCEPINFO",
                    "FILETIME",
                    "FUNCDESC",
                    "FUNCFLAGS",
                    "FUNCKIND",
                    "IBindCtx",
                    "IConnectionPoint",
                    "IConnectionPointContainer",
                    "IDLDESC",
                    "IDLFLAG",
                    "IEnumConnectionPoints",
                    "IEnumConnections",
                    "IEnumMoniker",
                    "IEnumString",
                    "IEnumVARIANT",
                    "IMPLTYPEFLAGS",
                    "IMoniker",
                    "INVOKEKIND",
                    "IPersistFile",
                    "IRunningObjectTable",
                    "IStream",
                    "ITypeComp",
                    "ITypeInfo",
                    "ITypeInfo2",
                    "ITypeLib",
                    "ITypeLib2",
                    "LIBFLAGS",
                    "PARAMDESC",
                    "PARAMFLAG",
                    "STATSTG",
                    "SYSKIND",
                    "TYPEATTR",
                    "TYPEDESC",
                    "TYPEFLAGS",
                    "TYPEKIND",
                    "TYPELIBATTR",
                    "VARDESC",
                    "VARFLAGS",
                    "VARKIND",
                },

                new string[1] { // System.Runtime.InteropServices.Expando
                    "IExpando",
                },

                new string[20] { // System.Runtime.Remoting
                    "ActivatedClientTypeEntry",
                    "ActivatedServiceTypeEntry",
                    "CustomErrorsModes",
                    "IChannelInfo",
                    "IEnvoyInfo",
                    "IObjectHandle",
                    "IRemotingTypeInfo",
                    "InternalRemotingServices",
                    "ObjRef",
                    "ObjectHandle",
                    "RemotingConfiguration",
                    "RemotingException",
                    "RemotingServices",
                    "RemotingTimeoutException",
                    "ServerException",
                    "SoapServices",
                    "TypeEntry",
                    "WellKnownClientTypeEntry",
                    "WellKnownObjectMode",
                    "WellKnownServiceTypeEntry",
                },

                new string[5] { // System.Runtime.Remoting.Activation
                    "ActivatorLevel",
                    "IActivator",
                    "IConstructionCallMessage",
                    "IConstructionReturnMessage",
                    "UrlAttribute",
                },

                new string[29] { // System.Runtime.Remoting.Channels
                    "BaseChannelObjectWithProperties",
                    "BaseChannelSinkWithProperties",
                    "BaseChannelWithProperties",
                    "ChannelDataStore",
                    "ChannelServices",
                    "ClientChannelSinkStack",
                    "IChannel",
                    "IChannelDataStore",
                    "IChannelReceiver",
                    "IChannelReceiverHook",
                    "IChannelSender",
                    "IChannelSinkBase",
                    "IClientChannelSink",
                    "IClientChannelSinkProvider",
                    "IClientChannelSinkStack",
                    "IClientFormatterSink",
                    "IClientFormatterSinkProvider",
                    "IClientResponseChannelSinkStack",
                    "ISecurableChannel",
                    "IServerChannelSink",
                    "IServerChannelSinkProvider",
                    "IServerChannelSinkStack",
                    "IServerFormatterSinkProvider",
                    "IServerResponseChannelSinkStack",
                    "ITransportHeaders",
                    "ServerChannelSinkStack",
                    "ServerProcessing",
                    "SinkProviderData",
                    "TransportHeaders",
                },

                new string[15] { // System.Runtime.Remoting.Contexts
                    "Context",
                    "ContextAttribute",
                    "ContextProperty",
                    "CrossContextDelegate",
                    "IContextAttribute",
                    "IContextProperty",
                    "IContextPropertyActivator",
                    "IContributeClientContextSink",
                    "IContributeDynamicSink",
                    "IContributeEnvoySink",
                    "IContributeObjectSink",
                    "IContributeServerContextSink",
                    "IDynamicMessageSink",
                    "IDynamicProperty",
                    "SynchronizationAttribute",
                },

                new string[5] { // System.Runtime.Remoting.Lifetime
                    "ClientSponsor",
                    "ILease",
                    "ISponsor",
                    "LeaseState",
                    "LifetimeServices",
                },

                new string[24] { // System.Runtime.Remoting.Messaging
                    "AsyncResult",
                    "CallContext",
                    "ConstructionCall",
                    "ConstructionResponse",
                    "Header",
                    "HeaderHandler",
                    "ILogicalThreadAffinative",
                    "IMessage",
                    "IMessageCtrl",
                    "IMessageSink",
                    "IMethodCallMessage",
                    "IMethodMessage",
                    "IMethodReturnMessage",
                    "IRemotingFormatter",
                    "InternalMessageWrapper",
                    "LogicalCallContext",
                    "MessageSurrogateFilter",
                    "MethodCall",
                    "MethodCallMessageWrapper",
                    "MethodResponse",
                    "MethodReturnMessageWrapper",
                    "OneWayAttribute",
                    "RemotingSurrogateSelector",
                    "ReturnMessage",
                },

                new string[7] { // System.Runtime.Remoting.Metadata
                    "SoapAttribute",
                    "SoapFieldAttribute",
                    "SoapMethodAttribute",
                    "SoapOption",
                    "SoapParameterAttribute",
                    "SoapTypeAttribute",
                    "XmlFieldOrderOption",
                },

                new string[32] { // System.Runtime.Remoting.Metadata.W3cXsd2001
                    "ISoapXsd",
                    "SoapAnyUri",
                    "SoapBase64Binary",
                    "SoapDate",
                    "SoapDateTime",
                    "SoapDay",
                    "SoapDuration",
                    "SoapEntities",
                    "SoapEntity",
                    "SoapHexBinary",
                    "SoapId",
                    "SoapIdref",
                    "SoapIdrefs",
                    "SoapInteger",
                    "SoapLanguage",
                    "SoapMonth",
                    "SoapMonthDay",
                    "SoapName",
                    "SoapNcName",
                    "SoapNegativeInteger",
                    "SoapNmtoken",
                    "SoapNmtokens",
                    "SoapNonNegativeInteger",
                    "SoapNonPositiveInteger",
                    "SoapNormalizedString",
                    "SoapNotation",
                    "SoapPositiveInteger",
                    "SoapQName",
                    "SoapTime",
                    "SoapToken",
                    "SoapYear",
                    "SoapYearMonth",
                },

                new string[2] { // System.Runtime.Remoting.Proxies
                    "ProxyAttribute",
                    "RealProxy",
                },

                new string[3] { // System.Runtime.Remoting.Services
                    "EnterpriseServicesHelper",
                    "ITrackingHandler",
                    "TrackingServices",
                },

                new string[26] { // System.Runtime.Serialization
                    "Formatter",
                    "FormatterConverter",
                    "FormatterServices",
                    "IDeserializationCallback",
                    "IFormatter",
                    "IFormatterConverter",
                    "IObjectReference",
                    "ISerializable",
                    "ISerializationSurrogate",
                    "ISurrogateSelector",
                    "ObjectIDGenerator",
                    "ObjectManager",
                    "OnDeserializedAttribute",
                    "OnDeserializingAttribute",
                    "OnSerializedAttribute",
                    "OnSerializingAttribute",
                    "OptionalFieldAttribute",
                    "SerializationBinder",
                    "SerializationEntry",
                    "SerializationException",
                    "SerializationInfo",
                    "SerializationInfoEnumerator",
                    "SerializationObjectManager",
                    "StreamingContext",
                    "StreamingContextStates",
                    "SurrogateSelector",
                },

                new string[10] { // System.Runtime.Serialization.Formatters
                    "FormatterAssemblyStyle",
                    "FormatterTypeStyle",
                    "IFieldInfo",
                    "ISoapMessage",
                    "InternalRM",
                    "InternalST",
                    "ServerFault",
                    "SoapFault",
                    "SoapMessage",
                    "TypeFilterLevel",
                },

                new string[1] { // System.Runtime.Serialization.Formatters.Binary
                    "BinaryFormatter",
                },

                new string[4] { // System.Runtime.Versioning
                    "ResourceConsumptionAttribute",
                    "ResourceExposureAttribute",
                    "ResourceScope",
                    "VersioningHelper",
                },

                new string[27] { // System.Security
                    "AllowPartiallyTrustedCallersAttribute",
                    "CodeAccessPermission",
                    "HostProtectionException",
                    "HostSecurityManager",
                    "HostSecurityManagerOptions",
                    "IEvidenceFactory",
                    "IPermission",
                    "ISecurityEncodable",
                    "ISecurityPolicyEncodable",
                    "IStackWalk",
                    "NamedPermissionSet",
                    "PermissionSet",
                    "PolicyLevelType",
                    "SecureString",
                    "SecurityContext",
                    "SecurityCriticalAttribute",
                    "SecurityCriticalScope",
                    "SecurityElement",
                    "SecurityException",
                    "SecurityManager",
                    "SecurityTransparentAttribute",
                    "SecurityTreatAsSafeAttribute",
                    "SecurityZone",
                    "SuppressUnmanagedCodeSecurityAttribute",
                    "UnverifiableCodeAttribute",
                    "VerificationException",
                    "XmlSyntaxException",
                },

                new string[64] { // System.Security.AccessControl
                    "AccessControlActions",
                    "AccessControlModification",
                    "AccessControlSections",
                    "AccessControlType",
                    "AccessRule",
                    "AceEnumerator",
                    "AceFlags",
                    "AceQualifier",
                    "AceType",
                    "AuditFlags",
                    "AuditRule",
                    "AuthorizationRule",
                    "AuthorizationRuleCollection",
                    "CommonAce",
                    "CommonAcl",
                    "CommonObjectSecurity",
                    "CommonSecurityDescriptor",
                    "CompoundAce",
                    "CompoundAceType",
                    "ControlFlags",
                    "CryptoKeyAccessRule",
                    "CryptoKeyAuditRule",
                    "CryptoKeyRights",
                    "CryptoKeySecurity",
                    "CustomAce",
                    "DirectoryObjectSecurity",
                    "DirectorySecurity",
                    "DiscretionaryAcl",
                    "EventWaitHandleAccessRule",
                    "EventWaitHandleAuditRule",
                    "EventWaitHandleRights",
                    "EventWaitHandleSecurity",
                    "FileSecurity",
                    "FileSystemAccessRule",
                    "FileSystemAuditRule",
                    "FileSystemRights",
                    "FileSystemSecurity",
                    "GenericAce",
                    "GenericAcl",
                    "GenericSecurityDescriptor",
                    "InheritanceFlags",
                    "KnownAce",
                    "MutexAccessRule",
                    "MutexAuditRule",
                    "MutexRights",
                    "MutexSecurity",
                    "NativeObjectSecurity",
                    "ObjectAccessRule",
                    "ObjectAce",
                    "ObjectAceFlags",
                    "ObjectAuditRule",
                    "ObjectSecurity",
                    "PrivilegeNotHeldException",
                    "PropagationFlags",
                    "QualifiedAce",
                    "RawAcl",
                    "RawSecurityDescriptor",
                    "RegistryAccessRule",
                    "RegistryAuditRule",
                    "RegistryRights",
                    "RegistrySecurity",
                    "ResourceType",
                    "SecurityInfos",
                    "SystemAcl",
                },

                new string[78] { // System.Security.Cryptography
                    "AsymmetricAlgorithm",
                    "AsymmetricKeyExchangeDeformatter",
                    "AsymmetricKeyExchangeFormatter",
                    "AsymmetricSignatureDeformatter",
                    "AsymmetricSignatureFormatter",
                    "CipherMode",
                    "CryptoAPITransform",
                    "CryptoConfig",
                    "CryptoStream",
                    "CryptoStreamMode",
                    "CryptographicException",
                    "CryptographicUnexpectedOperationException",
                    "CspKeyContainerInfo",
                    "CspParameters",
                    "CspProviderFlags",
                    "DES",
                    "DESCryptoServiceProvider",
                    "DSA",
                    "DSACryptoServiceProvider",
                    "DSAParameters",
                    "DSASignatureDeformatter",
                    "DSASignatureFormatter",
                    "DeriveBytes",
                    "FromBase64Transform",
                    "FromBase64TransformMode",
                    "HMAC",
                    "HMACMD5",
                    "HMACRIPEMD160",
                    "HMACSHA1",
                    "HMACSHA256",
                    "HMACSHA384",
                    "HMACSHA512",
                    "HashAlgorithm",
                    "ICryptoTransform",
                    "ICspAsymmetricAlgorithm",
                    "KeyNumber",
                    "KeySizes",
                    "KeyedHashAlgorithm",
                    "MACTripleDES",
                    "MD5",
                    "MD5CryptoServiceProvider",
                    "MaskGenerationMethod",
                    "PKCS1MaskGenerationMethod",
                    "PaddingMode",
                    "PasswordDeriveBytes",
                    "RC2",
                    "RC2CryptoServiceProvider",
                    "RIPEMD160",
                    "RIPEMD160Managed",
                    "RNGCryptoServiceProvider",
                    "RSA",
                    "RSACryptoServiceProvider",
                    "RSAOAEPKeyExchangeDeformatter",
                    "RSAOAEPKeyExchangeFormatter",
                    "RSAPKCS1KeyExchangeDeformatter",
                    "RSAPKCS1KeyExchangeFormatter",
                    "RSAPKCS1SignatureDeformatter",
                    "RSAPKCS1SignatureFormatter",
                    "RSAParameters",
                    "RandomNumberGenerator",
                    "Rfc2898DeriveBytes",
                    "Rijndael",
                    "RijndaelManaged",
                    "RijndaelManagedTransform",
                    "SHA1",
                    "SHA1CryptoServiceProvider",
                    "SHA1Managed",
                    "SHA256",
                    "SHA256Managed",
                    "SHA384",
                    "SHA384Managed",
                    "SHA512",
                    "SHA512Managed",
                    "SignatureDescription",
                    "SymmetricAlgorithm",
                    "ToBase64Transform",
                    "TripleDES",
                    "TripleDESCryptoServiceProvider",
                },

                new string[3] { // System.Security.Cryptography.X509Certificates
                    "X509Certificate",
                    "X509ContentType",
                    "X509KeyStorageFlags",
                },

                new string[56] { // System.Security.Permissions
                    "CodeAccessSecurityAttribute",
                    "EnvironmentPermission",
                    "EnvironmentPermissionAccess",
                    "EnvironmentPermissionAttribute",
                    "FileDialogPermission",
                    "FileDialogPermissionAccess",
                    "FileDialogPermissionAttribute",
                    "FileIOPermission",
                    "FileIOPermissionAccess",
                    "FileIOPermissionAttribute",
                    "GacIdentityPermission",
                    "GacIdentityPermissionAttribute",
                    "HostProtectionAttribute",
                    "HostProtectionResource",
                    "IUnrestrictedPermission",
                    "IsolatedStorageContainment",
                    "IsolatedStorageFilePermission",
                    "IsolatedStorageFilePermissionAttribute",
                    "IsolatedStoragePermission",
                    "IsolatedStoragePermissionAttribute",
                    "KeyContainerPermission",
                    "KeyContainerPermissionAccessEntry",
                    "KeyContainerPermissionAccessEntryCollection",
                    "KeyContainerPermissionAccessEntryEnumerator",
                    "KeyContainerPermissionAttribute",
                    "KeyContainerPermissionFlags",
                    "PermissionSetAttribute",
                    "PermissionState",
                    "PrincipalPermission",
                    "PrincipalPermissionAttribute",
                    "PublisherIdentityPermission",
                    "PublisherIdentityPermissionAttribute",
                    "ReflectionPermission",
                    "ReflectionPermissionAttribute",
                    "ReflectionPermissionFlag",
                    "RegistryPermission",
                    "RegistryPermissionAccess",
                    "RegistryPermissionAttribute",
                    "SecurityAction",
                    "SecurityAttribute",
                    "SecurityPermission",
                    "SecurityPermissionAttribute",
                    "SecurityPermissionFlag",
                    "SiteIdentityPermission",
                    "SiteIdentityPermissionAttribute",
                    "StrongNameIdentityPermission",
                    "StrongNameIdentityPermissionAttribute",
                    "StrongNamePublicKeyBlob",
                    "UIPermission",
                    "UIPermissionAttribute",
                    "UIPermissionClipboard",
                    "UIPermissionWindow",
                    "UrlIdentityPermission",
                    "UrlIdentityPermissionAttribute",
                    "ZoneIdentityPermission",
                    "ZoneIdentityPermissionAttribute",
                },

                new string[40] { // System.Security.Policy
                    "AllMembershipCondition",
                    "ApplicationDirectory",
                    "ApplicationDirectoryMembershipCondition",
                    "ApplicationSecurityInfo",
                    "ApplicationSecurityManager",
                    "ApplicationTrust",
                    "ApplicationTrustCollection",
                    "ApplicationTrustEnumerator",
                    "ApplicationVersionMatch",
                    "CodeConnectAccess",
                    "CodeGroup",
                    "Evidence",
                    "FileCodeGroup",
                    "FirstMatchCodeGroup",
                    "GacInstalled",
                    "GacMembershipCondition",
                    "Hash",
                    "HashMembershipCondition",
                    "IApplicationTrustManager",
                    "IIdentityPermissionFactory",
                    "IMembershipCondition",
                    "NetCodeGroup",
                    "PermissionRequestEvidence",
                    "PolicyException",
                    "PolicyLevel",
                    "PolicyStatement",
                    "PolicyStatementAttribute",
                    "Publisher",
                    "PublisherMembershipCondition",
                    "Site",
                    "SiteMembershipCondition",
                    "StrongName",
                    "StrongNameMembershipCondition",
                    "TrustManagerContext",
                    "TrustManagerUIContext",
                    "UnionCodeGroup",
                    "Url",
                    "UrlMembershipCondition",
                    "Zone",
                    "ZoneMembershipCondition",
                },

                new string[18] { // System.Security.Principal
                    "GenericIdentity",
                    "GenericPrincipal",
                    "IIdentity",
                    "IPrincipal",
                    "IdentityNotMappedException",
                    "IdentityReference",
                    "IdentityReferenceCollection",
                    "NTAccount",
                    "PrincipalPolicy",
                    "SecurityIdentifier",
                    "TokenAccessLevels",
                    "TokenImpersonationLevel",
                    "WellKnownSidType",
                    "WindowsAccountType",
                    "WindowsBuiltInRole",
                    "WindowsIdentity",
                    "WindowsImpersonationContext",
                    "WindowsPrincipal",
                },

                new string[25] { // System.Text
                    "ASCIIEncoding",
                    "Decoder",
                    "DecoderExceptionFallback",
                    "DecoderExceptionFallbackBuffer",
                    "DecoderFallback",
                    "DecoderFallbackBuffer",
                    "DecoderFallbackException",
                    "DecoderReplacementFallback",
                    "DecoderReplacementFallbackBuffer",
                    "Encoder",
                    "EncoderExceptionFallback",
                    "EncoderExceptionFallbackBuffer",
                    "EncoderFallback",
                    "EncoderFallbackBuffer",
                    "EncoderFallbackException",
                    "EncoderReplacementFallback",
                    "EncoderReplacementFallbackBuffer",
                    "Encoding",
                    "EncodingInfo",
                    "NormalizationForm",
                    "StringBuilder",
                    "UTF32Encoding",
                    "UTF7Encoding",
                    "UTF8Encoding",
                    "UnicodeEncoding",
                },

                new string[41] { // System.Threading
                    "AbandonedMutexException",
                    "ApartmentState",
                    "AsyncFlowControl",
                    "AutoResetEvent",
                    "CompressedStack",
                    "ContextCallback",
                    "EventResetMode",
                    "EventWaitHandle",
                    "ExecutionContext",
                    "HostExecutionContext",
                    "HostExecutionContextManager",
                    "IOCompletionCallback",
                    "Interlocked",
                    "LockCookie",
                    "ManualResetEvent",
                    "Monitor",
                    "Mutex",
                    "NativeOverlapped",
                    "Overlapped",
                    "ParameterizedThreadStart",
                    "ReaderWriterLock",
                    "RegisteredWaitHandle",
                    "SendOrPostCallback",
                    "SynchronizationContext",
                    "SynchronizationLockException",
                    "Thread",
                    "ThreadAbortException",
                    "ThreadInterruptedException",
                    "ThreadPool",
                    "ThreadPriority",
                    "ThreadStart",
                    "ThreadStartException",
                    "ThreadState",
                    "ThreadStateException",
                    "Timeout",
                    "Timer",
                    "TimerCallback",
                    "WaitCallback",
                    "WaitHandle",
                    "WaitHandleCannotBeOpenedException",
                    "WaitOrTimerCallback",
                },

            };
            TypeName [] orcasTypes = {
                new TypeName("System", "DateTimeOffset"),
                new TypeName("System", "GCCollectionMode"),
                new TypeName("System.Runtime", "GCLatencyMode"),
            };

            return GetTypeNames(namespaces, types, orcasTypes);
        }

        static IEnumerable<TypeName> Get_System_TypeNames() {
            // System, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089
            // Total number of types = 889

            string [] namespaces = new string[36] {
                "Microsoft.CSharp",
                "Microsoft.VisualBasic",
                "Microsoft.Win32",
                "System",
                "System.CodeDom",
                "System.CodeDom.Compiler",
                "System.Collections.Generic",
                "System.Collections.Specialized",
                "System.ComponentModel",
                "System.ComponentModel.Design",
                "System.ComponentModel.Design.Serialization",
                "System.Configuration",
                "System.Diagnostics",
                "System.IO",
                "System.IO.Compression",
                "System.IO.Ports",
                "System.Media",
                "System.Net",
                "System.Net.Cache",
                "System.Net.Configuration",
                "System.Net.Mail",
                "System.Net.Mime",
                "System.Net.NetworkInformation",
                "System.Net.Security",
                "System.Net.Sockets",
                "System.Runtime.InteropServices",
                "System.Runtime.InteropServices.ComTypes",
                "System.Security.AccessControl",
                "System.Security.Authentication",
                "System.Security.Cryptography",
                "System.Security.Cryptography.X509Certificates",
                "System.Security.Permissions",
                "System.Text.RegularExpressions",
                "System.Threading",
                "System.Timers",
                "System.Web",
            };

            string [][] types = new string[36][] {

                new string[1] { // Microsoft.CSharp
                    "CSharpCodeProvider",
                },

                new string[1] { // Microsoft.VisualBasic
                    "VBCodeProvider",
                },

                new string[20] { // Microsoft.Win32
                    "IntranetZoneCredentialPolicy",
                    "PowerModeChangedEventArgs",
                    "PowerModeChangedEventHandler",
                    "PowerModes",
                    "SessionEndReasons",
                    "SessionEndedEventArgs",
                    "SessionEndedEventHandler",
                    "SessionEndingEventArgs",
                    "SessionEndingEventHandler",
                    "SessionSwitchEventArgs",
                    "SessionSwitchEventHandler",
                    "SessionSwitchReason",
                    "SystemEvents",
                    "TimerElapsedEventArgs",
                    "TimerElapsedEventHandler",
                    "UserPreferenceCategory",
                    "UserPreferenceChangedEventArgs",
                    "UserPreferenceChangedEventHandler",
                    "UserPreferenceChangingEventArgs",
                    "UserPreferenceChangingEventHandler",
                },

                new string[20] { // System
                    "FileStyleUriParser",
                    "FtpStyleUriParser",
                    "GenericUriParser",
                    "GenericUriParserOptions",
                    "GopherStyleUriParser",
                    "HttpStyleUriParser",
                    "LdapStyleUriParser",
                    "NetPipeStyleUriParser",
                    "NetTcpStyleUriParser",
                    "NewsStyleUriParser",
                    "Uri",
                    "UriBuilder",
                    "UriComponents",
                    "UriFormat",
                    "UriFormatException",
                    "UriHostNameType",
                    "UriKind",
                    "UriParser",
                    "UriPartial",
                    "UriTypeConverter",
                },

                new string[86] { // System.CodeDom
                    "CodeArgumentReferenceExpression",
                    "CodeArrayCreateExpression",
                    "CodeArrayIndexerExpression",
                    "CodeAssignStatement",
                    "CodeAttachEventStatement",
                    "CodeAttributeArgument",
                    "CodeAttributeArgumentCollection",
                    "CodeAttributeDeclaration",
                    "CodeAttributeDeclarationCollection",
                    "CodeBaseReferenceExpression",
                    "CodeBinaryOperatorExpression",
                    "CodeBinaryOperatorType",
                    "CodeCastExpression",
                    "CodeCatchClause",
                    "CodeCatchClauseCollection",
                    "CodeChecksumPragma",
                    "CodeComment",
                    "CodeCommentStatement",
                    "CodeCommentStatementCollection",
                    "CodeCompileUnit",
                    "CodeConditionStatement",
                    "CodeConstructor",
                    "CodeDefaultValueExpression",
                    "CodeDelegateCreateExpression",
                    "CodeDelegateInvokeExpression",
                    "CodeDirectionExpression",
                    "CodeDirective",
                    "CodeDirectiveCollection",
                    "CodeEntryPointMethod",
                    "CodeEventReferenceExpression",
                    "CodeExpression",
                    "CodeExpressionCollection",
                    "CodeExpressionStatement",
                    "CodeFieldReferenceExpression",
                    "CodeGotoStatement",
                    "CodeIndexerExpression",
                    "CodeIterationStatement",
                    "CodeLabeledStatement",
                    "CodeLinePragma",
                    "CodeMemberEvent",
                    "CodeMemberField",
                    "CodeMemberMethod",
                    "CodeMemberProperty",
                    "CodeMethodInvokeExpression",
                    "CodeMethodReferenceExpression",
                    "CodeMethodReturnStatement",
                    "CodeNamespace",
                    "CodeNamespaceCollection",
                    "CodeNamespaceImport",
                    "CodeNamespaceImportCollection",
                    "CodeObject",
                    "CodeObjectCreateExpression",
                    "CodeParameterDeclarationExpression",
                    "CodeParameterDeclarationExpressionCollection",
                    "CodePrimitiveExpression",
                    "CodePropertyReferenceExpression",
                    "CodePropertySetValueReferenceExpression",
                    "CodeRegionDirective",
                    "CodeRegionMode",
                    "CodeRemoveEventStatement",
                    "CodeSnippetCompileUnit",
                    "CodeSnippetExpression",
                    "CodeSnippetStatement",
                    "CodeSnippetTypeMember",
                    "CodeStatement",
                    "CodeStatementCollection",
                    "CodeThisReferenceExpression",
                    "CodeThrowExceptionStatement",
                    "CodeTryCatchFinallyStatement",
                    "CodeTypeConstructor",
                    "CodeTypeDeclaration",
                    "CodeTypeDeclarationCollection",
                    "CodeTypeDelegate",
                    "CodeTypeMember",
                    "CodeTypeMemberCollection",
                    "CodeTypeOfExpression",
                    "CodeTypeParameter",
                    "CodeTypeParameterCollection",
                    "CodeTypeReference",
                    "CodeTypeReferenceCollection",
                    "CodeTypeReferenceExpression",
                    "CodeTypeReferenceOptions",
                    "CodeVariableDeclarationStatement",
                    "CodeVariableReferenceExpression",
                    "FieldDirection",
                    "MemberAttributes",
                },

                new string[19] { // System.CodeDom.Compiler
                    "CodeCompiler",
                    "CodeDomProvider",
                    "CodeGenerator",
                    "CodeGeneratorOptions",
                    "CodeParser",
                    "CompilerError",
                    "CompilerErrorCollection",
                    "CompilerInfo",
                    "CompilerParameters",
                    "CompilerResults",
                    "Executor",
                    "GeneratedCodeAttribute",
                    "GeneratorSupport",
                    "ICodeCompiler",
                    "ICodeGenerator",
                    "ICodeParser",
                    "IndentedTextWriter",
                    "LanguageOptions",
                    "TempFileCollection",
                },

                new string[6] { // System.Collections.Generic
                    "LinkedListNode`1",
                    "LinkedList`1",
                    "Queue`1",
                    "SortedDictionary`2",
                    "SortedList`2",
                    "Stack`1",
                },

                new string[11] { // System.Collections.Specialized
                    "BitVector32",
                    "CollectionsUtil",
                    "HybridDictionary",
                    "IOrderedDictionary",
                    "ListDictionary",
                    "NameObjectCollectionBase",
                    "NameValueCollection",
                    "OrderedDictionary",
                    "StringCollection",
                    "StringDictionary",
                    "StringEnumerator",
                },

                new string[172] { // System.ComponentModel
                    "AddingNewEventArgs",
                    "AddingNewEventHandler",
                    "AmbientValueAttribute",
                    "ArrayConverter",
                    "AsyncCompletedEventArgs",
                    "AsyncCompletedEventHandler",
                    "AsyncOperation",
                    "AsyncOperationManager",
                    "AttributeCollection",
                    "AttributeProviderAttribute",
                    "BackgroundWorker",
                    "BaseNumberConverter",
                    "BindableAttribute",
                    "BindableSupport",
                    "BindingDirection",
                    "BindingList`1",
                    "BooleanConverter",
                    "BrowsableAttribute",
                    "ByteConverter",
                    "CancelEventArgs",
                    "CancelEventHandler",
                    "CategoryAttribute",
                    "CharConverter",
                    "CollectionChangeAction",
                    "CollectionChangeEventArgs",
                    "CollectionChangeEventHandler",
                    "CollectionConverter",
                    "ComplexBindingPropertiesAttribute",
                    "Component",
                    "ComponentCollection",
                    "ComponentConverter",
                    "ComponentEditor",
                    "ComponentResourceManager",
                    "Container",
                    "ContainerFilterService",
                    "CultureInfoConverter",
                    "CustomTypeDescriptor",
                    "DataObjectAttribute",
                    "DataObjectFieldAttribute",
                    "DataObjectMethodAttribute",
                    "DataObjectMethodType",
                    "DateTimeConverter",
                    "DecimalConverter",
                    "DefaultBindingPropertyAttribute",
                    "DefaultEventAttribute",
                    "DefaultPropertyAttribute",
                    "DefaultValueAttribute",
                    "DescriptionAttribute",
                    "DesignOnlyAttribute",
                    "DesignTimeVisibleAttribute",
                    "DesignerAttribute",
                    "DesignerCategoryAttribute",
                    "DesignerSerializationVisibility",
                    "DesignerSerializationVisibilityAttribute",
                    "DisplayNameAttribute",
                    "DoWorkEventArgs",
                    "DoWorkEventHandler",
                    "DoubleConverter",
                    "EditorAttribute",
                    "EditorBrowsableAttribute",
                    "EditorBrowsableState",
                    "EnumConverter",
                    "EventDescriptor",
                    "EventDescriptorCollection",
                    "EventHandlerList",
                    "ExpandableObjectConverter",
                    "ExtenderProvidedPropertyAttribute",
                    "GuidConverter",
                    "HandledEventArgs",
                    "HandledEventHandler",
                    "IBindingList",
                    "IBindingListView",
                    "ICancelAddNew",
                    "IChangeTracking",
                    "IComNativeDescriptorHandler",
                    "IComponent",
                    "IContainer",
                    "ICustomTypeDescriptor",
                    "IDataErrorInfo",
                    "IEditableObject",
                    "IExtenderProvider",
                    "IIntellisenseBuilder",
                    "IListSource",
                    "INestedContainer",
                    "INestedSite",
                    "INotifyPropertyChanged",
                    "IRaiseItemChangedEvents",
                    "IRevertibleChangeTracking",
                    "ISite",
                    "ISupportInitialize",
                    "ISupportInitializeNotification",
                    "ISynchronizeInvoke",
                    "ITypeDescriptorContext",
                    "ITypedList",
                    "ImmutableObjectAttribute",
                    "InheritanceAttribute",
                    "InheritanceLevel",
                    "InitializationEventAttribute",
                    "InstallerTypeAttribute",
                    "InstanceCreationEditor",
                    "Int16Converter",
                    "Int32Converter",
                    "Int64Converter",
                    "InvalidAsynchronousStateException",
                    "InvalidEnumArgumentException",
                    "LicFileLicenseProvider",
                    "License",
                    "LicenseContext",
                    "LicenseException",
                    "LicenseManager",
                    "LicenseProvider",
                    "LicenseProviderAttribute",
                    "LicenseUsageMode",
                    "ListBindableAttribute",
                    "ListChangedEventArgs",
                    "ListChangedEventHandler",
                    "ListChangedType",
                    "ListSortDescription",
                    "ListSortDescriptionCollection",
                    "ListSortDirection",
                    "LocalizableAttribute",
                    "LookupBindingPropertiesAttribute",
                    "MarshalByValueComponent",
                    "MaskedTextProvider",
                    "MaskedTextResultHint",
                    "MemberDescriptor",
                    "MergablePropertyAttribute",
                    "MultilineStringConverter",
                    "NestedContainer",
                    "NotifyParentPropertyAttribute",
                    "NullableConverter",
                    "ParenthesizePropertyNameAttribute",
                    "PasswordPropertyTextAttribute",
                    "ProgressChangedEventArgs",
                    "ProgressChangedEventHandler",
                    "PropertyChangedEventArgs",
                    "PropertyChangedEventHandler",
                    "PropertyDescriptor",
                    "PropertyDescriptorCollection",
                    "PropertyTabAttribute",
                    "PropertyTabScope",
                    "ProvidePropertyAttribute",
                    "ReadOnlyAttribute",
                    "RecommendedAsConfigurableAttribute",
                    "ReferenceConverter",
                    "RefreshEventArgs",
                    "RefreshEventHandler",
                    "RefreshProperties",
                    "RefreshPropertiesAttribute",
                    "RunInstallerAttribute",
                    "RunWorkerCompletedEventArgs",
                    "RunWorkerCompletedEventHandler",
                    "SByteConverter",
                    "SettingsBindableAttribute",
                    "SingleConverter",
                    "StringConverter",
                    "SyntaxCheck",
                    "TimeSpanConverter",
                    "ToolboxItemAttribute",
                    "ToolboxItemFilterAttribute",
                    "ToolboxItemFilterType",
                    "TypeConverter",
                    "TypeConverterAttribute",
                    "TypeDescriptionProvider",
                    "TypeDescriptionProviderAttribute",
                    "TypeDescriptor",
                    "TypeListConverter",
                    "UInt16Converter",
                    "UInt32Converter",
                    "UInt64Converter",
                    "WarningException",
                    "Win32Exception",
                },

                new string[57] { // System.ComponentModel.Design
                    "ActiveDesignerEventArgs",
                    "ActiveDesignerEventHandler",
                    "CheckoutException",
                    "CommandID",
                    "ComponentChangedEventArgs",
                    "ComponentChangedEventHandler",
                    "ComponentChangingEventArgs",
                    "ComponentChangingEventHandler",
                    "ComponentEventArgs",
                    "ComponentEventHandler",
                    "ComponentRenameEventArgs",
                    "ComponentRenameEventHandler",
                    "DesignerCollection",
                    "DesignerEventArgs",
                    "DesignerEventHandler",
                    "DesignerOptionService",
                    "DesignerTransaction",
                    "DesignerTransactionCloseEventArgs",
                    "DesignerTransactionCloseEventHandler",
                    "DesignerVerb",
                    "DesignerVerbCollection",
                    "DesigntimeLicenseContext",
                    "DesigntimeLicenseContextSerializer",
                    "HelpContextType",
                    "HelpKeywordAttribute",
                    "HelpKeywordType",
                    "IComponentChangeService",
                    "IComponentDiscoveryService",
                    "IComponentInitializer",
                    "IDesigner",
                    "IDesignerEventService",
                    "IDesignerFilter",
                    "IDesignerHost",
                    "IDesignerOptionService",
                    "IDictionaryService",
                    "IEventBindingService",
                    "IExtenderListService",
                    "IExtenderProviderService",
                    "IHelpService",
                    "IInheritanceService",
                    "IMenuCommandService",
                    "IReferenceService",
                    "IResourceService",
                    "IRootDesigner",
                    "ISelectionService",
                    "IServiceContainer",
                    "ITreeDesigner",
                    "ITypeDescriptorFilterService",
                    "ITypeDiscoveryService",
                    "ITypeResolutionService",
                    "MenuCommand",
                    "SelectionTypes",
                    "ServiceContainer",
                    "ServiceCreatorCallback",
                    "StandardCommands",
                    "StandardToolWindows",
                    "ViewTechnology",
                },

                new string[18] { // System.ComponentModel.Design.Serialization
                    "ComponentSerializationService",
                    "ContextStack",
                    "DefaultSerializationProviderAttribute",
                    "DesignerLoader",
                    "DesignerSerializerAttribute",
                    "IDesignerLoaderHost",
                    "IDesignerLoaderService",
                    "IDesignerSerializationManager",
                    "IDesignerSerializationProvider",
                    "IDesignerSerializationService",
                    "INameCreationService",
                    "InstanceDescriptor",
                    "MemberRelationship",
                    "MemberRelationshipService",
                    "ResolveNameEventArgs",
                    "ResolveNameEventHandler",
                    "RootDesignerSerializerAttribute",
                    "SerializationStore",
                },

                new string[54] { // System.Configuration
                    "AppSettingsReader",
                    "ApplicationScopedSettingAttribute",
                    "ApplicationSettingsBase",
                    "ApplicationSettingsGroup",
                    "ClientSettingsSection",
                    "ConfigXmlDocument",
                    "ConfigurationException",
                    "ConfigurationSettings",
                    "DefaultSettingValueAttribute",
                    "DictionarySectionHandler",
                    "IApplicationSettingsProvider",
                    "IConfigurationSectionHandler",
                    "IConfigurationSystem",
                    "IPersistComponentSettings",
                    "ISettingsProviderService",
                    "IgnoreSectionHandler",
                    "LocalFileSettingsProvider",
                    "NameValueFileSectionHandler",
                    "NameValueSectionHandler",
                    "NoSettingsVersionUpgradeAttribute",
                    "SettingAttribute",
                    "SettingChangingEventArgs",
                    "SettingChangingEventHandler",
                    "SettingElement",
                    "SettingElementCollection",
                    "SettingValueElement",
                    "SettingsAttributeDictionary",
                    "SettingsBase",
                    "SettingsContext",
                    "SettingsDescriptionAttribute",
                    "SettingsGroupDescriptionAttribute",
                    "SettingsGroupNameAttribute",
                    "SettingsLoadedEventArgs",
                    "SettingsLoadedEventHandler",
                    "SettingsManageability",
                    "SettingsManageabilityAttribute",
                    "SettingsProperty",
                    "SettingsPropertyCollection",
                    "SettingsPropertyIsReadOnlyException",
                    "SettingsPropertyNotFoundException",
                    "SettingsPropertyValue",
                    "SettingsPropertyValueCollection",
                    "SettingsPropertyWrongTypeException",
                    "SettingsProvider",
                    "SettingsProviderAttribute",
                    "SettingsProviderCollection",
                    "SettingsSavingEventHandler",
                    "SettingsSerializeAs",
                    "SettingsSerializeAsAttribute",
                    "SingleTagSectionHandler",
                    "SpecialSetting",
                    "SpecialSettingAttribute",
                    "UserScopedSettingAttribute",
                    "UserSettingsGroup",
                },

                new string[76] { // System.Diagnostics
                    "BooleanSwitch",
                    "ConsoleTraceListener",
                    "CorrelationManager",
                    "CounterCreationData",
                    "CounterCreationDataCollection",
                    "CounterSample",
                    "CounterSampleCalculator",
                    "DataReceivedEventArgs",
                    "DataReceivedEventHandler",
                    "Debug",
                    "DefaultTraceListener",
                    "DelimitedListTraceListener",
                    "DiagnosticsConfigurationHandler",
                    "EntryWrittenEventArgs",
                    "EntryWrittenEventHandler",
                    "EventInstance",
                    "EventLog",
                    "EventLogEntry",
                    "EventLogEntryCollection",
                    "EventLogEntryType",
                    "EventLogPermission",
                    "EventLogPermissionAccess",
                    "EventLogPermissionAttribute",
                    "EventLogPermissionEntry",
                    "EventLogPermissionEntryCollection",
                    "EventLogTraceListener",
                    "EventSourceCreationData",
                    "EventTypeFilter",
                    "FileVersionInfo",
                    "ICollectData",
                    "InstanceData",
                    "InstanceDataCollection",
                    "InstanceDataCollectionCollection",
                    "MonitoringDescriptionAttribute",
                    "OverflowAction",
                    "PerformanceCounter",
                    "PerformanceCounterCategory",
                    "PerformanceCounterCategoryType",
                    "PerformanceCounterInstanceLifetime",
                    "PerformanceCounterManager",
                    "PerformanceCounterPermission",
                    "PerformanceCounterPermissionAccess",
                    "PerformanceCounterPermissionAttribute",
                    "PerformanceCounterPermissionEntry",
                    "PerformanceCounterPermissionEntryCollection",
                    "PerformanceCounterType",
                    "Process",
                    "ProcessModule",
                    "ProcessModuleCollection",
                    "ProcessPriorityClass",
                    "ProcessStartInfo",
                    "ProcessThread",
                    "ProcessThreadCollection",
                    "ProcessWindowStyle",
                    "SourceFilter",
                    "SourceLevels",
                    "SourceSwitch",
                    "Stopwatch",
                    "Switch",
                    "SwitchAttribute",
                    "SwitchLevelAttribute",
                    "TextWriterTraceListener",
                    "ThreadPriorityLevel",
                    "ThreadState",
                    "ThreadWaitReason",
                    "Trace",
                    "TraceEventCache",
                    "TraceEventType",
                    "TraceFilter",
                    "TraceLevel",
                    "TraceListener",
                    "TraceListenerCollection",
                    "TraceOptions",
                    "TraceSource",
                    "TraceSwitch",
                    "XmlWriterTraceListener",
                },

                new string[13] { // System.IO
                    "ErrorEventArgs",
                    "ErrorEventHandler",
                    "FileSystemEventArgs",
                    "FileSystemEventHandler",
                    "FileSystemWatcher",
                    "IODescriptionAttribute",
                    "InternalBufferOverflowException",
                    "InvalidDataException",
                    "NotifyFilters",
                    "RenamedEventArgs",
                    "RenamedEventHandler",
                    "WaitForChangedResult",
                    "WatcherChangeTypes",
                },

                new string[3] { // System.IO.Compression
                    "CompressionMode",
                    "DeflateStream",
                    "GZipStream",
                },

                new string[13] { // System.IO.Ports
                    "Handshake",
                    "Parity",
                    "SerialData",
                    "SerialDataReceivedEventArgs",
                    "SerialDataReceivedEventHandler",
                    "SerialError",
                    "SerialErrorReceivedEventArgs",
                    "SerialErrorReceivedEventHandler",
                    "SerialPinChange",
                    "SerialPinChangedEventArgs",
                    "SerialPinChangedEventHandler",
                    "SerialPort",
                    "StopBits",
                },

                new string[3] { // System.Media
                    "SoundPlayer",
                    "SystemSound",
                    "SystemSounds",
                },

                new string[87] { // System.Net
                    "AuthenticationManager",
                    "AuthenticationSchemeSelector",
                    "AuthenticationSchemes",
                    "Authorization",
                    "BindIPEndPoint",
                    "Cookie",
                    "CookieCollection",
                    "CookieContainer",
                    "CookieException",
                    "CredentialCache",
                    "DecompressionMethods",
                    "Dns",
                    "DnsPermission",
                    "DnsPermissionAttribute",
                    "DownloadDataCompletedEventArgs",
                    "DownloadDataCompletedEventHandler",
                    "DownloadProgressChangedEventArgs",
                    "DownloadProgressChangedEventHandler",
                    "DownloadStringCompletedEventArgs",
                    "DownloadStringCompletedEventHandler",
                    "EndPoint",
                    "EndpointPermission",
                    "FileWebRequest",
                    "FileWebResponse",
                    "FtpStatusCode",
                    "FtpWebRequest",
                    "FtpWebResponse",
                    "GlobalProxySelection",
                    "HttpContinueDelegate",
                    "HttpListener",
                    "HttpListenerBasicIdentity",
                    "HttpListenerContext",
                    "HttpListenerException",
                    "HttpListenerPrefixCollection",
                    "HttpListenerRequest",
                    "HttpListenerResponse",
                    "HttpRequestHeader",
                    "HttpResponseHeader",
                    "HttpStatusCode",
                    "HttpVersion",
                    "HttpWebRequest",
                    "HttpWebResponse",
                    "IAuthenticationModule",
                    "ICertificatePolicy",
                    "ICredentialPolicy",
                    "ICredentials",
                    "ICredentialsByHost",
                    "IPAddress",
                    "IPEndPoint",
                    "IPHostEntry",
                    "IWebProxy",
                    "IWebProxyScript",
                    "IWebRequestCreate",
                    "NetworkAccess",
                    "NetworkCredential",
                    "OpenReadCompletedEventArgs",
                    "OpenReadCompletedEventHandler",
                    "OpenWriteCompletedEventArgs",
                    "OpenWriteCompletedEventHandler",
                    "ProtocolViolationException",
                    "SecurityProtocolType",
                    "ServicePoint",
                    "ServicePointManager",
                    "SocketAddress",
                    "SocketPermission",
                    "SocketPermissionAttribute",
                    "TransportType",
                    "UploadDataCompletedEventArgs",
                    "UploadDataCompletedEventHandler",
                    "UploadFileCompletedEventArgs",
                    "UploadFileCompletedEventHandler",
                    "UploadProgressChangedEventArgs",
                    "UploadProgressChangedEventHandler",
                    "UploadStringCompletedEventArgs",
                    "UploadStringCompletedEventHandler",
                    "UploadValuesCompletedEventArgs",
                    "UploadValuesCompletedEventHandler",
                    "WebClient",
                    "WebException",
                    "WebExceptionStatus",
                    "WebHeaderCollection",
                    "WebPermission",
                    "WebPermissionAttribute",
                    "WebProxy",
                    "WebRequest",
                    "WebRequestMethods",
                    "WebResponse",
                },

                new string[5] { // System.Net.Cache
                    "HttpCacheAgeControl",
                    "HttpRequestCacheLevel",
                    "HttpRequestCachePolicy",
                    "RequestCacheLevel",
                    "RequestCachePolicy",
                },

                new string[29] { // System.Net.Configuration
                    "AuthenticationModuleElement",
                    "AuthenticationModuleElementCollection",
                    "AuthenticationModulesSection",
                    "BypassElement",
                    "BypassElementCollection",
                    "ConnectionManagementElement",
                    "ConnectionManagementElementCollection",
                    "ConnectionManagementSection",
                    "DefaultProxySection",
                    "FtpCachePolicyElement",
                    "HttpCachePolicyElement",
                    "HttpWebRequestElement",
                    "Ipv6Element",
                    "MailSettingsSectionGroup",
                    "ModuleElement",
                    "NetSectionGroup",
                    "PerformanceCountersElement",
                    "ProxyElement",
                    "RequestCachingSection",
                    "ServicePointManagerElement",
                    "SettingsSection",
                    "SmtpNetworkElement",
                    "SmtpSection",
                    "SmtpSpecifiedPickupDirectoryElement",
                    "SocketElement",
                    "WebProxyScriptElement",
                    "WebRequestModuleElement",
                    "WebRequestModuleElementCollection",
                    "WebRequestModulesSection",
                },

                new string[22] { // System.Net.Mail
                    "AlternateView",
                    "AlternateViewCollection",
                    "Attachment",
                    "AttachmentBase",
                    "AttachmentCollection",
                    "DeliveryNotificationOptions",
                    "LinkedResource",
                    "LinkedResourceCollection",
                    "MailAddress",
                    "MailAddressCollection",
                    "MailMessage",
                    "MailPriority",
                    "SendCompletedEventHandler",
                    "SmtpAccess",
                    "SmtpClient",
                    "SmtpDeliveryMethod",
                    "SmtpException",
                    "SmtpFailedRecipientException",
                    "SmtpFailedRecipientsException",
                    "SmtpPermission",
                    "SmtpPermissionAttribute",
                    "SmtpStatusCode",
                },

                new string[5] { // System.Net.Mime
                    "ContentDisposition",
                    "ContentType",
                    "DispositionTypeNames",
                    "MediaTypeNames",
                    "TransferEncoding",
                },

                new string[45] { // System.Net.NetworkInformation
                    "DuplicateAddressDetectionState",
                    "GatewayIPAddressInformation",
                    "GatewayIPAddressInformationCollection",
                    "IPAddressCollection",
                    "IPAddressInformation",
                    "IPAddressInformationCollection",
                    "IPGlobalProperties",
                    "IPGlobalStatistics",
                    "IPInterfaceProperties",
                    "IPStatus",
                    "IPv4InterfaceProperties",
                    "IPv4InterfaceStatistics",
                    "IPv6InterfaceProperties",
                    "IcmpV4Statistics",
                    "IcmpV6Statistics",
                    "MulticastIPAddressInformation",
                    "MulticastIPAddressInformationCollection",
                    "NetBiosNodeType",
                    "NetworkAddressChangedEventHandler",
                    "NetworkAvailabilityChangedEventHandler",
                    "NetworkAvailabilityEventArgs",
                    "NetworkChange",
                    "NetworkInformationAccess",
                    "NetworkInformationException",
                    "NetworkInformationPermission",
                    "NetworkInformationPermissionAttribute",
                    "NetworkInterface",
                    "NetworkInterfaceComponent",
                    "NetworkInterfaceType",
                    "OperationalStatus",
                    "PhysicalAddress",
                    "Ping",
                    "PingCompletedEventArgs",
                    "PingCompletedEventHandler",
                    "PingException",
                    "PingOptions",
                    "PingReply",
                    "PrefixOrigin",
                    "SuffixOrigin",
                    "TcpConnectionInformation",
                    "TcpState",
                    "TcpStatistics",
                    "UdpStatistics",
                    "UnicastIPAddressInformation",
                    "UnicastIPAddressInformationCollection",
                },

                new string[8] { // System.Net.Security
                    "AuthenticatedStream",
                    "AuthenticationLevel",
                    "LocalCertificateSelectionCallback",
                    "NegotiateStream",
                    "ProtectionLevel",
                    "RemoteCertificateValidationCallback",
                    "SslPolicyErrors",
                    "SslStream",
                },

                new string[24] { // System.Net.Sockets
                    "AddressFamily",
                    "IOControlCode",
                    "IPPacketInformation",
                    "IPv6MulticastOption",
                    "LingerOption",
                    "MulticastOption",
                    "NetworkStream",
                    "ProtocolFamily",
                    "ProtocolType",
                    "SelectMode",
                    "Socket",
                    "SocketError",
                    "SocketException",
                    "SocketFlags",
                    "SocketInformation",
                    "SocketInformationOptions",
                    "SocketOptionLevel",
                    "SocketOptionName",
                    "SocketShutdown",
                    "SocketType",
                    "TcpClient",
                    "TcpListener",
                    "TransmitFileOptions",
                    "UdpClient",
                },

                new string[3] { // System.Runtime.InteropServices
                    "DefaultParameterValueAttribute",
                    "HandleCollector",
                    "StandardOleMarshalObject",
                },

                new string[11] { // System.Runtime.InteropServices.ComTypes
                    "ADVF",
                    "DATADIR",
                    "DVASPECT",
                    "FORMATETC",
                    "IAdviseSink",
                    "IDataObject",
                    "IEnumFORMATETC",
                    "IEnumSTATDATA",
                    "STATDATA",
                    "STGMEDIUM",
                    "TYMED",
                },

                new string[4] { // System.Security.AccessControl
                    "SemaphoreAccessRule",
                    "SemaphoreAuditRule",
                    "SemaphoreRights",
                    "SemaphoreSecurity",
                },

                new string[6] { // System.Security.Authentication
                    "AuthenticationException",
                    "CipherAlgorithmType",
                    "ExchangeAlgorithmType",
                    "HashAlgorithmType",
                    "InvalidCredentialException",
                    "SslProtocols",
                },

                new string[6] { // System.Security.Cryptography
                    "AsnEncodedData",
                    "AsnEncodedDataCollection",
                    "AsnEncodedDataEnumerator",
                    "Oid",
                    "OidCollection",
                    "OidEnumerator",
                },

                new string[33] { // System.Security.Cryptography.X509Certificates
                    "OpenFlags",
                    "PublicKey",
                    "StoreLocation",
                    "StoreName",
                    "X500DistinguishedName",
                    "X500DistinguishedNameFlags",
                    "X509BasicConstraintsExtension",
                    "X509Certificate2",
                    "X509Certificate2Collection",
                    "X509Certificate2Enumerator",
                    "X509CertificateCollection",
                    "X509Chain",
                    "X509ChainElement",
                    "X509ChainElementCollection",
                    "X509ChainElementEnumerator",
                    "X509ChainPolicy",
                    "X509ChainStatus",
                    "X509ChainStatusFlags",
                    "X509EnhancedKeyUsageExtension",
                    "X509Extension",
                    "X509ExtensionCollection",
                    "X509ExtensionEnumerator",
                    "X509FindType",
                    "X509IncludeOption",
                    "X509KeyUsageExtension",
                    "X509KeyUsageFlags",
                    "X509NameType",
                    "X509RevocationFlag",
                    "X509RevocationMode",
                    "X509Store",
                    "X509SubjectKeyIdentifierExtension",
                    "X509SubjectKeyIdentifierHashAlgorithm",
                    "X509VerificationFlags",
                },

                new string[5] { // System.Security.Permissions
                    "ResourcePermissionBase",
                    "ResourcePermissionBaseEntry",
                    "StorePermission",
                    "StorePermissionAttribute",
                    "StorePermissionFlags",
                },

                new string[12] { // System.Text.RegularExpressions
                    "Capture",
                    "CaptureCollection",
                    "Group",
                    "GroupCollection",
                    "Match",
                    "MatchCollection",
                    "MatchEvaluator",
                    "Regex",
                    "RegexCompilationInfo",
                    "RegexOptions",
                    "RegexRunner",
                    "RegexRunnerFactory",
                },

                new string[4] { // System.Threading
                    "Semaphore",
                    "SemaphoreFullException",
                    "ThreadExceptionEventArgs",
                    "ThreadExceptionEventHandler",
                },

                new string[4] { // System.Timers
                    "ElapsedEventArgs",
                    "ElapsedEventHandler",
                    "Timer",
                    "TimersDescriptionAttribute",
                },

                new string[3] { // System.Web
                    "AspNetHostingPermission",
                    "AspNetHostingPermissionAttribute",
                    "AspNetHostingPermissionLevel",
                },

            };
            TypeName [] orcasTypes = {
                new TypeName("System.ComponentModel.Design.Serialization", "IDesignerLoaderHost2"),
                new TypeName("System.ComponentModel", "INotifyPropertyChanging"),
                new TypeName("System.ComponentModel", "PropertyChangingEventArgs"),
                new TypeName("System.ComponentModel", "PropertyChangingEventHandler"),
                new TypeName("System.Configuration", "IdnElement"),
                new TypeName("System.Configuration", "IriParsingElement"),
                new TypeName("System.Configuration", "UriSection"),
                new TypeName("System.Net.Sockets", "SendPacketsElement"),
                new TypeName("System.Net.Sockets", "SocketAsyncEventArgs"),
                new TypeName("System.Net.Sockets", "SocketAsyncOperation"),
                new TypeName("System", "UriIdnScope"),
            };

            return GetTypeNames(namespaces, types, orcasTypes);
        }

        static IEnumerable<TypeName> Get_SystemDrawing_TypeNames() {
            // System.Drawing, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a
            // Total number of types = 177

            string [] namespaces = new string[6] {
                "System.Drawing",
                "System.Drawing.Design",
                "System.Drawing.Drawing2D",
                "System.Drawing.Imaging",
                "System.Drawing.Printing",
                "System.Drawing.Text",
            };

            string [][] types = new string[6][] {

                new string[54] { // System.Drawing
                    "Bitmap",
                    "Brush",
                    "Brushes",
                    "BufferedGraphics",
                    "BufferedGraphicsContext",
                    "BufferedGraphicsManager",
                    "CharacterRange",
                    "Color",
                    "ColorConverter",
                    "ColorTranslator",
                    "ContentAlignment",
                    "CopyPixelOperation",
                    "Font",
                    "FontConverter",
                    "FontFamily",
                    "FontStyle",
                    "Graphics",
                    "GraphicsUnit",
                    "IDeviceContext",
                    "Icon",
                    "IconConverter",
                    "Image",
                    "ImageAnimator",
                    "ImageConverter",
                    "ImageFormatConverter",
                    "KnownColor",
                    "Pen",
                    "Pens",
                    "Point",
                    "PointConverter",
                    "PointF",
                    "Rectangle",
                    "RectangleConverter",
                    "RectangleF",
                    "Region",
                    "RotateFlipType",
                    "Size",
                    "SizeConverter",
                    "SizeF",
                    "SizeFConverter",
                    "SolidBrush",
                    "StringAlignment",
                    "StringDigitSubstitute",
                    "StringFormat",
                    "StringFormatFlags",
                    "StringTrimming",
                    "StringUnit",
                    "SystemBrushes",
                    "SystemColors",
                    "SystemFonts",
                    "SystemIcons",
                    "SystemPens",
                    "TextureBrush",
                    "ToolboxBitmapAttribute",
                },

                new string[18] { // System.Drawing.Design
                    "CategoryNameCollection",
                    "IPropertyValueUIService",
                    "IToolboxItemProvider",
                    "IToolboxService",
                    "IToolboxUser",
                    "PaintValueEventArgs",
                    "PropertyValueUIHandler",
                    "PropertyValueUIItem",
                    "PropertyValueUIItemInvokeHandler",
                    "ToolboxComponentsCreatedEventArgs",
                    "ToolboxComponentsCreatedEventHandler",
                    "ToolboxComponentsCreatingEventArgs",
                    "ToolboxComponentsCreatingEventHandler",
                    "ToolboxItem",
                    "ToolboxItemCollection",
                    "ToolboxItemCreatorCallback",
                    "UITypeEditor",
                    "UITypeEditorEditStyle",
                },

                new string[36] { // System.Drawing.Drawing2D
                    "AdjustableArrowCap",
                    "Blend",
                    "ColorBlend",
                    "CombineMode",
                    "CompositingMode",
                    "CompositingQuality",
                    "CoordinateSpace",
                    "CustomLineCap",
                    "DashCap",
                    "DashStyle",
                    "FillMode",
                    "FlushIntention",
                    "GraphicsContainer",
                    "GraphicsPath",
                    "GraphicsPathIterator",
                    "GraphicsState",
                    "HatchBrush",
                    "HatchStyle",
                    "InterpolationMode",
                    "LineCap",
                    "LineJoin",
                    "LinearGradientBrush",
                    "LinearGradientMode",
                    "Matrix",
                    "MatrixOrder",
                    "PathData",
                    "PathGradientBrush",
                    "PathPointType",
                    "PenAlignment",
                    "PenType",
                    "PixelOffsetMode",
                    "QualityMode",
                    "RegionData",
                    "SmoothingMode",
                    "WarpMode",
                    "WrapMode",
                },

                new string[33] { // System.Drawing.Imaging
                    "BitmapData",
                    "ColorAdjustType",
                    "ColorChannelFlag",
                    "ColorMap",
                    "ColorMapType",
                    "ColorMatrix",
                    "ColorMatrixFlag",
                    "ColorMode",
                    "ColorPalette",
                    "EmfPlusRecordType",
                    "EmfType",
                    "Encoder",
                    "EncoderParameter",
                    "EncoderParameterValueType",
                    "EncoderParameters",
                    "EncoderValue",
                    "FrameDimension",
                    "ImageAttributes",
                    "ImageCodecFlags",
                    "ImageCodecInfo",
                    "ImageFlags",
                    "ImageFormat",
                    "ImageLockMode",
                    "MetaHeader",
                    "Metafile",
                    "MetafileFrameUnit",
                    "MetafileHeader",
                    "MetafileType",
                    "PaletteFlags",
                    "PixelFormat",
                    "PlayRecordCallback",
                    "PropertyItem",
                    "WmfPlaceableFileHeader",
                },

                new string[30] { // System.Drawing.Printing
                    "Duplex",
                    "InvalidPrinterException",
                    "Margins",
                    "MarginsConverter",
                    "PageSettings",
                    "PaperKind",
                    "PaperSize",
                    "PaperSource",
                    "PaperSourceKind",
                    "PreviewPageInfo",
                    "PreviewPrintController",
                    "PrintAction",
                    "PrintController",
                    "PrintDocument",
                    "PrintEventArgs",
                    "PrintEventHandler",
                    "PrintPageEventArgs",
                    "PrintPageEventHandler",
                    "PrintRange",
                    "PrinterResolution",
                    "PrinterResolutionKind",
                    "PrinterSettings",
                    "PrinterUnit",
                    "PrinterUnitConvert",
                    "PrintingPermission",
                    "PrintingPermissionAttribute",
                    "PrintingPermissionLevel",
                    "QueryPageSettingsEventArgs",
                    "QueryPageSettingsEventHandler",
                    "StandardPrintController",
                },

                new string[6] { // System.Drawing.Text
                    "FontCollection",
                    "GenericFontFamilies",
                    "HotkeyPrefix",
                    "InstalledFontCollection",
                    "PrivateFontCollection",
                    "TextRenderingHint",
                },

            };
            TypeName [] orcasTypes = {
            };

            return GetTypeNames(namespaces, types, orcasTypes);
        }

        static IEnumerable<TypeName> Get_SystemWindowsForms_TypeNames() {
            // System.Windows.Forms, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089
            // Total number of types = 777

            string [] namespaces = new string[7] {
                "System.Resources",
                "System.Windows.Forms",
                "System.Windows.Forms.ComponentModel.Com2Interop",
                "System.Windows.Forms.Design",
                "System.Windows.Forms.Layout",
                "System.Windows.Forms.PropertyGridInternal",
                "System.Windows.Forms.VisualStyles",
            };

            string [][] types = new string[7][] {

                new string[5] { // System.Resources
                    "ResXDataNode",
                    "ResXFileRef",
                    "ResXResourceReader",
                    "ResXResourceSet",
                    "ResXResourceWriter",
                },

                new string[705] { // System.Windows.Forms
                    "AccessibleEvents",
                    "AccessibleNavigation",
                    "AccessibleObject",
                    "AccessibleRole",
                    "AccessibleSelection",
                    "AccessibleStates",
                    "AmbientProperties",
                    "AnchorStyles",
                    "Appearance",
                    "Application",
                    "ApplicationContext",
                    "ArrangeDirection",
                    "ArrangeStartingPosition",
                    "ArrowDirection",
                    "AutoCompleteMode",
                    "AutoCompleteSource",
                    "AutoCompleteStringCollection",
                    "AutoScaleMode",
                    "AutoSizeMode",
                    "AutoValidate",
                    "AxHost",
                    "BaseCollection",
                    "BatteryChargeStatus",
                    "Binding",
                    "BindingCompleteContext",
                    "BindingCompleteEventArgs",
                    "BindingCompleteEventHandler",
                    "BindingCompleteState",
                    "BindingContext",
                    "BindingManagerBase",
                    "BindingManagerDataErrorEventArgs",
                    "BindingManagerDataErrorEventHandler",
                    "BindingMemberInfo",
                    "BindingNavigator",
                    "BindingSource",
                    "BindingsCollection",
                    "BootMode",
                    "Border3DSide",
                    "Border3DStyle",
                    "BorderStyle",
                    "BoundsSpecified",
                    "Button",
                    "ButtonBase",
                    "ButtonBorderStyle",
                    "ButtonRenderer",
                    "ButtonState",
                    "CacheVirtualItemsEventArgs",
                    "CacheVirtualItemsEventHandler",
                    "CaptionButton",
                    "CharacterCasing",
                    "CheckBox",
                    "CheckBoxRenderer",
                    "CheckState",
                    "CheckedListBox",
                    "Clipboard",
                    "CloseReason",
                    "ColorDepth",
                    "ColorDialog",
                    "ColumnClickEventArgs",
                    "ColumnClickEventHandler",
                    "ColumnHeader",
                    "ColumnHeaderAutoResizeStyle",
                    "ColumnHeaderConverter",
                    "ColumnHeaderStyle",
                    "ColumnReorderedEventArgs",
                    "ColumnReorderedEventHandler",
                    "ColumnStyle",
                    "ColumnWidthChangedEventArgs",
                    "ColumnWidthChangedEventHandler",
                    "ColumnWidthChangingEventArgs",
                    "ColumnWidthChangingEventHandler",
                    "ComboBox",
                    "ComboBoxRenderer",
                    "ComboBoxStyle",
                    "CommonDialog",
                    "ContainerControl",
                    "ContentsResizedEventArgs",
                    "ContentsResizedEventHandler",
                    "ContextMenu",
                    "ContextMenuStrip",
                    "Control",
                    "ControlBindingsCollection",
                    "ControlEventArgs",
                    "ControlEventHandler",
                    "ControlPaint",
                    "ControlStyles",
                    "ControlUpdateMode",
                    "ConvertEventArgs",
                    "ConvertEventHandler",
                    "CreateParams",
                    "CurrencyManager",
                    "Cursor",
                    "CursorConverter",
                    "Cursors",
                    "DataFormats",
                    "DataGrid",
                    "DataGridBoolColumn",
                    "DataGridCell",
                    "DataGridColumnStyle",
                    "DataGridLineStyle",
                    "DataGridParentRowsLabelStyle",
                    "DataGridPreferredColumnWidthTypeConverter",
                    "DataGridTableStyle",
                    "DataGridTextBox",
                    "DataGridTextBoxColumn",
                    "DataGridView",
                    "DataGridViewAdvancedBorderStyle",
                    "DataGridViewAdvancedCellBorderStyle",
                    "DataGridViewAutoSizeColumnMode",
                    "DataGridViewAutoSizeColumnModeEventArgs",
                    "DataGridViewAutoSizeColumnModeEventHandler",
                    "DataGridViewAutoSizeColumnsMode",
                    "DataGridViewAutoSizeColumnsModeEventArgs",
                    "DataGridViewAutoSizeColumnsModeEventHandler",
                    "DataGridViewAutoSizeModeEventArgs",
                    "DataGridViewAutoSizeModeEventHandler",
                    "DataGridViewAutoSizeRowMode",
                    "DataGridViewAutoSizeRowsMode",
                    "DataGridViewBand",
                    "DataGridViewBindingCompleteEventArgs",
                    "DataGridViewBindingCompleteEventHandler",
                    "DataGridViewButtonCell",
                    "DataGridViewButtonColumn",
                    "DataGridViewCell",
                    "DataGridViewCellBorderStyle",
                    "DataGridViewCellCancelEventArgs",
                    "DataGridViewCellCancelEventHandler",
                    "DataGridViewCellCollection",
                    "DataGridViewCellContextMenuStripNeededEventArgs",
                    "DataGridViewCellContextMenuStripNeededEventHandler",
                    "DataGridViewCellErrorTextNeededEventArgs",
                    "DataGridViewCellErrorTextNeededEventHandler",
                    "DataGridViewCellEventArgs",
                    "DataGridViewCellEventHandler",
                    "DataGridViewCellFormattingEventArgs",
                    "DataGridViewCellFormattingEventHandler",
                    "DataGridViewCellMouseEventArgs",
                    "DataGridViewCellMouseEventHandler",
                    "DataGridViewCellPaintingEventArgs",
                    "DataGridViewCellPaintingEventHandler",
                    "DataGridViewCellParsingEventArgs",
                    "DataGridViewCellParsingEventHandler",
                    "DataGridViewCellStateChangedEventArgs",
                    "DataGridViewCellStateChangedEventHandler",
                    "DataGridViewCellStyle",
                    "DataGridViewCellStyleContentChangedEventArgs",
                    "DataGridViewCellStyleContentChangedEventHandler",
                    "DataGridViewCellStyleConverter",
                    "DataGridViewCellStyleScopes",
                    "DataGridViewCellToolTipTextNeededEventArgs",
                    "DataGridViewCellToolTipTextNeededEventHandler",
                    "DataGridViewCellValidatingEventArgs",
                    "DataGridViewCellValidatingEventHandler",
                    "DataGridViewCellValueEventArgs",
                    "DataGridViewCellValueEventHandler",
                    "DataGridViewCheckBoxCell",
                    "DataGridViewCheckBoxColumn",
                    "DataGridViewClipboardCopyMode",
                    "DataGridViewColumn",
                    "DataGridViewColumnCollection",
                    "DataGridViewColumnDesignTimeVisibleAttribute",
                    "DataGridViewColumnDividerDoubleClickEventArgs",
                    "DataGridViewColumnDividerDoubleClickEventHandler",
                    "DataGridViewColumnEventArgs",
                    "DataGridViewColumnEventHandler",
                    "DataGridViewColumnHeaderCell",
                    "DataGridViewColumnHeadersHeightSizeMode",
                    "DataGridViewColumnSortMode",
                    "DataGridViewColumnStateChangedEventArgs",
                    "DataGridViewColumnStateChangedEventHandler",
                    "DataGridViewComboBoxCell",
                    "DataGridViewComboBoxColumn",
                    "DataGridViewComboBoxDisplayStyle",
                    "DataGridViewComboBoxEditingControl",
                    "DataGridViewContentAlignment",
                    "DataGridViewDataErrorContexts",
                    "DataGridViewDataErrorEventArgs",
                    "DataGridViewDataErrorEventHandler",
                    "DataGridViewEditMode",
                    "DataGridViewEditingControlShowingEventArgs",
                    "DataGridViewEditingControlShowingEventHandler",
                    "DataGridViewElement",
                    "DataGridViewElementStates",
                    "DataGridViewHeaderBorderStyle",
                    "DataGridViewHeaderCell",
                    "DataGridViewHitTestType",
                    "DataGridViewImageCell",
                    "DataGridViewImageCellLayout",
                    "DataGridViewImageColumn",
                    "DataGridViewLinkCell",
                    "DataGridViewLinkColumn",
                    "DataGridViewPaintParts",
                    "DataGridViewRow",
                    "DataGridViewRowCancelEventArgs",
                    "DataGridViewRowCancelEventHandler",
                    "DataGridViewRowCollection",
                    "DataGridViewRowContextMenuStripNeededEventArgs",
                    "DataGridViewRowContextMenuStripNeededEventHandler",
                    "DataGridViewRowDividerDoubleClickEventArgs",
                    "DataGridViewRowDividerDoubleClickEventHandler",
                    "DataGridViewRowErrorTextNeededEventArgs",
                    "DataGridViewRowErrorTextNeededEventHandler",
                    "DataGridViewRowEventArgs",
                    "DataGridViewRowEventHandler",
                    "DataGridViewRowHeaderCell",
                    "DataGridViewRowHeadersWidthSizeMode",
                    "DataGridViewRowHeightInfoNeededEventArgs",
                    "DataGridViewRowHeightInfoNeededEventHandler",
                    "DataGridViewRowHeightInfoPushedEventArgs",
                    "DataGridViewRowHeightInfoPushedEventHandler",
                    "DataGridViewRowPostPaintEventArgs",
                    "DataGridViewRowPostPaintEventHandler",
                    "DataGridViewRowPrePaintEventArgs",
                    "DataGridViewRowPrePaintEventHandler",
                    "DataGridViewRowStateChangedEventArgs",
                    "DataGridViewRowStateChangedEventHandler",
                    "DataGridViewRowsAddedEventArgs",
                    "DataGridViewRowsAddedEventHandler",
                    "DataGridViewRowsRemovedEventArgs",
                    "DataGridViewRowsRemovedEventHandler",
                    "DataGridViewSelectedCellCollection",
                    "DataGridViewSelectedColumnCollection",
                    "DataGridViewSelectedRowCollection",
                    "DataGridViewSelectionMode",
                    "DataGridViewSortCompareEventArgs",
                    "DataGridViewSortCompareEventHandler",
                    "DataGridViewTextBoxCell",
                    "DataGridViewTextBoxColumn",
                    "DataGridViewTextBoxEditingControl",
                    "DataGridViewTopLeftHeaderCell",
                    "DataGridViewTriState",
                    "DataObject",
                    "DataSourceUpdateMode",
                    "DateBoldEventArgs",
                    "DateBoldEventHandler",
                    "DateRangeEventArgs",
                    "DateRangeEventHandler",
                    "DateTimePicker",
                    "DateTimePickerFormat",
                    "Day",
                    "DialogResult",
                    "DockStyle",
                    "DockingAttribute",
                    "DockingBehavior",
                    "DomainUpDown",
                    "DragAction",
                    "DragDropEffects",
                    "DragEventArgs",
                    "DragEventHandler",
                    "DrawItemEventArgs",
                    "DrawItemEventHandler",
                    "DrawItemState",
                    "DrawListViewColumnHeaderEventArgs",
                    "DrawListViewColumnHeaderEventHandler",
                    "DrawListViewItemEventArgs",
                    "DrawListViewItemEventHandler",
                    "DrawListViewSubItemEventArgs",
                    "DrawListViewSubItemEventHandler",
                    "DrawMode",
                    "DrawToolTipEventArgs",
                    "DrawToolTipEventHandler",
                    "DrawTreeNodeEventArgs",
                    "DrawTreeNodeEventHandler",
                    "ErrorBlinkStyle",
                    "ErrorIconAlignment",
                    "ErrorProvider",
                    "FeatureSupport",
                    "FileDialog",
                    "FixedPanel",
                    "FlatButtonAppearance",
                    "FlatStyle",
                    "FlowDirection",
                    "FlowLayoutPanel",
                    "FlowLayoutSettings",
                    "FolderBrowserDialog",
                    "FontDialog",
                    "Form",
                    "FormBorderStyle",
                    "FormClosedEventArgs",
                    "FormClosedEventHandler",
                    "FormClosingEventArgs",
                    "FormClosingEventHandler",
                    "FormCollection",
                    "FormStartPosition",
                    "FormWindowState",
                    "FrameStyle",
                    "GetChildAtPointSkip",
                    "GiveFeedbackEventArgs",
                    "GiveFeedbackEventHandler",
                    "GridColumnStylesCollection",
                    "GridItem",
                    "GridItemCollection",
                    "GridItemType",
                    "GridTableStylesCollection",
                    "GridTablesFactory",
                    "GroupBox",
                    "GroupBoxRenderer",
                    "HScrollBar",
                    "HScrollProperties",
                    "HandledMouseEventArgs",
                    "Help",
                    "HelpEventArgs",
                    "HelpEventHandler",
                    "HelpNavigator",
                    "HelpProvider",
                    "HorizontalAlignment",
                    "HtmlDocument",
                    "HtmlElement",
                    "HtmlElementCollection",
                    "HtmlElementErrorEventArgs",
                    "HtmlElementErrorEventHandler",
                    "HtmlElementEventArgs",
                    "HtmlElementEventHandler",
                    "HtmlElementInsertionOrientation",
                    "HtmlHistory",
                    "HtmlWindow",
                    "HtmlWindowCollection",
                    "IBindableComponent",
                    "IButtonControl",
                    "ICommandExecutor",
                    "IComponentEditorPageSite",
                    "IContainerControl",
                    "ICurrencyManagerProvider",
                    "IDataGridColumnStyleEditingNotificationService",
                    "IDataGridEditingService",
                    "IDataGridViewEditingCell",
                    "IDataGridViewEditingControl",
                    "IDataObject",
                    "IDropTarget",
                    "IFeatureSupport",
                    "IFileReaderService",
                    "IMessageFilter",
                    "IWin32Window",
                    "IWindowTarget",
                    "ImageIndexConverter",
                    "ImageKeyConverter",
                    "ImageLayout",
                    "ImageList",
                    "ImageListStreamer",
                    "ImeMode",
                    "InputLanguage",
                    "InputLanguageChangedEventArgs",
                    "InputLanguageChangedEventHandler",
                    "InputLanguageChangingEventArgs",
                    "InputLanguageChangingEventHandler",
                    "InputLanguageCollection",
                    "InsertKeyMode",
                    "InvalidateEventArgs",
                    "InvalidateEventHandler",
                    "ItemActivation",
                    "ItemBoundsPortion",
                    "ItemChangedEventArgs",
                    "ItemChangedEventHandler",
                    "ItemCheckEventArgs",
                    "ItemCheckEventHandler",
                    "ItemCheckedEventArgs",
                    "ItemCheckedEventHandler",
                    "ItemDragEventArgs",
                    "ItemDragEventHandler",
                    "KeyEventArgs",
                    "KeyEventHandler",
                    "KeyPressEventArgs",
                    "KeyPressEventHandler",
                    "Keys",
                    "KeysConverter",
                    "Label",
                    "LabelEditEventArgs",
                    "LabelEditEventHandler",
                    "LayoutEventArgs",
                    "LayoutEventHandler",
                    "LayoutSettings",
                    "LeftRightAlignment",
                    "LinkArea",
                    "LinkBehavior",
                    "LinkClickedEventArgs",
                    "LinkClickedEventHandler",
                    "LinkConverter",
                    "LinkLabel",
                    "LinkLabelLinkClickedEventArgs",
                    "LinkLabelLinkClickedEventHandler",
                    "LinkState",
                    "ListBindingConverter",
                    "ListBindingHelper",
                    "ListBox",
                    "ListControl",
                    "ListControlConvertEventArgs",
                    "ListControlConvertEventHandler",
                    "ListView",
                    "ListViewAlignment",
                    "ListViewGroup",
                    "ListViewGroupCollection",
                    "ListViewHitTestInfo",
                    "ListViewHitTestLocations",
                    "ListViewInsertionMark",
                    "ListViewItem",
                    "ListViewItemConverter",
                    "ListViewItemMouseHoverEventArgs",
                    "ListViewItemMouseHoverEventHandler",
                    "ListViewItemSelectionChangedEventArgs",
                    "ListViewItemSelectionChangedEventHandler",
                    "ListViewItemStates",
                    "ListViewVirtualItemsSelectionRangeChangedEventArgs",
                    "ListViewVirtualItemsSelectionRangeChangedEventHandler",
                    "MainMenu",
                    "MaskFormat",
                    "MaskInputRejectedEventArgs",
                    "MaskInputRejectedEventHandler",
                    "MaskedTextBox",
                    "MdiClient",
                    "MdiLayout",
                    "MeasureItemEventArgs",
                    "MeasureItemEventHandler",
                    "Menu",
                    "MenuGlyph",
                    "MenuItem",
                    "MenuMerge",
                    "MenuStrip",
                    "MergeAction",
                    "Message",
                    "MessageBox",
                    "MessageBoxButtons",
                    "MessageBoxDefaultButton",
                    "MessageBoxIcon",
                    "MessageBoxOptions",
                    "MethodInvoker",
                    "MonthCalendar",
                    "MouseButtons",
                    "MouseEventArgs",
                    "MouseEventHandler",
                    "NativeWindow",
                    "NavigateEventArgs",
                    "NavigateEventHandler",
                    "NodeLabelEditEventArgs",
                    "NodeLabelEditEventHandler",
                    "NotifyIcon",
                    "NumericUpDown",
                    "NumericUpDownAcceleration",
                    "NumericUpDownAccelerationCollection",
                    "OSFeature",
                    "OpacityConverter",
                    "OpenFileDialog",
                    "Orientation",
                    "OwnerDrawPropertyBag",
                    "Padding",
                    "PaddingConverter",
                    "PageSetupDialog",
                    "PaintEventArgs",
                    "PaintEventHandler",
                    "Panel",
                    "PictureBox",
                    "PictureBoxSizeMode",
                    "PopupEventArgs",
                    "PopupEventHandler",
                    "PowerLineStatus",
                    "PowerState",
                    "PowerStatus",
                    "PreProcessControlState",
                    "PreviewKeyDownEventArgs",
                    "PreviewKeyDownEventHandler",
                    "PrintControllerWithStatusDialog",
                    "PrintDialog",
                    "PrintPreviewControl",
                    "PrintPreviewDialog",
                    "ProfessionalColorTable",
                    "ProfessionalColors",
                    "ProgressBar",
                    "ProgressBarRenderer",
                    "ProgressBarStyle",
                    "PropertyGrid",
                    "PropertyManager",
                    "PropertySort",
                    "PropertyTabChangedEventArgs",
                    "PropertyTabChangedEventHandler",
                    "PropertyValueChangedEventArgs",
                    "PropertyValueChangedEventHandler",
                    "QueryAccessibilityHelpEventArgs",
                    "QueryAccessibilityHelpEventHandler",
                    "QueryContinueDragEventArgs",
                    "QueryContinueDragEventHandler",
                    "QuestionEventArgs",
                    "QuestionEventHandler",
                    "RadioButton",
                    "RadioButtonRenderer",
                    "RelatedImageListAttribute",
                    "RetrieveVirtualItemEventArgs",
                    "RetrieveVirtualItemEventHandler",
                    "RichTextBox",
                    "RichTextBoxFinds",
                    "RichTextBoxLanguageOptions",
                    "RichTextBoxScrollBars",
                    "RichTextBoxSelectionAttribute",
                    "RichTextBoxSelectionTypes",
                    "RichTextBoxStreamType",
                    "RichTextBoxWordPunctuations",
                    "RightToLeft",
                    "RowStyle",
                    "SaveFileDialog",
                    "Screen",
                    "ScreenOrientation",
                    "ScrollBar",
                    "ScrollBarRenderer",
                    "ScrollBars",
                    "ScrollButton",
                    "ScrollEventArgs",
                    "ScrollEventHandler",
                    "ScrollEventType",
                    "ScrollOrientation",
                    "ScrollProperties",
                    "ScrollableControl",
                    "SearchDirectionHint",
                    "SearchForVirtualItemEventArgs",
                    "SearchForVirtualItemEventHandler",
                    "SecurityIDType",
                    "SelectedGridItemChangedEventArgs",
                    "SelectedGridItemChangedEventHandler",
                    "SelectionMode",
                    "SelectionRange",
                    "SelectionRangeConverter",
                    "SendKeys",
                    "Shortcut",
                    "SizeGripStyle",
                    "SizeType",
                    "SortOrder",
                    "SplitContainer",
                    "Splitter",
                    "SplitterCancelEventArgs",
                    "SplitterCancelEventHandler",
                    "SplitterEventArgs",
                    "SplitterEventHandler",
                    "SplitterPanel",
                    "StatusBar",
                    "StatusBarDrawItemEventArgs",
                    "StatusBarDrawItemEventHandler",
                    "StatusBarPanel",
                    "StatusBarPanelAutoSize",
                    "StatusBarPanelBorderStyle",
                    "StatusBarPanelClickEventArgs",
                    "StatusBarPanelClickEventHandler",
                    "StatusBarPanelStyle",
                    "StatusStrip",
                    "StructFormat",
                    "SystemInformation",
                    "SystemParameter",
                    "TabAlignment",
                    "TabAppearance",
                    "TabControl",
                    "TabControlAction",
                    "TabControlCancelEventArgs",
                    "TabControlCancelEventHandler",
                    "TabControlEventArgs",
                    "TabControlEventHandler",
                    "TabDrawMode",
                    "TabPage",
                    "TabRenderer",
                    "TabSizeMode",
                    "TableLayoutCellPaintEventArgs",
                    "TableLayoutCellPaintEventHandler",
                    "TableLayoutColumnStyleCollection",
                    "TableLayoutControlCollection",
                    "TableLayoutPanel",
                    "TableLayoutPanelCellBorderStyle",
                    "TableLayoutPanelCellPosition",
                    "TableLayoutPanelGrowStyle",
                    "TableLayoutRowStyleCollection",
                    "TableLayoutSettings",
                    "TableLayoutStyle",
                    "TableLayoutStyleCollection",
                    "TextBox",
                    "TextBoxBase",
                    "TextBoxRenderer",
                    "TextDataFormat",
                    "TextFormatFlags",
                    "TextImageRelation",
                    "TextRenderer",
                    "ThreadExceptionDialog",
                    "TickStyle",
                    "Timer",
                    "ToolBar",
                    "ToolBarAppearance",
                    "ToolBarButton",
                    "ToolBarButtonClickEventArgs",
                    "ToolBarButtonClickEventHandler",
                    "ToolBarButtonStyle",
                    "ToolBarTextAlign",
                    "ToolStrip",
                    "ToolStripArrowRenderEventArgs",
                    "ToolStripArrowRenderEventHandler",
                    "ToolStripButton",
                    "ToolStripComboBox",
                    "ToolStripContainer",
                    "ToolStripContentPanel",
                    "ToolStripContentPanelRenderEventArgs",
                    "ToolStripContentPanelRenderEventHandler",
                    "ToolStripControlHost",
                    "ToolStripDropDown",
                    "ToolStripDropDownButton",
                    "ToolStripDropDownCloseReason",
                    "ToolStripDropDownClosedEventArgs",
                    "ToolStripDropDownClosedEventHandler",
                    "ToolStripDropDownClosingEventArgs",
                    "ToolStripDropDownClosingEventHandler",
                    "ToolStripDropDownDirection",
                    "ToolStripDropDownItem",
                    "ToolStripDropDownItemAccessibleObject",
                    "ToolStripDropDownMenu",
                    "ToolStripGripDisplayStyle",
                    "ToolStripGripRenderEventArgs",
                    "ToolStripGripRenderEventHandler",
                    "ToolStripGripStyle",
                    "ToolStripItem",
                    "ToolStripItemAlignment",
                    "ToolStripItemClickedEventArgs",
                    "ToolStripItemClickedEventHandler",
                    "ToolStripItemCollection",
                    "ToolStripItemDisplayStyle",
                    "ToolStripItemEventArgs",
                    "ToolStripItemEventHandler",
                    "ToolStripItemImageRenderEventArgs",
                    "ToolStripItemImageRenderEventHandler",
                    "ToolStripItemImageScaling",
                    "ToolStripItemOverflow",
                    "ToolStripItemPlacement",
                    "ToolStripItemRenderEventArgs",
                    "ToolStripItemRenderEventHandler",
                    "ToolStripItemTextRenderEventArgs",
                    "ToolStripItemTextRenderEventHandler",
                    "ToolStripLabel",
                    "ToolStripLayoutStyle",
                    "ToolStripManager",
                    "ToolStripManagerRenderMode",
                    "ToolStripMenuItem",
                    "ToolStripOverflow",
                    "ToolStripOverflowButton",
                    "ToolStripPanel",
                    "ToolStripPanelRenderEventArgs",
                    "ToolStripPanelRenderEventHandler",
                    "ToolStripPanelRow",
                    "ToolStripProfessionalRenderer",
                    "ToolStripProgressBar",
                    "ToolStripRenderEventArgs",
                    "ToolStripRenderEventHandler",
                    "ToolStripRenderMode",
                    "ToolStripRenderer",
                    "ToolStripSeparator",
                    "ToolStripSeparatorRenderEventArgs",
                    "ToolStripSeparatorRenderEventHandler",
                    "ToolStripSplitButton",
                    "ToolStripStatusLabel",
                    "ToolStripStatusLabelBorderSides",
                    "ToolStripSystemRenderer",
                    "ToolStripTextBox",
                    "ToolStripTextDirection",
                    "ToolTip",
                    "ToolTipIcon",
                    "TrackBar",
                    "TrackBarRenderer",
                    "TreeNode",
                    "TreeNodeCollection",
                    "TreeNodeConverter",
                    "TreeNodeMouseClickEventArgs",
                    "TreeNodeMouseClickEventHandler",
                    "TreeNodeMouseHoverEventArgs",
                    "TreeNodeMouseHoverEventHandler",
                    "TreeNodeStates",
                    "TreeView",
                    "TreeViewAction",
                    "TreeViewCancelEventArgs",
                    "TreeViewCancelEventHandler",
                    "TreeViewDrawMode",
                    "TreeViewEventArgs",
                    "TreeViewEventHandler",
                    "TreeViewHitTestInfo",
                    "TreeViewHitTestLocations",
                    "TreeViewImageIndexConverter",
                    "TreeViewImageKeyConverter",
                    "TypeValidationEventArgs",
                    "TypeValidationEventHandler",
                    "UICues",
                    "UICuesEventArgs",
                    "UICuesEventHandler",
                    "UnhandledExceptionMode",
                    "UpDownBase",
                    "UpDownEventArgs",
                    "UpDownEventHandler",
                    "UserControl",
                    "VScrollBar",
                    "VScrollProperties",
                    "ValidationConstraints",
                    "View",
                    "WebBrowser",
                    "WebBrowserBase",
                    "WebBrowserDocumentCompletedEventArgs",
                    "WebBrowserDocumentCompletedEventHandler",
                    "WebBrowserEncryptionLevel",
                    "WebBrowserNavigatedEventArgs",
                    "WebBrowserNavigatedEventHandler",
                    "WebBrowserNavigatingEventArgs",
                    "WebBrowserNavigatingEventHandler",
                    "WebBrowserProgressChangedEventArgs",
                    "WebBrowserProgressChangedEventHandler",
                    "WebBrowserReadyState",
                    "WebBrowserRefreshOption",
                    "WebBrowserSiteBase",
                    "WindowsFormsSection",
                    "WindowsFormsSynchronizationContext",
                },

                new string[3] { // System.Windows.Forms.ComponentModel.Com2Interop
                    "Com2Variant",
                    "ICom2PropertyPageDisplayService",
                    "IComPropertyBrowser",
                },

                new string[9] { // System.Windows.Forms.Design
                    "ComponentEditorForm",
                    "ComponentEditorPage",
                    "EventsTab",
                    "IUIService",
                    "IWindowsFormsEditorService",
                    "PropertyTab",
                    "ToolStripItemDesignerAvailability",
                    "ToolStripItemDesignerAvailabilityAttribute",
                    "WindowsFormsComponentEditor",
                },

                new string[3] { // System.Windows.Forms.Layout
                    "ArrangedElementCollection",
                    "LayoutEngine",
                    "TableLayoutSettingsTypeConverter",
                },

                new string[3] { // System.Windows.Forms.PropertyGridInternal
                    "IRootGridEntry",
                    "PropertiesTab",
                    "PropertyGridCommands",
                },

                new string[49] { // System.Windows.Forms.VisualStyles
                    "BackgroundType",
                    "BooleanProperty",
                    "BorderType",
                    "CheckBoxState",
                    "ColorProperty",
                    "ComboBoxState",
                    "ContentAlignment",
                    "EdgeEffects",
                    "EdgeStyle",
                    "Edges",
                    "EnumProperty",
                    "FilenameProperty",
                    "FillType",
                    "FontProperty",
                    "GlyphFontSizingType",
                    "GlyphType",
                    "GroupBoxState",
                    "HitTestCode",
                    "HitTestOptions",
                    "HorizontalAlign",
                    "IconEffect",
                    "ImageOrientation",
                    "ImageSelectType",
                    "IntegerProperty",
                    "MarginProperty",
                    "OffsetType",
                    "PointProperty",
                    "PushButtonState",
                    "RadioButtonState",
                    "ScrollBarArrowButtonState",
                    "ScrollBarSizeBoxState",
                    "ScrollBarState",
                    "SizingType",
                    "StringProperty",
                    "TabItemState",
                    "TextBoxState",
                    "TextMetrics",
                    "TextMetricsCharacterSet",
                    "TextMetricsPitchAndFamilyValues",
                    "TextShadowType",
                    "ThemeSizeType",
                    "ToolBarState",
                    "TrackBarThumbState",
                    "TrueSizeScalingType",
                    "VerticalAlignment",
                    "VisualStyleElement",
                    "VisualStyleInformation",
                    "VisualStyleRenderer",
                    "VisualStyleState",
                },

            };
            TypeName [] orcasTypes = {
                new TypeName("System.Windows.Forms", "FileDialogCustomPlace"),
                new TypeName("System.Windows.Forms", "FileDialogCustomPlacesCollection"),
            };

            return GetTypeNames(namespaces, types, orcasTypes);
        }

        static IEnumerable<TypeName> Get_SystemXml_TypeNames() {
            // System.Xml, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089
            // Total number of types = 255

            string [] namespaces = new string[7] {
                "System.Xml",
                "System.Xml.Schema",
                "System.Xml.Serialization",
                "System.Xml.Serialization.Advanced",
                "System.Xml.Serialization.Configuration",
                "System.Xml.XPath",
                "System.Xml.Xsl",
            };

            string [][] types = new string[7][] {

                new string[61] { // System.Xml
                    "ConformanceLevel",
                    "EntityHandling",
                    "Formatting",
                    "IHasXmlNode",
                    "IXmlLineInfo",
                    "IXmlNamespaceResolver",
                    "NameTable",
                    "NewLineHandling",
                    "ReadState",
                    "ValidationType",
                    "WhitespaceHandling",
                    "WriteState",
                    "XmlAttribute",
                    "XmlAttributeCollection",
                    "XmlCDataSection",
                    "XmlCharacterData",
                    "XmlComment",
                    "XmlConvert",
                    "XmlDateTimeSerializationMode",
                    "XmlDeclaration",
                    "XmlDocument",
                    "XmlDocumentFragment",
                    "XmlDocumentType",
                    "XmlElement",
                    "XmlEntity",
                    "XmlEntityReference",
                    "XmlException",
                    "XmlImplementation",
                    "XmlLinkedNode",
                    "XmlNameTable",
                    "XmlNamedNodeMap",
                    "XmlNamespaceManager",
                    "XmlNamespaceScope",
                    "XmlNode",
                    "XmlNodeChangedAction",
                    "XmlNodeChangedEventArgs",
                    "XmlNodeChangedEventHandler",
                    "XmlNodeList",
                    "XmlNodeOrder",
                    "XmlNodeReader",
                    "XmlNodeType",
                    "XmlNotation",
                    "XmlOutputMethod",
                    "XmlParserContext",
                    "XmlProcessingInstruction",
                    "XmlQualifiedName",
                    "XmlReader",
                    "XmlReaderSettings",
                    "XmlResolver",
                    "XmlSecureResolver",
                    "XmlSignificantWhitespace",
                    "XmlSpace",
                    "XmlText",
                    "XmlTextReader",
                    "XmlTextWriter",
                    "XmlTokenizedType",
                    "XmlUrlResolver",
                    "XmlValidatingReader",
                    "XmlWhitespace",
                    "XmlWriter",
                    "XmlWriterSettings",
                },

                new string[87] { // System.Xml.Schema
                    "IXmlSchemaInfo",
                    "ValidationEventArgs",
                    "ValidationEventHandler",
                    "XmlAtomicValue",
                    "XmlSchema",
                    "XmlSchemaAll",
                    "XmlSchemaAnnotated",
                    "XmlSchemaAnnotation",
                    "XmlSchemaAny",
                    "XmlSchemaAnyAttribute",
                    "XmlSchemaAppInfo",
                    "XmlSchemaAttribute",
                    "XmlSchemaAttributeGroup",
                    "XmlSchemaAttributeGroupRef",
                    "XmlSchemaChoice",
                    "XmlSchemaCollection",
                    "XmlSchemaCollectionEnumerator",
                    "XmlSchemaCompilationSettings",
                    "XmlSchemaComplexContent",
                    "XmlSchemaComplexContentExtension",
                    "XmlSchemaComplexContentRestriction",
                    "XmlSchemaComplexType",
                    "XmlSchemaContent",
                    "XmlSchemaContentModel",
                    "XmlSchemaContentProcessing",
                    "XmlSchemaContentType",
                    "XmlSchemaDatatype",
                    "XmlSchemaDatatypeVariety",
                    "XmlSchemaDerivationMethod",
                    "XmlSchemaDocumentation",
                    "XmlSchemaElement",
                    "XmlSchemaEnumerationFacet",
                    "XmlSchemaException",
                    "XmlSchemaExternal",
                    "XmlSchemaFacet",
                    "XmlSchemaForm",
                    "XmlSchemaFractionDigitsFacet",
                    "XmlSchemaGroup",
                    "XmlSchemaGroupBase",
                    "XmlSchemaGroupRef",
                    "XmlSchemaIdentityConstraint",
                    "XmlSchemaImport",
                    "XmlSchemaInclude",
                    "XmlSchemaInference",
                    "XmlSchemaInferenceException",
                    "XmlSchemaInfo",
                    "XmlSchemaKey",
                    "XmlSchemaKeyref",
                    "XmlSchemaLengthFacet",
                    "XmlSchemaMaxExclusiveFacet",
                    "XmlSchemaMaxInclusiveFacet",
                    "XmlSchemaMaxLengthFacet",
                    "XmlSchemaMinExclusiveFacet",
                    "XmlSchemaMinInclusiveFacet",
                    "XmlSchemaMinLengthFacet",
                    "XmlSchemaNotation",
                    "XmlSchemaNumericFacet",
                    "XmlSchemaObject",
                    "XmlSchemaObjectCollection",
                    "XmlSchemaObjectEnumerator",
                    "XmlSchemaObjectTable",
                    "XmlSchemaParticle",
                    "XmlSchemaPatternFacet",
                    "XmlSchemaRedefine",
                    "XmlSchemaSequence",
                    "XmlSchemaSet",
                    "XmlSchemaSimpleContent",
                    "XmlSchemaSimpleContentExtension",
                    "XmlSchemaSimpleContentRestriction",
                    "XmlSchemaSimpleType",
                    "XmlSchemaSimpleTypeContent",
                    "XmlSchemaSimpleTypeList",
                    "XmlSchemaSimpleTypeRestriction",
                    "XmlSchemaSimpleTypeUnion",
                    "XmlSchemaTotalDigitsFacet",
                    "XmlSchemaType",
                    "XmlSchemaUnique",
                    "XmlSchemaUse",
                    "XmlSchemaValidationException",
                    "XmlSchemaValidationFlags",
                    "XmlSchemaValidator",
                    "XmlSchemaValidity",
                    "XmlSchemaWhiteSpaceFacet",
                    "XmlSchemaXPath",
                    "XmlSeverityType",
                    "XmlTypeCode",
                    "XmlValueGetter",
                },

                new string[75] { // System.Xml.Serialization
                    "CodeExporter",
                    "CodeGenerationOptions",
                    "CodeIdentifier",
                    "CodeIdentifiers",
                    "IXmlSerializable",
                    "IXmlTextParser",
                    "ImportContext",
                    "SchemaImporter",
                    "SoapAttributeAttribute",
                    "SoapAttributeOverrides",
                    "SoapAttributes",
                    "SoapCodeExporter",
                    "SoapElementAttribute",
                    "SoapEnumAttribute",
                    "SoapIgnoreAttribute",
                    "SoapIncludeAttribute",
                    "SoapReflectionImporter",
                    "SoapSchemaExporter",
                    "SoapSchemaImporter",
                    "SoapSchemaMember",
                    "SoapTypeAttribute",
                    "UnreferencedObjectEventArgs",
                    "UnreferencedObjectEventHandler",
                    "XmlAnyAttributeAttribute",
                    "XmlAnyElementAttribute",
                    "XmlAnyElementAttributes",
                    "XmlArrayAttribute",
                    "XmlArrayItemAttribute",
                    "XmlArrayItemAttributes",
                    "XmlAttributeAttribute",
                    "XmlAttributeEventArgs",
                    "XmlAttributeEventHandler",
                    "XmlAttributeOverrides",
                    "XmlAttributes",
                    "XmlChoiceIdentifierAttribute",
                    "XmlCodeExporter",
                    "XmlDeserializationEvents",
                    "XmlElementAttribute",
                    "XmlElementAttributes",
                    "XmlElementEventArgs",
                    "XmlElementEventHandler",
                    "XmlEnumAttribute",
                    "XmlIgnoreAttribute",
                    "XmlIncludeAttribute",
                    "XmlMapping",
                    "XmlMappingAccess",
                    "XmlMemberMapping",
                    "XmlMembersMapping",
                    "XmlNamespaceDeclarationsAttribute",
                    "XmlNodeEventArgs",
                    "XmlNodeEventHandler",
                    "XmlReflectionImporter",
                    "XmlReflectionMember",
                    "XmlRootAttribute",
                    "XmlSchemaEnumerator",
                    "XmlSchemaExporter",
                    "XmlSchemaImporter",
                    "XmlSchemaProviderAttribute",
                    "XmlSchemas",
                    "XmlSerializationCollectionFixupCallback",
                    "XmlSerializationFixupCallback",
                    "XmlSerializationGeneratedCode",
                    "XmlSerializationReadCallback",
                    "XmlSerializationReader",
                    "XmlSerializationWriteCallback",
                    "XmlSerializationWriter",
                    "XmlSerializer",
                    "XmlSerializerAssemblyAttribute",
                    "XmlSerializerFactory",
                    "XmlSerializerImplementation",
                    "XmlSerializerNamespaces",
                    "XmlSerializerVersionAttribute",
                    "XmlTextAttribute",
                    "XmlTypeAttribute",
                    "XmlTypeMapping",
                },

                new string[2] { // System.Xml.Serialization.Advanced
                    "SchemaImporterExtension",
                    "SchemaImporterExtensionCollection",
                },

                new string[6] { // System.Xml.Serialization.Configuration
                    "DateTimeSerializationSection",
                    "SchemaImporterExtensionElement",
                    "SchemaImporterExtensionElementCollection",
                    "SchemaImporterExtensionsSection",
                    "SerializationSectionGroup",
                    "XmlSerializerSection",
                },

                new string[13] { // System.Xml.XPath
                    "IXPathNavigable",
                    "XPathDocument",
                    "XPathException",
                    "XPathExpression",
                    "XPathItem",
                    "XPathNamespaceScope",
                    "XPathNavigator",
                    "XPathNodeIterator",
                    "XPathNodeType",
                    "XPathResultType",
                    "XmlCaseOrder",
                    "XmlDataType",
                    "XmlSortOrder",
                },

                new string[11] { // System.Xml.Xsl
                    "IXsltContextFunction",
                    "IXsltContextVariable",
                    "XslCompiledTransform",
                    "XslTransform",
                    "XsltArgumentList",
                    "XsltCompileException",
                    "XsltContext",
                    "XsltException",
                    "XsltMessageEncounteredEventArgs",
                    "XsltMessageEncounteredEventHandler",
                    "XsltSettings",
                },

            };
            TypeName [] orcasTypes = {
                new TypeName("System.Xml.Serialization.Configuration", "RootedPathValidator"),
            };

            return GetTypeNames(namespaces, types, orcasTypes);
        }


        // *** END GENERATED CODE ***

        #endregion

#endif
    }
}
