#include "config.h"

#ifdef HAVE_LIBCDDA_PARANOIA

#include "cdda.h"
#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <fcntl.h>
#include <qthread.h>
#include <qregexp.h>
#include <qpainter.h>
#include <qpixmap.h>
#include "mainwnd.h"
#include "pump.h"

#define BLOCK_SKIP 4096
#define DEFAULT_PARANOIA_MODE   (PARANOIA_MODE_DISABLE)

pthread_mutex_t CDDAStreamSource::cdlock=PTHREAD_MUTEX_INITIALIZER;

extern "C" {
static void paranoia_callback(long inpos, int function)
{
    /**/
}
}

CDDAStreamSource::CDDAStreamSource(QWidget *dest) : StreamSource(dest)
{
	drive=NULL;
	p=NULL;
	secs_in_buffer=0;
}

CDDAStreamSource::~CDDAStreamSource()
{
	if(p)
		paranoia_free(p);
	if(drive)
		cdda_close(drive);
}

int CDDAStreamSource::attach(QString file)
{
	sample_rate=44100;

	filename=file;

	if(file.left(7) != "cdda://")
		return -1;
	
	file=file.mid(7);
	int cpos=file.find(":");
	if(cpos < 0)
		return -1;
	QString strtrack=file.mid(cpos+1);
	file=file.left(cpos);
	strtrack=strtrack.replace(QRegExp("[^0-9]*"), "");
	track=atoi(strtrack);

	drive=cdda_identify(file, CDDA_MESSAGE_FORGETIT, NULL);
	if(!drive)
		return -1;
	cdda_verbose_set(drive, CDDA_MESSAGE_FORGETIT, CDDA_MESSAGE_FORGETIT);
	if(cdda_open(drive))
	{
		cdda_close(drive);
		drive=NULL;
		return -1;
	}
	if(track < 1 || track > drive->tracks)
	{
		cdda_close(drive);
		drive=NULL;
		return -1;
	}

	max_frames=(cdda_track_lastsector(drive, track)-cdda_track_firstsector(drive, track)) * CD_FRAMESAMPLES;
	p=paranoia_init(drive);
	paranoia_modeset(p, DEFAULT_PARANOIA_MODE);
//	cdda_speed_set(drive, drive->maxspeed);
	paranoia_seek(p, cdda_track_firstsector(drive, track), SEEK_SET);
	sector=cdda_track_firstsector(drive, track);
	offset=0;
	secs_in_buffer=0;

	fillBuffer();

	return 0;
}

int CDDAStreamSource::get_buffer(char *buf, int max)
{
	if(!playing)
		return 0;
	
	if(sample_pos+max > max_frames)
		max=max_frames-sample_pos;

	if(!max)
	{
		if(playing)
			QThread::postEvent(this, new QEvent(QEvent::User));
		playing=0;
		return 0;
	}

	if(play_samples)
	{
		if((unsigned long)max > play_samples)
			max=(int)play_samples;
	}

	fillBuffer();
	if(!playing)
		return(0);

	int i;

	for(i=0;i<max;i++)
	{
		*(((short *)buf)+i)=buffer[offset*2];
		*(((short *)buf)+max+i)=buffer[offset*2+1];
		++offset;
	}


	sample_pos+=max;

	if(play_samples)
	{
		play_samples-=max;
		if(play_samples <= 0)
		{
			if(loop_play)
				play(last_start, last_samples, true);
			else
			{
				playing=0;
				QThread::postEvent(this, new QEvent(QEvent::User));
			}
		}
	}
	
	calcAgc(buf, max);
	return max;
}

int CDDAStreamSource::play(unsigned long start_frame, unsigned long samples, bool loop)
{
	if(!drive)
		return -1;
	
	if(!loop_play || !loop)
		stop();

	if(start_frame >= max_frames)
		return -1;

	loop_play=loop;
	last_start=start_frame;

	int sec=start_frame/CD_FRAMESAMPLES+cdda_track_firstsector(drive, track);
	if(sec >= buffer_first_sector && sec < (buffer_first_sector + secs_in_buffer))
	{
		offset=(start_frame%CD_FRAMESAMPLES)+CD_FRAMESAMPLES*(sec-buffer_first_sector);
	}
	else
	{
		paranoia_seek(p, sec, SEEK_SET);
		sector=sec;
		offset=start_frame%CD_FRAMESAMPLES;
		secs_in_buffer=0;
	}

	sample_pos=start_frame;
	frame_pos=start_frame;
	play_samples=samples;
	last_samples=play_samples;

	bumpCDRom();
	playing=1;
//	prefetch();
	app_window->get_pump()->work();
	return 0;
}

int CDDAStreamSource::stop(void)
{
	if(!drive || !playing)
		return 0;

	StreamSource::stop();
	playing=0;
	return 0;
}

StreamSource *CDDAStreamSource::get_source(void)
{
	CDDAStreamSource *s=new CDDAStreamSource(app_window);
	s->attach(filename);
	return (StreamSource *)s;
}

void CDDAStreamSource::exit_loop(void)
{
	pthread_mutex_lock(&control_lock);
	play_samples=0;
	loop_play=false;
	pthread_mutex_unlock(&control_lock);
}

void CDDAStreamSource::fillBuffer()
{
	if(pthread_mutex_trylock(&cdlock) < 0)
		return;
	int cursec=sample_pos/CD_FRAMESAMPLES+cdda_track_firstsector(drive, track);
	if(cursec >= (buffer_first_sector+secs_in_buffer))
	{
		secs_in_buffer=0;
		sector=cursec;
		buffer_first_sector=cursec;
		offset=sample_pos%CD_FRAMESAMPLES;
		paranoia_seek(p, cursec, SEEK_SET);
	}
	else if(cursec < buffer_first_sector)
	{
		secs_in_buffer=0;
		sector=cursec;
		buffer_first_sector=cursec;
		offset=sample_pos%CD_FRAMESAMPLES;
		paranoia_seek(p, cursec, SEEK_SET);
	}
	else
	{
//		if(buffer_first_sector+secs_in_buffer-PREBUFFER/2-1 <= cursec)
//		{
			while(buffer_first_sector+PREBUFFER/4 < cursec)
			{
				memmove((char *)buffer, (char *)&buffer[CD_FRAMESAMPLES*2], (secs_in_buffer-1)*CD_FRAMESIZE_RAW);
				--secs_in_buffer;
				++buffer_first_sector;
			}
			offset=sample_pos%CD_FRAMESAMPLES;
			offset+=CD_FRAMESAMPLES*(cursec-buffer_first_sector);
//		}
	}

	while(secs_in_buffer < PREBUFFER)
	{
		if(sector >= cdda_track_lastsector(drive, track))
			break;
		short *readbuf=paranoia_read_limited(p, paranoia_callback, 1);
		char *err = cdda_errors(drive);
		char *mes = cdda_messages(drive);
		if(err)
			free(err);
		if(mes)
			free(mes);
		if(!readbuf)
		{
			stop();
		}
		++sector;
		memcpy((char *)buffer+CD_FRAMESIZE_RAW*secs_in_buffer, (char *)readbuf, CD_FRAMESIZE_RAW);
		++secs_in_buffer;
	}
	pthread_mutex_unlock(&cdlock);

	int start_sample=(buffer_first_sector-cdda_track_firstsector(drive, track))*CD_FRAMESAMPLES;
	int first_wavelet=start_sample/32768;
	if(first_wavelet*32768 < start_sample)
		++first_wavelet;
	int buffer_samples=secs_in_buffer*CD_FRAMESAMPLES;
	int samples_left=buffer_samples-(first_wavelet*32768-start_sample);
	int wavelets_in_buffer=samples_left/32768;

	int w;
	if(wave)
	{
		for(w=first_wavelet;w<(first_wavelet+wavelets_in_buffer);++w)
		{
			if(wave[w])
				continue;
			wave[w]=new Wavelet(this, w*32768, 32768);
			wave[w]->loadBuffer((char *)buffer+((w*32768-start_sample)*sizeof(short)*2), 32768);
		}
	}
}

void CDDAStreamSource::prefetch()
{
	if(playing)
		fillBuffer();
}

void CDDAStreamSource::generate_wave(void)
{
        if(!max_frames)
                return;
        if(wave)
                return;

		int frames;
        wave=new Wavelet*[(frames=(max_frames+32767)/32768)];
        memset((char *)wave, 0, sizeof(Wavelet *)*frames);
        wavelets=frames;
		fillBuffer();
}

void CDDAStreamSource::get_wave_pixmap(QPixmap *pm, int display_width)
{
	int frame_position=frame_pos;
	if(app_window->is_seeking())
		frame_position=app_window->transport_position();
	int pixel_pos=frame_position/512-display_width/2;
	int w=pixel_pos/64;
	int pixel_offset=pixel_pos-w*64;
	int pos=0;
	int peak_w=frame_position/512/64;
	int peak_offset=frame_position/512-peak_w*64;

	if(peak_w < 0 || peak_w >= wavelets)
		peak_value=0;
	else
	{
		if(wave && wave[peak_w] && wave[peak_w]->pixmap)
		{
			peak_value=wave[peak_w]->pixmap[peak_offset];
			if(peak_value < 0)
				peak_value=-peak_value;
		}
		else
			peak_value=0;
	}

	QPainter p;

	QPen wap=QColor(36, 152, 255);
	
	p.begin((const QPaintDevice *)pm, (const QWidget *)app_window);
	p.setPen(wap);

	while(pos-pixel_offset < display_width)
	{
		if(w < 0)
		{
			pos+=64;
			w++;
			continue;
		}
		if(w >= wavelets)
			break;

		if(wave[w] && wave[w]->pixmap)
		{
			int j;

			for(j=0;j<64;j++)
			{
				p.drawPoint(pos-pixel_offset+j, 15-(wave[w]->pixmap[j]));
				p.moveTo(pos-pixel_offset+j, 15-(wave[w]->pixmap[j]));
				p.lineTo(pos-pixel_offset+j, 15+(wave[w]->pixmap[j]));
			}
		}
		pos+=64;
		w++;
	}
	QPen rp(QColor(255, 0, 0));
	p.setPen(rp);
	p.moveTo(display_width/2, 0);
	p.lineTo(display_width/2, pm->height());
}

void CDDAStreamSource::bumpCDRom()
{
	paranoia_seek(p, sector+512, SEEK_SET);
	paranoia_read_limited(p, paranoia_callback, 1);
	paranoia_seek(p, sector, SEEK_SET);
	paranoia_read_limited(p, paranoia_callback, 1);
}

bool CDDAStreamSource::can_do_multiple_reads()
{
	return false;
}

#endif /* HAVE_LIBCDDA_PARANOIA */
