/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.digikam.org
 *
 * Date        : 2007-10-24
 * Description : XMP origin settings page.
 *
 * Copyright (C) 2007-2011 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "xmporigin.moc"

// Qt includes

#include <QCheckBox>
#include <QMap>
#include <QPushButton>
#include <QGridLayout>

// KDE includes

#include <kaboutdata.h>
#include <kcombobox.h>
#include <kcomponentdata.h>
#include <kdatetimewidget.h>
#include <kdialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klocale.h>
#include <kseparator.h>

// LibKExiv2 includes

#include <libkexiv2/version.h>
#include <libkexiv2/kexiv2.h>
#include <libkexiv2/countryselector.h>

// LibKDcraw includes

#include <libkdcraw/squeezedcombobox.h>

// Local includes

#include "metadatacheckbox.h"

using namespace KDcrawIface;
using namespace KExiv2Iface;

namespace KIPIMetadataEditPlugin
{

class XMPOrigin::XMPOriginPriv
{
public:

    XMPOriginPriv()
    {
        cityEdit               = 0;
        sublocationEdit        = 0;
        provinceEdit           = 0;
        cityCheck              = 0;
        sublocationCheck       = 0;
        provinceCheck          = 0;
        countryCheck           = 0;
        dateCreatedSel         = 0;
        dateDigitalizedSel     = 0;
        dateCreatedCheck       = 0;
        dateDigitalizedCheck   = 0;
        syncHOSTDateCheck      = 0;
        syncEXIFDateCheck      = 0;
        setTodayCreatedBtn     = 0;
        setTodayDigitalizedBtn = 0;
    }

    QCheckBox*                     dateCreatedCheck;
    QCheckBox*                     dateDigitalizedCheck;
    QCheckBox*                     syncHOSTDateCheck;
    QCheckBox*                     syncEXIFDateCheck;
    QCheckBox*                     cityCheck;
    QCheckBox*                     sublocationCheck;
    QCheckBox*                     provinceCheck;

    QPushButton*                   setTodayCreatedBtn;
    QPushButton*                   setTodayDigitalizedBtn;

    KDateTimeWidget*               dateCreatedSel;
    KDateTimeWidget*               dateDigitalizedSel;

    KLineEdit*                     cityEdit;
    KLineEdit*                     sublocationEdit;
    KLineEdit*                     provinceEdit;

    MetadataCheckBox*              countryCheck;

    CountrySelector*               countryCB;
};

XMPOrigin::XMPOrigin(QWidget* parent)
    : QWidget(parent), d(new XMPOriginPriv)
{
    QGridLayout* grid = new QGridLayout(this);

    // --------------------------------------------------------

    d->dateDigitalizedCheck   = new QCheckBox(i18n("Digitization date"), this);
    d->dateDigitalizedSel     = new KDateTimeWidget(this);

    d->setTodayDigitalizedBtn = new QPushButton();
    d->setTodayDigitalizedBtn->setIcon(SmallIcon("go-jump-today"));
    d->setTodayDigitalizedBtn->setWhatsThis(i18n("Set digitization date to today"));

    d->dateDigitalizedSel->setWhatsThis(i18n("Set here the creation date of "
                                             "digital representation."));

    slotSetTodayDigitalized();

    // --------------------------------------------------------

    d->dateCreatedCheck   = new QCheckBox(i18n("Creation date"), this);
    d->dateCreatedSel     = new KDateTimeWidget(this);
    d->syncHOSTDateCheck  = new QCheckBox(i18n("Sync creation date hosted by %1",
                                               KGlobal::mainComponent().aboutData()->programName()),
                                               this);
    d->syncEXIFDateCheck  = new QCheckBox(i18n("Sync EXIF creation date"), this);

    d->setTodayCreatedBtn = new QPushButton();
    d->setTodayCreatedBtn->setIcon(SmallIcon("go-jump-today"));
    d->setTodayCreatedBtn->setWhatsThis(i18n("Set creation date to today"));

    d->dateCreatedSel->setWhatsThis(i18n("Set here the creation date of "
                                         "intellectual content."));

    slotSetTodayCreated();

    // --------------------------------------------------------

    d->cityCheck = new QCheckBox(i18n("City:"), this);
    d->cityEdit  = new KLineEdit(this);
    d->cityEdit->setClearButtonShown(true);
    d->cityEdit->setWhatsThis(i18n("Set here the content's city of origin."));

    // --------------------------------------------------------

    d->sublocationCheck = new QCheckBox(i18n("Sublocation:"), this);
    d->sublocationEdit  = new KLineEdit(this);
    d->sublocationEdit->setClearButtonShown(true);
    d->sublocationEdit->setWhatsThis(i18n("Set here the content's location within the city."));

    // --------------------------------------------------------

    d->provinceCheck = new QCheckBox(i18n("State/Province:"), this);
    d->provinceEdit  = new KLineEdit(this);
    d->provinceEdit->setClearButtonShown(true);
    d->provinceEdit->setWhatsThis(i18n("Set here the content's Province or State of origin."));

    // --------------------------------------------------------

    d->countryCheck = new MetadataCheckBox(i18n("Country:"), this);
    d->countryCB    = new CountrySelector(this);
    d->countryCB->setWhatsThis(i18n("Select here country name of content origin."));
    // Remove 2 last items for the list (separator + Unknown item)
    d->countryCB->removeItem(d->countryCB->count()-1);
    d->countryCB->removeItem(d->countryCB->count()-1);

    // --------------------------------------------------------

    grid->addWidget(d->dateDigitalizedCheck,                0, 0, 1, 5);
    grid->addWidget(d->dateDigitalizedSel,                  1, 0, 1, 3);
    grid->addWidget(d->setTodayDigitalizedBtn,              1, 4, 1, 1);
    grid->addWidget(d->dateCreatedCheck,                    2, 0, 1, 5);
    grid->addWidget(d->dateCreatedSel,                      3, 0, 1, 3);
    grid->addWidget(d->setTodayCreatedBtn,                  3, 4, 1, 1);
    grid->addWidget(d->syncHOSTDateCheck,                   4, 0, 1, 5);
    grid->addWidget(d->syncEXIFDateCheck,                   5, 0, 1, 5);
    grid->addWidget(new KSeparator(Qt::Horizontal, this),   6, 0, 1, 5);
    grid->addWidget(d->cityCheck,                           7, 0, 1, 1);
    grid->addWidget(d->cityEdit,                            7, 1, 1, 4);
    grid->addWidget(d->sublocationCheck,                    8, 0, 1, 1);
    grid->addWidget(d->sublocationEdit,                     8, 1, 1, 4);
    grid->addWidget(d->provinceCheck,                       9, 0, 1, 1);
    grid->addWidget(d->provinceEdit,                        9, 1, 1, 4);
    grid->addWidget(d->countryCheck,                       10, 0, 1, 1);
    grid->addWidget(d->countryCB,                          10, 1, 1, 4);
    grid->setColumnStretch(3, 10);
    grid->setRowStretch(11, 10);
    grid->setMargin(0);
    grid->setSpacing(KDialog::spacingHint());

    // --------------------------------------------------------

    connect(d->dateCreatedCheck, SIGNAL(toggled(bool)),
            d->dateCreatedSel, SLOT(setEnabled(bool)));

    connect(d->dateDigitalizedCheck, SIGNAL(toggled(bool)),
            d->dateDigitalizedSel, SLOT(setEnabled(bool)));

    connect(d->dateCreatedCheck, SIGNAL(toggled(bool)),
            d->syncHOSTDateCheck, SLOT(setEnabled(bool)));

    connect(d->dateCreatedCheck, SIGNAL(toggled(bool)),
            d->syncEXIFDateCheck, SLOT(setEnabled(bool)));

    connect(d->cityCheck, SIGNAL(toggled(bool)),
            d->cityEdit, SLOT(setEnabled(bool)));

    connect(d->sublocationCheck, SIGNAL(toggled(bool)),
            d->sublocationEdit, SLOT(setEnabled(bool)));

    connect(d->provinceCheck, SIGNAL(toggled(bool)),
            d->provinceEdit, SLOT(setEnabled(bool)));

    connect(d->countryCheck, SIGNAL(toggled(bool)),
            d->countryCB, SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->dateCreatedCheck, SIGNAL(toggled(bool)),
            this, SIGNAL(signalModified()));

    connect(d->dateDigitalizedCheck, SIGNAL(toggled(bool)),
            this, SIGNAL(signalModified()));

    connect(d->cityCheck, SIGNAL(toggled(bool)),
            this, SIGNAL(signalModified()));

    connect(d->sublocationCheck, SIGNAL(toggled(bool)),
            this, SIGNAL(signalModified()));

    connect(d->provinceCheck, SIGNAL(toggled(bool)),
            this, SIGNAL(signalModified()));

    connect(d->countryCheck, SIGNAL(toggled(bool)),
            this, SIGNAL(signalModified()));

    // --------------------------------------------------------

    connect(d->dateCreatedSel, SIGNAL(valueChanged(QDateTime)),
            this, SIGNAL(signalModified()));

    connect(d->dateDigitalizedSel, SIGNAL(valueChanged(QDateTime)),
            this, SIGNAL(signalModified()));

    // --------------------------------------------------------

    connect(d->setTodayCreatedBtn, SIGNAL(clicked()),
            this, SLOT(slotSetTodayCreated()));

    connect(d->setTodayDigitalizedBtn, SIGNAL(clicked()),
            this, SLOT(slotSetTodayDigitalized()));

    // --------------------------------------------------------

    connect(d->countryCB, SIGNAL(activated(int)),
            this, SIGNAL(signalModified()));

    connect(d->cityEdit, SIGNAL(textChanged(QString)),
            this, SIGNAL(signalModified()));

    connect(d->sublocationEdit, SIGNAL(textChanged(QString)),
            this, SIGNAL(signalModified()));

    connect(d->provinceEdit, SIGNAL(textChanged(QString)),
            this, SIGNAL(signalModified()));
}

XMPOrigin::~XMPOrigin()
{
    delete d;
}

void XMPOrigin::slotSetTodayCreated()
{
    d->dateCreatedSel->setDateTime(QDateTime::currentDateTime());
}

void XMPOrigin::slotSetTodayDigitalized()
{
    d->dateDigitalizedSel->setDateTime(QDateTime::currentDateTime());
}

bool XMPOrigin::syncHOSTDateIsChecked()
{
    return d->syncHOSTDateCheck->isChecked();
}

bool XMPOrigin::syncEXIFDateIsChecked()
{
    return d->syncEXIFDateCheck->isChecked();
}

void XMPOrigin::setCheckedSyncHOSTDate(bool c)
{
    d->syncHOSTDateCheck->setChecked(c);
}

void XMPOrigin::setCheckedSyncEXIFDate(bool c)
{
    d->syncEXIFDateCheck->setChecked(c);
}

QDateTime XMPOrigin::getXMPCreationDate()
{
    return d->dateCreatedSel->dateTime();
}

void XMPOrigin::readMetadata(QByteArray& xmpData)
{
    blockSignals(true);
    KExiv2 exiv2Iface;
    exiv2Iface.setXmp(xmpData);

    QString     data;
    QStringList code, list;
    QDateTime   dateTime;
    QString     dateTimeStr;

    dateTimeStr = exiv2Iface.getXmpTagString("Xmp.photoshop.DateCreated", false);
    if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.xmp.CreateDate", false);
    else if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.xmp.ModifyDate", false);
    else if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.exif.DateTimeCreated", false);
    else if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.exif.DateTimeOriginal", false);
    else if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.tiff.DateTime", false);
    else if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.xmp.ModifyDate", false);
    else if (dateTimeStr.isEmpty())
        dateTimeStr = exiv2Iface.getXmpTagString("Xmp.xmp.MetadataDate", false);

    d->dateCreatedSel->setDateTime(QDateTime::currentDateTime());
    d->dateCreatedCheck->setChecked(false);
    if (!dateTimeStr.isEmpty())
    {
        dateTime = QDateTime::fromString(dateTimeStr, Qt::ISODate);
        if (dateTime.isValid())
        {
            d->dateCreatedSel->setDateTime(dateTime);
            d->dateCreatedCheck->setChecked(true);
        }
    }
    d->dateCreatedSel->setEnabled(d->dateCreatedCheck->isChecked());
    d->syncHOSTDateCheck->setEnabled(d->dateCreatedCheck->isChecked());
    d->syncEXIFDateCheck->setEnabled(d->dateCreatedCheck->isChecked());

    dateTimeStr = exiv2Iface.getXmpTagString("Xmp.exif.DateTimeDigitized", false);

    d->dateDigitalizedSel->setDateTime(QDateTime::currentDateTime());
    d->dateDigitalizedCheck->setChecked(false);
    if (!dateTimeStr.isEmpty())
    {
        dateTime = QDateTime::fromString(dateTimeStr, Qt::ISODate);
        if (dateTime.isValid())
        {
            d->dateDigitalizedSel->setDateTime(dateTime);
            d->dateDigitalizedCheck->setChecked(true);
        }
    }
    d->dateDigitalizedSel->setEnabled(d->dateDigitalizedCheck->isChecked());

    d->cityEdit->clear();
    d->cityCheck->setChecked(false);
    data = exiv2Iface.getXmpTagString("Xmp.photoshop.City", false);
    if (!data.isNull())
    {
        d->cityEdit->setText(data);
        d->cityCheck->setChecked(true);
    }
    d->cityEdit->setEnabled(d->cityCheck->isChecked());

    d->sublocationEdit->clear();
    d->sublocationCheck->setChecked(false);
    data = exiv2Iface.getXmpTagString("Xmp.iptc.Location", false);
    if (!data.isNull())
    {
        d->sublocationEdit->setText(data);
        d->sublocationCheck->setChecked(true);
    }
    d->sublocationEdit->setEnabled(d->sublocationCheck->isChecked());

    d->provinceEdit->clear();
    d->provinceCheck->setChecked(false);
    data = exiv2Iface.getXmpTagString("Xmp.photoshop.State", false);
    if (!data.isNull())
    {
        d->provinceEdit->setText(data);
        d->provinceCheck->setChecked(true);
    }
    d->provinceEdit->setEnabled(d->provinceCheck->isChecked());

    d->countryCB->setCurrentIndex(0);
    d->countryCheck->setChecked(false);
    data = exiv2Iface.getXmpTagString("Xmp.iptc.CountryCode", false);
    if (!data.isNull())
    {
        int item = -1;
        for (int i = 0 ; i < d->countryCB->count() ; ++i)
            if (d->countryCB->itemText(i).left(3) == data)
                item = i;

        if (item != -1)
        {
            d->countryCB->setCurrentIndex(item);
            d->countryCheck->setChecked(true);
        }
        else
            d->countryCheck->setValid(false);
    }
    d->countryCB->setEnabled(d->countryCheck->isChecked());

    blockSignals(false);
}

void XMPOrigin::applyMetadata(QByteArray& exifData, QByteArray& xmpData)
{
    KExiv2 exiv2Iface;
    exiv2Iface.setExif(exifData);
    exiv2Iface.setXmp(xmpData);

    if (d->dateCreatedCheck->isChecked())
    {
        exiv2Iface.setXmpTagString("Xmp.photoshop.DateCreated",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        exiv2Iface.setXmpTagString("Xmp.xmp.CreateDate",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        exiv2Iface.setXmpTagString("Xmp.exif.DateTimeCreated",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        exiv2Iface.setXmpTagString("Xmp.exif.DateTimeOriginal",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        exiv2Iface.setXmpTagString("Xmp.tiff.DateTime",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        exiv2Iface.setXmpTagString("Xmp.xmp.ModifyDate",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        exiv2Iface.setXmpTagString("Xmp.xmp.MetadataDate",
                                   getXMPCreationDate().toString("yyyy:MM:dd hh:mm:ss"));
        if (syncEXIFDateIsChecked())
        {
            exiv2Iface.setExifTagString("Exif.Image.DateTime",
                    getXMPCreationDate().toString(QString("yyyy:MM:dd hh:mm:ss")).toAscii());
        }
    }
    else
    {
        exiv2Iface.removeXmpTag("Xmp.photoshop.DateCreated");
        exiv2Iface.removeXmpTag("Xmp.xmp.CreateDate");
        exiv2Iface.removeXmpTag("Xmp.exif.DateTimeCreated");
        exiv2Iface.removeXmpTag("Xmp.exif.DateTimeOriginal");
        exiv2Iface.removeXmpTag("Xmp.tiff.DateTime");
        exiv2Iface.removeXmpTag("Xmp.xmp.ModifyDate");
        exiv2Iface.removeXmpTag("Xmp.xmp.MetadataDate");
    }

    if (d->dateDigitalizedCheck->isChecked())
        exiv2Iface.setXmpTagString("Xmp.exif.DateTimeDigitized",
                                   d->dateDigitalizedSel->dateTime().toString("yyyy:MM:dd hh:mm:ss"));
    else
        exiv2Iface.removeXmpTag("Xmp.exif.DateTimeDigitized");

    if (d->cityCheck->isChecked())
        exiv2Iface.setXmpTagString("Xmp.photoshop.City", d->cityEdit->text());
    else
        exiv2Iface.removeXmpTag("Xmp.photoshop.City");

    if (d->sublocationCheck->isChecked())
        exiv2Iface.setXmpTagString("Xmp.iptc.Location", d->sublocationEdit->text());
    else
        exiv2Iface.removeXmpTag("Xmp.iptc.Location");

    if (d->provinceCheck->isChecked())
        exiv2Iface.setXmpTagString("Xmp.photoshop.State", d->provinceEdit->text());
    else
        exiv2Iface.removeXmpTag("Xmp.photoshop.State");

    if (d->countryCheck->isChecked())
    {
        QString countryName = d->countryCB->currentText().mid(6);
        QString countryCode = d->countryCB->currentText().left(3);
        exiv2Iface.setXmpTagString("Xmp.iptc.CountryCode", countryCode);
        exiv2Iface.setXmpTagString("Xmp.photoshop.Country", countryName);
    }
    else if (d->countryCheck->isValid())
    {
        exiv2Iface.removeXmpTag("Xmp.iptc.CountryCode");
        exiv2Iface.removeXmpTag("Xmp.photoshop.Country");
    }

#if KEXIV2_VERSION >= 0x010000
    exifData = exiv2Iface.getExifEncoded();
#else
    exifData = exiv2Iface.getExif();
#endif

    xmpData  = exiv2Iface.getXmp();
}

}  // namespace KIPIMetadataEditPlugin
