/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2012-01-13
 * Description : progress manager
 *
 * Copyright (C) 2007-2012 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2004 Till Adam <adam at kde dot org>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "progressview.moc"

// Qt includes

#include <QApplication>
#include <QCloseEvent>
#include <QEvent>
#include <QFrame>
#include <QLabel>
#include <QLayout>
#include <QObject>
#include <QProgressBar>
#include <QPushButton>
#include <QScrollBar>
#include <QTimer>
#include <QToolButton>
#include <QMap>
#include <QPixmap>

// KDE includes

#include <kdebug.h>
#include <kdialog.h>
#include <khbox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kstandardguiitem.h>

// Local includes

#include "progressmanager.h"

namespace Digikam
{

static const int MAX_LABEL_WIDTH = 650;

class TransactionItem;

TransactionItemView::TransactionItemView(QWidget* parent, const char* name)
    : QScrollArea( parent )
{
    setObjectName( name );
    setFrameStyle( NoFrame );
    m_bigBox = new KVBox( this );
    setWidget( m_bigBox );
    setWidgetResizable( true );
    setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Fixed );
}

TransactionItem* TransactionItemView::addTransactionItem(ProgressItem* item, bool first)
{
    TransactionItem* ti = new TransactionItem(m_bigBox, item, first);
    m_bigBox->layout()->addWidget(ti);

    resize(m_bigBox->width(), m_bigBox->height());

    return ti;
}

void TransactionItemView::resizeEvent(QResizeEvent* event)
{
    // Tell the layout in the parent (progressview) that our size changed
    updateGeometry();

    QSize sz         = parentWidget()->sizeHint();
    int currentWidth = parentWidget()->width();

    // Don't resize to sz.width() every time when it only reduces a little bit
    if ( currentWidth < sz.width() || currentWidth > sz.width() + 100 )
    {
        currentWidth = sz.width();
    }
    parentWidget()->resize( currentWidth, sz.height() );

    QScrollArea::resizeEvent( event );
}

QSize TransactionItemView::sizeHint() const
{
    return minimumSizeHint();
}

QSize TransactionItemView::minimumSizeHint() const
{
    int f      = 2 * frameWidth();
    // Make room for a vertical scrollbar in all cases, to avoid a horizontal one
    int vsbExt = verticalScrollBar()->sizeHint().width();
    int minw   = topLevelWidget()->width() / 3;
    int maxh   = topLevelWidget()->height() / 2;
    QSize sz( m_bigBox->minimumSizeHint() );
    sz.setWidth( qMax( sz.width(), minw ) + f + vsbExt );
    sz.setHeight( qMin( sz.height(), maxh ) + f );
    return sz;
}

void TransactionItemView::slotLayoutFirstItem()
{
    //This slot is called whenever a TransactionItem is deleted, so this is a
    //good place to call updateGeometry(), so our parent takes the new size
    //into account and resizes.
    updateGeometry();

    /*
        The below relies on some details in Qt's behaviour regarding deleting
        objects. This slot is called from the destroyed signal of an item just
        going away. That item is at that point still in the  list of chilren, but
        since the vtable is already gone, it will have type QObject. The first
        one with both the right name and the right class therefor is what will
        be the first item very shortly. That's the one we want to remove the
        hline for.
    */
    TransactionItem* ti = m_bigBox->findChild<TransactionItem*>("TransactionItem");
    if ( ti )
    {
        ti->hideHLine();
    }
}

// ----------------------------------------------------------------------------

class TransactionItem::TransactionItemPriv
{
public:

    TransactionItemPriv() :
        progress(0),
        cancelButton(0),
        itemLabel(0),
        itemStatus(0),
        itemThumb(0),
        frame(0),
        item(0)
    {
    }

    QProgressBar* progress;
    QPushButton*  cancelButton;
    QLabel*       itemLabel;
    QLabel*       itemStatus;
    QLabel*       itemThumb;
    QFrame*       frame;
    ProgressItem* item;
};

TransactionItem::TransactionItem(QWidget* parent, ProgressItem* item, bool first)
    : KVBox(parent), d(new TransactionItemPriv)
{
    d->item = item;
    setSpacing(2);
    setMargin(2);
    setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed));

    d->frame = new QFrame(this);
    d->frame->setFrameShape(QFrame::HLine);
    d->frame->setFrameShadow(QFrame::Raised);
    d->frame->show();
    setStretchFactor(d->frame, 3);
    layout()->addWidget(d->frame);

    KHBox* h = new KHBox(this);
    h->setSpacing(5);
    layout()->addWidget(h);

    if (item->hasThumbnail())
    {
        d->itemThumb = new QLabel(h);
        d->itemThumb->setFixedSize(QSize(KIconLoader::SizeSmallMedium, KIconLoader::SizeSmallMedium));
        h->layout()->addWidget(d->itemThumb);
        h->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
    }

    d->itemLabel = new QLabel(fontMetrics().elidedText(item->label(), Qt::ElideRight, MAX_LABEL_WIDTH), h);
    h->layout()->addWidget(d->itemLabel);
    h->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed));

    d->progress = new QProgressBar(h);
    d->progress->setMaximum(100);
    d->progress->setValue(item->progress());
    h->layout()->addWidget(d->progress);

    if (item->canBeCanceled())
    {
        d->cancelButton = new QPushButton(SmallIcon("dialog-cancel"), QString(), h);
        d->cancelButton->setToolTip( i18n("Cancel this operation."));
        connect(d->cancelButton, SIGNAL( clicked()),
                this, SLOT(slotItemCanceled()));
        h->layout()->addWidget(d->cancelButton);
    }

    h = new KHBox(this);
    h->setSpacing(5);
    h->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed));
    layout()->addWidget(h);

    d->itemStatus = new QLabel(h);
    d->itemStatus->setTextFormat(Qt::RichText);
    d->itemStatus->setText(fontMetrics().elidedText(item->status(), Qt::ElideRight, MAX_LABEL_WIDTH));
    h->layout()->addWidget(d->itemStatus);
    if (first)
    {
        hideHLine();
    }
}

TransactionItem::~TransactionItem()
{
    delete d;
}

ProgressItem* TransactionItem::item() const
{
    return d->item;
}

void TransactionItem::hideHLine()
{
    d->frame->hide();
}

void TransactionItem::setProgress(int progress)
{
    d->progress->setValue(progress);
}

void TransactionItem::setItemComplete()
{
    d->item = 0;
}

void TransactionItem::setLabel(const QString& label)
{
    d->itemLabel->setText(fontMetrics().elidedText(label, Qt::ElideRight, MAX_LABEL_WIDTH));
}

void TransactionItem::setThumbnail(const QPixmap& thumb)
{
    d->itemThumb->setPixmap(thumb);
}

void TransactionItem::setStatus(const QString& status)
{
    d->itemStatus->setText(fontMetrics().elidedText(status, Qt::ElideRight, MAX_LABEL_WIDTH));
}

void TransactionItem::setTotalSteps(int totalSteps)
{
    d->progress->setMaximum(totalSteps);
}

void TransactionItem::slotItemCanceled()
{
    if ( d->item )
    {
        d->item->cancel();
    }
}

void TransactionItem::addSubTransaction(ProgressItem* item)
{
    Q_UNUSED(item);
}

// ---------------------------------------------------------------------------

class ProgressView::ProgressViewPriv
{
public:

    ProgressViewPriv() :
        wasLastShown(false),
        scrollView(0),
        previousItem(0)
    {
    }

    bool                                        wasLastShown;
    TransactionItemView*                        scrollView;
    TransactionItem*                            previousItem;
    QMap<const ProgressItem*, TransactionItem*> transactionsToListviewItems;
};

ProgressView::ProgressView(QWidget* alignWidget, QWidget* parent, const char* name)
    : OverlayWidget(alignWidget, parent, name), d(new ProgressViewPriv)
{
    setFrameStyle(QFrame::Panel | QFrame::Sunken);
    setAutoFillBackground(true);

    d->scrollView = new TransactionItemView( this, "ProgressScrollView" );
    layout()->addWidget( d->scrollView );

    // No more close button for now, since there is no more autoshow
    /*
        QVBox* rightBox = new QVBox( this );
        QToolButton* pbClose = new QToolButton( rightBox );
        pbClose->setAutoRaise(true);
        pbClose->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed ) );
        pbClose->setFixedSize( 16, 16 );
        pbClose->setIcon( KIconLoader::global()->loadIconSet( "window-close", KIconLoader::Small, 14 ) );
        pbClose->setToolTip( i18n( "Hide detailed progress window" ) );
        connect(pbClose, SIGNAL(clicked()), this, SLOT(slotClose()));
        QWidget* spacer = new QWidget( rightBox ); // don't let the close button take up all the height
        rightBox->setStretchFactor( spacer, 100 );
    */

    /*
    * Get the singleton ProgressManager item which will inform us of
    * appearing and vanishing items.
    */
    ProgressManager* pm = ProgressManager::instance();

    connect(pm, SIGNAL(progressItemAdded(ProgressItem*)),
            this, SLOT(slotTransactionAdded(ProgressItem*)));

    connect(pm, SIGNAL(progressItemCompleted(ProgressItem*)),
            this, SLOT(slotTransactionCompleted(ProgressItem*)));

    connect(pm, SIGNAL(progressItemProgress(ProgressItem*, unsigned int)),
            this, SLOT(slotTransactionProgress(ProgressItem*, unsigned int)));

    connect(pm, SIGNAL(progressItemStatus(ProgressItem*, const QString&)),
            this, SLOT(slotTransactionStatus(ProgressItem*, const QString&)));

    connect(pm, SIGNAL(progressItemLabel(ProgressItem*, const QString&)),
            this, SLOT(slotTransactionLabel(ProgressItem*, const QString&)));

    connect(pm, SIGNAL(progressItemUsesBusyIndicator(ProgressItem*, bool)),
            this, SLOT(slotTransactionUsesBusyIndicator(ProgressItem*, bool)));

    connect(pm, SIGNAL(progressItemThumbnail(ProgressItem*, const QPixmap&)),
            this, SLOT(slotTransactionThumbnail(ProgressItem*, const QPixmap&)));

    connect(pm, SIGNAL(showProgressView()),
            this, SLOT(slotShow()));
}

ProgressView::~ProgressView()
{
    // NOTE: no need to delete child widgets.
    delete d;
}

void ProgressView::closeEvent(QCloseEvent* e)
{
    e->accept();
    hide();
}

void ProgressView::slotTransactionAdded(ProgressItem* item)
{
    TransactionItem* parent = 0;
    if ( item->parent() )
    {
        if ( d->transactionsToListviewItems.contains( item->parent() ) )
        {
            parent = d->transactionsToListviewItems[ item->parent() ];
            parent->addSubTransaction( item );
        }
    }
    else
    {
        const bool first    = d->transactionsToListviewItems.empty();
        TransactionItem* ti = d->scrollView->addTransactionItem( item, first );
        if ( ti )
        {
            d->transactionsToListviewItems.insert( item, ti );
        }
        if ( first && d->wasLastShown )
        {
            QTimer::singleShot( 1000, this, SLOT( slotShow() ) );
        }
    }
}

void ProgressView::slotTransactionCompleted(ProgressItem* item)
{
    if ( d->transactionsToListviewItems.contains( item ) )
    {
        TransactionItem* ti = d->transactionsToListviewItems[item];
        d->transactionsToListviewItems.remove( item );
        ti->setItemComplete();
        QTimer::singleShot( 3000, ti, SLOT( deleteLater() ) );

        // see the slot for comments as to why that works
        connect ( ti, SIGNAL( destroyed() ),
                d->scrollView, SLOT( slotLayoutFirstItem() ) );
    }
    // This was the last item, hide.
    if ( d->transactionsToListviewItems.empty() )
    {
        QTimer::singleShot( 3000, this, SLOT( slotHide() ) );
    }
}

void ProgressView::slotTransactionCanceled(ProgressItem*)
{
}

void ProgressView::slotTransactionProgress(ProgressItem* item, unsigned int progress)
{
    if (d->transactionsToListviewItems.contains(item))
    {
        TransactionItem* ti = d->transactionsToListviewItems[item];
        ti->setProgress(progress);
    }
}

void ProgressView::slotTransactionStatus(ProgressItem* item, const QString& status)
{
    if (d->transactionsToListviewItems.contains(item))
    {
        TransactionItem* ti = d->transactionsToListviewItems[item];
        ti->setStatus(status);
    }
}

void ProgressView::slotTransactionLabel(ProgressItem* item, const QString& label )
{
    if ( d->transactionsToListviewItems.contains(item))
    {
        TransactionItem* ti = d->transactionsToListviewItems[item];
        ti->setLabel(label);
    }
}

void ProgressView::slotTransactionUsesBusyIndicator(ProgressItem* item, bool value)
{
    if (d->transactionsToListviewItems.contains(item))
    {
        TransactionItem* ti = d->transactionsToListviewItems[item];
        if (value)
        {
            ti->setTotalSteps(0);
        }
        else
        {
            ti->setTotalSteps(100);
        }
    }
}

void ProgressView::slotTransactionThumbnail(ProgressItem* item, const QPixmap& thumb)
{
    if (d->transactionsToListviewItems.contains(item))
    {
        TransactionItem* ti = d->transactionsToListviewItems[item];
        ti->setThumbnail(thumb);
    }
}

void ProgressView::slotShow()
{
    setVisible(true);
}

void ProgressView::slotHide()
{
    // check if a new item showed up since we started the timer. If not, hide
    if ( d->transactionsToListviewItems.isEmpty() )
    {
        setVisible(false);
    }
}

void ProgressView::slotClose()
{
    d->wasLastShown = false;
    setVisible(false);
}

void ProgressView::setVisible(bool b)
{
    OverlayWidget::setVisible(b);
    emit visibilityChanged(b);
}

void ProgressView::slotToggleVisibility()
{
    /* Since we are only hiding with a timeout, there is a short period of
    * time where the last item is still visible, but clicking on it in
    * the statusbarwidget should not display the dialog, because there
    * are no items to be shown anymore. Guard against that.
    */
    d->wasLastShown = isHidden();
    if ( !isHidden() || !d->transactionsToListviewItems.isEmpty() )
    {
        setVisible( isHidden() );
    }
}

} // namespace Digikam
