/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/logging_mutex.hpp
 *
 * $Id: logging_mutex.hpp,v 1.9 2005/06/23 09:54:29 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] the two macros
 * @ref DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD
 * @ref DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED
 *
 * These two macros come in four flavors:
 *
 * @arg DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 0 : Null-calls
 *
 * @arg DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 3 : Nesting is
 * dectectable, but not mutexing happens.
 *
 * @arg DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 1 : Nesting is not
 * detected, but the calls are mutexed. In particular, nested calls
 * will result in a deadlock.
 *
 * @arg DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 2 : Nesting is
 * dectectable and the calls are mutexed.
 *
 * @test none
 */
#ifndef DIAGNOSTICS__FRAME__LOGGING_MUTEX_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__LOGGING_MUTEX_HPP__INCLUDE_GUARD

#include <diagnostics/frame/platform_config.hpp>


////////////////////////////////////////////////////////////////////////////////
//
// No mutex
//
#if DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 0

#  define DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(NAME)     ((void)0)
#  define DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED(NAME) false

#endif



////////////////////////////////////////////////////////////////////////////////
//
// check for recursion but no mutex
//
#if DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 3

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;
class Lock_Guard
{
private:
	static int m_nesting_count;
public:
    inline Lock_Guard()   { ++m_nesting_count; }
    inline ~Lock_Guard()  { --m_nesting_count;}
    inline bool nesting() { return m_nesting_count!=1;}
};
INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

#  define DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(NAME) ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Lock_Guard NAME
#  define DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED(NAME) (NAME.nesting())

#endif




#if DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE == 1

////////////////////////////////////////////////////////////////////////////////
//
// Use ACE for non-recursive, multithreaded solution
//
#  if DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 1
#    include <ace/Synch.h>
DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;
class Lock_Guard
{
private:
    typedef ACE_Thread_Mutex Mutex_t;
    static Mutex_t *m_mutex;
public:
    inline Lock_Guard() {
		if(m_mutex==NULL) m_mutex=new Mutex_t();
		m_mutex->acquire();
    }
    inline ~Lock_Guard() {
		m_mutex->release();
    }
};
INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

#    define DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(NAME) ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Lock_Guard NAME
#    define DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED(NAME) false
#  endif


////////////////////////////////////////////////////////////////////////////////
//
// Use ACE for recursive, multithreaded solution
//
#  if DIAGNOSTICS_INTERNAL_SWITCH_MUTEX == 2
#    include <ace/Synch.h>
DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;
class Lock_Guard
{
private:
    typedef ACE_Recursive_Thread_Mutex Mutex_t;
    static Mutex_t *m_mutex;
	static int m_nesting_count;
public:
    inline Lock_Guard() {
		if(m_mutex==NULL) {
			m_mutex=new Mutex_t();
			m_nesting_count=0;
		}
		m_mutex->acquire();
		++m_nesting_count;
    }
    inline ~Lock_Guard() {
		--m_nesting_count;
		m_mutex->release();
    }
    inline bool nesting() {
		return m_nesting_count!=1;
	}
};
INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;
#    define DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(NAME) ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Lock_Guard NAME
#    define DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED(NAME) (NAME.nesting())
#  endif
#endif

#endif
// vim:ts=4:sw=4
