# Copyright 2009-2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Manuel de la Pena<manuel@canonical.com>
"""Test the database code mocking the db access."""

from couchdb.http import ResourceNotFound, PreconditionFailed
from mocker import MockerTestCase, ANY, KWARGS

from desktopcouch.application.server import DesktopDatabase
from desktopcouch.records.database import NoSuchDatabase


# disable pylint warnings that are common due to the nature of the tests
# pylint: disable=W0104, W0201, W0212
def set_database_expectations(testcase):
    """Set the expectations for the db creation."""
    testcase.server_class(testcase.uri, ctx=testcase.ctx,
        oauth_tokens=testcase.oauth_tokens)
    testcase.mocker.result(testcase.server_class)
    testcase.server_class.create(testcase.dbname)
    testcase.server_class[testcase.dbname]
    testcase.mocker.result(testcase.server_class)
    testcase.server_class.info()
    testcase.mocker.result({'update_seq': []})


def set_create_view_expectations(testcase):
    """Set the expectations for when a view is created."""
    # set expectations for the creation of the view
    testcase.views_factory(ANY, ANY, ANY, ANY)
    testcase.mocker.result(testcase.views_factory)
    testcase.views_factory.sync(testcase.server_class)
    # assert that is was added, shall we be mocking this?
    testcase.server_class[ANY]['views']
    testcase.mocker.result(testcase.server_class)
    ANY in testcase.server_class
    testcase.mocker.result(True)


def create_database(testcase):
    """Create a database to be used in the testcase."""
    testcase.database = DesktopDatabase(testcase.dbname, uri=testcase.uri,
        record_factory=testcase.record_factory, create=True,
        server_class=testcase.server_class,
        oauth_tokens=testcase.oauth_tokens, ctx=testcase.ctx,
        views_factory=testcase.views_factory)


class TestMockedCouchDatabaseCreateStates(MockerTestCase):
    def setUp(self):
        """Set up tests."""
        super(TestMockedCouchDatabaseCreateStates, self).setUp()
        self.record_factory = self.mocker.mock()
        self.server_class = self.mocker.mock()
        self.oauth_tokens = self.mocker.mock()
        self.ctx = self.mocker.mock()
        self.uri = 'uri'
        self.dbname = self._testMethodName

    def test_create_already_exists(self):
        """Access DB with create=True, when it already exists."""
        self.server_class(self.uri, ctx=self.ctx,
            oauth_tokens=self.oauth_tokens)
        self.mocker.result(self.server_class)
        self.server_class.create(self.dbname)
        self.mocker.throw(PreconditionFailed)

        self.server_class[self.dbname]
        self.mocker.result(self.server_class)
        self.server_class.info()
        self.mocker.result({'update_seq': []})

        self.mocker.replay()
        database = DesktopDatabase(self.dbname, uri=self.uri,
            record_factory=self.record_factory, create=True,
            server_class=self.server_class,
            oauth_tokens=self.oauth_tokens, ctx=self.ctx)

    def test_create_new(self):
        """Access DB with create=True, when it doesn't exist."""
        self.server_class(self.uri, ctx=self.ctx,
            oauth_tokens=self.oauth_tokens)
        self.mocker.result(self.server_class)
        self.server_class.create(self.dbname)
        self.mocker.result(self.server_class)
        self.server_class[self.dbname]
        self.mocker.result(self.server_class)
        info = self.mocker.mock()
        self.server_class.info()
        self.mocker.result(info)
        info["update_seq"]
        self.mocker.result({})
        self.mocker.replay()
        database = DesktopDatabase(self.dbname, uri=self.uri,
            record_factory=self.record_factory, create=True,
            server_class=self.server_class,
            oauth_tokens=self.oauth_tokens, ctx=self.ctx)

    def test_no_create_not_exists(self):
        """Access DB with create=False, when it already exists."""
        self.server_class(self.uri, ctx=self.ctx,
            oauth_tokens=self.oauth_tokens)
        self.mocker.result(self.server_class)

        self.server_class[self.dbname]
        self.mocker.throw(ResourceNotFound)

        self.mocker.replay()
        self.assertRaises(NoSuchDatabase,
            DesktopDatabase, self.dbname, uri=self.uri,
            record_factory=self.record_factory, create=False,
            server_class=self.server_class,
            oauth_tokens=self.oauth_tokens, ctx=self.ctx)

    def test_no_create_already_exists(self):
        """Access DB with create=False, when it doesn't exist."""
        self.server_class(self.uri, ctx=self.ctx,
            oauth_tokens=self.oauth_tokens)
        self.mocker.result(self.server_class)
        self.server_class[self.dbname]
        self.mocker.result(self.server_class)
        info = self.mocker.mock()
        self.server_class.info()
        self.mocker.result(info)
        info["update_seq"]
        self.mocker.result({})

        self.mocker.replay()
        database = DesktopDatabase(self.dbname, uri=self.uri,
            record_factory=self.record_factory, create=False,
            server_class=self.server_class,
            oauth_tokens=self.oauth_tokens, ctx=self.ctx)


class TestMockedCouchDatabaseDeprecated(MockerTestCase):
    """Test the deprecated API."""

    def setUp(self):
        """Setup each test."""
        super(TestMockedCouchDatabaseDeprecated, self).setUp()
        # set mocked objects
        self.record_factory = self.mocker.mock()
        self.server_class = self.mocker.mock()
        self.oauth_tokens = self.mocker.mock()
        self.ctx = self.mocker.mock()
        self.views_factory = self.mocker.mock()
        # set not mocked required data
        self.uri = 'uri'
        self.dbname = self._testMethodName
        self.database = None

    def set_database_expectations(self):
        """Set the expectations for the creation of the db."""
        self.server_class(self.uri, ctx=self.ctx,
            oauth_tokens=self.oauth_tokens)
        self.mocker.result(self.server_class)
        self.dbname in self.server_class
        self.mocker.result(True)
        self.server_class[self.dbname]
        self.mocker.result(self.server_class)
        self.server_class.info()
        self.mocker.result({'update_seq': []})

    def test_get_records_by_record_type_saved_view(self):
        """Test getting multiple records by type."""
        record_type = 'test.com'
        set_database_expectations(self)
        # set expectations for the view_exists operation
        self.server_class[ANY]['views']
        self.mocker.result(self.server_class)
        ANY in self.server_class
        self.mocker.result(True)
        # set expectations for the execution
        self.server_class.view(ANY, KWARGS)
        self.mocker.result({record_type: record_type})
        self.mocker.replay()
        create_database(self)
        records = self.database.get_records(record_type=record_type,
            create_view=True)
        self.assertEqual(record_type, records)

    def test_get_records_by_record_type_new_view(self):
        """Test getting multiple record and create the view."""
        record_type = 'test.com'
        set_database_expectations(self)
        # set expectations for the view_exists operation
        self.server_class[ANY]['views']
        self.mocker.result(self.server_class)
        ANY in self.server_class
        self.mocker.result(False)
        set_create_view_expectations(self)
        # set expectations for the execution
        self.server_class.view(ANY, KWARGS)
        self.mocker.result({record_type: record_type})
        self.mocker.replay()
        create_database(self)
        records = self.database.get_records(record_type=record_type,
            create_view=True)
        self.assertEqual(record_type, records)

    def test_get_records_by_record_type_new_view_not_create(self):
        """Test getting the records when the view does not exist."""
        record_type = 'test.com'
        set_database_expectations(self)
        # assert that is was added, shall we be mocking this?
        self.server_class[ANY]['views']
        self.mocker.result(self.server_class)
        ANY in self.server_class
        self.mocker.result(True)
        # set expectations for the execution
        self.server_class.view(ANY, KWARGS)
        self.mocker.result({record_type: record_type})
        self.mocker.replay()
        create_database(self)
        self.assertRaises(KeyError, self.database.get_records,
            (record_type, False))

    def test_get_records_by_record_type_saved_view_false_create(self):
        """Test calling the view and do not create it."""
        record_type = 'test.com'
        set_database_expectations(self)
        # set expectations for the view_exists operation
        self.server_class[ANY]['views']
        self.mocker.result(self.server_class)
        ANY in self.server_class
        self.mocker.result(True)
        # set expectations for the execution
        self.server_class.view(ANY, KWARGS)
        self.mocker.result({record_type: record_type})
        self.mocker.replay()
        create_database(self)
        records = self.database.get_records(record_type=record_type,
            create_view=False)
        self.assertEqual(record_type, records)


class TestMockedDesktopDatabase(MockerTestCase):
    """Test case for DesktopDatabase."""

    def setUp(self):
        """setup each test"""
        super(TestMockedDesktopDatabase, self).setUp()
        # set mocked objects
        self.record_factory = self.mocker.mock()
        self.server_class = self.mocker.mock()
        self.oauth_tokens = self.mocker.mock()
        self.ctx = self.mocker.mock()
        self.views_factory = self.mocker.mock()
        # set not mocked required data
        self.uri = 'uri'
        self.dbname = self._testMethodName
        self.database = None

    def _set_get_all_records_expectations(self, record_type, view_result,
        records):
        """Set the common expectations for the get_all_records call."""
        # replace the transform to records funtion which is used to
        # return records instead of view results
        transform_to_records = self.mocker.replace(
            'desktopcouch.records.database.transform_to_records')
        self.server_class.view(ANY, KWARGS)
        self.mocker.result(view_result)
        transform_to_records(record_type)
        self.mocker.result(records)

    def test_database_not_exists(self):
        """Test that the database does not exist."""
        # set expectations so that the server class returns that
        # the db does ot exist.
        self.server_class(self.uri, ctx=self.ctx,
            oauth_tokens=self.oauth_tokens)
        self.mocker.result(self.server_class)
        self.server_class[self.dbname]
        self.mocker.throw(ResourceNotFound)
        self.mocker.replay()
        self.assertRaises(NoSuchDatabase, DesktopDatabase, self.dbname,
            uri=self.uri, record_factory=self.record_factory, create=False,
            server_class=self.server_class, oauth_tokens=self.oauth_tokens,
            ctx=self.ctx, views_factory=self.views_factory)

    def test_get_records_by_record_type_present_view(self):
        """Test getting mutliple records by type with a present view."""
        record_type = 'test.com'
        view_result = {record_type: record_type}
        records = ['first', 'second']
        set_database_expectations(self)
        self._set_get_all_records_expectations(record_type, view_result,
            records)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(records,
            self.database.get_all_records(record_type=record_type))

    def test_get_records_by_record_type_missing_view(self):
        """Test getting mutliple records by type with a present view."""
        record_type = 'test.com'
        view_result = {record_type: record_type}
        records = ['first', 'second']
        set_database_expectations(self)
        self.server_class.view(ANY, KWARGS)
        # raise exception so that the view has to be created.
        self.mocker.throw(ResourceNotFound)
        set_create_view_expectations(self)
        # set the expectations to be used after the creation of the view
        self._set_get_all_records_expectations(record_type, view_result,
            records)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(records,
            self.database.get_all_records(record_type=record_type))

    def test_get_present_record_no_attachments(self):
        """Test getting a record."""
        record_id = 'test_record'
        data = {record_id: record_id}
        record_mock = self.mocker.mock()
        set_database_expectations(self)
        self.server_class[record_id]
        self.mocker.result(data)
        self.record_factory(data=data)
        self.mocker.result(record_mock)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(record_mock, self.database.get_record(record_id))

    def test_get_deleted_record(self):
        """Test getting a deleted record."""
        record_id = 'random_id'
        set_database_expectations(self)
        self.server_class[record_id]
        self.mocker.throw(ResourceNotFound)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(None, self.database.get_record(record_id))

    def test_put_record_no_attachments_or_id(self):
        """Test putting a record with no attachments."""
        record_id = 'random_id'
        record_mock = self.mocker.mock()
        # replace the base_n funtion which is used to
        # return the string representation of the uuid
        base_n = self.mocker.replace(
            'desktopcouch.records.database.base_n')
        set_database_expectations(self)
        record_mock.record_id
        self.mocker.result(None)
        base_n(ANY, ANY)
        self.mocker.result(record_id)
        record_mock.record_id = record_id
        record_mock._detached
        self.mocker.result([])
        record_mock.list_attachments()
        self.mocker.result([])
        record_mock._data
        self.mocker.result({})
        self.server_class[record_id] = {}
        self.mocker.replay()
        create_database(self)
        self.assertEqual(record_id, self.database.put_record(record_mock))

    def test_put_record_no_attachments(self):
        """Test putting a record with no attachments."""
        record_id = 'random_id'
        record_mock = self.mocker.mock()
        set_database_expectations(self)
        record_mock.record_id
        self.mocker.result(record_id)
        record_mock._detached
        self.mocker.result([])
        record_mock.list_attachments()
        self.mocker.result([])
        record_mock._data
        self.mocker.result({})
        self.server_class[record_id] = {}
        self.mocker.replay()
        create_database(self)
        self.assertEqual(record_id, self.database.put_record(record_mock))

    def test_put_record_with_attachments(self):
        """Test putting a record with attachments."""
        record_id = 'random_id'
        record_mock = self.mocker.mock()
        set_database_expectations(self)
        record_mock.record_id
        self.mocker.result(record_id)
        record_mock._data
        self.mocker.result({})
        record_mock._detached
        self.mocker.result(['to_delete'])
        self.server_class.delete_attachment({}, 'to_delete')
        record_mock.list_attachments()
        self.mocker.result(['to_add'])
        record_mock.attachment_data('to_add')
        self.mocker.result(('data', 'type'))
        self.server_class.put_attachment({}, 'data', 'to_add', 'type')
        self.server_class[record_id] = {}
        self.mocker.replay()
        create_database(self)
        self.assertEqual(record_id, self.database.put_record(record_mock))

    def test_put_records_batch(self):
        """Test putting a batch of records."""
        records = {'1': (self.mocker.mock(), {'name': '1'}),
                   '2': (self.mocker.mock(), {'name': '2'})}
        put_result = []
        for record_id in records:
            put_result.append((True, record_id, record_id))
        set_database_expectations(self)
        for record_id in records:
            records[record_id][0].record_id
            self.mocker.result(record_id)
            # TODO: we currently call _data twice, we should get
            # this number down
            records[record_id][0]._data
            self.mocker.result(records[record_id][1])
            records[record_id][0]._data
            self.mocker.result(records[record_id][1])
            records[record_id][0]._detached
            self.mocker.result(['to_delete'])
            self.server_class.delete_attachment(
                records[record_id][1], 'to_delete')
            records[record_id][0].list_attachments()
            self.mocker.result(['to_add'])
            records[record_id][0].attachment_data('to_add')
            self.mocker.result(('data', 'type'))
            self.server_class.put_attachment({'_id': record_id,
                '_rev': record_id}, 'data', 'to_add', 'type')
        self.server_class.update([records[record_id][1]
            for record_id in records])
        self.mocker.result(put_result)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(put_result, self.database.put_records_batch(
            [records[record_id][0] for record_id in records]))

    def test_delete_record_no_attachments(self):
        """Test deleting a record with no attachments."""
        # replace the base_n funtion which is used to
        # return the string representation of the uuid
        base_n = self.mocker.replace(
            'desktopcouch.records.database.base_n')
        # replace deepcopy because it is used to create a duplicate of
        # the record
        deepcopy = self.mocker.replace('copy.deepcopy')
        record_id = 'test_record'
        data = {record_id: record_id}
        record_mock = self.mocker.mock()
        trash_mock = self.mocker.mock()
        set_database_expectations(self)
        self.server_class[record_id]
        self.mocker.result(data)
        self.record_factory(data=data)
        self.mocker.result(record_mock)
        deepcopy(record_mock)
        self.mocker.result(record_mock)
        base_n(ANY, ANY)
        self.mocker.result(record_id)
        record_mock.record_id = record_id
        del record_mock._data['_rev']
        del record_mock._data['_attachments']
        record_mock.application_annotations['desktopcouch'] = ANY
        del self.server_class[record_id]
        trash_mock.put_record(record_mock)
        self.mocker.replay()
        create_database(self)
        self.database._dctrash = trash_mock
        self.database.delete_record(record_id)

    def test_delete_record_not_exist(self):
        """Nonexistent keys should not fail and should return None as
        deleted record id."""
        record_id = 'notexist'
        set_database_expectations(self)
        self.server_class[record_id]
        self.mocker.throw(ResourceNotFound)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(None, self.database.delete_record(record_id))

    def test_record_exists(self):
        """Test checking whether a record exists."""
        record_id = 'test_record'
        set_database_expectations(self)
        record_id in self.server_class
        self.mocker.result(True)
        record_id in self.server_class
        self.mocker.result(False)
        self.mocker.replay()
        create_database(self)
        self.assertTrue(self.database.record_exists(record_id))
        self.assertFalse(self.database.record_exists(record_id))

    def test_update_fields(self):
        """Test the update_fields method."""
        dictionary = {'record_number': 0, 'field1': 1, 'field2': 2}
        record_id = 'test_record'
        set_database_expectations(self)
        self.server_class[record_id]
        self.mocker.result(dictionary)
        self.server_class[record_id] = {'record_number': 0, 'field1': 11,
             'field2': 2}
        self.mocker.replay()
        create_database(self)
        # no assertions are required because mocker will make sure that
        # the new saved dict matches the expected one.
        self.database.update_fields(record_id, {'field1': 11})

    def test_delete_not_present_view(self):
        """Test deleting a view that is not in the db."""
        view_name = 'my_view'
        design_doc = 'my_design'
        doc_id = "_design/%s" % design_doc
        set_database_expectations(self)
        self.server_class[doc_id]['views']
        self.mocker.throw(ResourceNotFound)
        self.mocker.replay()
        create_database(self)
        self.assertRaises(KeyError, self.database.delete_view, view_name,
            design_doc)

    def test_delete_present_view_not_last(self):
        """Test deleting a view that is not the last in the doc."""
        view_name = 'my_view'
        design_doc = 'my_design'
        doc_id = "_design/%s" % design_doc
        container_mock = self.mocker.mock()
        key_mock = self.mocker.mock()
        value_mock = self.mocker.mock()
        iteritems = [(key_mock, value_mock)]
        set_database_expectations(self)
        self.server_class[doc_id]['views']
        self.mocker.result(container_mock)
        container_mock.pop(view_name)
        self.mocker.result(view_name)
        len(container_mock)
        self.mocker.result(3)
        container_mock.iteritems()
        self.mocker.result(iteritems)
        value_mock.get('map')
        self.mocker.result('map')
        value_mock.get('reduce')
        self.mocker.result('reduce')
        self.views_factory(design_doc, key_mock, 'map', 'reduce')
        self.mocker.result(container_mock)
        self.views_factory.sync_many(ANY, ANY, remove_missing=True)
        # set expectations for the view_exists operation
        self.server_class[ANY]['views']
        self.mocker.result(self.server_class)
        ANY in self.server_class
        self.mocker.result(False)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(view_name, self.database.delete_view(
            view_name, design_doc))

    def test_delete_present_view_is_last(self):
        """Test deleting a view that is the last of the doc."""
        view_name = 'my_view'
        design_doc = 'my_design'
        doc_id = "_design/%s" % design_doc
        container_mock = self.mocker.mock()
        set_database_expectations(self)
        self.server_class[doc_id]['views']
        self.mocker.result(container_mock)
        container_mock.pop(view_name)
        self.mocker.result(view_name)
        len(container_mock)
        self.mocker.result(0)
        del self.server_class[doc_id]
        # set expectations for the view_exists operation
        self.server_class[ANY]['views']
        self.mocker.result(self.server_class)
        ANY in self.server_class
        self.mocker.result(False)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(view_name, self.database.delete_view(
            view_name, design_doc))

    def test_list_views_no_design_doc(self):
        """Test the list views when they do not exist.."""
        design_doc = 'my_design'
        doc_id = "_design/%s" % design_doc
        set_database_expectations(self)
        self.server_class[doc_id]['views']
        self.mocker.throw(KeyError)
        self.mocker.replay()
        create_database(self)
        self.assertEqual([], self.database.list_views(design_doc))

    def test_list_views(self):
        """Test the list views."""
        design_doc = 'my_design'
        doc_id = "_design/%s" % design_doc
        views = ['one', 'two', 'three']
        set_database_expectations(self)
        self.server_class[doc_id]['views']
        self.mocker.result(views)
        self.mocker.replay()
        create_database(self)
        self.assertEqual(views, self.database.list_views(design_doc))
