/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      Neil Jagdish Patel <neil.patel@canonical.com>
 *      Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
/**
 * SECTION:dee-peer
 * @short_description: Finds other objects with the same swarm-name on the bus.
 * @include: dee.h
 *
 * #DeePeer allows you to build objects that can rendevouz on DBus
 * without the need for an central registration service. Think of it like
 * peer-to-peer for your application. The DBus session bus will also implicitly
 * elect a swarm leader - namely the one owning the swarm name on the bus, but
 * it's up to the consumer of this API to determine whether swarm leadership has
 * any concrete responsibilities associated.
 *
 * Peers find eachother through a well-known "swarm-name", which is a
 * well known DBus name, such as: org.myapp.MyPeers. Choose a namespaced
 * name that would not normally be used outside of your program.
 *
 * For example:
 * <informalexample><programlisting>
 * {
 *   DeePeer *peer;
 *
 *   peer = g_object_new (DBUS_TYPE_PEER,
 *                        "swarm-name", "org.myapp.MyPeers",
 *                        NULL);
 *
 *   g_signal_connect (peer, "peer-found",
 *                     G_CALLBACK (on_peer_found), NULL);
 *   g_signal_connect (peer, "peer-lost",
 *                     G_CALLBACK (on_peer_lost), NULL);
 * }
 * </programlisting></informalexample>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gio/gio.h>

#include "dee-peer.h"
#include "dee-marshal.h"
#include "trace-log.h"

G_DEFINE_TYPE (DeePeer, dee_peer, G_TYPE_OBJECT)

#define DEE_PEER_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_PEER, DeePeerPrivate))

#define _DeePeerIter GSequenceIter

/**
 * DeePeerPrivate:
 *
 * Ignore this structure.
 **/
struct _DeePeerPrivate
{
  GDBusConnection *connection;

  /* Used as a hash set with the unique addresses of the peers.
   * This hash table must be protected from concurrent access,
   * since it's used in the GDBus message dispatch thread */
  GHashTable *peers;
  
  /* A List with the string-formatted match rules we've installed */
  GSList *match_rules;
  
  /* The GDBus filter id, so we can uninstall our message filter again */
  guint   filter_id;
  
  /* GDBus id for the DBus signal subscriptions */
  guint   dbus_signals_id;

  /* The GDBus name owner id for g_bus_own_name() */
  guint   name_owner_id;

  /* The GDBus name watcher id from g_bus_watch_name() */
  guint name_watcher_id;

  const gchar *own_name;
  gchar       *swarm_name;
  gchar       *swarm_path;
  gchar       *swarm_leader;

  gboolean connected;
  gboolean is_swarm_leader;
  gboolean has_been_leader;
  gboolean is_first_name_check;

  /* if priv->head_count != NULL it indicates that we are in
   * "head counting mode" in which case priv->head_count_source will be a
   * GSource id for a timeout that completes the head count */
  GSList *head_count;
  guint   head_count_source;

  /* Protecting the priv->peers table from concurrent access in
   * the GDBus message dispatch thread */
  GMutex *lock;
};

/* Globals */
enum
{
  PROP_0,
  PROP_SWARM_NAME,
  PROP_SWARM_LEADER
};

enum
{
  CONNECTED,
  PEER_FOUND,
  PEER_LOST,

  LAST_SIGNAL
};

static guint32 _peer_signals[LAST_SIGNAL] = { 0 };

/* Forwards */
static void                 remove_match_rule        (GDBusConnection *conn,
                                                      const gchar     *rule);

static void                 emit_peer_found          (DeePeer    *self,
                                                      const gchar *name);
                                                      
static void                 on_bus_acquired          (GDBusConnection *connection,
                                                      const gchar     *name,
                                                      gpointer         user_data);

static void                 on_leadership_lost       (GDBusConnection *connection,
                                                      const gchar     *name,
                                                      gpointer         user_data);

static void                 on_leadership_acquired   (GDBusConnection *connection,
                                                      const gchar     *name,
                                                      gpointer         user_data);

static void                 on_leadership_changed    (GDBusConnection *connection,
                                                      const gchar     *name,
                                                      const gchar     *name_owner,
                                                      gpointer         user_data);

static void                 on_join_received         (DeePeer    *self,
                                                      const gchar *peer_address);

static void                 on_bye_received          (DeePeer    *self,
                                                      const gchar *peer_address);

static void                 on_ping_received         (DeePeer    *self,
                                                      const gchar *leader_address);

static void                 on_pong_received         (DeePeer    *self,
                                                      const gchar *peer_address);

static void                 on_list_received         (GObject      *source_object,
                                                      GAsyncResult *res,
                                                      gpointer      user_data);

static void                 set_swarm_name           (DeePeer    *self,
                                                      const gchar *swarm_name);

static void                 emit_ping                (DeePeer    *self);

static void                 emit_pong                (DeePeer    *self);

static void                 on_dbus_peer_signal      (GDBusConnection *connection,
                                                      const gchar     *sender_name,
                                                      const gchar     *object_path,
                                                      const gchar     *interface_name,
                                                      const gchar     *signal_name,
                                                      GVariant        *parameters,
                                                      gpointer         user_data);

static GDBusMessage*        gdbus_message_filter    (GDBusConnection *connection,
                                                     GDBusMessage    *message,
                                                     gboolean         incoming,
                                                     gpointer         user_data);

/* GObject methods */
static void
dee_peer_finalize (GObject *object)
{
  DeePeerPrivate *priv;
  GSList *match_iter;

  priv = DEE_PEER (object)->priv;

  /* Remove match rules from the bus, and free the string repr. of the rule  */
  if (priv->connection)
    {
      /* Uninstall filter.
       * Implementation note: We must remove the filter and signal listener
       * _before_ dropping the swarm name because gdbus currently does a
       * sync dbus call to release the name which makes us race against
       * getting a NameOwnerChanged */
      g_dbus_connection_remove_filter (priv->connection, priv->filter_id);
      
      for (match_iter = priv->match_rules;
           match_iter != NULL;
           match_iter = match_iter->next)
        {
          remove_match_rule (priv->connection, match_iter->data);
          g_free (match_iter->data);
        }

      /* Stop listening for signals */
      if (priv->dbus_signals_id != 0)
        {
          g_dbus_connection_signal_unsubscribe (priv->connection,
                                                priv->dbus_signals_id);
          priv->dbus_signals_id = 0;
        }

      g_object_unref (priv->connection);
      priv->connection = NULL;
    }

  /* Stop trying to own the swarm name.
   * See implementation note above */
  if (priv->name_owner_id != 0)
    {
      g_bus_unown_name (priv->name_owner_id);
      priv->name_owner_id = 0;
    }

  /* Stop listening for swarm leadership changes */
  if (priv->name_watcher_id != 0)
    {
      g_bus_unwatch_name (priv->name_watcher_id);
      priv->name_watcher_id = 0;
    }

  /* Free resources */
  if (priv->swarm_name)
    {
      g_free (priv->swarm_name);
      priv->swarm_name = NULL;
    }
  if (priv->swarm_path)
    {
      g_free (priv->swarm_path);
      priv->swarm_path = NULL;
    }
  if (priv->swarm_leader)
      {
        g_free (priv->swarm_leader);
        priv->swarm_leader = NULL;
      }
  if (priv->peers)
    {
      g_hash_table_destroy (priv->peers);
      priv->peers = NULL;
    }
  if (priv->match_rules)
    {
      g_slist_free (priv->match_rules);
      priv->match_rules = NULL;
    }
  if (priv->lock != NULL)
    {
      g_mutex_free (priv->lock);
      priv->lock = NULL;
    }
  if (priv->head_count != NULL)
    {
      g_slist_foreach(priv->head_count, (GFunc) g_free, NULL);
      g_slist_free (priv->head_count);
      priv->head_count = NULL;
    }
  if (priv->head_count_source != 0)
    {
      g_source_remove (priv->head_count_source);
      priv->head_count_source = 0;
    }
  
  G_OBJECT_CLASS (dee_peer_parent_class)->finalize (object);
}

static void
dee_peer_constructed (GObject *self)
{
  DeePeerPrivate *priv;
  gpointer       *weak_self;

  priv = DEE_PEER (self)->priv;
  
  if (priv->swarm_name == NULL)
    {
      g_critical ("DeePeer created without a swarm name. You must specify "
                  "a non-NULL swarm name");
      return;
    }
  
  /* We need to use a weak pointer on self because g_bus_own_name() can
   * not be cancelled and we might end up finalizing @self before obtaining
   * a connection to the bus. And, no, adding a temporary ref on self is not
   * a good idea since we still can't know if we've been cancelled and we'd
   * just make the peer live on; happily ever after.
   * With a weak ref we can check if self == NULL and take that as a
   * cancellation.
   */
  weak_self = g_new0 (gpointer, 1);
  *weak_self = self;
  g_object_add_weak_pointer (self, weak_self);

  /* Contend to be swarm leaders. Pick me! Pick me! */
  priv->name_owner_id = g_bus_own_name (G_BUS_TYPE_SESSION,
                                        priv->swarm_name,      /* name to own */
                                        G_BUS_NAME_OWNER_FLAGS_ALLOW_REPLACEMENT,
                                        on_bus_acquired,
                                        on_leadership_acquired,
                                        on_leadership_lost,
                                        weak_self,               /* user data */
                                        NULL);                   /* free func */

  /* Listen for changes in leadership */
  priv->name_watcher_id = g_bus_watch_name(G_BUS_TYPE_SESSION,
                                           priv->swarm_name,
                                           G_BUS_NAME_WATCHER_FLAGS_NONE,
                                           on_leadership_changed,
                                           NULL, /* name vanished cb */
                                           self, /* user data */
                                           NULL); /* user data free func */
}


static void
dee_peer_set_property (GObject       *object,
                       guint          id,
                       const GValue  *value,
                       GParamSpec    *pspec)
{
  DeePeerPrivate *priv = DEE_PEER (object)->priv;
  
  
  switch (id)
    {
    case PROP_SWARM_NAME:
      set_swarm_name (DEE_PEER (object), g_value_get_string (value));
      break;
    case PROP_SWARM_LEADER:
      g_free (priv->swarm_leader);
      priv->swarm_leader = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_peer_get_property (GObject     *object,
                        guint        id,
                        GValue      *value,
                        GParamSpec  *pspec)
{
  switch (id)
    {
    case PROP_SWARM_NAME:
      g_value_set_string (value, DEE_PEER (object)->priv->swarm_name);
      break;
    case PROP_SWARM_LEADER:
      g_value_set_string (value, DEE_PEER (object)->priv->swarm_leader);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_peer_class_init (DeePeerClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = dee_peer_finalize;
  obj_class->set_property = dee_peer_set_property;
  obj_class->get_property = dee_peer_get_property;
  obj_class->constructed  = dee_peer_constructed;


  /* Add Signals */

  /**
   * DeePeer::peer-found:
   * @self: the #DeePeer on which the signal is emitted
   * @name: the DBus name of the object found
   *
   * Connect to this signal to be notified of existing and new peers that are
   *   in your swarm.
   **/
  _peer_signals[PEER_FOUND] =
    g_signal_new ("peer-found",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeePeerClass, peer_found),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DeePeer::peer-lost:
   * @self: the #DeePeer on which the signal is emitted
   * @name: the DBus name of the object that disconnected
   *
   * Connect to this signal to be notified when peers disconnect from the swarm
   **/
  _peer_signals[PEER_LOST] =
    g_signal_new ("peer-lost",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DeePeerClass, peer_lost),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);
  
  /* Add properties */
  /**
   * DeePeer::swarm-name:
   *
   * The name of the swarm that this peer is connected to. All swarm members
   * will try and own this name on the session bus. The one owning the name
   * is the swarm leader.
   */
  pspec = g_param_spec_string ("swarm-name", "Swarm Name",
                               "Well-known name to find other peers with",
                               NULL,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT
                               | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_SWARM_NAME, pspec);

  /**
   * DeePeer::swarm-leader:
   *
   * The name of the swarm that this peer is connected to. All swarm members
   * will try and own this name on the session bus. The one owning the name
   * is the swarm leader.
   **/
  pspec = g_param_spec_string ("swarm-leader", "Swarm Leader",
                               "Unique DBus address of the swarm leader",
                               NULL,
                               G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_SWARM_LEADER, pspec);

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeePeerPrivate));
}

static void
dee_peer_init (DeePeer *peer)
{
  DeePeerPrivate *priv;

  priv = peer->priv = DEE_PEER_GET_PRIVATE (peer);

  priv->swarm_name = NULL;
  priv->swarm_leader = NULL;
  priv->own_name = NULL;
  priv->match_rules = NULL;
  priv->peers = g_hash_table_new_full (g_str_hash,
                                       g_str_equal,
                                       (GDestroyNotify) g_free,
                                       NULL);
  
  priv->connected = FALSE;
  priv->is_swarm_leader = FALSE;
  priv->has_been_leader = FALSE;
  priv->is_first_name_check = TRUE;

  priv->lock = g_mutex_new ();

  priv->head_count_source = 0;
}

/* Private Methods */


/* Async callback for com.canonical.Dee.Peer.List */
static void
on_list_received (GObject      *source_object,
                  GAsyncResult *res,
                  gpointer      user_data)
{
  DeePeer        *self = DEE_PEER (user_data);
  DeePeerPrivate *priv;
  GHashTable     *peers;
  GSList         *new_peers, *iter;
  guint           i;
  GVariant       *val, *_val;
  const gchar    **names;
  gsize           n_names;
  GError         *error;
  GHashTableIter  hiter;
  gpointer        hkey, hval;

  g_return_if_fail (DEE_IS_PEER (self));
  priv = self->priv;

  error = NULL;
  _val = g_dbus_connection_call_finish (priv->connection, res, &error);
  if (error != NULL)
    {
      g_warning ("%s: Unable to list peers: %s", G_STRLOC, error->message);
      g_error_free (error);
      return;
    }
  
  /* Unpack the wrapping struct from the reply */
  val = g_variant_get_child_value (_val, 0);
  g_variant_unref (_val);

  names = g_variant_get_strv (val, &n_names);
  trace_object (self, "Got list of %d peers", n_names);
  
  /* We diff the current list of peers against the new list
   * and emit signals accordingly. New peers are added to new_peers,
   * and lost peers will remain in priv->peers: */
  new_peers = NULL;
  peers = g_hash_table_new_full (g_str_hash,
                                 g_str_equal,
                                 (GDestroyNotify)g_free,
                                 NULL);                                     
  for (i = 0; i < n_names; i++)
    {
      g_hash_table_insert (peers, g_strdup (names[i]), NULL);
      if (!g_hash_table_remove (priv->peers, names[i]))
        {
          /* The peer was not previously known */
          new_peers = g_slist_prepend (new_peers, (gchar *) names[i]);
        }      
    }

  /* Signal about lost peers */
  g_hash_table_iter_init (&hiter, priv->peers);
  while (g_hash_table_iter_next (&hiter, &hkey, &hval))
    {
      g_signal_emit (self, _peer_signals[PEER_LOST], 0, hkey);
    }

  /* Signal about new peers */
  for (iter = new_peers; iter; iter = iter->next)
    {
      emit_peer_found (self, (const gchar*)iter->data);
    }

  /* The return value of g_variant_get_strv() is a shallow copy */
  g_free (names);
  g_variant_unref (val);

  /* Free just the array, not the strings - they are owned by 'peers' now */
  g_slist_free (new_peers);
  g_hash_table_destroy (priv->peers);
  priv->peers = peers;
}

/* Install a DBus match rule, async, described by a printf-like format string.
 * The match rule will be properly retracted when @self is finalized */
static void
install_match_rule (DeePeer *self, const char *rule, ...)
{
  DeePeerPrivate *priv;
  gchar          *f_rule;
  va_list         args;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (rule != NULL);

  priv = self->priv;
  
	va_start (args, rule);
  f_rule = g_strdup_vprintf (rule, args);
  va_end (args);

  /* By setting the error argument to NULL libdbus will use async mode
   * for adding the match rule. We want that. */
  g_dbus_connection_call (priv->connection,
                          "org.freedesktop.DBus",
                          "/org/freedesktop/dbus",
                          "org.freedesktop.DBus",
                          "AddMatch",
                          g_variant_new ("(s)", f_rule),
                          NULL, /* reply type */
                          G_DBUS_CALL_FLAGS_NONE,
                          -1,
                          NULL,  /* cancellable */
                          NULL,  /* callback */
                          NULL); /* user_data */
  
  priv->match_rules = g_slist_prepend (priv->match_rules, f_rule);
}

static void
remove_match_rule (GDBusConnection *conn, const gchar *rule)
{
  g_dbus_connection_call (conn,
                          "org.freedesktop.DBus",
                          "/org/freedesktop/dbus",
                          "org.freedesktop.DBus",
                          "RemoveMatch",
                          g_variant_new ("(s)", rule),
                          NULL, /* reply type */
                          G_DBUS_CALL_FLAGS_NONE,
                          -1,
                          NULL,  /* cancellable */
                          NULL,  /* callback */
                          NULL); /* user_data */
}

/* Public Methods */

/**
 * dee_peer_new
 * @swarm_name: The name of the swarm to join.
 *              Fx &quot;org.example.DataProviders&quot;
 *
 * Create a new #DeePeer. The peer will immediately connect to the swarm
 * and start the peer discovery.
 *
 * Return value: (transfer full): A newly constructed #DeePeer.
 *               Free with g_object_unref().
 */
DeePeer*
dee_peer_new (const gchar* swarm_name)
{
  g_return_val_if_fail (swarm_name != NULL, NULL);

  return g_object_new (DEE_TYPE_PEER, "swarm-name", swarm_name, NULL);
}

/**
 * dee_peer_is_swarm_leader
 * @self: a #DeePeer
 *
 * Return value: %TRUE if and only if this peer owns the swarm name on
 *               the session bus
 */
const gboolean
dee_peer_is_swarm_leader  (DeePeer *self)
{
  g_return_val_if_fail (DEE_PEER (self), FALSE);

  return self->priv->is_swarm_leader;
}

/**
 * dee_peer_get_swarm_leader
 * @self: a #DeePeer
 *
 * Gets the unique DBus address of the current swarm leader.
 *
 * This function can only be used after dee_peer_connect() has been called.
 *
 * Return value: Unique DBus address of the current swarm leader, possibly %NULL
 *    if the leader has not been detected yet
 */
const gchar*
dee_peer_get_swarm_leader (DeePeer *self)
{
  g_return_val_if_fail (DEE_PEER (self), NULL);

  return self->priv->swarm_leader;
}

/**
 * dee_peer_get_swarm_name:
 * @self: a #DeePeer
 *
 * Gets the unique name for this swarm. The swarm leader is the Peer owning
 * this name on the session bus.
 *
 * Return value: The swarm name
 **/
const gchar*
dee_peer_get_swarm_name (DeePeer *self)
{
  g_return_val_if_fail (DEE_PEER (self), NULL);

  return self->priv->swarm_name;
}

static void
emit_peer_found (DeePeer     *self,
                 const gchar *name)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER(self));
  g_return_if_fail (name != NULL);

  priv = self->priv;
  
  if (!g_str_equal (name, priv->own_name))
    {
      g_signal_emit (self, _peer_signals[PEER_FOUND], 0, name);
    }
}

static void
set_swarm_name (DeePeer    *self,
                const gchar *swarm_name)
{
  DeePeerPrivate *priv;
  gchar           *dummy;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (swarm_name != NULL);
  priv = self->priv;

  if (priv->swarm_name)
    {
      g_warning ("%s: Unable to set previously set swarm_name (%s) to (%s)",
                 G_STRLOC,
                 priv->swarm_name,
                 swarm_name);
      return;
    }

  /* If swarm_name is org.example.MyService then the swarm_path will
   * become /com/canonical/dee/org/example/MyService. Note that
   * the actual object path of the peer is not used in the Swarm spec */
  
  priv->swarm_name = g_strdup (swarm_name);
  dummy = g_strdelimit (g_strdup(swarm_name), ".", '/');
  priv->swarm_path = g_strdup_printf ("/com/canonical/dee/peer/%s", dummy);

  g_free (dummy);
}

/* Called when we get the bus connection the first time.
 * Note that user_data is a weak pointer to self */
static void
on_bus_acquired (GDBusConnection *connection,
                 const gchar     *name,
                 gpointer         user_data)
{
  DeePeer        *self;
  DeePeerPrivate *priv;
  gpointer       *weak_self;
  
  g_return_if_fail (user_data != NULL);

  weak_self = user_data;
  if (*weak_self == NULL)
    {
      trace ("DeePeer finalized before getting a connection to the bus");
      return;
    }

  g_return_if_fail (DEE_IS_PEER (*weak_self));

  self = DEE_PEER (*weak_self);
  priv = self->priv;
  priv->connection = g_object_ref (connection);
  priv->own_name = g_strdup (g_dbus_connection_get_unique_name (connection));

  priv->filter_id = g_dbus_connection_add_filter (priv->connection,
                                                  gdbus_message_filter,
                                                  self, /* self */
                                                  NULL);
  
  /* Detect when someone joins the swarm */
  install_match_rule (self,
                      "interface='org.freedesktop.DBus',"
                      "member='RequestName',"
                      "arg0='%s'",
                      priv->swarm_name);

  /* Listen for all signals on the Dee interface concerning this swarm */
  priv->dbus_signals_id =
      g_dbus_connection_signal_subscribe(priv->connection,
                                         NULL,                /* sender */
                                         DEE_PEER_DBUS_IFACE, /* iface */
                                         NULL,                /* member */
                                         NULL,                /* object path */
                                         priv->swarm_name,    /* arg0 */
                                         G_DBUS_SIGNAL_FLAGS_NONE,
                                         on_dbus_peer_signal,      /* callback */
                                         self,                /* user_data */
                                         NULL);               /* user data free */
}

static void
assume_leadership (DeePeer *self)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (self));

  priv = self->priv;
  if (priv->is_swarm_leader)
    {
      trace_object (self, "Leadership acquired, but we are already leaders");
    }
  else
    {
      trace_object (self, "Got swarm leadership");

      /* The first time we become leaders we install a broad match rule
       * that triggers any time someone drops off the bus.
       * Please note that we don't bother cleaning up that rule in the
       * rare case we loose leadership (which only happens if someone
       * forcefully grabs the swarm name) */
      if (!priv->has_been_leader)
        {
          install_match_rule (self, "interface='org.freedesktop.DBus',"
                                    "member='NameOwnerChanged',"
                                    "arg2=''");
        }

      priv->is_swarm_leader = TRUE;
      priv->has_been_leader = TRUE;

      g_free (priv->swarm_leader);
      priv->swarm_leader = g_strdup (priv->own_name);

      /* Emit a Ping so we can do a head count */
      emit_ping (self);

      /* Signal emission must be a "tail call"
       * because we can in theory be finalized by
       * one of the callbacks */
      g_object_notify (G_OBJECT (self), "swarm-leader");
    }

}

/* GDBus callback from the name owner installed with g_bus_own_name().
 * Called when losing leadership.
 * Note that user_data is a weak pointer to self */
static void
on_leadership_lost (GDBusConnection *connection,
                    const gchar     *name,
                    gpointer         user_data)
{
  DeePeer        *self;
  DeePeerPrivate *priv;
  gpointer       *weak_self;
  
  g_return_if_fail (user_data != NULL);

  weak_self = user_data;
  if (*weak_self == NULL)
    {
      /* We where finalized before getting the name */
      return;
    }

  g_return_if_fail (DEE_IS_PEER (*weak_self));

  self = DEE_PEER (*weak_self);
  priv = self->priv;
  
  if (priv->is_swarm_leader)
    {      
      /* We signal the change of swarm leadership in on_ping_received(),
       * only at that point do we know the unique name of the new leader */
      trace_object (self, "Lost swarm leadership");      
      // FIXME. We ought to remove the Pong match rule, but it's not paramount
      priv->is_swarm_leader = FALSE;      
    }
  else
    {
      trace_object (self, "Did not become leader");
    }

  /* If this is the first time we are notified that we are not the leader
   * then request a roster from the leader */
  if (priv->is_first_name_check)
    {
      trace_object (self, "Requesting peer roster from leader");
      g_dbus_connection_call (priv->connection,
                              priv->swarm_name,
                              priv->swarm_path,
                              DEE_PEER_DBUS_IFACE,
                              "List",
                              g_variant_new ("()"),
                              NULL,              /* reply type */
                              G_DBUS_CALL_FLAGS_NONE,
                              -1,
                              NULL,              /* cancellable */
                              on_list_received,  /* callback */
                              self);             /* user_data */
      priv->is_first_name_check = FALSE;
    }
}

/* GDBus callback from the name owner installed with g_bus_own_name().
 * Called when we become leaders.
 * Note that user_data is a weak pointer to self */
static void
on_leadership_acquired (GDBusConnection *connection,
                        const gchar     *name,
                        gpointer         user_data)
{
  gpointer       *weak_self;

  g_return_if_fail (user_data != NULL);

  weak_self = user_data;
  if (*weak_self == NULL)
    {
      /* We where finalized before connecting to the bus */
      return;
    }

  g_return_if_fail (DEE_IS_PEER (*weak_self));

  assume_leadership (DEE_PEER (*weak_self));
}

/* Callback from the GDBus name watcher installed with g_bus_watch_name() */
static void
on_leadership_changed (GDBusConnection *connection,
                       const gchar     *name,
                       const gchar     *name_owner,
                       gpointer         user_data)
{
  DeePeer        *self;
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (user_data));

  self = DEE_PEER (user_data);
  priv = self->priv;

  /* Don't bother if we already know this leader */
  if (g_strcmp0 (priv->swarm_leader, name_owner) == 0)
    return;

  /* At this point we assume we have a new leader */
  if (g_strcmp0 (priv->own_name, name_owner) == 0)
    assume_leadership (self);
  else
    {
      g_free (priv->swarm_leader);
      priv->swarm_leader = g_strdup (name_owner);
      priv->is_swarm_leader = FALSE;
      g_object_notify (G_OBJECT (self), "swarm-leader");
    }
}

/* Callback from gdbus_message_filter() for custom match rules
 * Indicates that @peer_address joined the swarm.
 * This method is thread safe */
static void
on_join_received (DeePeer     *self,
                  const gchar *peer_address)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Found peer %s", peer_address);
  priv = self->priv;

  /* If the peer is already known it must have tried to acquire the swarm name
   * twice...  Just ignore it */
  g_mutex_lock (priv->lock);
  if (g_hash_table_lookup_extended (priv->peers, peer_address, NULL, NULL))
    {
      g_mutex_unlock (priv->lock);
      return;
    }

  g_hash_table_insert (priv->peers, g_strdup (peer_address), NULL);
  g_mutex_unlock (priv->lock);

  emit_peer_found (self, peer_address);
}

/* Callback from _gdbus_message_filter() for custom match rules
 * Indicates that @peer_address left the swarm */
static void
on_bye_received (DeePeer    *self,
                 const gchar *peer_address)
{
  DeePeerPrivate *priv;
  gboolean        removed;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Bye %s", peer_address);
  priv = self->priv;
  
  g_mutex_lock (priv->lock);
  removed = g_hash_table_remove (self->priv->peers, peer_address);
  g_mutex_unlock (priv->lock);

  if (removed)
    {
      trace_object (self, "Leader said Bye to %s", peer_address);
      g_signal_emit (self, _peer_signals[PEER_LOST], 0, peer_address);
    }
  else
    {
      trace_object (self, "Unknown peer '%s' dropped out of the swarm",
                    peer_address);
    }

}

/* Broadcast a Bye signal to to notify the swarm that someone left.
 * Only call this method as swarm leader - that's the contract
 * of the Swarm spec.
 * This method is thread safe */
static void
emit_bye (DeePeer     *self,
          const gchar *peer_address)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (self->priv->is_swarm_leader);
  g_return_if_fail (self->priv->connection != NULL);
  g_return_if_fail (peer_address != NULL);

  trace_object (self, "Emit Bye(%s)", peer_address);

  g_signal_emit (self, _peer_signals[PEER_LOST], 0, peer_address);

  priv = self->priv;
  g_dbus_connection_emit_signal (priv->connection,
                                 NULL,                 /* destination */
                                 priv->swarm_path,     /* object path */
                                 DEE_PEER_DBUS_IFACE,  /* interface */
                                 "Bye",                /* signal name */
                                 g_variant_new ("(ss)",
                                                priv->swarm_name, peer_address),
                                 NULL);                /* error */
}

/* Timeout started when we receive a Ping. When this timeout triggers
 * we do a diff of priv->head_count and priv->peers and emit peer-lost
 * as appropriate. We don't need to emit peer-found because that is already
 * done in when receiving the Pongs from the peers */
static gboolean
on_head_count_complete (DeePeer *self)
{
  DeePeerPrivate *priv;
  GHashTable     *new_peers;
  gpointer        hkey, hval;
  GHashTableIter  hiter;
  GSList         *iter;

  g_return_val_if_fail (DEE_IS_PEER (self), FALSE);

  priv = self->priv;

  /* First we build a new_peers hash set with the names of the
   * head counted peers. Then we diff the old and the new peers
   * sets and emit peer-lost appropriately */
  new_peers = g_hash_table_new_full (g_str_hash,
                                       g_str_equal,
                                       (GDestroyNotify) g_free,
                                       NULL);

  /* Build new_peers hash set */
  iter = priv->head_count;
  for (iter = priv->head_count; iter; iter = iter->next)
    {
      g_hash_table_insert (new_peers, g_strdup (iter->data), NULL);
    }

  /* Emit peer-lost and Bye on peers that didn't emit Pong in due time */
  g_hash_table_iter_init (&hiter, priv->peers);
  while (g_hash_table_iter_next (&hiter, &hkey, &hval))
    {
      if (!g_hash_table_lookup_extended (new_peers, hkey, NULL, NULL))
        {
          if (priv->is_swarm_leader)
            emit_bye (self, hkey);
          else
            g_signal_emit (self, _peer_signals[PEER_LOST], 0, hkey);

        }
    }

  /* Swap old and new peers hash sets */
  g_hash_table_destroy (priv->peers);
  priv->peers = new_peers;

  /* Unregister the head count timeout source. And reset head counting mode */
  priv->head_count_source = 0;
  g_slist_foreach (priv->head_count, (GFunc) g_free, NULL);
  g_slist_free (priv->head_count);
  priv->head_count = NULL;

  return FALSE;
}

/* Indicates that @leader_address send a Ping to the swarm to
 * initiate a head count */
static void
on_ping_received (DeePeer    *self,
                  const gchar *leader_address)
{
  DeePeerPrivate *priv;
  
  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (leader_address != NULL);
  
  priv = self->priv;

  trace_object (self, "Got Ping from: %s", leader_address);

  /* When we receive a Ping (and note that the swarm leader will receive its
   * own Ping as well) we enter a "head count mode" where we will consider
   * all peers that haven't given us a Pong within a certain timeout for lost.
   *
   * We indicate that we are in head counting mode by setting
   * priv->head_count != NULL
   */
  if (priv->head_count)
    {
      g_slist_foreach (priv->head_count, (GFunc) g_free, NULL);
      g_slist_free (priv->head_count);
    }

  priv->head_count = g_slist_prepend (NULL, g_strdup (priv->own_name));
  priv->head_count_source = g_timeout_add (500,
                                           (GSourceFunc) on_head_count_complete,
                                           self);

  /* The DeePeer protocol says we must respond with a Pong to any Ping */
  emit_pong(self);
}

/* Indicates that @peer_address has emitted a Pong  */
static void
on_pong_received (DeePeer    *self,
                  const gchar *peer_address)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (peer_address != NULL);

  priv = self->priv;
  trace_object (self, "Got pong %s", peer_address);
  
  if (!g_hash_table_lookup_extended (priv->peers, peer_address, NULL, NULL))
    {
      g_mutex_lock (priv->lock);
      g_hash_table_insert (priv->peers, g_strdup (peer_address), NULL);
      g_mutex_unlock (priv->lock);

      emit_peer_found (self, peer_address);
    }

  /* If we are in head counting mode register this Ping */
  if (priv->head_count)
    priv->head_count = g_slist_prepend (priv->head_count,
                                        g_strdup (peer_address));
}

static void
on_dbus_peer_signal (GDBusConnection *connection,
                    const gchar      *sender_name,
                    const gchar      *object_path,
                    const gchar      *interface_name,
                    const gchar      *signal_name,
                    GVariant         *parameters,
                    gpointer          user_data)
{
  DeePeer          *self;
  DeePeerPrivate   *priv;
  gchar            *peer_address = NULL;

  g_return_if_fail (DEE_IS_PEER (user_data));

  self = DEE_PEER (user_data);
  priv = self->priv;

  if (g_strcmp0 ("Bye", signal_name) == 0)
    {
      g_variant_get (parameters, "(ss)", NULL, &peer_address);
      on_bye_received (self, peer_address);
    }
  else if (g_strcmp0 ("Ping", signal_name) == 0)
    on_ping_received (self, sender_name);
  else if (g_strcmp0 ("Pong", signal_name) == 0)
    on_pong_received (self, sender_name);
  else
    g_critical ("Unexpected signal from peer %s: %s.%s",
                sender_name, interface_name, signal_name);
}

/* Broadcast a Ping signal to do a head-count on the swarm.
 * Only call this method as swarm leader - that's the contract
 * of the Swarm spec.
 * This method is thread safe */
static void
emit_ping (DeePeer    *self)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (self->priv->is_swarm_leader);
  g_return_if_fail (self->priv->connection != NULL);

  trace_object (self, "Emit ping");
  
  priv = self->priv;
  g_dbus_connection_emit_signal (priv->connection,
                                 NULL,                 /* destination */
                                 priv->swarm_path,     /* object path */
                                 DEE_PEER_DBUS_IFACE,  /* interface */
                                 "Ping",               /* signal name */
                                 g_variant_new ("(s)", priv->swarm_name),
                                 NULL);                /* error */
}

/* Broadcast a Pong signal as a response to a Ping.
 * This method is thread safe */
static void
emit_pong (DeePeer    *self)
{
  DeePeerPrivate *priv;

  g_return_if_fail (DEE_IS_PEER (self));
  g_return_if_fail (self->priv->connection != NULL);

  trace_object (self, "Emit pong");
  
  priv = self->priv;
  g_dbus_connection_emit_signal (priv->connection,
                                 NULL,                 /* destination */
                                 priv->swarm_path,     /* object path */
                                 DEE_PEER_DBUS_IFACE,  /* interface */
                                 "Pong",               /* signal name */
                                 g_variant_new ("(s)", priv->swarm_name),
                                 NULL);                /* error */
}

/* Return floating variant of type '(as)' with unique DBus names of all peers.
 * This method is thread safe */
static GVariant*
build_peer_list (DeePeer *self)
{
  DeePeerPrivate  *priv;
  GHashTableIter   iter;
  GVariantBuilder  b;
  gpointer         key, val;

  g_return_val_if_fail (DEE_IS_PEER (self), FALSE);

  priv = self->priv;

  g_variant_builder_init (&b, G_VARIANT_TYPE ("(as)"));
  g_variant_builder_open (&b, G_VARIANT_TYPE ("as"));

  g_mutex_lock (priv->lock);
  g_hash_table_iter_init (&iter, priv->peers);
  while (g_hash_table_iter_next (&iter, &key, &val))
  {
    g_variant_builder_add (&b, "s", key);
  }
  g_mutex_unlock (priv->lock);

  g_variant_builder_close (&b);
  return g_variant_builder_end (&b);
}

/* This method is thread safe */
static gboolean
check_method (GDBusMessage     *msg,
               const gchar      *iface,
               const gchar      *member,
               const gchar      *path)
{
  return msg != NULL &&
         G_DBUS_MESSAGE_TYPE_METHOD_CALL == g_dbus_message_get_message_type (msg) &&
         (iface == NULL || g_strcmp0 (g_dbus_message_get_interface (msg), iface) == 0) &&
         (member == NULL || g_strcmp0 (g_dbus_message_get_member (msg), member) == 0) &&
         (path == NULL || g_strcmp0 (g_dbus_message_get_path (msg), path) == 0);
}

/* This method is thread safe */
static gboolean
check_signal (GDBusMessage     *msg,
               const gchar      *iface,
               const gchar      *member,
               const gchar      *path)
{
  return msg != NULL &&
         G_DBUS_MESSAGE_TYPE_SIGNAL == g_dbus_message_get_message_type (msg) &&
         (iface == NULL || g_strcmp0 (g_dbus_message_get_interface (msg), iface) == 0) &&
         (member == NULL || g_strcmp0 (g_dbus_message_get_member (msg), member) == 0) &&
         (path == NULL || g_strcmp0 (g_dbus_message_get_path (msg), path) == 0);
}

/* Used to transfer data to the mainloop.
 * Use only for good, not evil, and only from gdbus_message_filter() */
static gboolean
transfer_to_mainloop (gpointer *args)
{
  GFunc cb = (GFunc) args[0];

  cb (args[1], args[2]);
  g_object_unref (args[1]);
  g_free (args[2]);
  g_free (args);

  return FALSE;
}

/* Callback applied to all incoming DBus messages. We use this to grab
 * messages for our match rules and dispatch to the right on_*_received
 * function.
 * WARNING: This callback is run in the GDBus message handling thread -
 *          and NOT in the mainloop! */
static GDBusMessage*
gdbus_message_filter (GDBusConnection *connection,
                      GDBusMessage    *msg,
                      gboolean         incoming,
                       gpointer        user_data)
{
  DeePeer          *self;
  DeePeerPrivate   *priv;
  GVariant         *body;
  GDBusMessageType  msg_type;
  gchar            *swarm_name = NULL;
  gchar            *peer_address = NULL;
  const gchar      *sender_address;
  gpointer         *data;

  g_return_val_if_fail (DEE_IS_PEER (user_data), msg);

  self = DEE_PEER (user_data);
  priv = self->priv;
  body = g_dbus_message_get_body (msg);
  sender_address = g_dbus_message_get_sender (msg);
  msg_type = g_dbus_message_get_message_type (msg);

  /* We have no business with outgoing messages */
  if (!incoming)
    return msg;

  /* We're only interested in method calls and signals */
  if (msg_type != G_DBUS_MESSAGE_TYPE_METHOD_CALL &&
      msg_type != G_DBUS_MESSAGE_TYPE_SIGNAL)
    return msg;

  /*trace ("FILTER: %p", user_data);
    trace ("Msg filter: From: %s, Iface: %s, Member: %s",
           dbus_message_get_sender (msg),
           dbus_message_get_interface (msg),
           dbus_message_get_member (msg));*/

  /* Important note: Apps consuming this lib will likely install custom match
   *                 rules which will trigger this filter. Hence we must do very
   *                 strict matching before we dispatch our methods */
  
  if (check_method (msg, "org.freedesktop.DBus", "RequestName", NULL) &&
      g_strcmp0 (sender_address, g_dbus_connection_get_unique_name (connection)) != 0 &&
      body != NULL)
    {
      g_variant_get (body, "(su)", &swarm_name, NULL);
      if (g_strcmp0 (swarm_name, priv->swarm_name) == 0)
        {
          /* Call on_join_received() in the main loop */
          data = g_new (gpointer, 2);
          data[0] = on_join_received;
          data[1] = g_object_ref (self);
          data[2] = g_strdup (sender_address);
          g_idle_add ((GSourceFunc) transfer_to_mainloop, data);
        }
    }
  else if (check_signal (msg, "org.freedesktop.DBus", "NameOwnerChanged", NULL) && body != NULL)
    {      
      gchar *old_address, *new_address;
      gboolean should_emit_bye;

      g_variant_get (body, "(sss)", &peer_address, &old_address, &new_address);

      /* Check if a known peer dropped off the bus and emit the Bye signal
       * if we are the swarm leaders */
      g_mutex_lock (priv->lock);
      should_emit_bye = priv->is_swarm_leader &&
                        g_strcmp0 (peer_address, old_address) == 0 &&
                        g_strcmp0 (new_address, "") == 0 &&
                        g_strcmp0 (peer_address, g_dbus_connection_get_unique_name (connection)) != 0 &&
                        g_hash_table_lookup_extended (priv->peers,
                                                      peer_address,
                                                      NULL,
                                                      NULL);
      g_mutex_unlock (priv->lock);

      if (should_emit_bye)
        {
          /* Call emit_bye() in the main loop */
          data = g_new (gpointer, 2);
          data[0] = emit_bye;
          data[1] = g_object_ref (self);
          data[2] = peer_address; // own
          g_idle_add ((GSourceFunc) transfer_to_mainloop, data);
          peer_address = NULL;
        }
      g_free (old_address);
      g_free (new_address);
    }
  else if (check_method (msg, DEE_PEER_DBUS_IFACE, "List", priv->swarm_path))
    {
      /* We don't want to go through the whole GDBus interface/introspection
       * setup just to export the List method. We just handle this particular
       * method inline */
      GDBusMessage *reply;
      reply = g_dbus_message_new_method_reply (msg);
      g_dbus_message_set_body (reply, build_peer_list (self));
      g_dbus_connection_send_message (connection,
                                      reply,
                                      G_DBUS_SEND_MESSAGE_FLAGS_NONE,
                                      NULL,   /* out serial */
                                      NULL);  /* error */
      g_object_unref (reply);

      /* Convince GDBus that we handled this message by returning NULL */
      return NULL;
    }
  
  g_free (swarm_name);
  g_free (peer_address);

  return msg;
}
