{-# OPTIONS_GHC -cpp #-}
{-# LANGUAGE CPP #-}

module Darcs.Repository.Repair ( replayRepository,
                                 RepositoryConsistency(..) )
       where
       
import Control.Monad ( when, unless )
import Control.Exception ( finally )
import Data.Maybe ( catMaybes )
import Data.List ( sort )
import System.Directory ( createDirectoryIfMissing )

import Darcs.SlurpDirectory ( empty_slurpy, withSlurpy, Slurpy, SlurpMonad, syncSlurpy )
import Darcs.Lock( rm_recursive )
import Darcs.Hopefully ( PatchInfoAnd, info )

import Darcs.Ordered ( FL(..), RL(..), lengthFL, reverseFL, reverseRL, concatRL,
                     mapRL )
import Darcs.Patch.Depends ( get_patches_beyond_tag )
import Darcs.Patch.Patchy ( applyAndTryToFix )
import Darcs.Patch.Info ( PatchInfo( .. ), human_friendly )
import Darcs.Patch.Set ( PatchSet )
import Darcs.Patch ( RepoPatch, patch2patchinfo )

import Darcs.Repository.Format ( identifyRepoFormat, 
                                 RepoProperty ( HashedInventory ), format_has )
import Darcs.Repository.Cache ( Cache, HashedDir( HashedPristineDir ) )
import Darcs.Repository.HashedIO ( slurpHashedPristine, writeHashedPristine,
                                   clean_hashdir )
import Darcs.Repository.HashedRepo ( readHashedPristineRoot )
import Darcs.Repository.Checkpoint ( get_checkpoint_by_default )
import Darcs.Repository.InternalTypes ( extractCache )
import Darcs.Repository ( Repository, read_repo,
                          checkPristineAgainstSlurpy,
                          makePatchLazy )

import Darcs.Sealed ( Sealed(..), unsafeUnflippedseal )
import Progress ( debugMessage, beginTedious, endTedious, tediousSize, finishedOneIO )
import Darcs.Utils ( catchall )
import Darcs.Global ( darcsdir )
import Darcs.Flags ( compression )
import Printer ( Doc, putDocLn, text )
import Darcs.Arguments ( DarcsFlag( Verbose, Quiet ) )
#include "impossible.h"

run_slurpy :: Slurpy -> SlurpMonad a -> IO (Slurpy, a)
run_slurpy s f =
    case withSlurpy s f of
      Left err -> fail err
      Right x -> return x

update_slurpy :: Repository p -> Cache -> [DarcsFlag] -> Slurpy -> IO Slurpy
update_slurpy r c opts s = do
  current <- readHashedPristineRoot r
  h <- writeHashedPristine c (compression opts) s
  s' <- slurpHashedPristine c (compression opts) h
  clean_hashdir c HashedPristineDir $ catMaybes [Just h, current]
  return s'

replaceInFL :: FL (PatchInfoAnd a)
            -> [(PatchInfo, PatchInfoAnd a)]
            -> FL (PatchInfoAnd a)
replaceInFL orig [] = orig
replaceInFL NilFL _ = impossible
replaceInFL (o:>:orig) ch@((o',c):ch_rest)
    | info o == o' = c:>:replaceInFL orig ch_rest
    | otherwise = o:>:replaceInFL orig ch

applyAndFix :: RepoPatch p => Cache -> [DarcsFlag] -> Slurpy -> Repository p -> FL (PatchInfoAnd p) -> IO (FL (PatchInfoAnd p), Slurpy, Bool)
applyAndFix _ _ s _ NilFL = return (NilFL, s, True)
applyAndFix c opts s_ r psin =
    do beginTedious k
       tediousSize k $ lengthFL psin
       (repaired, slurpy, ok) <- aaf s_ psin
       endTedious k
       orig <- (reverseRL . concatRL) `fmap` read_repo r
       return (replaceInFL orig repaired, slurpy, ok)
    where k = "Replaying patch"
          aaf s NilFL = return ([], s, True)
          aaf s (p:>:ps) = do
            (s', mp') <- run_slurpy s $ applyAndTryToFix p
            finishedOneIO k $ show $ human_friendly $ info p
            s'' <- syncSlurpy (update_slurpy r c opts) s'
            (ps', s''', restok) <- aaf s'' ps
            case mp' of
              Nothing -> return (ps', s''', restok)
              Just (e,pp) -> do putStrLn e
                                p' <- makePatchLazy r pp
                                return ((info p, p'):ps', s''', False)

data RepositoryConsistency p =
    RepositoryConsistent
  | BrokenPristine Slurpy
  | BrokenPatches Slurpy (PatchSet p)

check_uniqueness :: RepoPatch p => (Doc -> IO ()) -> (Doc -> IO ()) -> Repository p -> IO ()
check_uniqueness putVerbose putInfo repository =
    do putVerbose $ text "Checking that patch names are unique..."
       r <- read_repo repository
       case has_duplicate $ mapRL info $ concatRL r of
         Nothing -> return ()
         Just pinf -> do putInfo $ text "Error! Duplicate patch name:"
                         putInfo $ human_friendly pinf
                         fail "Duplicate patches found."

has_duplicate :: Ord a => [a] -> Maybe a
has_duplicate li = hd $ sort li
    where hd [_] = Nothing
          hd [] = Nothing
          hd (x1:x2:xs) | x1 == x2 = Just x1
                        | otherwise = hd (x2:xs)
replayRepository' :: (RepoPatch p) => Repository p -> [DarcsFlag] -> IO (RepositoryConsistency p)
replayRepository' repo opts = do
  let putVerbose s = when (Verbose `elem` opts) $ putDocLn s
      putInfo s = when (not $ Quiet `elem` opts) $ putDocLn s
  check_uniqueness putVerbose putInfo repo
  maybe_chk <- get_checkpoint_by_default repo
  let c = extractCache repo
  createDirectoryIfMissing False $ darcsdir ++ "/pristine.hashed"
  rooth <- writeHashedPristine c (compression opts) empty_slurpy
  s <- slurpHashedPristine c (compression opts) rooth
  putVerbose $ text "Applying patches..."
  patches <- read_repo repo
  (s', newpatches, patches_ok) <- case maybe_chk of
    Just (Sealed chk) ->
        do let chtg = patch2patchinfo chk
           putVerbose $ text "I am repairing from a checkpoint."
           (s'', _) <- run_slurpy s $ applyAndTryToFix chk
           (_, s_, ok) <- applyAndFix c opts s'' repo
                      (reverseRL $ concatRL $ unsafeUnflippedseal $ get_patches_beyond_tag chtg patches)
           return (s_, patches, ok)
    Nothing -> do debugMessage "Fixing any broken patches..."
                  let psin = reverseRL $ concatRL patches
                  (ps, s_, ok) <- applyAndFix c opts s repo psin
                  debugMessage "Done fixing broken patches..."
                  return (s_, (reverseFL ps :<: NilRL), ok)
  debugMessage "Checking pristine agains slurpy"
  is_same <- checkPristineAgainstSlurpy repo s' `catchall` return False
  -- TODO is the latter condition needed? Does a broken patch imply pristine
  -- difference? Why, or why not?
  return (if is_same && patches_ok
     then RepositoryConsistent
     else if patches_ok
            then BrokenPristine s'
            else BrokenPatches s' newpatches)

cleanupRepositoryReplay :: Repository p -> IO ()
cleanupRepositoryReplay r = do
  let c = extractCache r
  rf_or_e <- identifyRepoFormat "."
  rf <- case rf_or_e of Left e -> fail e
                        Right x -> return x
  unless (format_has HashedInventory rf) $
         rm_recursive $ darcsdir ++ "/pristine.hashed" 
  when (format_has HashedInventory rf) $ do
       current <- readHashedPristineRoot r
       clean_hashdir c HashedPristineDir $ catMaybes [current]

replayRepository :: (RepoPatch p) => Repository p -> [DarcsFlag] -> (RepositoryConsistency p -> IO a) -> IO a
replayRepository r opt f = run `finally` cleanupRepositoryReplay r
    where run = do
            st <- replayRepository' r opt
            f st
