# backend.rb : The base of the arcitecture of the Backends
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Filters and descriptions
require 'MetaBuilder/metabuilder'
require 'SciYAG/Backends/descriptions'
require 'SciYAG/Backends/filters'
require 'SciYAG/Backends/dataset'
require 'SciYAG/Backends/cache'

# Document-class: SciYAG
# The SciYAG module contains all the classes related to Backend and Filter.
# It will also contain additionnal features in a not-so-distant future.
module SciYAG

  module Backends

    # A simple class to describe a backend. See the
    # Descriptions::Description
    class BackendDescription < MetaBuilder::Description
      # The backend's banner.
      def banner(instance) 
        "Backend '#{instance.long_name}'"
      end

      # Creates a new backend and optionnally registers it
      def initialize(cls, name, long_name, desc, register = true)
        super(cls, name, long_name, desc)
        if register
          Backend.register_class(self)
        end
      end
    end


    # This class provides the infrastructure for accessing data sets. It
    # shouldn't be used directly, but rather subclassed and reimplemented.
    # The aim of this class is to provide any software which is interested
    # to retrive data from some source with a consistent way to do so,
    # independent the kind of source accessed.
    #
    # Subclasses should:
    # * provide a consistent method for creating themselves,
    #   with as much information as necessary, including options and default
    #   parameters. Actually, their initialize function should take no value
    #   bu on the other side, the BackendDescription associated with it
    #   should make it easy to set all the parameters necessary to
    #   get one set of data.
    # * provide a way to fill an OptionParser with their own parameters
    # * provide a way to retrieve the data via named 'sets' (either 2D or 3D
    #   data, depending on the subclass)
    # * provide a way to obtain all meta-informations on one dataset,
    #   such as the date, the meaning of the columns (if any), and so on.
    # * provide a way to know which named sets are available, or at least
    #   a subset (or nothing if we don't know a thing).
    # * wether the actual reading of the data is done at initialization time
    #   or at query time is left to the implementor ;-) !

    class Backend

      # Import the main description functions into the appropriate
      # namespaces
      extend  MetaBuilder::DescriptionExtend
      include MetaBuilder::DescriptionInclude

      # Backend is a factory, but no autoregistering is made.
      create_factory false

      # Sets up a few things, such as the filters.
      def initialize()
        @xy_filters = []
        @xy_filters_apply = false

        @xyz_filters = []
        @xyz_filters_apply = false

        # If set, uses this set as a baseline for all the others.
        @base_line = ""
        @base_line_cache = false

        # Filters are classes that have one method apply, which takes a
        # pair/triplet of vectors and return the same number of stuff,
        # modified or not.


        # We create a cache by default. Doesn't take up much space anyway:
        @cache = Cache.new
      end

      def push_xy_filter(f)
        @xy_filters << f
        @xy_filters_apply = true
      end

      # Removes a filter from the top
      def pop_xy_filter
        return @xy_filters.pop
      end

      # Removes all filters on the list.
      def clear_xy_filters
        @xy_filters = []
        @xy_filters_apply = false
      end


      # Creates a description object with the given texts and associates
      # it with the class. It is necessary to have this statement
      # *before* any parameter declaration. If you don't set any description,
      # you will not be able to benefit from the plugin system.
      # To be used in Backend subclasses, simply this way:
      #
      #  describe "biniou", "Biniou backend", "A backend to deal with Binious"
      #
      def Backend.describe(name, longname, desc, register = true)
        d = BackendDescription.new(self, name, longname, desc, register)
        set_description(d)
      end

      # Returns a hash containing the description of all available backends
      def Backend.list_backends
        return factory_description_hash
      end

      def Backend.list_descriptions
        warn "Backend.list_descriptions should not be used, use Backend.list_backends instead"
        list_backends
      end

      describe 'backend', 'The base class for backends', <<EOD, false
This is the base class for backends. It should never be used directly.
EOD

      # A hook to set a baseline:
      param_reader :base_line=, :base_line, "baseline", "Base line",
      {:type => :string, }, "Sets a baseline for subsequent data sets"


      def base_line=(str)
        if str =~ /^no$/ or str.empty?
          @base_line = ""
        else
          @base_line = expand_sets(str)[0]
          # Fill the cache.
          ary = query_xy_data(@base_line)
          @base_line_cache = if ary.is_a?(Array)
                               ary[0]
                             else
                               ary
                             end
        end
      end

      # This function should return a hash containing all meta-information
      # available about the given set. The hash can contain elements such as:
      # 
      # <tt>:date</tt>:: the date at which the set was recorded
      # <tt>:x_legend</tt>:: legend of the X axis
      # <tt>:x_unit</tt>:: unit of the X axis
      # (and the same kind for Y and Z axis)
      # 
      # Of course, this list is not limitative; you are encouraged at any rate
      # to add as much metadata as possible. Later on, I'll provide a method
      # to register metadata types. 

      def meta_data(set)
        warn "No metadata implementation for backend " +
          "#{description.name}"
      end

      # Returns true if the backend can provide data for the given set.
      def has_set?(set)
        return false
      end
      
      alias set? has_set?

      # Returns :xy or :xyz depending on the kind of the given set.
      def set_type(set)
        raise "Shouldn't be called for the base class"
      end

      # This function must be redefined by children who provide
      # 2D datasets. This function must return either:
      # * a Function alone, representing the 2D data of the set
      # * or an Array, whose first element is a Function and whose
      #   second element is an array containing metadata about the
      #   dataset.
      def query_xy_data(set)
        raise "query_xy_data must be redefined by children !"
      end

      # Used by other classes to query for the Backends data. This may
      # include processing with data filters. Please note that the
      # query_xy_data functions *must* return a Dobjects::Function.
      # The return value *must* be modifiable without consequences to
      # the backend.
      def xy_data(set)
        retval = query_xy_data(set)
        if retval.is_a?(Array) 
          ary,errors,meta = *retval
        else
          ary = retval
          errors = {}
          meta = {}
        end
        if (not @base_line.empty?) and @base_line_cache
          ary.y.sub!(@base_line_cache.y)
        end
        # Now, this is the fun part: we create the Dataset object
        dataset = DataSet2D.new(self, ary, errors, meta)
        # apply the filters if necessary
        if @xy_filters_apply
          for filter in @xy_filters
            filter.apply!(dataset)
          end
        end
        return dataset
      end

      # Returns the XYZ data for the given set
      def xyz_data(set)
        ary = query_xyz_data(set)
        # apply the filters if necessary
        if @xyz_filters_apply
          for filter in @xyz_filters
            ary = filter.apply(ary)
          end
        end
        return ary
      end


      # When converting a user input into a set, a program should
      # *always* use this function, unless it has really good reasons for
      # that.
      #
      # The default implementation is to expand 2##4 to 2, 3, 4. Can be
      # useful even for mathematical stuff.
      #
      # Another thing is recognised and expanded:
      # #<2<i*2>,5> runs the code i*2 with the values from 2 to 5 and
      # returns the result. The code in the middle is a Ruby block, and therefore
      # should be valid ! 
      #
      # A third expansion is now available:
      # #<a = 2<a * sin(x)>10> will expand into 2*sin(x) , 3*sin(x) ... 10*sin(x)
      # it is different than the previous in the sense that the code in the
      # middle is not a Ruby code, but a mere string, which means there won't be
      # compilation problems.
      #
      # Unless your backend can't accomodate for that, all redefinitions
      # of this function should check for their specific signatures
      # first and call  this function if they fail. This way, they
      # will profit from improvements in this code while keeping
      # old stuff working.
      def expand_sets(spec)
        if m = /(\d+)##(\d+)/.match(spec)
          debug "Using expansion rule #1"
          a = m[1].to_i
          b = m[2].to_i
          ret = []
          a.upto(b) do |i|
            ret << m.pre_match + i.to_s + m.post_match
          end
          return ret
        elsif m = /\#<(\d+)<(.*?)>(\d+)>/.match(spec)
          debug "Using expansion rule #2"
          from = m[1].to_i
          to = m[3].to_i
          debug "Ruby code used for expansion: {|i| #{m[2]} }"
          code = eval "proc {|i| #{m[2]} }"
          ret = []
          from.upto(to) do |i|
            ret << m.pre_match + code.call(i).to_s + m.post_match
          end
          return ret
        elsif m = /\#<\s*(\w+)\s*=\s*(\d+)\s*<(.*?)>\s*(\d+)\s*>/.match(spec)
          debug "Using expansion rule #3"
          var = m[1]
          from = m[2].to_i
          to = m[4].to_i
          # Then we replace all occurences of the variable
          literal = '"' + m[3].gsub(/\b#{var}\b/, '#{' + var + '}') + '"'
          debug "Ruby code used for expansion: {|#{var}| #{literal} }"
          code = eval "proc {|#{var}| #{literal} }"
          ret = []
          from.upto(to) do |i|
            ret << m.pre_match + code.call(i).to_s + m.post_match
          end
          return ret
        end
        # Fallback
        return [spec]
      rescue  Exception => ex
        # In case something went wrong in the eval.
        warn "An error occured during expansion of '#{spec}': #{ex.message}"
        debug "Error backtrace: #{ex.backtrace.join "\n"}"
        warn "Ignoring, but you're nearly garanteed something will "+
          "fail later on"
        return [spec]
      end

      # Some backends have a pretty good idea of the sets available for use.
      # Some really don't. You can choose to reimplement this function if
      # you can provide a useful list of sets for your backend. This list
      # doesn't need to be exhaustive (and is most unlikely to be). It can
      # also return something that would need further expansion using
      # expand_sets.
      def sets_available
        return []
      end

      # Gets a cached entry or generate it and cache it. See Cache#cache
      # for more details. The cache's meta_data is constructed as following:
      # * the current state of the backend is taken
      # * keys inside _exclude_ are removed.
      # * _supp_info_ is added
      def get_cached_entry(name, exclude = [], supp_info = {}, &code)
        state = save_state
        for k in exclude
          state.delete(k)
        end
        state.merge!(supp_info)
        return @cache.get_cache(name, state, &code)
      end

      # Returns the default state of the named backend,
      # or nil if it wasn't found.
      def self.default_state(name)
        desc = factory_description_hash[name]
        if desc
          return desc.default_state
        else
          return nil
        end
      end
      
      @@log = nil

      # Set the logger for Backends
      def self.logger= (logger)
        @@log = logger
      end

      # Facilities for logging: we forward all functions that look
      # like logging facility to the @@log class variable, if that
      # one responds to it. 
      methods = %w(warn info debug error fatal)
      for meth in methods
        eval <<"EOE"
def #{meth}(*args)
  if @@log.respond_to? :#{meth}
    @@log.#{meth}(*args)
  end
end
EOE
      end
 
      # ruby.el really isn't happy with this one
      private *methods.map {|t| t.to_sym}
    
    end
  end
end
