/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#include <wx/wx.h>
#include <wx/regex.h>

#include <wx/treectrl.h>
#include <wx/filename.h>
#include <wx/dir.h>
#include <wx/listctrl.h>
#include <wx/imaglist.h>
#include <wx/statline.h>

#include "PropertiesFrame.hh"

#include "Cryptonit.hh"
#include "CertificateViewer.hh"
#include "Pkcs12Dlg.hh"
#include "CertificateRequestWizard.hh"
#include "AddIdentity.hh"
#include "wxPasswordManager.hh"

#include "Erreur.hh"
#include "Common.hh"

#include "../Certificate.hh"
#include "../User.hh"
#include "../Utils.hh"
#include "../pkcs12.hh"
#include "../P11Devices.hh"

#ifndef __WXMSW__ // en attendant
#include "pics/identity.xpm"
#include "pics/authorities.xpm"
#include "pics/language.xpm"
#include "pics/algos.xpm"
#include "pics/addressbook.xpm"
#include "pics/add.xpm"
#include "pics/remove.xpm"
#include "pics/up.xpm"
#include "pics/down.xpm"
#include "pics/search_contact.xpm"
#include "pics/cert.xpm"
#include "pics/hardware.xpm"
#include "pics/folder.xpm"
#include "pics/folder_mini.xpm"
#include "pics/sign.xpm"
#endif

#include <vector>
#include "config.h"

#if defined(HAVE_EXT_HASH_MAP)
#include <ext/hash_map>
using namespace __gnu_cxx;
#else
#if defined(HAVE_HASH_MAP)
#include <hash_map>
#else
#error Cannot find hash_map header !
#endif
#endif


BEGIN_EVENT_TABLE(PropertiesFrame, wxDialog)
    /* General events */
    EVT_LIST_ITEM_SELECTED ( PF_MENU_ID, PropertiesFrame::onSelectMenuItem )
    EVT_BUTTON ( PF_APPLY_BTN_ID, PropertiesFrame::onApply )
    EVT_BUTTON ( PF_CANCEL_BTN_ID, PropertiesFrame::onCancel )

    /* CA page events */
    EVT_TREE_ITEM_ACTIVATED ( PF_CERT_TREE_ID, PropertiesFrame::onActivateCA )
    EVT_TREE_SEL_CHANGED ( PF_CERT_TREE_ID, PropertiesFrame::onSelectCA )
    EVT_BUTTON ( PF_VIEW_CERT_DETAILS_BTN_ID, PropertiesFrame::onViewCertDetails )
    EVT_BUTTON ( PF_ADD_CA_BTN_ID, PropertiesFrame::onAddCA )
    EVT_BUTTON ( PF_REMOVE_CA_BTN_ID, PropertiesFrame::onRemoveCA )

    /* Algorithms page events */
    EVT_COMBOBOX( PF_ENCRYPT_ALGO_ID, PropertiesFrame::onCipherChange)
    EVT_COMBOBOX( PF_SIGNATURE_ALGO_ID, PropertiesFrame::onDigestChange)

    /* AddressBook page events */
    EVT_LISTBOX ( PF_AVAILABLE_FIELDS_ID, PropertiesFrame::onAvailableFieldsSelected )
    EVT_LISTBOX ( PF_SELECTED_FIELDS_ID, PropertiesFrame::onSelectedFieldsSelected )
    EVT_BUTTON ( PF_ADD_FIELDS_BTN_ID, PropertiesFrame::onAddAvailableFields )
    EVT_BUTTON ( PF_REMOVE_FIELDS_BTN_ID, PropertiesFrame::onRemoveSelectedFields )
    EVT_BUTTON ( PF_UP_SELECTED_FIELD_BTN_ID, PropertiesFrame::onUpSelectedField )
    EVT_BUTTON ( PF_DOWN_SELECTED_FIELD_BTN_ID, PropertiesFrame::onDownSelectedField )

    /* Identity page events */
    EVT_BUTTON ( PF_ID_ADD_BTN_ID, PropertiesFrame::onAddID )
    EVT_BUTTON ( PF_ID_DEL_BTN_ID, PropertiesFrame::onDeleteID )
    EVT_BUTTON ( PF_ID_VIEW_BTN_ID, PropertiesFrame::onViewCert )
    EVT_BUTTON ( PF_ID_CERT_REQ_BTN_ID, PropertiesFrame::onCertRequest )
    EVT_BUTTON ( PF_ID_EXPORT_BTN_ID, PropertiesFrame::onExportP12 )
    EVT_LISTBOX_DCLICK( PF_ID_LIST_ID, PropertiesFrame::onViewCert )

  /* LDAP page events */
  EVT_BUTTON ( PF_LDAP_CONNECTION_TEST_ID, PropertiesFrame::onConnectionTest )
  EVT_CHECKBOX( PF_LDAP_V2, PropertiesFrame::onLDAPv2)

    /* Crypto hardware events */
    EVT_BUTTON( PF_CRYPTO_LOAD_BTN_ID , PropertiesFrame::loadDriver )
    EVT_BUTTON( PF_CRYPTO_UNLOAD_BTN_ID , PropertiesFrame::unloadDriver )
	 EVT_BUTTON( PF_CRYPTO_RESTART_BTN_ID , PropertiesFrame::onRestartP11 )
    EVT_LIST_ITEM_SELECTED( PF_CRYPTO_LIST_ID , PropertiesFrame::onSelectDriver )
	 EVT_LISTBOX_DCLICK( PF_CRYPTO_CERTLIST_ID , PropertiesFrame::onSelectCertList )

  /* Default parameters events */  
  EVT_BUTTON ( PF_DEFAULT_FOLDER_ID, PropertiesFrame::onDefaultFolder )
  EVT_CHECKBOX ( PF_AUTOFILL_ID, PropertiesFrame::onAutofill )

  /* Signing profile */
  EVT_RADIOBUTTON ( PF_ATTACHED_RADIO_ID, PropertiesFrame::onSigningAttached )
  EVT_RADIOBUTTON ( PF_DETACHED_RADIO_ID, PropertiesFrame::onSigningDetached )
  EVT_CHECKBOX ( PF_DELETE_CB_ID, PropertiesFrame::onSigningDeleteDB )
  EVT_CHECKBOX ( PF_LEAVE_CB_ID, PropertiesFrame::onSigningLeaveCB )
  EVT_BUTTON ( PF_TARGETDIR_BTN_ID, PropertiesFrame::onSigningTarget )

END_EVENT_TABLE();



PropertiesFrame::PropertiesFrame( wxWindow* parent, wxWindowID id,
				  const wxString& title, User* u,
				  std::vector< KeyStore *> *_keyStores,
				  const wxPoint& pos, const wxSize& size,
				  long style, const wxString& name) :
    wxDialog(parent, id, title, pos, wxSize(600,650), style, name)
{
    keyStores = _keyStores;
    ks = getSoftKeyStore();
    user = u;
    caTree = NULL;
    caDNValue = NULL;
    caSNValue = NULL;
    caINValue = NULL;
    cipher = NULL;
    selectedCipher = "";
    selectedDigest = "";

    //CreateStatusBar();
    wxBoxSizer* globalSizer = new wxBoxSizer(wxVERTICAL);
    mainSizer = new wxBoxSizer(wxHORIZONTAL);

    wxBoxSizer* menuSizer = new wxBoxSizer(wxHORIZONTAL);
    menu = new wxListView( this, PF_MENU_ID, wxDefaultPosition, wxDefaultSize,
			   wxLC_ICON | wxLC_NO_HEADER | wxLC_SINGLE_SEL );

    wxImageList* imageList = new wxImageList( 32, 32, TRUE );
    imageList->Add(wxBITMAP(identity) );
    imageList->Add(wxBITMAP(authorities) );
    imageList->Add(wxBITMAP(language), wxColour(0xC0,0xC0,0xC0));
    imageList->Add(wxBITMAP(algos), wxColour(0xC0,0xC0,0xC0));
    // This images already have a transparent background
    imageList->Add(wxBITMAP(addressbook));
    imageList->Add(wxBITMAP(search_contact));
    imageList->Add(wxBITMAP( hardware ) );
    imageList->Add(wxBITMAP( folder ) );
    imageList->Add(wxBITMAP( sign ) );
    menu->AssignImageList( imageList, wxIMAGE_LIST_NORMAL );

    /* Try to setup the wxListView, but the SetAlign()
     * seems to have no effect ?!
     */
    wxListItem item;
    item.SetMask(wxLIST_MASK_FORMAT);
    item.SetAlign(wxLIST_FORMAT_CENTRE);
    menu->SetColumn(0, item);

    menu->InsertItem( 0, _("Identities"), 0);
    menu->EnsureVisible( 0 );
    menu->InsertItem( 1, _("Authorities"), 1);
    menu->EnsureVisible( 1 );
    menu->InsertItem( 2, _("Language"), 2);
    menu->EnsureVisible( 2 );
    menu->InsertItem( 3, _("Algorithms"), 3);
    menu->EnsureVisible( 3 );
    menu->InsertItem( 4, _("Contacts"), 4);
    menu->EnsureVisible( 4 );
    menu->InsertItem( 5, _("LDAP"), 5);
    menu->EnsureVisible( 5 );
    menu->InsertItem( 6, _("Device"), 6);
    menu->EnsureVisible( 6 );
    menu->InsertItem( 7, _("Folder"), 7);
    menu->EnsureVisible( 7 );
    menu->InsertItem( 8, _("Signing\nprofile"), 8);
    menu->EnsureVisible( 8 );


     // Ensure that the list box is focused on the first item
    menu->Focus(0);

// This method produce random segfaults...
// #ifdef __WXMSW__
//     This method is only implemented under MS Windows
//     but segfaults when used under GTK...
//     menu->Select(0, TRUE);
// #endif


#ifdef __WXGTK__
//    menu->SetSize( menu->GetBestSize().GetWidth() * 2,
//		   menu->GetBestSize().GetHeight() );
      menu->SetSize(wxSIZE_USE_EXISTING,wxSIZE_USE_EXISTING);
#else
    menu->SetSize( menu->GetBestSize().GetWidth(),
		   menu->GetBestSize().GetHeight() );
#endif
    menuSizer->Add( menu, 3, wxEXPAND | wxALL, 0 );
    /* Prepare the bottom panel */
    wxPanel* bottomPanel = new wxPanel( this, -1 );

    helpText = new wxStaticText( bottomPanel, -1, _T(""));
    wxFont font = helpText->GetFont();
    font.SetWeight(wxBOLD);
    helpText->SetFont( font );

    applyBtn = new wxButton( bottomPanel, PF_APPLY_BTN_ID, _("Apply") );
    applyBtn->SetDefault(); applyBtn->Disable();

    /* Display the default page */
    parametersPanel = new wxPanel(this,-1);
    createIDPage( parametersPanel );
    currentPage = 0;

    mainSizer->Add( parametersPanel, 3, wxEXPAND | wxALL, 0);
    menuSizer->Add(mainSizer,12, wxEXPAND | wxALL, 0);
    globalSizer->Add( menuSizer, 15, wxEXPAND | wxALL, 0 );

    wxStaticLine* staticLine = new wxStaticLine( this, -1 );
    globalSizer->Add( staticLine, 0, wxEXPAND | wxALL, 0 );

    /* Finalize the bottom panel */
    bottomSizer = new wxBoxSizer( wxHORIZONTAL );

    bottomSizer->Add( helpText, 1,  wxALIGN_CENTER_VERTICAL | wxALIGN_LEFT | wxEXPAND |  wxALL, 5 );

    /* Buttons */
    wxBoxSizer* btnSizer = new wxBoxSizer( wxHORIZONTAL );

    wxButton* closeBtn = new wxButton( bottomPanel, PF_CANCEL_BTN_ID, _("Close") );
    btnSizer->Add( applyBtn, 0, wxALIGN_RIGHT | wxALL, 6 );
    btnSizer->Add( closeBtn, 0, wxALIGN_RIGHT | wxALL, 6 );
    bottomSizer->Add( btnSizer, 0, wxALIGN_CENTER_VERTICAL | wxALIGN_RIGHT /*| wxALL */ | wxEXPAND, 0 );

    bottomPanel->SetSizer( bottomSizer );
    globalSizer->Add( bottomPanel, 1, wxALIGN_CENTER_VERTICAL | wxALL | wxEXPAND, 0 );

    SetSizer(globalSizer);
    //     general->Fit(this);
    Centre();

#ifdef __WXMSW__
    SetIcon(wxICON(aaa));
#endif

#ifdef __WXMAC__
    Show(false);
#else
    Show();
#endif
}

PropertiesFrame::~PropertiesFrame()
{
    deleteParametersPanel();

    //     if( cipher != NULL ) {
    // 	cipher->clear();
    // 	digest->clear();
    // 	delete cipher;
    // 	delete digest;
    //     }
}

/* Return the certificate's CommonName or the
 * organization name if there is no CN.
 */
std::string getCertName( Certificate cert )
{
    /* Use the origanization name, if there is no CN */
    std::string name = cert.getSubjectName().getValue("CN");
    if( name == "" ) {
	name = cert.getSubjectName().getValue("O");
	if( cert.getSubjectName().getValue("OU") != "" )
	    name += " / " + cert.getSubjectName().getValue("OU");
    }
    return name;
}


/*
 * CA Tree page
 */
void PropertiesFrame::buildCATree()
{
    if( user->authorities != NULL ) {
	std::vector<std::string>* ca = user->authorities->getAllAuhtorityName();
	std::vector<std::string>::iterator itr;

	// Add algorithms for the isIssued() method
	OpenSSL_add_all_algorithms();

	hash_map<const char*, std::vector<std::string>, hash<const char*>, PropertiesFrame::eqstr> adjacencies;
	std::vector<std::string> roots;

	for( itr = ca->begin(); itr != ca->end(); itr++ ) {
	    std::vector<std::string>::iterator i;

	    // Load the father's certificate
	    wxFileName certFilename( std2wx(user->authorities->getAuthorityInfo( *itr, "certificate" )) );
	    certFilename=wxMyPrepend(certFilename, std2wx(user->getCACertificatesDir()) );

	    Certificate father;
	    if( father.load( wx2std(certFilename.GetFullPath()).c_str() ) != SUCCESS ) {
		return;
	    }
	    // If the certificate is self-signed,
	    // add it in the roots list
	    if( father.isIssued(father) == X509_V_OK ) {
		roots.push_back( *itr );
	    }

	    // If there is only one CA
	    if( ca->size() == 1 ) {
		// Add it in the adjancies list, it will be inserted into tree
		// in the "orphans loop"

		// Force the 'itr' insertion into the 'adjacencies' list
		if( adjacencies[(*itr).c_str()].empty() );
	    }
	    else {
		for( i = ca->begin(); i != ca->end(); i++ ) {
		    if( *i != *itr ) {
			// Load the child certificate
			wxFileName certFilename( std2wx(user->authorities->getAuthorityInfo( *i, "certificate" )) );
			certFilename=wxMyPrepend(certFilename, std2wx(user->getCACertificatesDir()) );

			Certificate child;
			if( child.load( wx2std(certFilename.GetFullPath()).c_str() ) != SUCCESS ) {
			    return;
			}

			// Force the 'itr' insertion into the 'adjacencies' list
			if( adjacencies[(*itr).c_str()].empty() );

			// Is 'itr' father of 'i'
			if( child.isIssued(father) == X509_V_OK ) {
			    // Add 'child' into the father's adjacencies list
			    adjacencies[(*itr).c_str()].push_back(*i);
			}
		    }
		}
	    }
	}

	delete ca;

	// Construct the wxTreeCtrl
	if( caTree != NULL ) {
	    caTree->DeleteAllItems();
	    wxTreeItemId rootItem = caTree->AddRoot(_T(""));

	    // Step into all 'roots'
	    std::vector<std::string>::iterator root;
	    for( root = roots.begin(); root != roots.end(); root++ ) {

		// Load the certificate
		wxFileName certFilename( std2wx(user->authorities->getAuthorityInfo( *root, "certificate" )) );
		certFilename=wxMyPrepend(certFilename, std2wx(user->getCACertificatesDir()) );

		Certificate cert;
		wxTreeItemId item;
		if( cert.load( wx2std(certFilename.GetFullPath()).c_str() ) == SUCCESS ) {
				std::string s1(wx2std(certFilename.GetFullPath()));
				std::string s2(cert.getSubjectName().getValues(DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ','));
				std::string s3(cert.getSubjectName().getValues(DN_DISPLAY_SHORT | DN_DISPLAY_VALUE, ' '));
		    item = caTree->AppendItem
						( rootItem, std2wx(getCertName(cert)), 0, -1, new CATreeItemData(s1, s2, s3) );
		} else {
		    item = caTree->AppendItem( rootItem, _("Certificate not found! ") );
		}
		addAllChilds( item, *root, adjacencies );
	    }

	    // Add all 'orphans' nodes
	    hash_map<const char*,
		std::vector<std::string>,
		hash<const char*>,
		PropertiesFrame::eqstr>::iterator orphan;

	    for( orphan = adjacencies.begin(); orphan != adjacencies.end(); orphan++ ) {

		// Load the certificate
					wxFileName certFilename( std2wx(user->authorities->getAuthorityInfo(orphan->first, "certificate")) );
		certFilename=wxMyPrepend(certFilename, std2wx(user->getCACertificatesDir()) );

		Certificate cert;
		if( cert.load( wx2std(certFilename.GetFullPath()).c_str() ) == SUCCESS ) {
				std::string s1(wx2std(certFilename.GetFullPath()));
				std::string s2(cert.getSubjectName().getValues(DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ','));
				std::string s3(cert.getSubjectName().getValues(DN_DISPLAY_SHORT | DN_DISPLAY_VALUE, ' '));
				caTree->AppendItem( rootItem, std2wx(getCertName(cert)), 0, -1,
														new CATreeItemData(s1, s2, s3) );
		} else {
		    caTree->AppendItem( rootItem, _("Certificate not found! ") );
		}
	    }
	}
    }
}

void PropertiesFrame::addAllChilds( wxTreeItemId& parent, std::string certDN,
				    hash_map<const char*,
				    std::vector<std::string>,
				    hash<const char*>,
				    PropertiesFrame::eqstr>& adjacencies )
{
    if( adjacencies[certDN.c_str()].size() > 0 ) {
	std::vector<std::string>::iterator itr;

	for( itr = adjacencies[certDN.c_str()].begin();
	     itr != adjacencies[certDN.c_str()].end();
	     itr ++ ) {

	    // Load the certificate
	    wxFileName certFilename( std2wx(user->authorities->getAuthorityInfo(*itr, "certificate")) );
	    certFilename=wxMyPrepend(certFilename, std2wx(user->getCACertificatesDir()) );

	    Certificate cert;
	    wxTreeItemId item;

	    if( cert.load( wx2std(certFilename.GetFullPath()).c_str() ) == SUCCESS ) {
					std::string s1(wx2std(certFilename.GetFullPath()));
					std::string s2(cert.getSubjectName().getValues(DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ','));
					std::string s3(cert.getSubjectName().getValues(DN_DISPLAY_SHORT | DN_DISPLAY_VALUE, ' '));
					item = caTree->AppendItem
						( parent, std2wx(getCertName(cert)), 0, -1, new CATreeItemData(s1, s2, s3) );
	    } else {
					item = caTree->AppendItem( parent, _("Certificate not found! ") );
	    }
	    // Recursive call for all 'itr''s childs
	    addAllChilds( item, *itr, adjacencies );
	}
    }
    // Delete the current node from the 'adjacencies' list
    adjacencies[certDN.c_str()].clear();
    adjacencies.erase(certDN.c_str());
}

void PropertiesFrame::createCAPage( wxPanel *panel)
{
    applyBtn->Disable();
    helpText->SetLabel(_("In this panel you can view information about "
			 "your certification authorities tree."));

    wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);

    wxStaticBox *frame = new wxStaticBox(panel, -1, _("Certification Authority Tree"));
    wxStaticBoxSizer* caTreeSizer = new wxStaticBoxSizer(frame, wxVERTICAL);

    wxImageList *certImageList = new wxImageList( 16 , 16 , TRUE , 1 );
    certImageList->Add( wxBITMAP( cert ) );

    caTree = new wxTreeCtrl( panel, PF_CERT_TREE_ID, wxDefaultPosition, wxDefaultSize,
			     wxTR_HAS_BUTTONS | wxTR_HIDE_ROOT | wxTR_LINES_AT_ROOT );

    caTree->SetImageList( certImageList );

    buildCATree();

    caTreeSizer->Add( caTree, 1, wxEXPAND | wxALL, 5);
    sizerPanel->Add( caTreeSizer, 1, wxEXPAND | wxALL , 5);

    wxBoxSizer* btnSizer = new wxBoxSizer( wxHORIZONTAL );
    wxButton* addCA = new wxButton( panel, PF_ADD_CA_BTN_ID, _("Add a new authority") );
    btnSizer->Add( addCA, 0, wxALL, 10 );

    removeCA = new wxButton( panel, PF_REMOVE_CA_BTN_ID, _("Remove an authority") );
    removeCA->Disable();
    btnSizer->Add( removeCA, 0, wxALL, 10 );

    sizerPanel->Add( btnSizer, 0, wxEXPAND | wxALL , 1);

    wxStaticBox *frame1 = new wxStaticBox(panel, -1, _("Certificate Summary"));
    wxStaticBoxSizer* caSummarySizer = new wxStaticBoxSizer(frame1, wxVERTICAL);
    wxFlexGridSizer* caSummaryFlexSizer = new wxFlexGridSizer( 2, 3 );

    /* DN */
    wxStaticText* caDNText = new wxStaticText(panel, -1, _("Distinguished name:"));
    caSummaryFlexSizer->Add( caDNText, 0, wxEXPAND | wxALL | wxALIGN_RIGHT
			     | wxALIGN_CENTER_VERTICAL, 5);

    caDNValue = new wxStaticText(panel, -1, _T(""));
    caSummaryFlexSizer->Add( caDNValue, 0, wxEXPAND | wxALL | wxALIGN_RIGHT
			     | wxALIGN_CENTER_VERTICAL, 5);

    /* SN */
    wxStaticText* caSNText = new wxStaticText(panel, -1, _("Subject name:"));
    caSummaryFlexSizer->Add( caSNText, 0, wxEXPAND | wxALL | wxALIGN_RIGHT
			     | wxALIGN_CENTER_VERTICAL, 5);

    caSNValue = new wxStaticText(panel, -1, _T(""));
    caSummaryFlexSizer->Add( caSNValue, 0, wxEXPAND | wxALL | wxALIGN_RIGHT
			     | wxALIGN_CENTER_VERTICAL, 5);

    /* IN */
    wxStaticText* caINText = new wxStaticText(panel, -1, _("Issuer name:"));
    caSummaryFlexSizer->Add( caINText, 0, wxEXPAND | wxALL | wxALIGN_RIGHT
			     | wxALIGN_CENTER_VERTICAL, 5);

    caINValue = new wxStaticText(panel, -1, _T(""));
    caSummaryFlexSizer->Add( caINValue, 0, wxEXPAND | wxALL | wxALIGN_RIGHT
			     | wxALIGN_CENTER_VERTICAL, 5);

    caSummarySizer->Add( caSummaryFlexSizer, 0, wxEXPAND | wxALL, 5 );

    /* View Details Button */
    wxButton* viewDetailsBtn = new  wxButton( panel, PF_VIEW_CERT_DETAILS_BTN_ID, _("View Details"));
    caSummarySizer->Add( viewDetailsBtn, 0, wxALIGN_RIGHT | wxALL, 10 );

    sizerPanel->Add( caSummarySizer, 0,  wxADJUST_MINSIZE | wxEXPAND | wxALL, 5);
    sizerPanel->Fit( panel );
    panel->SetSizer(sizerPanel);
}

void PropertiesFrame::onSelectCA(wxTreeEvent& event)
{
    wxTreeItemId itemId = event.GetItem();
    CATreeItemData *item = (CATreeItemData *) caTree->GetItemData(itemId);

    if ( item != NULL ) {
	Certificate cert;
	if( cert.load( item->GetCertFilename().c_str(), der_format ) == SUCCESS ) {
	    caDNValue->SetLabel( std2wx(item->GetShortDN()) );
	    caSNValue->SetLabel( std2wx(cert.getSubjectName().getValue("CN")) );
	    caINValue->SetLabel( std2wx(cert.getIssuerName().getValue("CN")) );
	}
	removeCA->Enable();
    }
}

void PropertiesFrame::onActivateCA(wxTreeEvent& event)
{
    wxTreeItemId itemId = event.GetItem();
    CATreeItemData *item = (CATreeItemData *) caTree->GetItemData(itemId);

    if ( item != NULL ) {
	Certificate cert;

	if( cert.load( item->GetCertFilename().c_str(), der_format ) == SUCCESS ) {
	    wxString title = _("Certificate properties: ");
	    title += std2wx(item->GetShortDN());

	    bool showCrl = true;
	    if ( cert.isIssued( cert ) ==  X509_V_OK)
		showCrl = false;

	    CertificateViewer cv( cert, user, this, -1 , title, showCrl);
	    cv.showModal(wxSize(435,500) , _("Close"), false);
	}
	removeCA->Enable();
    }
}

void PropertiesFrame::onViewCertDetails(wxCommandEvent& WXUNUSED(event))
{
    CATreeItemData* item = (CATreeItemData*) caTree->GetItemData(caTree->GetSelection());

    if( item != NULL ) {
	Certificate cert;

	if( cert.load( item->GetCertFilename().c_str(), der_format ) == SUCCESS ) {
	    wxString title = _("Certificate properties: ");
	    title += std2wx(item->GetShortDN());

	    CertificateViewer cv( cert, user ,this, -1 , title);
	    cv.showModal(wxSize(435,500) , _T("Close"), false);
	}
    }
}

void PropertiesFrame::onAddCA(wxCommandEvent &WXUNUSED(event))
{
    wxString wildcard;
    wildcard << _("All supported formats") << _T("|*.cer;*.der;*.pem")
						 << _T("|") << _("Certificate") << _T(" (*.cer)|*.cer")
						 << _T("|") << _("DER encoded certificate") << _T(" (*.der)|*.der")
						 << _T("|") << _("PEM encoded certificate") << _T(" (*.pem)|*.pem";)

    wxFileDialog fileDlg( this, _("Choose a Certification Authority certificate to import"), wxGetCwd(), _T(""),
			  wildcard, wxOPEN | wxCHANGE_DIR);

    if( fileDlg.ShowModal() == wxID_OK ) {
	wxFileName filename(fileDlg.GetFilename());
	filename=wxMyPrepend(filename,fileDlg.GetDirectory());

	Certificate certificate;
	if( certificate.load( wx2std(filename.GetFullPath()).c_str() ) == SUCCESS ) {
	    if( ! certificate.isCA() ) {
		wxMessageDialog errorMsg
		    (this, _("The selected file doesn't seem to be a Certification "
			     "Authority certificate. Do you want to continue anyway? "),
		     _("Warning"), wxYES_NO | wxICON_EXCLAMATION );
		if( errorMsg.ShowModal() != wxID_YES ) {
		    return;
		}
	    }
	    std::string certHash = certificate.getHash();

	    wxFileName CAFilename( std2wx(certHash) );
	    CAFilename.SetExt(_T("der"));
	    CAFilename=wxMyPrepend(CAFilename,std2wx(user->getCACertificatesDir()));

	    if(certificate.save(wx2std(CAFilename.GetFullPath()).c_str())!=0) {
		wxString buffer = _("Cannot write certificate on disk at: ");
		buffer += CAFilename.GetFullPath();
		wxMessageDialog errorMsg(this, buffer, _("Error"), wxOK | wxICON_ERROR);
		errorMsg.ShowModal();
		return;
	    } else {
		user->authorities->addAuthority
		    (certificate.getSubjectName().getValues
		     (DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ','),
		     wx2std(CAFilename.GetFullName()));
		buildCATree();
	    }
	} else {
	    wxString buffer = _("Cannot load certificate: ");
	    buffer += filename.GetFullPath();
	    wxMessageDialog errorMsg(this, buffer, _("Error"), wxOK | wxICON_ERROR);
	    errorMsg.ShowModal();
	    return;
	}
    }
}

void PropertiesFrame::onRemoveCA(wxCommandEvent &WXUNUSED(event))
{
    CATreeItemData* item = (CATreeItemData*) caTree->GetItemData
	(caTree->GetSelection());
    if( item != NULL ) {
	wxString msg = _("Are you sure to delete this authority: ");
	msg += _("\n");
	msg += std2wx(item->GetShortDN());
	msg += _(" ?");

	wxMessageDialog errorMsg(this, msg, _("Warning"), wxYES_NO | wxICON_EXCLAMATION );

	if( errorMsg.ShowModal() != wxID_YES ) {
	    return;
	}

	if( ! user->authorities->removeAuthority( item->GetDN() ) ) {
	    wxString msg = _("Cannot remove this authority: ");
	    msg += std2wx(item->GetDN());
	    wxMessageDialog errorMsg(this, msg, _("Error"), wxOK | wxICON_ERROR);
	    errorMsg.ShowModal();
	    return;
	}

	wxRemoveFile( std2wx(item->GetCertFilename()) );
	buildCATree();
	removeCA->Disable();
    }
}


/*
 *  Identity page.
 */
void PropertiesFrame::createIDPage( wxPanel *panel )
{
    applyBtn->Disable();
    helpText->SetLabel(_("In this panel you can view and edit information "
			 "about your identities (private keys + certificates)."));

    wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);

    wxStaticBox *frame = new wxStaticBox(panel , -1 , _("Identities Manager"));
    wxStaticBoxSizer *IDSizer = new wxStaticBoxSizer(frame , wxHORIZONTAL);


    wxBoxSizer *innerSizer = new wxBoxSizer( wxVERTICAL );
    IDList = new wxListBox( panel, PF_ID_LIST_ID, wxDefaultPosition, wxDefaultSize,
			    0, NULL, wxLB_SINGLE | wxLB_NEEDED_SB );
    innerSizer->Add(IDList , 1, wxCENTER | wxEXPAND | wxALL , 5);

    wxGridSizer *btnSizer = new wxGridSizer( 1, 5 );

    addID = new wxButton(panel, PF_ID_ADD_BTN_ID , _("Add"));
    btnSizer->Add(addID,1 , wxEXPAND | wxTOP | wxBOTTOM, 20);

    deleteID = new wxButton(panel , PF_ID_DEL_BTN_ID , _("Delete"));
    btnSizer->Add(deleteID,1, wxEXPAND | wxTOP | wxBOTTOM, 20);

    viewCert = new wxButton(panel , PF_ID_VIEW_BTN_ID , _("View details"));
    btnSizer->Add(viewCert, 1 , wxEXPAND | wxTOP | wxBOTTOM, 20);

    exportP12 = new wxButton(panel , PF_ID_EXPORT_BTN_ID , _("Export"));
    btnSizer->Add(exportP12 , 1 , wxEXPAND | wxTOP | wxBOTTOM, 20);

    certRequest = new wxButton(panel , PF_ID_CERT_REQ_BTN_ID , _("Certificate request"));
    btnSizer->Add(certRequest , 1 , wxEXPAND | wxTOP | wxBOTTOM, 20);

    IDSizer->Add(innerSizer, 1 , wxCENTER | wxEXPAND | wxALL , 5 );
    IDSizer->Add(btnSizer, 0, wxCENTER | wxEXPAND | wxALL , 5);

    fillIDList();

    sizerPanel->Add( IDSizer, 1, wxEXPAND | wxALL, 5 );
    panel->SetSizer(sizerPanel);
}


void PropertiesFrame::fillIDList()
{
    if(ks){
				std::vector<Certificate> certificates = ks->listCertificates();
				std::vector<Certificate>::iterator certIt;
				std::vector<std::string> identities;
				std::vector<std::string>::iterator IDIt;
				
				
				identities = user->getInfos( "Identities" );
				
				if( ! certificates.empty() ){
						for (certIt = certificates.begin(), IDIt = identities.begin();
								 certIt != certificates.end() , IDIt != identities.end();
								 certIt++, IDIt++){
								std::string* h = new std::string( certIt->getHash() );
								if( *h != "")
										IDList->Append( std2wx(*IDIt), (void*)h );
						}
				}
    }
}


void PropertiesFrame::onAddID(wxCommandEvent &WXUNUSED(event))
{
    AddIdentity addIdentity( user, this, -1, _("Add a new identity to Cryptonit"));
    if( addIdentity.showModal() == wxID_OK ) {
	Certificate c = addIdentity.getCertificate();
	pkcs12 p12 = addIdentity.getKey();
	wxString label = addIdentity.getIdentity();
	if( c.getHash() != "" && p12.getHash() != "" ){
	    if( label == _T("")){
		label = _("Default");
	    }

	    ks->addPair( c , p12 );
	    updateIDList(c , p12, label);
	}
    }
}



void PropertiesFrame::onDeleteID(wxCommandEvent &WXUNUSED(event))
{
    std::string test;
    int ret = ks->removeKey( IDList->GetSelection() , test) ;

    switch( ret ){
    case -1:{
	wxMessageDialog errorMsg(this, _("Unable to delete certificate file:")
													 + wxString(std2wx(test)) ,_("Error"),
				 wxOK | wxICON_ERROR);
	errorMsg.ShowModal();
    }
	break;
    case -2: {
	wxMessageDialog errorMsg1(this, _("Unable to delete PKCS#12 file:")
				  + wxString(std2wx(test)) ,_("Error"),
				  wxOK | wxICON_ERROR);
	errorMsg1.ShowModal();
    }
	break;
    case -3:{
	wxMessageDialog errorMsg2(this, _("Out of bound"), _("Error"),
				  wxOK | wxICON_ERROR);
	errorMsg2.ShowModal();
    }
	break;

    case -7: std::cerr << "Canceled by user" << std::endl; break;

    default:{
	// refresh the listbox
	std::string *data = (std::string*)IDList->GetClientData( IDList->GetSelection() );
	if( data )
	    delete data;

	IDList->Delete( IDList->GetSelection() );
    }
	break;
    }
}


void PropertiesFrame::onViewCert(wxCommandEvent &event)
{
    std::string *h = (std::string*)IDList->GetClientData(IDList->GetSelection());
    if(h != NULL) {
				wxFileName certificateFilename ( std2wx(*h) );
				certificateFilename.SetExt( _T("der") );
				certificateFilename=wxMyPrepend(certificateFilename, std2wx(user->getCertificatesDir()) );
				
				Certificate c;
				if( c.load( wx2std(certificateFilename.GetFullPath()).c_str() ) == SUCCESS){
						CertificateViewer cv( c , user ,this, -1 , _("Certificate properties"), true);
						cv.showModal(wxSize(435,500) , _("Close"), false);
				}
    }
}


void PropertiesFrame::onCertRequest(wxCommandEvent &WXUNUSED(event))
{
  CertificateRequestWizard* wizi = new CertificateRequestWizard( user , this );
    delete wizi;

}


void PropertiesFrame::updateIDList(Certificate &c, pkcs12 &pkey, const wxString &label)
{
    Certificate cert = c;
    std::string* h = new std::string( c.getHash().c_str() );
    IDList->Append( label, (void*)h );
}


void PropertiesFrame::onExportP12(wxCommandEvent &WXUNUSED(event))
{
    std::string *h = (std::string*)IDList->GetClientData(IDList->GetSelection());
    if( h ) {
				wxString wildcard = _T("PKCS#12 file");
				wildcard += _T(" (*.pkcs12)|*.pkcs12|");
				wildcard += _T("PKCS#12 file");
				wildcard += _T(" (*.p12)|*.p12");
				wildcard += _T("PKCS#12 file");
				wildcard += _T(" (*.pfx)|*.pfx");

				wxString filename = wxFileSelector(_("Export your personnal data"), wxGetCwd(), _T("") ,
								   _T(".pkcs12"), wildcard ,
								   wxSAVE | wxOVERWRITE_PROMPT );
	if ( !filename.empty() ) {
	    wxFileName pkcs12File( std2wx(*h) );
	    pkcs12File.SetExt( _T("p12") );
	    pkcs12File=wxMyPrepend(pkcs12File, std2wx(user->getP12Dir()) );
	    char buffer[1024];
	    FILE *fin=NULL, *fout=NULL;
	    if((fin = fopen(wx2std(pkcs12File.GetFullPath()).c_str(), "rb")) == NULL) {
		wxMessageDialog errorMsg(this, _("Error opening: ") +
					 pkcs12File.GetFullPath() ,
					 _("Error"), wxOK | wxICON_ERROR);
		errorMsg.ShowModal();
		return;
	    }

	    if((fout=fopen(wx2std(filename).c_str(),"wb")) == NULL){
		wxMessageDialog errorMsg(this, _("Error opening: ") +
					 filename , _("Error"),
					 wxOK | wxICON_ERROR);
		errorMsg.ShowModal();
		fclose(fin);
		return;
	    }

	    size_t i=0;

	    for(;;) {
		i = fread(buffer, sizeof(char), 1024, fin);
		if (i <= 0) {
		    break;
		}
		fwrite(buffer, sizeof(char),i,fout);
	    }

	    fclose(fin);
	    fclose(fout);
	    wxMessageDialog successMsg
		(this, _("Personnal data exported to ") + filename,
		 _("Exportation done"), wxOK | wxICON_INFORMATION);
	    successMsg.ShowModal();
	}
    }

}



/*
 * Language page
 */
void PropertiesFrame::createLanguagePage( wxPanel *panel )
{
    applyBtn->Enable();
    helpText->SetLabel(_("In this panel you can set your preferred language."));

    /* Get a reference on the app
     * for accessing locales methods
     */
    CryptonitGui& app = wxGetApp();

    wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);

    wxStaticBox *frame = new wxStaticBox(panel, -1, _("Language selection"));
    wxStaticBoxSizer* languageSizer = new wxStaticBoxSizer(frame, wxVERTICAL);

    wxString buffer = _("The current selected language is ");
    buffer += app.getCurrentLocale();

    wxStaticText* currentLangText = new wxStaticText( panel, -1, buffer );
    languageSizer->Add( currentLangText, 0, wxEXPAND | wxALL, 5  );

    wxStaticText* text = new wxStaticText( panel, -1, _("Select your language:"));
    languageSizer->Add( text, 0, wxEXPAND | wxALL, 5  );

    langList = new wxListBox( panel, -1, wxDefaultPosition, wxDefaultSize,
			      CryptonitGui::nbLangs, CryptonitGui::langs,
			      wxLB_SINGLE | wxLB_NEEDED_SB | wxLB_SORT );
    languageSizer->Add( langList, 1, wxEXPAND | wxALL, 5 );

    sizerPanel->Add( languageSizer, 1, wxEXPAND | wxALL, 5 );
    panel->SetSizer(sizerPanel);
}


/*
 * Algorithms page
 */
void PropertiesFrame::createAlgosPage( wxPanel *panel )
{
    applyBtn->Enable();
    helpText->SetLabel(_("In this panel you can choose your encryption and signature algorithms."));

    wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);
    wxStaticBox *box = new wxStaticBox(panel , -1, _("Algorithms"));
    wxStaticBoxSizer *mSizer = new wxStaticBoxSizer(box, wxVERTICAL);
    int i = -1;

    cipher = new std::vector<std::string>;
    cipher_name = new wxString[CIPHER_NUM];

    std::vector< std::pair< std::string, std::string> > allCiphers;
    std::vector< std::pair< std::string, std::string> >::iterator itr;
    allCiphers = getAllCiphers();

    for( itr = allCiphers.begin(); itr != allCiphers.end(); itr++ ) {
				cipher_name[++i] = std2wx((itr->first));
				cipher->push_back( itr->second );
    }


    int j=-1;
    digest = new std::vector<std::string>;
    digest_name = new wxString[DIGEST_NUM];

    std::vector< std::pair< std::string, std::string> > allDigests;
    allDigests = getAllDigests();

    for( itr = allDigests.begin(); itr != allDigests.end(); itr++ ) {
				digest_name[++j] = std2wx(itr->first);
				digest->push_back( itr->second );
    }

    wxFlexGridSizer *gSizer = new wxFlexGridSizer(2);
		
    wxString selectedCipherName;
		
    if(selectedCipher == ""){
				selectedCipherName = getCipherName(user->getInfo("CipherAlgo"));
				if(selectedCipherName == _T("")){
						selectedCipherName = _T("AES 128 bits");
				}
    } else {
				selectedCipherName = getCipherName(selectedCipher);
    }
		
    wxString selectedDigestName;
    if(selectedDigest == ""){
				selectedDigestName= getDigestName(user->getInfo("DigestAlgo"));
				if(selectedDigestName == _T("")){
						selectedDigestName = _T("SHA1");
				}
    } else {
				selectedDigestName = getDigestName(selectedDigest);
    }

    gSizer->Add(new wxStaticText(panel,-1,_("Encryption algorithm: ")),
								1, wxEXPAND |  wxTOP | wxLEFT | wxBOTTOM , 5);
    if(i >= 0) {
				encryptAlgo = new wxComboBox
						(panel, PF_ENCRYPT_ALGO_ID, selectedCipherName ,wxDefaultPosition,
						 wxSize(200,20), i+1, cipher_name, wxCB_READONLY);
				gSizer->Add(encryptAlgo, 1, wxEXPAND | wxRIGHT | wxBOTTOM | wxLEFT, 5);
				
	//setting up data
	std::vector<std::string>::iterator it;
	int k=0;

	for(it=cipher->begin(); it!=cipher->end();  it++){
	    std::string *data = new std::string(*it);
	    encryptAlgo->SetClientData(k, (void*)data);
	    k++;
	}
    } else {
	wxStaticText* text = new wxStaticText(panel, -1,
					      _("Sorry, no encryption algorithms available?!?"));
	gSizer->Add(text , 1, wxEXPAND | wxALIGN_CENTER_HORIZONTAL | wxALL , 5);
    }

    gSizer->Add(new wxStaticText(panel,-1,_("Digest type for RSA signature: ")),
		1, wxEXPAND |  wxBOTTOM | wxTOP | wxLEFT , 5);
    signatureAlgo = new wxComboBox(panel, PF_SIGNATURE_ALGO_ID, selectedDigestName,
				   wxDefaultPosition, wxSize(200,20), j+1,
				   digest_name, wxCB_READONLY);

    std::vector<std::string>::iterator it;
    j=0;

    for(it=digest->begin(); it!=digest->end();  it++){
	std::string *data = new std::string(*it);
	signatureAlgo->SetClientData(j, (void*)data);
	j++;
    }

    gSizer->Add( signatureAlgo, 1, wxEXPAND | wxRIGHT | wxBOTTOM | wxLEFT, 5);
    mSizer->Add( gSizer, 1, wxEXPAND | wxALL | wxCENTER, 5);
    sizerPanel->Add( mSizer, 1, wxEXPAND | wxALL, 5);
    panel->SetSizer(sizerPanel);
}

void PropertiesFrame::onCipherChange(wxCommandEvent &event)
{
  selectedCipher = ((std::string *)encryptAlgo->GetClientData(event.GetInt()))->c_str();
}

void PropertiesFrame::onDigestChange(wxCommandEvent &event)
{
  selectedDigest = ((std::string *)signatureAlgo->GetClientData(event.GetInt()))->c_str();
    //signatureAlgo->GetStringSelection().c_str();
}

wxString PropertiesFrame::getCipherName(std::string c){
    int i=0;
    for(std::vector<std::string>::iterator it = cipher->begin(); it!=cipher->end(); it++){
	if(c == *it) return cipher_name[i];
	i++;
    }
    return _T("");
}

wxString PropertiesFrame::getDigestName(std::string c){
    int i=0;
    for(std::vector<std::string>::iterator it = digest->begin(); it!=digest->end(); it++){
	if(c == *it) return digest_name[i];
	i++;
    }
    return _T("");
}



/*
 * AddressBook page
 */
void PropertiesFrame::createAddressBookPage( wxPanel *panel )
{
    applyBtn->Enable();
    helpText->SetLabel(_("In this panel you can choose the displayed "
			 "fields in the address book."));

    wxBoxSizer* sizerPanel = new wxBoxSizer(wxVERTICAL);

    wxStaticBox* box = new wxStaticBox(panel , -1, _("Address Book Fields"));
    wxStaticBoxSizer* mSizer = new wxStaticBoxSizer(box, wxHORIZONTAL);

    /* Available list */
    wxBoxSizer* availableSizer = new wxBoxSizer( wxVERTICAL );
    availableSizer->Add ( new wxStaticText
			  (panel, -1, _("Available fields: ")),
			  0, wxALIGN_LEFT, 5 );
    availableFields = new wxListBox( panel, PF_AVAILABLE_FIELDS_ID, wxDefaultPosition,
				     wxDefaultSize, 0, NULL, wxLB_SORT | wxLB_EXTENDED);

    /* Fill the list with all supported attributes */
    for( unsigned int i = 0; ContactInfo::AttributeList[i][1] != NULL; i++ ) {
				availableFields->Append( std2wx(ContactInfo::AttributeList[i][1]),
																 (void*)ContactInfo::AttributeList[i][0] );
    }

    availableSizer->Add( availableFields, 1, wxEXPAND | wxALL, 5 );
    mSizer->Add( availableSizer, 3, wxEXPAND | wxALL, 5 );

    /* Buttons */
    wxBoxSizer* buttonsSizer = new wxBoxSizer( wxVERTICAL );
    wxBoxSizer* buttonsSizer1 = new wxBoxSizer( wxVERTICAL );
    /* Add */
    wxBitmap addIcon = wxBITMAP(add);
#ifdef __WXMSW__
    addIcon.SetMask( new wxMask(addIcon, wxColour(0xC0,0xC0,0xC0)) );
#endif
    addBtn = new wxBitmapButton(panel, PF_ADD_FIELDS_BTN_ID, addIcon);
    addBtn->Disable();
    buttonsSizer1->Add( addBtn, 1, wxALL | wxALIGN_CENTER_HORIZONTAL |
			wxALIGN_CENTER_VERTICAL, 5 );

    /* Remove */
    wxBitmap removeIcon = wxBITMAP(remove);
#ifdef __WXMSW__
    removeIcon.SetMask( new wxMask(removeIcon, wxColour(0xC0,0xC0,0xC0)) );
#endif
    removeBtn = new wxBitmapButton(panel, PF_REMOVE_FIELDS_BTN_ID, removeIcon);
    removeBtn->Disable();
    buttonsSizer1->Add( removeBtn, 1, wxALL | wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL, 5 );


    wxBoxSizer* buttonsSizer2 = new wxBoxSizer( wxVERTICAL );
    /* Up */
    wxBitmap upIcon = wxBITMAP(up);
#ifdef __WXMSW__
    upIcon.SetMask( new wxMask(upIcon, wxColour(0xC0,0xC0,0xC0)) );
#endif
    upBtn = new wxBitmapButton(panel, PF_UP_SELECTED_FIELD_BTN_ID, upIcon);
    upBtn->Disable();
    buttonsSizer2->Add( upBtn, 1, wxALL | wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL, 5 );

    /* Down */
    wxBitmap downIcon = wxBITMAP(down);
#ifdef __WXMSW__
    downIcon.SetMask( new wxMask(downIcon, wxColour(0xC0,0xC0,0xC0)) );
#endif
    downBtn = new wxBitmapButton(panel, PF_DOWN_SELECTED_FIELD_BTN_ID, downIcon);
    downBtn->Disable();
    buttonsSizer2->Add( downBtn, 1, wxALL | wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL, 5 );

    buttonsSizer->Add( buttonsSizer1, 1, wxALL | wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL, 2 );
    buttonsSizer->Add( buttonsSizer2, 1, wxALL | wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL, 2 );
    mSizer->Add( buttonsSizer, 1, wxALL | wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL, 2 );

    /* Selected list */
    wxBoxSizer* selectedSizer = new wxBoxSizer( wxVERTICAL );
    selectedSizer->Add( new wxStaticText(panel, -1, _("Selected fields: ")), 0, wxALIGN_LEFT, 5 );
    selectedFields = new wxListBox( panel, PF_SELECTED_FIELDS_ID, wxDefaultPosition, wxDefaultSize,
				    0, NULL, wxLB_EXTENDED );

    /* Add user's fields into the selectedFields list */
    std::vector<std::string> fields = user->getInfos( "AddressBookFields" );
    if( fields.size() > 0 ) {
	std::vector<std::string>::iterator it;
	for( it = fields.begin(); it != fields.end(); it++ ) {

	    /* Search information's details from the ContactInfo::AttributeList */
	    unsigned int i = 0;
	    while( ContactInfo::AttributeList[i][0] != NULL &&
		   strcmp(ContactInfo::AttributeList[i][0], it->c_str()) != 0 )
		i++;

	    if( ContactInfo::AttributeList[i][0] != NULL ) {
		/* Add the info into the selectedFields list, and remove it from the availableFields list */
					selectedFields->Append( std2wx(ContactInfo::AttributeList[i][1]),
																	(void*)ContactInfo::AttributeList[i][0] );
					availableFields->Delete( availableFields->FindString
																	 ( std2wx(ContactInfo::AttributeList[i][1]) ) );
	    }
	}
    }
    selectedSizer->Add( selectedFields, 1, wxEXPAND | wxALL, 5 );

    mSizer->Add( selectedSizer, 3, wxEXPAND | wxALL, 5 );
    sizerPanel->Add( mSizer, 1, wxEXPAND | wxALL | wxCENTER, 5 );
    panel->SetSizer( sizerPanel );
}


void PropertiesFrame::onAvailableFieldsSelected(wxCommandEvent& event)
{
    addBtn->Enable();
}


void PropertiesFrame::onSelectedFieldsSelected(wxCommandEvent& WXUNUSED(event))
{
    if( selectedFields->GetCount() > 0 )
	removeBtn->Enable();
    else
	removeBtn->Disable();


    wxArrayInt index ;
    int i = selectedFields->GetSelections( index );


    if( i > 0 && index[0] > 0 )
	upBtn->Enable();
    else
	upBtn->Disable();


    if(  i> 0 && index[i-1]  == selectedFields->GetCount() - 1 )
	downBtn->Disable();
    else
	downBtn->Enable();
}


void PropertiesFrame::onAddAvailableFields(wxCommandEvent& WXUNUSED(event))
{
    addBtn->Disable();

    wxArrayInt index ;
    int i = availableFields->GetSelections( index );

    wxArrayString stArray;
    for(int  j = 0 ; j < i ; j++){
	wxString st = availableFields->GetString( index[j] );
	stArray.Add( st );
	selectedFields->Append( st, availableFields->GetClientData( index[j]) );
    }

    for(int j = 0 ; j < i ; j++){
	availableFields->Delete( availableFields->FindString( stArray.Item(j)  ) );
    }


    addBtn->Disable();
}


void PropertiesFrame::onRemoveSelectedFields(wxCommandEvent& WXUNUSED(event))
{
    removeBtn->Disable();
    upBtn->Disable();
    downBtn->Disable();

    wxArrayInt index ;
    int i = selectedFields->GetSelections( index );

    wxArrayString stArray;
    for(int  j = 0 ; j < i ; j++){
	wxString st = selectedFields->GetString( index[j] );
	stArray.Add( st );
	availableFields->Append( st, selectedFields->GetClientData( index[j]) );
    }

    for(int j = 0 ; j < i ; j++){
	selectedFields->Delete( selectedFields->FindString( stArray.Item(j)  ) );
    }

    /* A "selection" event seems to be generated, so we need to re-disable the buttons */
    removeBtn->Disable();
    upBtn->Disable();
    downBtn->Disable();
}


void PropertiesFrame::onUpSelectedField(wxCommandEvent& WXUNUSED(event))
{
    wxArrayInt index ;
    int i = selectedFields->GetSelections( index );

    if( i > 0 ) {
	wxString name[i];
	void *data[i];

	//saving name and data;
	for( int j = 0 ; j < i ; j++ ){
	    name[j] = selectedFields->GetString( index[j] );
	    data[j] = selectedFields->GetClientData( index[j] );
	}

	//deleting old items
	for( int j = 0 ; j < i ; j++ ){
	    selectedFields->Delete( selectedFields->FindString( name[j] ) );
	}

	//re-adding items with pos--
	for( int j = 0 ; j < i ; j++ ){
	    selectedFields->InsertItems( 1, name+j, index[j] - 1 );
	    selectedFields->SetClientData( index[j] - 1, data[j] );
	    selectedFields->SetSelection( index[j] - 1, TRUE );
	}

	//updating pos index
	selectedFields->GetSelections( index );
	if( index[i-1] - 1 == selectedFields->GetCount() - 1 ) {
	    downBtn->Disable();
	} else {
	    downBtn->Enable();
	}
	if( index[0] == 0 ) {
	    upBtn->Disable();
	} else {
	    upBtn->Enable();
	}
    }
}


void PropertiesFrame::onDownSelectedField(wxCommandEvent& WXUNUSED(event))
{
    wxArrayInt index ;
    int i = selectedFields->GetSelections( index );

    if( i > 0 ) {
	wxString name[i];
	void *data[i];

	//saving name and data;
	for( int j = 0 ; j < i ; j++ ){
	    name[j] = selectedFields->GetString( index[j] );
	    data[j] = selectedFields->GetClientData( index[j] );
	}

	//deleting old items
	for( int j = 0 ; j < i ; j++ ){
	    selectedFields->Delete( selectedFields->FindString( name[j] ) );
	}

	//re-adding items with pos--
	for( int j = 0 ; j < i ; j++ ){
	    selectedFields->InsertItems( 1, name+j, index[j] + 1 );
	    selectedFields->SetClientData( index[j] + 1, data[j] );
	    selectedFields->SetSelection( index[j] + 1, TRUE );
	}

	//updating pos index
	selectedFields->GetSelections( index );
	if( index[i-1] + 1 == selectedFields->GetCount()  ) {
	    downBtn->Disable();
	} else {
	    downBtn->Enable();
	}

	if( index[i-1] + 1 == 0 ) {
	    upBtn->Disable();
	} else {
	    upBtn->Enable();
	}
    }
}



/*
 * Create LDAP page
 */
void PropertiesFrame::createLDAPPage( wxPanel *panel )
{
  applyBtn->Enable();
  helpText->SetLabel(_("In this panel you can set the default parameters for accessing the LDAP server."));

  std::string userServer;
  std::string userPort;
  std::string userRetainLDAPv2;
  std::string userDN;
  std::string userPassword;
  std::string userLogin;

    userServer = user->getInfo("LdapServer") != "" ? user->getInfo("LdapServer") : "localhost";
    userPort = user->getInfo("LdapPort") != "" ? user->getInfo("LdapPort") : "389";
    userRetainLDAPv2 = user->getInfo("Use LDAPv2") != "" ? user->getInfo("Use LDAPv2") : "0";
    userDN = user->getInfo("LdapDn") != "" ? user->getInfo("LdapDn") : "";
    userLogin = user->getInfo("LdapLogin");
    userPassword = user->getInfo("LdapPassword");


    wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);
    wxStaticBox *box = new wxStaticBox(panel , -1, _("LDAP"));
    wxStaticBoxSizer* mSizer = new wxStaticBoxSizer(box, wxVERTICAL);
    wxFlexGridSizer *gridSizer = new wxFlexGridSizer(2, 4);

    /* Server name */
    gridSizer->Add( new wxStaticText(panel, -1, _("Server name:")), 0,
		 wxEXPAND | wxALL | wxALIGN_LEFT, 5 );

    serverName = new wxTextCtrl(panel, -1, std2wx(userServer), wxDefaultPosition, wxSize(250,20));
    gridSizer->Add( serverName, 0, wxALL | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL , 5);

    /* Server port */
    gridSizer->Add( new wxStaticText(panel, -1, _("Port:")), 0,
		 wxEXPAND | wxALL | wxALIGN_LEFT, 5 );

    serverPort = new wxTextCtrl(panel, -1,  std2wx(userPort), wxDefaultPosition, wxSize(50,20));
    gridSizer->Add( serverPort, 0, wxALL | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL, 5);

    /* Base DN */
    gridSizer->Add( new wxStaticText(panel, -1, _("Base distinguished name:")), 0,
		 wxEXPAND | wxALL | wxALIGN_LEFT, 5 );

    dn = new wxTextCtrl(panel, -1,  std2wx(userDN), wxDefaultPosition, wxSize(250,20));
    gridSizer->Add( dn, 0, wxALL | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL  , 5);

    /* Server is retaining LDAPv2 */
    gridSizer->Add( new wxStaticText(panel, -1, wxEmptyString), 0, wxEXPAND | wxALL | wxALIGN_LEFT, 5 );
    retain_LDAPv2 = new wxCheckBox(panel, PF_LDAP_V2,_("retain LDAPv2 compatibility?"));
    retain_LDAPv2->SetValue(user->getInfo("Use LDAPv2") == "1" ? TRUE : FALSE);
    gridSizer->Add(retain_LDAPv2, 0, wxALL | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL, 5);


    /* Login */
    gridSizer->Add( new wxStaticText(panel, -1, _("Server login (if required): ")), 0,
		 wxEXPAND | wxALL | wxALIGN_LEFT, 5 );

    login = new wxTextCtrl(panel, -1, std2wx(userLogin), wxDefaultPosition, wxSize(250,20));
    gridSizer->Add( login, 0, wxALL | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL,   5);


    /* Password */
    gridSizer->Add( new wxStaticText(panel, -1, _("Server password (if required): ")), 0,
		 wxEXPAND | wxALL | wxALIGN_LEFT, 5 );

    passwd = new wxTextCtrl(panel, -1,  std2wx(userPassword), wxDefaultPosition, wxSize(250,20),wxTE_PASSWORD);
    gridSizer->Add( passwd, 0, wxALL | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL , 5);


    /* Connection test button */
    // Dummy element
    gridSizer->Add( new wxStaticText(panel, -1, wxEmptyString), 0, wxEXPAND | wxALL | wxALIGN_LEFT, 5 );
    connectionTestBtn = new wxButton( panel, PF_LDAP_CONNECTION_TEST_ID, _("Connection test") );
    gridSizer->Add( connectionTestBtn, 0, wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 0 );


    mSizer->Add( gridSizer, 1, wxEXPAND | wxALL | wxCENTER, 5 );


    mSizer->Add( new wxStaticText(panel, -1, _("Beware that this password will be stored in CLEAR text into your preferences file!")),
		    0, wxEXPAND | wxALL | wxALIGN_CENTER, 5 );

    /* LDAP Fields */
    wxStaticText *ldapFieldsText;
    ldapFieldsText = new wxStaticText(panel, -1,
				      _("Select in the list below the desired LDAP "
					"fields to be displayed in the import window\n "
					"(click on the \"Connection test\" above to refresh the list):") );
    mSizer->Add( ldapFieldsText, 0, wxEXPAND | wxALL | wxALIGN_LEFT, 5);


    ldapFieldsListBox = new wxListBox(panel, -1, wxDefaultPosition, wxDefaultSize,
				      0, NULL, wxLB_EXTENDED);
    ldapFieldsListBox->Set( ldapFields );
    mSizer->Add( ldapFieldsListBox, 1, wxEXPAND | wxALL | wxCENTER, 5 );


    sizerPanel->Add( mSizer, 1, wxEXPAND | wxALL | wxCENTER, 5 );
    panel->SetSizer( sizerPanel );
}



void PropertiesFrame::onConnectionTest(wxCommandEvent &WXUNUSED(event))
{
    ldapFields.Empty();

    if( serverName->GetValue() == _T("") || serverPort->GetValue() == _T("")
	|| dn->GetValue() == _T("") ) {

	wxMessageBox( _("Please fill all fields on this page."),
		      _("Error"), wxICON_ERROR | wxOK, this );
	return;
    }

    DirectoryService *dsLdap;
    dsLdap = dsLdap->factory("ldapcache");

    std::string params[] = { 
      wx2std(serverName->GetValue()),
      wx2std(serverPort->GetValue()),
      retain_LDAPv2->GetValue() == TRUE ? "1" : "0",
      wx2std(dn->GetValue()),
      std::string("objectClass=*"),
      "LDAP_SCOPE_SUBTREE", 
      "" 
    };
    
    dsLdap->setLogin( wx2std(login->GetValue()) );
    dsLdap->setPassword( wx2std(passwd->GetValue()) );

    if( ! dsLdap->read( params ) ) {
	wxMessageBox(_("Cannot connect to the LDAP server."),
		     _("Error"), wxOK | wxICON_ERROR, this);
    }
    else {

	int res = dsLdap->getNbEntry();

	if( res > 0 ) {
	    wxString msg;
	    msg << _("The connection was successfull, there is")
					<< _T(" ") << res << _T(" ") << _("entries in this server.");
	    wxMessageBox( msg, _("Connection successfull"),
			  wxOK | wxICON_INFORMATION, this);


	    // Search all used fields in the LDAP, and add them into
	    // the ldapFields array.
	    // The complexity is horrible
	    DirectoryService::iterator entry = dsLdap->begin();
	    while( entry != dsLdap->end() ) {
		Entry::iterator attribute = entry.second()->begin();
		while( attribute != entry.second()->end() ) {
		    if( ldapFields.Index(std2wx(attribute.first()), FALSE) == wxNOT_FOUND ) {
						ldapFields.Add(std2wx(attribute.first()));
		    }
		    attribute++;
		}
		entry++;
	    }

	    ldapFields.Sort();
	    ldapFieldsListBox->Set( ldapFields );

	    // If the specified server is same than the one in the Preference
	    // file, then highlight the preselected ones
	    if( user->getInfo("LdapServer") == wx2std(serverName->GetValue())
					&& user->getInfo("LdapPort") == wx2std(serverPort->GetValue())
					&& user->getInfo("LdapDn") == wx2std(dn->GetValue()) ) {

		std::vector<std::string> fields = user->getInfos( "LdapFields" );
		if( fields.size() > 0 ) {
		    std::vector<std::string>::iterator it;
		    for( it = fields.begin(); it != fields.end(); it++ ) {
						long item = ldapFieldsListBox->FindString(std2wx(*it));
						ldapFieldsListBox->SetSelection(item, TRUE);
		    }
		}
	    }


	}

	else {
	    wxMessageBox(_("Cannot fetch data in this LDAP directory."),
			 _("Error"), wxOK | wxICON_ERROR, this);
	}

    }

    delete dsLdap;
}

void PropertiesFrame::onLDAPv2(wxCommandEvent &WXUNUSED(event)) {
  if(retain_LDAPv2->GetValue()) {
    user->setInfo("Use LDAPv2", retain_LDAPv2->GetValue() == TRUE ? "1" : "0");
  }
}


void PropertiesFrame::onDefaultFolder(wxCommandEvent &WXUNUSED(event))
{
  std::string	userDefaultFolder(user->getInfo("DefaultFolder"));
  wxDirDialog	*dirDlg = new wxDirDialog( this, _("Choose a folder:"), std2wx(userDefaultFolder));
  
  if(dirDlg->ShowModal() == wxID_OK) 
    {
      folderNameStr = dirDlg->GetPath();
      folderName->Clear();
      folderName->WriteText(folderNameStr);
    }
  applyBtn->Enable();
  autoFill->Enable(userDefaultFolder != "" ? TRUE : FALSE);
}

void PropertiesFrame::onAutofill(wxCommandEvent &WXUNUSED(event)) {

  applyBtn->Enable();
}

void PropertiesFrame::onSigningAttached(wxCommandEvent &WXUNUSED(event)) {

  applyBtn->Enable();
  deleteCB->Enable(TRUE);
}

void PropertiesFrame::onSigningDetached(wxCommandEvent &WXUNUSED(event)) {

  applyBtn->Enable();
  deleteCB->Enable(FALSE);
}

void PropertiesFrame::onSigningTarget(wxCommandEvent &WXUNUSED(event)) {
  
  std::string	userSigningFolder(user->getInfo("Signing Profile Path").c_str());
  wxDirDialog	*dirDlg = new wxDirDialog( this, _("Choose a folder:"), std2wx(userSigningFolder));
  
  if(dirDlg->ShowModal() == wxID_OK) 
    {
      targetDir->Clear();
      targetDir->WriteText(dirDlg->GetPath());
    }
  applyBtn->Enable();
}

void PropertiesFrame::onSigningLeaveCB(wxCommandEvent &WXUNUSED(event)) {

  applyBtn->Enable();
  if (leaveCB->GetValue() == TRUE) {
    targetDir->Enable(FALSE);
    dirBtn->Enable(FALSE);
  }
  else {
    targetDir->Enable(TRUE);
    dirBtn->Enable(TRUE);
  }
}

void PropertiesFrame::onSigningDeleteDB(wxCommandEvent &WXUNUSED(event)) {

  applyBtn->Enable();
}

/*
 *  Device Plugins  page.
 */
void PropertiesFrame::createDevicePage( wxPanel *panel )
{
    applyBtn->Disable();
    helpText->SetLabel(_("In this panel you can view, add and remove "
			 "crypto hardware support."));

    wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);

    wxStaticBox *frame = new wxStaticBox(panel , -1 , _("Cryptographic hardware manager"));
    wxStaticBoxSizer *cryptoSizer = new wxStaticBoxSizer(frame , wxHORIZONTAL);

    wxBoxSizer *innerSizer = new wxBoxSizer( wxVERTICAL );
	innerSizer->Add(
		new wxStaticText(panel, -1, _("Driver list: click to view certificates")),
		0,
		wxALL,
		5);
    deviceList = new wxListCtrl( panel, PF_CRYPTO_LIST_ID, wxDefaultPosition,
				 wxDefaultSize, wxLC_REPORT | wxLC_SORT_ASCENDING );
    deviceList->InsertColumn( 0, _("Manufacturer"), wxLIST_FORMAT_LEFT);
    deviceList->InsertColumn( 1, _("Library Name"), wxLIST_FORMAT_LEFT);
    deviceList->InsertColumn( 2, _("Library File"), wxLIST_FORMAT_LEFT);
    innerSizer->Add(deviceList, 1, wxCENTER | wxEXPAND | wxALL, 5);

	 innerSizer->Add(
		new wxStaticText(panel, -1, _("Certificates present on connected devices: ")),
		0,
		wxALL,
		5);
	 certList = new wxListBox(panel, PF_CRYPTO_CERTLIST_ID, wxDefaultPosition, wxDefaultSize, 0, NULL, wxLB_SINGLE | wxLB_NEEDED_SB | wxLB_SORT);
	 innerSizer->Add(certList, 1, wxCENTER | wxEXPAND | wxALL, 5);
	 innerSizer->Add(
		new wxStaticText(panel, -1, _("Double-click on a certificate to view details")),
		0,
		wxALL,
		5);


    wxBoxSizer *btnSizer = new wxBoxSizer( wxVERTICAL );

    loadDevice = new wxButton(panel , PF_CRYPTO_LOAD_BTN_ID , _("Load"));
    btnSizer->Add(loadDevice, 0,  wxTOP | wxBOTTOM | wxEXPAND , 20);

    unloadDevice = new wxButton(panel , PF_CRYPTO_UNLOAD_BTN_ID , _("Unload"));
    btnSizer->Add(unloadDevice, 0,  wxTOP | wxBOTTOM | wxEXPAND , 20);

	 restartP11lib = new wxButton(panel , PF_CRYPTO_RESTART_BTN_ID , _("Restart"));
    btnSizer->Add(restartP11lib, 0,  wxTOP | wxBOTTOM | wxEXPAND , 20);

    cryptoSizer->Add(innerSizer, 1 , wxCENTER | wxEXPAND | wxALL , 5 );
    cryptoSizer->Add(btnSizer, 0, wxCENTER | wxALL | wxEXPAND, 5);

    fillDeviceList();

    sizerPanel->Add( cryptoSizer, 1, wxEXPAND | wxALL, 5 );
    panel->SetSizer(sizerPanel);

    unloadDevice->Disable();
	 restartP11lib->Disable();

}

/*
 *  Folder choice page.
 */
void PropertiesFrame::createFolderPage( wxPanel *panel )
{
  std::string	userDefaultFolder = user->getInfo("DefaultFolder");
  wxString	userDefaultFolderStr(std2wx(userDefaultFolder));
  std::string	userDefaultFolderAutofill(user->getInfo("DefaultFolderAutofill"));
  
#ifdef __WXMSW__
  wxRegEx		regExp(_T("/"));
  const wxString	replacement(_T("\\\\"));
  regExp.ReplaceAll(&userDefaultFolderStr, replacement); 
#endif
  
  applyBtn->Disable();
  helpText->SetLabel(_("In this panel you can set the Cryptonit default folder"));

  /* Boxes */
  wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);
  wxStaticBox *box = new wxStaticBox(panel , -1 , _("Default folder choice"));  
  wxStaticBoxSizer* mSizer = new wxStaticBoxSizer(box, wxVERTICAL);
  wxBoxSizer *sizerPath = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *sizerAutofill = new wxBoxSizer(wxHORIZONTAL);

  /* Default folder */
  mSizer->Add( new wxStaticText(panel, -1, _("Default folder:")),
 		  0, wxALIGN_LEFT | wxTOP | wxLEFT | wxRIGHT , 15);
  folderName = new wxTextCtrl(panel, -1, userDefaultFolderStr.c_str(), 
			      wxDefaultPosition, wxSize(350,25));
  folderName->SetEditable(FALSE);
  wxBitmap dirIcon = wxBITMAP(folder_mini);
  wxBitmapButton *dirBtn = new wxBitmapButton(panel, PF_DEFAULT_FOLDER_ID, dirIcon);
  sizerPath->Add( folderName, 1, wxCENTER | wxALIGN_LEFT | wxALL, 5);
  sizerPath->Add( dirBtn, 0, wxRIGHT | wxALL, 5);

  /* CheckBox */
  autoFill = new wxCheckBox(panel, PF_AUTOFILL_ID, _("Autofill the list of files at startup"));
  autoFill->SetToolTip( _("If you check this option, the zone containing the files will be automatically filled") );
  sizerAutofill->Add( autoFill, 0, wxALL | wxALIGN_LEFT | wxALIGN_TOP, 5);
  autoFill->SetValue(userDefaultFolderAutofill == "1"  ? TRUE : FALSE);
  autoFill->Enable(userDefaultFolder != "" ? TRUE : FALSE);

  /* Setting boxes */
  mSizer->Add( sizerPath, 0, wxALIGN_LEFT | wxALIGN_TOP | wxLEFT | wxRIGHT | wxBOTTOM, 5);
  mSizer->Add( sizerAutofill, 0, wxALIGN_LEFT | wxALIGN_BOTTOM | wxLEFT | wxRIGHT,5 );

  sizerPanel->Add( mSizer, 1, wxEXPAND | wxALL | wxCENTER, 5 );
  panel->SetSizer(sizerPanel);
}


/* Signing Profile */

void PropertiesFrame::createSigningProfilePage( wxPanel *panel ) {

		wxString	signingDefaultFolder(std2wx(user->getInfo("Signing Profile Path")));
#ifdef __WXMSW__
		wxRegEx		regExp(_T("/"));
		const wxString	replacement(_T("\\\\"));
  regExp.ReplaceAll(&signingDefaultFolder, replacement); 
#endif

  applyBtn->Disable();
  helpText->SetLabel(_("In this panel you can define a signing profile"));

  /* Boxes */
  wxBoxSizer *sizerPanel = new wxBoxSizer(wxVERTICAL);
  wxStaticBox *signingTypeBox = new wxStaticBox(panel,-1, _("Signature type"));
  wxStaticBoxSizer *detachedSizer = new wxStaticBoxSizer(signingTypeBox, wxVERTICAL);
  wxStaticBox *signingConfigBox = new wxStaticBox(panel,-1, _("Signature settings"));
  wxStaticBoxSizer *configSizer = new wxStaticBoxSizer(signingConfigBox, wxVERTICAL);
  wxBoxSizer *sizerPath = new wxBoxSizer(wxHORIZONTAL);
  
  /* Signing type parameters */
  attachedRadio = new wxRadioButton(panel, PF_ATTACHED_RADIO_ID,_("Attached"),wxDefaultPosition,wxDefaultSize, wxRB_GROUP);  
  attachedRadio->SetToolTip( _("Cryptonit will produce a file containing both original file and its signature."));
  detachedRadio = new wxRadioButton(panel, PF_DETACHED_RADIO_ID ,_("Detached"),wxDefaultPosition,wxDefaultSize);
  detachedRadio->SetToolTip( _("Cryptonit will produce a file only containing the signature."));

  attachedRadio->SetValue(user->getInfo("Signing Profile Attached") == "1" ? TRUE : FALSE);
  detachedRadio->SetValue(user->getInfo("Signing Profile Detached") == "1" ? TRUE : FALSE);

  detachedSizer->Add(attachedRadio, 0, wxALIGN_LEFT | wxALIGN_TOP | wxALL, 5);
  detachedSizer->Add(detachedRadio, 0, wxALIGN_LEFT | wxALIGN_TOP | wxALL, 5);

  /* Signing configuration */
  deleteCB = new wxCheckBox(panel, PF_DELETE_CB_ID, _("Delete signed files"));
  deleteCB->SetToolTip( _("If you check this option, original files will be deleted after signature.") );
  leaveCB = new wxCheckBox(panel, PF_LEAVE_CB_ID, _("Leave files at the same place"));
  leaveCB->SetToolTip( _("If you check this option, signed files will be put in the same directory than original ones.") );
  targetDir = new wxTextCtrl(panel, -1, signingDefaultFolder.c_str(),wxDefaultPosition , wxSize(350,20));
  wxBitmap dirIcon = wxBITMAP(folder_mini);
#ifdef __WXMSW__
  dirIcon.SetMask( new wxMask(dirIcon, wxColour(0xC0,0xC0,0xC0)) );
#endif
  dirBtn = new wxBitmapButton(panel, PF_TARGETDIR_BTN_ID,dirIcon);
  targetDir->SetEditable(FALSE);

  leaveCB->SetValue(user->getInfo("Signing Profile LeaveCB") == "1" ? TRUE : FALSE);
  deleteCB->SetValue(user->getInfo("Signing Profile DeleteCB") == "1" ? TRUE : FALSE);
  
  if (user->getInfo("Signing Profile Detached") == "1") {
    deleteCB->Enable(FALSE);
  }
  if (user->getInfo("Signing Profile LeaveCB") == "1") {
    dirBtn->Enable(FALSE);
    targetDir->Enable(FALSE);
  }

  configSizer->Add(new wxStaticText( panel, -1, _("Target Directory for signed files")), 0, wxALIGN_LEFT | wxTOP | wxLEFT | wxRIGHT , 5); 
  sizerPath->Add(targetDir, 1, wxALIGN_LEFT | wxALIGN_CENTER | wxALL, 5);
  sizerPath->Add(dirBtn,    0, wxRIGHT | wxALIGN_CENTER | wxALL, 5);
  configSizer->Add(sizerPath, 0, wxALIGN_LEFT | wxALIGN_TOP | wxALL, 5);
  configSizer->Add(leaveCB,     0, wxALIGN_LEFT | wxLEFT | wxRIGHT, 5 );
  configSizer->Add(deleteCB,    0, wxALIGN_LEFT | wxALL, 5);

  sizerPanel->Add( detachedSizer, 0, wxALIGN_LEFT | wxALIGN_TOP | wxALL | wxEXPAND, 5);
  sizerPanel->Add( configSizer,   0, wxALIGN_LEFT | wxALIGN_TOP | wxALL | wxEXPAND, 5);
  panel->SetSizer(sizerPanel);
}




void PropertiesFrame::fillP11CertList(P11Manager * p11man)
{
	// blank the list
	certList->Set(0, NULL, NULL);
	// empty the certificate vector
	p11cert.clear();
	// get a new one
	p11cert = p11man->listCertificates();
	// fill the list
	int n = p11cert.size();
	if(n > 0)
	{
		void ** table = (void**)malloc(n*sizeof(void*));
		wxString * choices = new wxString[n];
		std::vector<Certificate>::iterator i;
		int j = 0;
		for(i = p11cert.begin(); i != p11cert.end() ; i++) {
				// write the short description of the certificate in the list
				choices[j] = std2wx(i->getSubjectName().getValues(DN_DISPLAY_SHORT | DN_DISPLAY_VALUE , ','));
			// record a pointer to the corresponding certificate in the table
			table[j] = (void*)(&(*i));
			j++;
		}
		certList->Set(n, choices, table);
	}
	else certList->Set(0, NULL, NULL);
}



void PropertiesFrame::clearP11CertList()
{
	// blank the list
	certList->Set(0, NULL, NULL);
	// empty the certificate vector
	p11cert.clear();
}

void PropertiesFrame::fillDeviceList()
{
    deviceList->DeleteAllItems();
    std::vector< KeyStore *>::iterator i;

    for( i = keyStores->begin() ; i != keyStores->end() ; i++ ) {

	if( (*i)->getType() == PKCS11_KEYSTORE ){
	    int cryptokiVersionMajor, cryptokiVersionMinor;
	    int libraryVersionMajor, libraryVersionMinor;
	    std::string manufacturerID, libraryDescription;
	    wxString manufacturerIDCol, libraryDescriptionCol;
	    wxListItem item;
	    long index;

	    item.SetId(0);
	    item.SetColumn(0);
	    item.SetMask( wxLIST_MASK_DATA );

	    // Store a pointer to the keystore in the item data field
	    item.SetData( (long) (*i) );

	    index = deviceList->InsertItem( item );

	    if( ((P11KeyStore*)(*i))->getInfos(cryptokiVersionMajor,
																				 cryptokiVersionMinor,
																				 manufacturerID,
																				 libraryDescription,
																				 libraryVersionMajor,
																				 libraryVersionMinor) ) {
					// Fill the list with token information, if they exist
					libraryDescriptionCol << std2wx(libraryDescription) << _T("(v")
																<< libraryVersionMajor << _T(".")
																<< libraryVersionMinor << _T(")");
					manufacturerIDCol = std2wx(manufacturerID);
	    }
	    else {
					// Set fields to "unknown"
					manufacturerIDCol = _("Unknown");
					libraryDescriptionCol = _("Unknown");
	    }

	    deviceList->SetItem( index, 0, manufacturerIDCol );
	    deviceList->SetItem( index, 1, libraryDescriptionCol );
	    deviceList->SetItem( index, 2, std2wx(((P11KeyStore*)*i)->getLib()));
	}
    }
}


void PropertiesFrame::loadDriver(wxCommandEvent &WXUNUSED(event))
{
    /* Try to automatically detect PKCS#11 devices, and prompt the user for
     * which devices he wants to be installed.
     */
    P11Devices devices;
#ifdef USE_AUTODETECT
    int ret = devices.autodetect();
    if( ret < 0 ) {
	wxMessageBox( _("An error occured while attempting to autodetect "
			"PKCS#11 devices.") + _T(" ")
		      + _("Please provide the driver filename."),
		      _("Error"), wxOK | wxICON_ERROR, this );

    } else if ( ret == 0 ) {
	wxMessageDialog errorMsg( this, _("No device detected.") + _T("\n")
		      + _("Do you want to manully provide a driver? "),
				   _("Warning"), wxYES_NO | wxICON_EXCLAMATION);

	if( errorMsg.ShowModal() != wxID_YES )
	    return;

    } else {
	std::vector<std::string>::iterator i;
	std::vector<std::string> drivers = devices.getDetectedDriverList();
	std::vector<std::string> products = devices.getDevicesProductName();
	wxArrayString driverList, productList;
	wxArrayInt selected, final;

	for( i = drivers.begin(); i != drivers.end(); i++ ) {
	    // Prepend OS Windows root directory (return "" if not WIN32)
	    //driverList.Add( wxGetOSDirectory() + i->c_str() );
	    driverList.Add( i->c_str() );
	}

	for( i = products.begin(); i != products.end(); i++ ) {
	    productList.Add( i->c_str() );
	}

	size_t nb = wxGetMultipleChoices
	    (selected, _("Please select one or more device for installation, "
			 "if you want to specify a driver manually, click "
			 "Cancel: "), _("Detected devices"), productList, this);

	if( nb > 0 ) {

	    std::vector<std::string> pkcs11libs = user->getInfos( "PKCS11_KEYSTORE" );

	    // Step through selected devices
	    for( size_t i = 0; i < selected.GetCount(); i++ ) {
		// Check if driver is not already loaded
		if( find(pkcs11libs.begin(), pkcs11libs.end(),
			 driverList[selected[i]].c_str())->c_str()
		    == driverList[selected[i]]) {
		    wxMessageBox(_("The driver is already installed."), _("Error"),
				 wxOK | wxICON_ERROR, this);
		    continue;
		}

		// NOTE: does we need to check before if the file exists ?
		// Try to laod the PKCS#11 driver
		Erreur * errorManager = new Erreur(this);
		wxPasswordManager *pm  = new wxPasswordManager
		    (NULL, -1 , user , _("Enter your password"));
		P11KeyStore pks(driverList[selected[i]].c_str(), errorManager, pm);
		if( ! pks.isValid() ) {
		    wxMessageBox(_("The driver cannot be loaded."), _("Error"),
				 wxOK | wxICON_ERROR, this);
		    continue;
		}
		// Make the current driver to be finally added
		final.Add( selected[i] );
	    }

	    // Add only working drivers
	    for( size_t i = 0; i < final.GetCount(); i++ ) {

		wxPasswordManager *pm  = new wxPasswordManager
		    (NULL, -1 , user , _("Enter your password"));
		Erreur * errorManager = new Erreur(this);
		P11KeyStore *pks = new P11KeyStore( driverList[final[i]].c_str() , errorManager, pm);

		keyStores->push_back(  pks );
	    }

	    fillDeviceList();
	    return;
	}
    }
#endif

    /* Manual driver selection */
    wxString filename = wxFileSelector(_("Choose a pkcs11 driver"));

    if ( !filename.empty() ) {

#ifdef DEBUG	   
	std::cerr << "Loading driver from " << filename << std::endl;
#endif	
#ifdef __WXMSW__
	filename = wxFileName( filename ).GetFullName();
#endif
	wxPasswordManager *pm  = new wxPasswordManager(NULL, -1 , user , _("Enter your password"));
	Erreur * errorManager = new Erreur(this);
	P11KeyStore *pks = new P11KeyStore( wx2std(filename) , errorManager, pm);

	// Check if the provided library is a valid PKCS#11
	if( ! pks->isValid() ) {

	    wxMessageDialog errorMsg
		(this, _("It does not seems to be a valid PKCS#11 library."),
		 _("Warning"), wxOK | wxICON_EXCLAMATION );

	    errorMsg.ShowModal();
		delete pks;
		return;
	    }
	keyStores->push_back(  pks );
	fillDeviceList();
    }
}



void PropertiesFrame::unloadDriver(wxCommandEvent &WXUNUSED(event))
{
    if( deviceList->GetSelectedItemCount() == 0 )
	return;

	clearP11CertList();
    long item = -1;
    while( true ) {
	item = deviceList->GetNextItem( item, wxLIST_NEXT_ALL,
					wxLIST_STATE_SELECTED );
	if( item == -1 )
	    break;

	wxListItem currentSelectedItem;
	currentSelectedItem.SetId(item);
	currentSelectedItem.SetColumn(2);
	currentSelectedItem.SetMask(wxLIST_MASK_TEXT | wxLIST_MASK_DATA);
	currentSelectedItem.SetState(wxLIST_STATE_SELECTED);

	if( deviceList->GetItem(currentSelectedItem) ) {
#ifdef DEBUG		
	    std::cerr << "Unloading " << currentSelectedItem.GetText() << std::endl;
#endif	    

	    P11KeyStore *data =
		(P11KeyStore*)deviceList->GetItemData(currentSelectedItem.m_itemId);
	    std::vector< KeyStore *>::iterator i;
	    if( data ){
		for( i = keyStores->begin() ; i != keyStores->end() ; i++ ){
		  if( *i == data ){
		    delete data;
			keyStores->erase( i );
			data = NULL;
			deviceList->DeleteItem( currentSelectedItem );
			unloadDevice->Disable();
			restartP11lib->Disable();
			break;
		    }
		}
	    }
	}
    }
}

void PropertiesFrame::onRestartP11(wxCommandEvent &WXUNUSED(event))
{
	if( deviceList->GetSelectedItemCount() == 0 ) return;
    long item = -1;
	clearP11CertList();
	while( true )
	{
	item = deviceList->GetNextItem( item, wxLIST_NEXT_ALL,
					wxLIST_STATE_SELECTED );
		if( item == -1 ) break;
	wxListItem currentSelectedItem;
		currentSelectedItem.SetId(item);
		currentSelectedItem.SetColumn(2);
		currentSelectedItem.SetMask(wxLIST_MASK_TEXT | wxLIST_MASK_DATA);
		currentSelectedItem.SetState(wxLIST_STATE_SELECTED);
		if( deviceList->GetItem(currentSelectedItem) )
		{
#ifdef DEBUG			
			std::cerr << "Restarting " << currentSelectedItem.GetText() << std::endl;
#endif			
			P11KeyStore *data = (P11KeyStore*)(deviceList->GetItemData(currentSelectedItem.GetId()));
			std::vector< KeyStore *>::iterator i;
			if( data )
			{
				for( i = keyStores->begin() ; i != keyStores->end() ; i++ )
				{
					if( *i == data )
					{
						data->restart();
						break;
	}
    }
	    }
	}
    }
}

void PropertiesFrame::onSelectDriver(wxListEvent& event)
{
    P11KeyStore *data = (P11KeyStore*)deviceList->GetItemData(event.m_itemIndex);
	clearP11CertList();
	if( data )
	{
		fillP11CertList(data->getP11ManagerHandle());
	}
	unloadDevice->Enable();
	restartP11lib->Enable();
}

void PropertiesFrame::onSelectCertList(wxCommandEvent & event)
{
	if(event.GetClientData() != NULL)
	{
		// get the certificate
		Certificate c = *((Certificate*)event.GetClientData());
		// open a properties dialog for the certificate
		CertificateViewer cv(c, user, this, -1 , _("Certificate properties"));
		cv.showModal(wxSize(435,500) , _("Close"), false);
	}
}

/*
 * General methods
 */
void PropertiesFrame::deleteParametersPanel()
{
    switch( currentPage )	{
    case 0:
	for(int i = 0 ; i< IDList->GetCount() ; i++){
	    std::string *data = (std::string *) IDList->GetClientData( i );
	    if (data){
		delete data;
	    }
	}
	break;

    case 3:
	if( cipher != NULL ) {
	    cipher->clear();
	    digest->clear();
	    delete cipher;
	    delete digest;

	    for( int i=0 ; i< encryptAlgo->GetCount() ; i++){
		std::string *data = (std::string *)encryptAlgo->GetClientData( i );
		if( data ){
		    delete data;
		}
	    }
	    
	    for( int i=0 ; i< signatureAlgo->GetCount() ; i++){
		std::string *data = (std::string *)signatureAlgo->GetClientData( i );
		if( data ){
		  delete data;
		}
	    }
	}
	break;

    }
    delete parametersPanel;
}

void PropertiesFrame::changePage( unsigned int page )
{
    if( page != currentPage ) {
	mainSizer->Show( mainSizer, false );
	mainSizer->Remove( 1 );

	if( parametersPanel != NULL ){
	    deleteParametersPanel();
	    //    delete parametersPanel;
	}
	parametersPanel = new wxPanel(this,-1);

	switch( page ) {
	case 0: createIDPage( parametersPanel ); break;
	case 1: createCAPage( parametersPanel ); break;
	case 2: createLanguagePage( parametersPanel ); break;
	case 3: createAlgosPage( parametersPanel ); break;
	case 4: createAddressBookPage( parametersPanel ); break;
	case 5: createLDAPPage( parametersPanel ); break;
	case 6: createDevicePage( parametersPanel ); break;
	case 7: createFolderPage( parametersPanel ); break;
	case 8: createSigningProfilePage( parametersPanel ); break;
	}

	mainSizer->Add( parametersPanel, 5, wxEXPAND | wxALL, 0);
	mainSizer->Layout();
	bottomSizer->Layout();
	mainSizer->Show( mainSizer, true );

	currentPage = page;
    }
}

void PropertiesFrame::onSelectMenuItem(wxListEvent& event)
{
    wxListItem info;
    info.m_itemId = event.m_itemIndex;
    info.m_col = 0;
    info.m_mask = wxLIST_MASK_TEXT;


    if ( menu->GetItem(info) ) {
	changePage( info.m_itemId );
    }
}


void PropertiesFrame::onApply(wxCommandEvent &WXUNUSED(event))
{
    switch( currentPage ) {
    case 0: break;
    case 1: break;
    case 2: {/* select language */
	CryptonitGui& app = wxGetApp();
	app.setCurrentLocale(langList->GetStringSelection());
	user->setInfo("Language", wx2std(langList->GetStringSelection()));
	break;
    }
    case 3: {/* algos */
	user->setInfo("CipherAlgo", selectedCipher);
	user->setInfo("DigestAlgo", selectedDigest);
	break;
    }
    case 4: { /* Address Book */
	/* TODO: vrifier si l'ancien vecteur est correctement dsallou */
	std::vector<std::string> fields;
	for( int i = 0; i < selectedFields->GetCount(); i++ )
	    fields.push_back( std::string
			      ((char*) selectedFields->GetClientData( i )) );
	user->setInfos("AddressBookFields", fields);
	break;
    }
    case 5: { /* LDAP */
	user->setInfo("LdapServer", wx2std(serverName->GetValue()) );
	user->setInfo("LdapPort", wx2std(serverPort->GetValue()) );
	user->setInfo("LdapDn", wx2std(dn->GetValue()) );
	user->setInfo("LdapLogin", wx2std(login->GetValue()) );
	user->setInfo("LdapPassword", wx2std(passwd->GetValue()) );
	user->setInfo("Use LDAPv2", retain_LDAPv2->GetValue()== TRUE ? "1" : "0");

	// Store the selected LDAP fields
	std::vector<std::string> ldapFields;
	wxArrayInt selections;

	if( ldapFieldsListBox->GetSelections(selections) > 0 ) {
			for( unsigned int i = 0; i < selections.GetCount(); i++ ) {
					ldapFields.push_back( wx2std(ldapFieldsListBox->GetString(selections[i])) );
	    }
	}
	user->setInfos("LdapFields", ldapFields );

	break;
    }

      case 6: {
	break;
      }

    case 7: { /* Default folder */


      wxString		userDefaultFolder(folderName->GetValue());
#ifdef __WXMSW__
      wxRegEx		regExp(_T("\\\\"));
      const wxString	replacement(_T("/"));
      regExp.ReplaceAll(&userDefaultFolder, replacement); 
#endif
	user->setInfo("DefaultFolder", wx2std(userDefaultFolder).c_str());
	user->setInfo("DefaultFolderAutofill", autoFill->GetValue() == TRUE ? "1" : "0");
	autoFill->Enable(userDefaultFolder != _T("") ? TRUE : FALSE);
	applyBtn->Disable();
	break;
      }
      
    case 8: { /* Signing profile */
      
      wxString signingTargetDir(targetDir->GetValue());
#ifdef __WXMSW__
      wxRegEx		regExp(_T("\\\\"));
      const wxString	replacement(_T("/"));
      regExp.ReplaceAll(&signingTargetDir, replacement); 
#endif
      user->setInfo("Signing Profile Attached", attachedRadio->GetValue() == TRUE ? "1" : "0");
      user->setInfo("Signing Profile Detached", detachedRadio->GetValue() == TRUE ? "1" : "0");
      user->setInfo("Signing Profile Path", wx2std(signingTargetDir).c_str());
      user->setInfo("Signing Profile LeaveCB", leaveCB->GetValue() == TRUE ? "1" : "0");
      user->setInfo("Signing Profile DeleteCB", deleteCB->GetValue() == TRUE ? "1" : "0");
      applyBtn->Disable();
      break;
    }      
    }
}

void PropertiesFrame::onCancel(wxCommandEvent &WXUNUSED(event))
{
    wxCommandEvent e;
    
    onApply(e);
    Close(TRUE);
}



void PropertiesFrame::displayPage( int index )
{
	changePage( index );
}


SoftKeyStore* PropertiesFrame::getSoftKeyStore(){
    std::vector< KeyStore *>::iterator i;
    for( i = keyStores->begin() ; i != keyStores->end() ; i++ ){
	if ( (*i)->getType() == SOFT_KEYSTORE )
	    return (SoftKeyStore*)*i;
    }
    return NULL;
}

