!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \par History
!>      JGH (22-Feb-03) PW grid options added
!>      gt 16-nov-03 moved initialization in this new module
!> \author gloria,30.10.2004, previously in kg_rho_types
! *****************************************************************************
MODULE kg_rspw_types
  USE cube_utils,                      ONLY: cube_info_type,&
                                             destroy_cube_info
  USE f77_blas
  USE gaussian_gridlevels,             ONLY: destroy_gaussian_gridlevel,&
                                             gridlevel_info_type
  USE pw_poisson_types,                ONLY: pw_poisson_release,&
                                             pw_poisson_type
  USE pw_pool_types,                   ONLY: pw_pool_p_type,&
                                             pw_pool_type,&
                                             pw_pools_dealloc
  USE realspace_grid_types,            ONLY: realspace_grid_desc_p_type,&
                                             realspace_grid_desc_type,&
                                             rs_grid_release_descriptor
  USE timings,                         ONLY: timeset,&
                                             timestop
#include "cp_common_uses.h"

  IMPLICIT NONE

  PRIVATE
  LOGICAL, PRIVATE, PARAMETER :: debug_this_module=.TRUE.
  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'kg_rspw_types'
  PUBLIC :: kg_rspw_type, kg_rspw_release, kg_rspw_get, &
            kg_rspw_retain

! *****************************************************************************
  TYPE kg_rspw_type
     INTEGER :: auxbas_grid
     INTEGER :: ref_count
     TYPE(gridlevel_info_type),POINTER              :: gridlevel_info
     TYPE(cube_info_type), DIMENSION(:), POINTER    :: cube_info
     TYPE(pw_pool_p_type), DIMENSION(:), POINTER    :: pw_pools
     TYPE(realspace_grid_desc_p_type), DIMENSION(:), POINTER :: rs_descs
     TYPE(pw_poisson_type), POINTER :: poisson_env
  END TYPE kg_rspw_type
!-----------------------------------------------------------------------------!

CONTAINS

!-----------------------------------------------------------------------------!
! *****************************************************************************
SUBROUTINE kg_rspw_get(kg_rspw, cube_info, gridlevel_info,&
     auxbas_pw_pool,auxbas_grid,rs_descs,pw_pools,auxbas_rs_desc,&
     poisson_env,error)
    TYPE(kg_rspw_type), POINTER              :: kg_rspw
    TYPE(cube_info_type), DIMENSION(:), &
      OPTIONAL, POINTER                      :: cube_info
    TYPE(gridlevel_info_type), OPTIONAL, &
      POINTER                                :: gridlevel_info
    TYPE(pw_pool_type), OPTIONAL, POINTER    :: auxbas_pw_pool
    INTEGER, INTENT(out), OPTIONAL           :: auxbas_grid
    TYPE(realspace_grid_desc_p_type), &
      DIMENSION(:), OPTIONAL, POINTER        :: rs_descs
    TYPE(pw_pool_p_type), DIMENSION(:), &
      OPTIONAL, POINTER                      :: pw_pools
    TYPE(realspace_grid_desc_type), &
      OPTIONAL, POINTER                      :: auxbas_rs_desc
    TYPE(pw_poisson_type), OPTIONAL, POINTER :: poisson_env
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_rspw_get', &
      routineP = moduleN//':'//routineN

    LOGICAL                                  :: failure

  failure=.FALSE.

  CPPrecondition(ASSOCIATED(kg_rspw),cp_failure_level,routineP,error,failure)
  CPPrecondition(kg_rspw%ref_count>0,cp_failure_level,routineP,error,failure)

  IF (.NOT. failure) THEN
     IF (PRESENT(pw_pools)) pw_pools => kg_rspw%pw_pools
     IF (PRESENT(rs_descs)) rs_descs => kg_rspw%rs_descs
     IF (PRESENT(cube_info)) cube_info => kg_rspw%cube_info
     IF (PRESENT(gridlevel_info)) gridlevel_info => kg_rspw%gridlevel_info
     IF (PRESENT(auxbas_pw_pool)) THEN
        auxbas_pw_pool => kg_rspw%pw_pools(kg_rspw%auxbas_grid)%pool
     END IF
     IF (PRESENT(auxbas_rs_desc)) THEN
        auxbas_rs_desc => kg_rspw%rs_descs(kg_rspw%auxbas_grid)%rs_desc
     END IF
     IF (PRESENT(auxbas_grid)) auxbas_grid = kg_rspw%auxbas_grid
     IF (PRESENT(poisson_env)) poisson_env => kg_rspw%poisson_env
  END IF
END SUBROUTINE kg_rspw_get
! *****************************************************************************
SUBROUTINE kg_rspw_retain(kg_rspw,error)
    TYPE(kg_rspw_type), POINTER              :: kg_rspw
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_rspw_retain', &
      routineP = moduleN//':'//routineN

    LOGICAL                                  :: failure

  failure=.FALSE.

  CPPrecondition(ASSOCIATED(kg_rspw),cp_failure_level,routineP,error,failure)
  IF (.NOT. failure) THEN
     CPPrecondition(kg_rspw%ref_count>0,cp_failure_level,routineP,error,failure)
     kg_rspw%ref_count=kg_rspw%ref_count+1
  END IF
END SUBROUTINE kg_rspw_retain

! *****************************************************************************
!> \brief releases the memory used by the kg_rspw and kg_rspw
! *****************************************************************************
  SUBROUTINE kg_rspw_release(kg_rspw,error)

    TYPE(kg_rspw_type), POINTER              :: kg_rspw
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_rspw_release', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, i, igrid_level, stat
    LOGICAL                                  :: failure

!---------------------------------------------------------------------------

   CALL timeset(routineN,handle)
   failure=.FALSE.
   IF (ASSOCIATED(kg_rspw)) THEN
     CPPrecondition(kg_rspw%ref_count>0,cp_failure_level,routineP,error,failure)
     kg_rspw%ref_count=kg_rspw%ref_count-1
     IF (kg_rspw%ref_count<1) THEN
       IF (ASSOCIATED(kg_rspw%gridlevel_info)) THEN
         CALL destroy_gaussian_gridlevel(kg_rspw%gridlevel_info,error=error)
         DEALLOCATE(kg_rspw%gridlevel_info,stat=stat)
         CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
       END IF
       IF (ASSOCIATED(kg_rspw%cube_info)) THEN
         DO igrid_level=1,SIZE(kg_rspw%cube_info)
           CALL destroy_cube_info(kg_rspw%cube_info(igrid_level))
         END DO
        DEALLOCATE (kg_rspw%cube_info,STAT=stat)
        CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
      END IF
      CALL pw_pools_dealloc(kg_rspw%pw_pools,error=error)
      IF (ASSOCIATED(kg_rspw%rs_descs)) THEN
        DO i=1,SIZE(kg_rspw%rs_descs)
          CALL rs_grid_release_descriptor(kg_rspw%rs_descs(i)%rs_desc, error=error)
        END DO
        DEALLOCATE(kg_rspw%rs_descs,stat=stat)
        CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
      END IF
      CALL pw_poisson_release(kg_rspw%poisson_env,error=error)
      DEALLOCATE(kg_rspw, stat=stat)
      CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
    END IF
  END IF
  NULLIFY(kg_rspw)
  CALL timestop(handle)
  END SUBROUTINE kg_rspw_release

END MODULE kg_rspw_types

