!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \brief calculation of the Kinetic + Exchange correlation
!> \author gloria,30.09.2002
! *****************************************************************************
MODULE kg_kxc
  USE cell_types,                      ONLY: cell_type
  USE cp_linked_list_xc_deriv,         ONLY: cp_sll_xc_deriv_next,&
                                             cp_sll_xc_deriv_type
  USE f77_blas
  USE input_section_types,             ONLY: section_vals_type
  USE kinds,                           ONLY: dp
  USE message_passing,                 ONLY: mp_sum
  USE pw_grid_types,                   ONLY: PW_MODE_DISTRIBUTED
  USE pw_methods,                      ONLY: pw_zero
  USE pw_pool_types,                   ONLY: pw_pool_type
  USE pw_types,                        ONLY: pw_p_type,&
                                             pw_type
  USE termination,                     ONLY: stop_program
  USE timings,                         ONLY: timeset,&
                                             timestop
  USE xc,                              ONLY: xc_rho_set_and_dset_create
  USE xc_derivative_desc,              ONLY: MAX_DERIVATIVE_DESC_LENGTH
  USE xc_derivative_set_types,         ONLY: xc_derivative_set_type,&
                                             xc_dset_get_derivative,&
                                             xc_dset_release
  USE xc_derivative_types,             ONLY: xc_derivative_get,&
                                             xc_derivative_type
  USE xc_rho_set_types,                ONLY: xc_rho_set_release,&
                                             xc_rho_set_type
#include "cp_common_uses.h"

  IMPLICIT NONE

  PRIVATE
  PUBLIC :: calculate_kxc_derivatives
  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'kg_kxc'

CONTAINS

! *****************************************************************************
!> \brief Calculates the K-XC energy  and potential, and potentials associated
!>      to  second and third derivatives of the KXC kernel
!>      V1_r = dvol* rhop_r * (partial^2  (E_kxc) / partial^2 rho0)
!>      V2_r = dvol**2 * rhop_r**2 * (partial ^3 (E_kxc) /partial^3 rho0 )
!>      Full gradient functional for Exc and Vxc, only lda functional
!>      for higher order derivatives.
!> \author gt
! *****************************************************************************
 SUBROUTINE calculate_kxc_derivatives ( pw_pool,rho_r, rho_g, rhop_r, v_r, v1_r, &
                           v2_r, e1_xc, e2_xc, xc_section,calculate_forces,error)

    TYPE(pw_pool_type), POINTER              :: pw_pool
    TYPE(pw_p_type), INTENT(in)              :: rho_r, rho_g, rhop_r, v_r
    TYPE(pw_p_type), INTENT(inout)           :: v1_r, v2_r
    REAL(KIND=dp), INTENT(out)               :: e1_xc, e2_xc
    TYPE(section_vals_type), POINTER         :: xc_section
    LOGICAL, INTENT(in)                      :: calculate_forces
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'calculate_kxc_derivatives', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, order
    REAL(KIND=dp), DIMENSION(:, :, :), &
      POINTER                                :: v1xc, v2xc

!--------------------------------------------------------

  CALL timeset(routineN,handle)

  v1xc => v1_r%pw%cr3d(:,:,:)
  v2xc => v2_r%pw%cr3d(:,:,:)

  IF(calculate_forces) THEN
    order=3
  ELSE
    order=2
  END IF

  CALL xc_calculate_derivatives (rho_r%pw,rhop_r%pw,v_r%pw,v1_r%pw, &
                             v2_r%pw, e1_xc, e2_xc, order, xc_section, pw_pool, error)

  IF ( rho_r%pw%pw_grid%para%mode == PW_MODE_DISTRIBUTED ) THEN
    CALL mp_sum ( e1_xc, rho_r%pw%pw_grid%para%group )
    CALL mp_sum ( e2_xc, rho_r%pw%pw_grid%para%group )
  END IF

  e2_xc =  0.5_dp*rho_r%pw%pw_grid%dvol*e2_xc

  v1xc (:,:,:) =  v1xc(:,:,:) *rho_r%pw%pw_grid%dvol
  IF (calculate_forces)THEN
    v2xc (:,:,:) =  0.5_dp * v2xc(:,:,:) *rho_r%pw%pw_grid%dvol
  END IF
  CALL timestop(handle)
 END SUBROUTINE calculate_kxc_derivatives

! *****************************************************************************
!> \brief Exchange and Correlation functional derivatices
! *****************************************************************************
 SUBROUTINE xc_calculate_derivatives ( rho_r,rhop_r,v,v1,v2,e1,e2,order,&
      xc_section, pw_pool, error)

    TYPE(pw_type), POINTER                   :: rho_r, rhop_r, v
    TYPE(pw_type), INTENT(INOUT)             :: v1, v2
    REAL(KIND=dp), INTENT(out)               :: e1, e2
    INTEGER, INTENT(IN)                      :: order
    TYPE(section_vals_type), POINTER         :: xc_section
    TYPE(pw_pool_type), POINTER              :: pw_pool
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'xc_calculate_derivatives', &
      routineP = moduleN//':'//routineN

    CHARACTER&
      (len=MAX_DERIVATIVE_DESC_LENGTH)       :: desc
    INTEGER                                  :: deriv_order, i, j, k, stat
    INTEGER, DIMENSION(2, 3)                 :: bo
    INTEGER, DIMENSION(3, 3)                 :: nd
    LOGICAL                                  :: failure
    REAL(KIND=dp), DIMENSION(:, :, :), &
      POINTER                                :: pot
    TYPE(cell_type), POINTER                 :: cell
    TYPE(cp_sll_xc_deriv_type), POINTER      :: pos
    TYPE(pw_p_type), DIMENSION(:), POINTER   :: my_rho_g, my_rho_r, my_tau
    TYPE(xc_derivative_set_type), POINTER    :: derivative_set
    TYPE(xc_derivative_type), POINTER        :: deriv
    TYPE(xc_rho_set_type), POINTER           :: rho_set

!------------------------------------------------------------------------------

   failure=.FALSE.

   NULLIFY(derivative_set,rho_set)
   nd = RESHAPE ((/1,0,0,0,1,0,0,0,1/),(/3,3/))
   e1 = 0.0_dp
   e2 = 0.0_dp

   NULLIFY(rho_set,derivative_set,my_rho_g,my_rho_r,my_tau)
   ALLOCATE(my_rho_r(1),stat=stat)
   CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
   NULLIFY (my_rho_r(1)%pw)
   my_rho_r(1)%pw => rho_r
   bo=rho_r%pw_grid%bounds_local
   CALL xc_rho_set_and_dset_create(rho_set=rho_set,&
        deriv_set=derivative_set,&
        deriv_order=order,&
        rho_r=my_rho_r,rho_g=my_rho_g,tau=my_tau,xc_section=xc_section,&
        cell=cell,pw_pool=pw_pool,&
        needs_basic_components=.FALSE.,error=error)

   DEALLOCATE(my_rho_r,stat=stat)
   CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
   ! check for unsupported derivatives
   pos => derivative_set%derivs
   DO WHILE (cp_sll_xc_deriv_next(pos,el_att=deriv,error=error))
      CALL xc_derivative_get(deriv,order=deriv_order,&
           desc=desc,error=error)
      IF (deriv_order==2.AND.desc/="(rho)(rho)".OR.&
           order==3.and.deriv_order==3.AND.desc/="(rho)(rho)(rho)") THEN
         CALL stop_program(routine=routineP,&
              error_message="unsupported xc derivative: "//&
              TRIM(desc))
      END IF
   END DO

   CALL xc_rho_set_release(rho_set,error=error)
   deriv => xc_dset_get_derivative(derivative_set,"(rho)(rho)",error=error)
   IF (ASSOCIATED(deriv)) THEN
      CALL xc_derivative_get(deriv,deriv_data=pot,error=error)
      DO k = bo(1,3), bo(2,3)
         DO j = bo(1,2), bo(2,2)
            DO i = bo(1,1), bo(2,1)
               v1%cr3d(i,j,k) = pot(i,j,k)*rhop_r%cr3d(i,j,k)
            END DO
         END DO
      END DO
   ELSE
      CALL pw_zero(v1,error=error)
   END IF
   IF(order==3)THEN
      deriv => xc_dset_get_derivative(derivative_set,"(rho)(rho)(rho)",error=error)
      IF (ASSOCIATED(deriv)) THEN
         CALL xc_derivative_get(deriv,deriv_data=pot,error=error)
         DO k = bo(1,3), bo(2,3)
            DO j = bo(1,2), bo(2,2)
               DO i = bo(1,1), bo(2,1)
                  v2%cr3d(i,j,k) = pot(i,j,k)*rhop_r%cr3d(i,j,k)
               END DO
            END DO
         END DO
      ELSE
         CALL pw_zero(v2,error=error)
      END IF
   END IF
   DO k = bo(1,3), bo(2,3)
      DO j = bo(1,2), bo(2,2)
         DO i = bo(1,1), bo(2,1)
            IF(order==3)THEN
               v2%cr3d(i,j,k) =v2%cr3d(i,j,k)*rhop_r%cr3d(i,j,k)
            END IF
            e1= e1+ v%cr3d(i,j,k)*rhop_r%cr3d(i,j,k)
            e2= e2+ v1%cr3d(i,j,k)*rhop_r%cr3d(i,j,k)
         END DO
      END DO
   END DO
   CALL xc_dset_release(derivative_set,error=error)

 END SUBROUTINE xc_calculate_derivatives

END MODULE kg_kxc

