use 5.008;

use strict;
use warnings;

use Cwd;
use File::Spec;
use ExtUtils::MakeMaker;

# minimum required version of dependancies we need to build
our %build_reqs = (
    'perl-ExtUtils-Depends'   => '0.2',
    'perl-ExtUtils-PkgConfig' => '1.03',
    'perl-Glib'               => '1.180', # for Glib::MakeHelper
    'perl-Gtk2'               => '1.140', # for Gtk2::Pango
    'perl-Cairo'              => '1.00',
    'Clutter'                 => '0.8.0',
);

# Writing a fake Makefile ensures that CPAN will pick up the correct
# dependencies and install them.
unless (eval "use ExtUtils::Depends '$build_reqs{'perl-ExtUtils-Depends'}';"
           . "use ExtUtils::PkgConfig '$build_reqs{'perl-ExtUtils-PkgConfig'}';"
           # just seeing if Glib is available isn't enough, make sure
           # it's recent enough, too 
           . "use Glib '$build_reqs{'perl-Glib'}';"
           . "use Cairo '$build_reqs{'perl-Cairo'}';"
           . "use Gtk2 '$build_reqs{'perl-Gtk2'}';"
           . "use Glib::MakeHelper;"
           # we need Gtk2::CodeGen because we use GtkObject in ClutterGtk
           . "use Gtk2::CodeGen;"
           . "1") {
   warn "$@\n";
   WriteMakefile(
         PREREQ_FATAL => 1,
         PREREQ_PM    => {
            'Glib'                => $build_reqs{'perl-Glib'},
            'Cairo'               => $build_reqs{'perl-Cairo'},
            'ExtUtils::Depends'   => $build_reqs{'perl-ExtUtils-Depends'},
            'ExtUtils::PkgConfig' => $build_reqs{'perl-ExtUtils-PkgConfig'},
         },
   );
   exit 1; # not reached
}

mkdir 'build', 0777;

our @xs_files = ();

# Base module
our %pkgcfg =
    ExtUtils::PkgConfig->find('clutter-0.8 >= ' . $build_reqs{Clutter});

my @names = Glib::MakeHelper->read_source_list_file('xs_files');
print "Loaded " . scalar(@names) . " xs files for Clutter\n";
push @xs_files, @names;

our $module_cflags = $pkgcfg{cflags};
our $module_libs   = $pkgcfg{libs};

# Integration libraries: Clutter::Gst
our $have_clutter_gst = 0;
eval {
    my %gstpkgcfg = ExtUtils::PkgConfig->find('clutter-gst-0.8');

    $module_cflags .= ' ' . $gstpkgcfg{cflags} . '-DCLUTTERPERL_GST';
    $module_libs   .= ' ' . $gstpkgcfg{libs};

    $have_clutter_gst = 1;

    my @names = Glib::MakeHelper->read_source_list_file ('xs_files_gst');
    print "Loaded " . scalar(@names) . " xs files for Clutter::Gst\n";
    push @xs_files, @names;
};
if ($@) {
    warn("*** No clutter-gst found, skipping Clutter::Gst\n");
}

# Integration libraries: Clutter::Texture::Cairo
our $have_clutter_cairo = 0;
eval {
    my %cairopkgcfg = ExtUtils::PkgConfig->find('clutter-cairo-0.8');

    $module_cflags .= ' ' . $cairopkgcfg{cflags} . '-DCLUTTERPERL_CAIRO';
    $module_libs   .= ' ' . $cairopkgcfg{libs};

    $have_clutter_cairo = 1;

    my @names = Glib::MakeHelper->read_source_list_file ('xs_files_cairo');
    print "Loaded " . scalar(@names) . " xs files for Clutter::Cairo\n";
    push @xs_files, @names;
};
if ($@) {
    warn("*** No clutter-cairo found, skipping Clutter::Cairo\n");
}

# Integration libraries: Gtk2::ClutterEmbed
our $have_clutter_gtk = 0;
eval {
    my %gtkpkgcfg = ExtUtils::PkgConfig->find('clutter-gtk-0.8');

    $module_cflags .= ' ' . $gtkpkgcfg{cflags} . '-DCLUTTERPERL_GTK';
    $module_libs   .= ' ' . $gtkpkgcfg{libs};

    $have_clutter_gtk = 1;

    my @names = Glib::MakeHelper->read_source_list_file ('xs_files_gtk');
    print "Loaded " . scalar(@names) . " xs files for Clutter::Gtk\n";
    push @xs_files, @names;
};
if ($@) {
    warn("*** No clutter-gtk found, skipping Clutter::Gtk\n");
}


our %pm_files = (
    'Clutter.pm' => '$(INST_LIBDIR)/Clutter.pm',

    # Keysyms.pm is generated by Makefile.PL
    'build/Keysyms.pm' => '$(INST_LIBDIR)/Clutter/Keysyms.pm',

    # TestHelper is just a wrapper around Test::More
    'tools/TestHelper.pm' => '$(INST_LIBDIR)/Clutter/TestHelper.pm',
);

our %pod_files = (
    'Clutter.pm'   => '$(INST_MAN3DIR)/Clutter.$(MAN3EXT)',

    # Keysyms.pm is generated by Makefile.PL
    'build/Keysyms.pm' => '$(INST_MAN3DIR)/Clutter::Keysyms.$(MAN3EXT)',
    
    Glib::MakeHelper->do_pod_files(@xs_files),
);

# typemaps generation files
my @input_maps = qw/ maps /;
push @input_maps, 'maps-gst'   if $have_clutter_gst   == 1;
push @input_maps, 'maps-gtk'   if $have_clutter_gtk   == 1;
push @input_maps, 'maps-cairo' if $have_clutter_cairo == 1;

# autogeneration
Gtk2::CodeGen->parse_maps('clutterperl', input => \@input_maps);
Gtk2::CodeGen->write_boot(xs_files => \@xs_files);

# this one turns the long list of #defines in clutter-keysyms.h into a
# large hash of key names and values, %Clutter::Gdk::Keysyms.
system "$^X tools/genkeysyms.pl >\"build/Keysyms.pm\"";

my @deps = qw/ Glib /;
push @deps, 'GStreamer' if $have_clutter_gst   == 1;
push @deps, 'Cairo';    # core depends on Cairo
push @deps, 'Gtk2';     # core depends on Gtk2 for Pango

my $cwd = cwd();

our $clutter = ExtUtils::Depends->new('Clutter', @deps);
$clutter->set_inc($module_cflags . ' -I./build ');
$clutter->set_libs($module_libs);
$clutter->add_xs(@xs_files);
$clutter->add_pm(%pm_files);
$clutter->add_typemaps(
    map {
        File::Spec->catfile($cwd, $_)
    }
    'build/clutterperl.typemap',
    'clutter.typemap'
);
$clutter->install(qw(clutterperl.h build/clutterperl-autogen.h));
$clutter->save_config('build/IFiles.pm');

WriteMakefile(
    NAME          => 'Clutter',
    VERSION_FROM  => 'Clutter.pm', # finds $VERSION
    PREREQ_PM     => {}, # e.g., Module::Name => 1.1
    ABSTRACT_FROM => 'Clutter.pm', # retrieve abstract from module
    XSPROTOARG    => '-noprototypes',
    MAN3PODS	  => \%pod_files,
    $clutter->get_makefile_vars(),
);


=pod
#=unstable

print "
WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING

This is an unstable development release of Clutter.  The API is
not frozen and things are subject to change at any time.  Report any
bugs to clutter AT openedhand DOT com as soon as possible.

WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING
";

=cut

sub MY::postamble
{
    return Glib::MakeHelper->postamble_clean()
         . Glib::MakeHelper->postamble_docs_full(
            DEPENDS => $clutter,
            DOCTYPES => 'doctypes',
            COPYRIGHT_FROM => 'copyright.pod',
           )
         . Glib::MakeHelper->postamble_rpms(
            'CLUTTER' => $build_reqs{'Clutter'},
            'PERL_EXTUTILS_DEPENDS' => $build_reqs{'perl-ExtUtils-Depends'},
            'PERL_EXTUTILS_PKGCONFIG' => $build_reqs{'perl-ExtUtils-PkgConfig'},
            'PERL_GLIB' => $build_reqs{'perl-Glib'},
            'PERL_CAIRO' => $build_reqs{'perl-Cairo'},
            'PERL_GTK2' => $build_reqs{'perl-Gtk2'},
           );
}
