/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */

using Gtk;

namespace Clinica {

    public class DoctorListPage : Alignment, Page {
        
        private Alignment sidebar;
        private DoctorListView view;
        private VBox left_vbox;
        private ScrolledWindow scrolled_window;
        
        public ResourceManager resource_manager { get; set; }
        
        /**
         * @brief Page with the list of doctors. 
         */
        public class DoctorListPage (ResourceManager resources) {
            var hbox = new HBox (false, 6);
            resource_manager = resources;
            
            connect_signal ();
            
            var builder = new Builder ();
            
            try {
                builder.add_from_file (resources.get_ui_file ("doctor_list_toolbar.glade"));
                builder.add_from_file (resources.get_ui_file ("doctor_list_sidebar.glade"));
            } catch (Error e) {
                error ("Error loading doctor_list_toolbar.glade. Check your installation.");
            }
            
            left_vbox = builder.get_object ("left_vbox") as VBox;
            sidebar = builder.get_object ("sidebar") as Alignment;
            scrolled_window = builder.get_object ("treeview_scrolled_window") as ScrolledWindow;
            
            /* Get find_entry to filter the doctors */
            var find_entry = builder.get_object ("find_entry") as Entry;
            view = new DoctorListView (resource_manager, find_entry);
            scrolled_window.add (view);
            
            hbox.pack_start (left_vbox, true, true, 0);
            hbox.pack_start (sidebar, false, true, 0);
            
            add (hbox);
            show_all ();
            builder.connect_signals (this);
        }
        
        public void setup () {
            new_title (_("Doctors"));
        }
        
        [CCode (instance_pos = -1)]
        public void on_add_button_clicked (Button button) {
        	var new_doc_dialog = new DoctorEditor (resource_manager);
        	new_doc_dialog.dialog.set_transient_for (resource_manager.user_interface.window);
        	new_doc_dialog.run ();
			new_doc_dialog.destroy ();
        }
        
        [CCode (instance_pos = -1)]
        public void on_back_button_clicked (Button button) {
            start_page_requested ();
        }
        
        [CCode (instance_pos = -1)]
        public void on_modify_button_clicked (Button button) {
        	Doctor? doc = view.get_selected_doctor ();
        	if (doc == null)
        		return;
        	var edit_doc_dialog = new DoctorEditor.with_doctor (resource_manager, doc);
        	edit_doc_dialog.dialog.set_transient_for (resource_manager.user_interface.window);
        	edit_doc_dialog.run ();
        	edit_doc_dialog.destroy ();
        }
        
        /**
         * @brief Callback called when a user try to delete a doctor
         * clicking on the remove button. It handles removal asking
         * user if associated patients should be deleted. 
         */
        [CCode (instance_pos = -1)]
        public void on_remove_button_clicked (Button button) {
 			view.remove_selected_doctor ();
        }
    }
    
}
