(in-package :contextl)

(defun make-enclosing-package (name)
  (make-package name :use '()))

(defgeneric enclose-symbol (symbol package)
  (:method ((symbol symbol)
            (package package))
   (if (symbol-package symbol)
     (intern (format nil "~A::~A"
                     (package-name (symbol-package symbol))
                     (symbol-name symbol))
             package)
     (or (get symbol package)
         (setf (get symbol package) (gensym))))))

#|
Layers are represented as CLOS classes. To avoid nameclashes with plain
CLOS classes, the name of a layer is actually mappend to an internal
unambiguous name which is used instead of the regular name.
|#

(defvar *layer-class-definers*
  (make-enclosing-package "LAYER-CLASS-DEFINERS"))

(defun defining-layer (name)
  "Takes the name of a layer and returns its internal name."
  (if (eq name 't) 't
    (enclose-symbol name *layer-class-definers*)))

#|
Layered functions have two names: The name of the caller and the name of
the definer. The caller is just a function that adds a representation of
the active layers to the list of arguments and calls the definer. The
definer is a generic function that contains all the layered methods.

The caller has the name under which a user knows about a layered function.
The definer has an automatically generated name that can be unambiguously
determined from the caller's name. So for example, consider the following
layered function definition:

(define-layered-function foo (...))

The caller is named 'foo whereas the definer is named something like
'layered-function-definers::some-package\:\:foo. [The details of the
mapping should be considered an implementation detail, though, and not
part of the "official" API of ContextL.]
|#

(defvar *layered-function-definers*
  (make-enclosing-package "LAYERED-FUNCTION-DEFINERS"))

(defun get-layered-function-definer-name (name)
  "Takes the name of a layered function caller
   and returns the name of the corresponding definer."
  (cond ((plain-function-name-p name)
         (enclose-symbol name *layered-function-definers*))
        ((setf-function-name-p name)
         `(setf ,(enclose-symbol (cadr name) *layered-function-definers*)))
        (t (error "Illegal function name: ~S." name))))

(defun bind-layered-function-names (name)
  "Takes the name of a layered function caller
   and ensures that it can be retrieved again
   from the name of a corresponding definer."
  (let ((plain-function-name (plain-function-name name)))
    (setf (get (enclose-symbol plain-function-name *layered-function-definers*)
               'layered-function-caller)
          plain-function-name)))

(defun get-layered-function-caller-name (name)
  "Takes the name of a layered function definer
   and returns the name of the corresponding caller."
  (cond ((plain-function-name-p name)
         (get name 'layered-function-caller))
        ((setf-function-name-p name)
         `(setf ,(get (cadr name) 'layered-function-caller)))
        (t (error "Illegal function name: ~S." name))))

#|
The following are utility functions to distingush between
the two kinds of function names available in Common Lisp.
|#

(declaim (inline plain-function-name-p))

(defun plain-function-name-p (name)
  (symbolp name))

(defun setf-function-name-p (name)
  (and (consp name)
       (eq (car name) 'setf)
       (symbolp (cadr name))
       (null (cddr name))))

(defun plain-function-name (name)
  (cond ((plain-function-name-p name) name)
        ((setf-function-name-p name) (cadr name))
        (t (error "Illegal function name ~S." name))))
