#!/usr/bin/python
# Copyright (c) 2011 The Chromium OS Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""This program generates a C++ header file containing the input method
whitelist information from chromeos-assets/input_methods/whitelist.txt

It will produce output that looks like:

// This file is automatically generated by gen_input_method_whitelist.py
#ifndef CHROMEOS_INPUT_METHOD_WHITELIST_H_
#define CHROMEOS_INPUT_METHOD_WHITELIST_H_

namespace chromeos {
const char* kInputMethodIdsWhitelist[] = {
  "chewing",
  "hangul",
  "mozc",
  ...
};
const char* kXkbLayoutsWhitelist[] = {
  "us",
  "us(dvorak)",
  "fr",
  ...
};
}  // namespace chromeos

#endif  // CHROMEOS_INPUT_METHOD_WHITELIST_H_
"""

import fileinput
import re
import sys


OUTPUT_HEADER = """
// This file is automatically generated by gen_input_method_whitelist.py
#ifndef CHROMEOS_INPUT_METHOD_WHITELIST_H_
#define CHROMEOS_INPUT_METHOD_WHITELIST_H_

namespace chromeos {"""

OUTPUT_INPUT_METHOD_IDS = """
const char* kInputMethodIdsWhitelist[] = {
%s
};"""

OUTPUT_XKB_LAYOUTS = """
const char* kXkbLayoutsWhitelist[] = {
%s
};"""

OUTPUT_FOOTER = """
}  // namespace chromeos

#endif  // CHROMEOS_INPUT_METHOD_WHITELIST_H_"""


def main():
  if len(sys.argv) != 2:
    print >> sys.stderr, 'Usage: gen_input_method_whitelist.py [file]'
    sys.exit(1)
  print OUTPUT_HEADER

  # Write the kInputMethodIdsWhitelist array.
  ids = []
  for line in fileinput.input(sys.argv[1]):
    line = re.sub(r'#.*', '', line)  # Remove comments.
    line = line.split()
    if len(line) == 0:
      # The line only contains white spaces.
      continue
    # line[0] is a IBus input method name (e.g. "mozc", "xkb:us::eng"),
    # and line[1] is an ID for a keyboard overlay help (e.g. "en_US", "ja").
    ids.append('  "%s",' % line[0])
  print OUTPUT_INPUT_METHOD_IDS % '\n'.join(ids)

  # Write the kXkbLayoutsWhitelist array.
  layouts = []
  for line in fileinput.input(sys.argv[1]):
    line = re.sub(r'#.*', '', line)  # Remove comments.
    line = line.split()
    if len(line) == 0 or (not line[0].startswith('xkb:')):
      continue

    # 'xkb:us::eng' would be splitted into ['xkb', 'us', '', 'eng'].
    # 'xkb:us:dvorak:eng' woulb be ['xkb', 'us', 'dvorak', 'eng'].
    line = line[0].split(':')
    assert(len(line) == 4);

    if len(line[2]) == 0:
      layouts.append('  "%s",' % line[1])
    else:
      layouts.append('  "%s(%s)",' % (line[1], line[2]))
  print OUTPUT_XKB_LAYOUTS % '\n'.join(layouts)

  print OUTPUT_FOOTER


if __name__ == '__main__':
  main()
