// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_HISTORY_H_
#define CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_HISTORY_H_
#pragma once

// This class gets redirect chain for urls from the history service.

#include <string>
#include <vector>

#include "base/hash_tables.h"
#include "base/memory/linked_ptr.h"
#include "base/memory/ref_counted.h"
#include "chrome/browser/history/history.h"
#include "content/browser/browser_thread.h"
#include "content/common/notification_observer.h"
#include "content/common/notification_registrar.h"
#include "net/base/completion_callback.h"

class TabContents;

namespace safe_browsing {
typedef std::vector<GURL> RedirectChain;
}

class MalwareDetailsRedirectsCollector
    : public base::RefCountedThreadSafe<MalwareDetailsRedirectsCollector,
                                        BrowserThread::DeleteOnUIThread>,
      public NotificationObserver {

 public:
  explicit MalwareDetailsRedirectsCollector(Profile* profile);
  virtual ~MalwareDetailsRedirectsCollector();

  // Collects urls' redirects chain information from the history service.
  // We get access to history service via tab_contents in UI thread.
  // Notice the callback will be posted to the IO thread.
  void StartHistoryCollection(const std::vector<GURL>& urls,
                              Task* callback);

  // Returns whether or not StartCacheCollection has been called.
  bool HasStarted() const;

  // Returns the redirect urls we get from history service
  const std::vector<safe_browsing::RedirectChain>& GetCollectedUrls() const;

 private:
  friend struct BrowserThread::DeleteOnThread<BrowserThread::UI>;
  friend class DeleteTask<MalwareDetailsRedirectsCollector>;

  Profile* profile_;
  CancelableRequestConsumer request_consumer_;

  // Method we call when we are done. The caller must be alive for the
  // whole time, we are modifying its state (see above).
  Task* callback_;

  // Sets to true once StartHistoryCollection is called
  bool has_started_;

  // The urls we need to get redirects for
  std::vector<GURL> urls_;
  // The iterator goes over urls_
  std::vector<GURL>::iterator urls_it_;
  // The collected directs from history service
  std::vector<safe_browsing::RedirectChain> redirects_urls_;

  NotificationRegistrar registrar_;

  void StartGetRedirects(const std::vector<GURL>& urls);
  void GetRedirects(const GURL& url);
  void OnGotQueryRedirectsTo(HistoryService::Handle handle,
                             GURL url,
                             bool success,
                             history::RedirectList* redirect_list);

  // Posts the callback method back to IO thread when redirects collecting
  // is all done.
  void AllDone();

  virtual void Observe(NotificationType type,
                       const NotificationSource& source,
                       const NotificationDetails& details);

  DISALLOW_COPY_AND_ASSIGN(MalwareDetailsRedirectsCollector);
};

#endif  // CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_HISTORY_H_
